package hitters.multi;

import java.util.Arrays;
import java.util.HashSet;

public class SysParameter extends Parameter {

	// Die verwendeten Dimensionen sind frei waehlbar,
	// die passende interne Darstellung (String/int) und
	// die Anzahl der Bits je Hierarchieebene werden
	// automatisch festgelegt.
	DimType[] usedDims;
	int[] cap = null; // Begrenzung Hierarchietiefe
		
	// Calls, die zum Aufloesen der Filedeskriptoren
	// unbedingt geloggt werden muessen.
	HashSet<String> minCalls;
	
	// Die tats�chlich verwendeten Calls, die uebrigen 
	// werden ignoriert, sofern sie nicht in minCalls liegen.
	HashSet<String> usedCalls; 	
	
	// Verteilung der Bits auf die Hierarchieebenen
	final byte[] callPlength = {0,4,12,18,24,28,32};
	final byte[] seqPlength = {0,8,16,24,32};
	final byte[] retPlength = {0,2,32};

	int pathDim = -1;
	int callDim = -1;
	int seqDim = -1;
	int retDim = -1;
	
	// Gewichtung der Elemente nach Haeufigkeit der Calls
	// exp ist aus [0,1], das Gewicht gew wird berechnet
	// als: gew = ( 1 / Haeufigkeit ) ^ exp.
	// Fuer exp = 0 werden also alle Calls mit eins gewichtet,
	// fuer exp = 1 werden alle mit eins/Haeufgkeit gewichtet.
	// Z. Zt. nicht verwendet.
 	double exp = 0.0;	
	
	private Taxonomy taxonomy;
	private Weights myWeights;
		
	
	public SysParameter( DimType[] dims ){
		
		// usedDims kopieren, dabei interne Darstellung als String 
		// an die unteren Positionen
		usedDims = new DimType[dims.length];
		int counter = 0;
		for( DimType d : dims ) 
			if( d == DimType.PATH ) usedDims[counter++] = DimType.PATH;
		for( DimType d : dims ) 
			if( d != DimType.PATH ) usedDims[counter++] = d;
				
		dimS = 0;
		dimI = 0;
		
		for( int i = 0; i < usedDims.length; i++ ){
			if( usedDims[i] == DimType.PATH ) dimS = 1;
			else if( usedDims[i] != DimType.NONE ) dimI++;
		}

		plength = new byte[ dimI ][];
		cap = new int[ usedDims.length ];
		
		for( int i = 0; i < usedDims.length; i++ ){
			
			switch( usedDims[i] ){
			
			case PATH: 				
				pathDim = i;
				cap[i] = stringCap;
				break;
				
			case CALL: 
				plength[ i - dimS ] = callPlength;
				callDim = i;
				cap[i] = callPlength.length -1;
				break;
			
			case SEQUENCE: 
				plength[ i - dimS ] = seqPlength;
				seqDim = i;
				cap[i] = seqPlength.length -1;
				break;
				
			case RETURN: 
				plength[ i - dimS ] = retPlength;
				retDim = i;
				cap[i] = retPlength.length -1;
				break;
			}			
		}
				
		mask = calcMasks();
		taxonomy = loadTaxonomy();
		myWeights = loadWeights(); 	
		minCalls = getMinCalls();
		usedCalls = loadAllCalls();
		usedCalls.addAll( minCalls );
	}

	
	public SysParameter( DimType[] dims, int[] caps ){
		
		this( dims );				

		if( dims.length != caps.length ) 
			throw new RuntimeException( "Dimensionen von caps " +
					"und dims bitte passend waehlen." );
		
		// caps kopieren, dabei anpassen an umsortierte Dimensionen, 
		// siehe Standardkonstruktor.
		//cap = new int[ caps.length ];
		for( int i = 0; i < usedDims.length; i++ )
			for( int j = 0; j < dims.length; j++ )
				if( usedDims[i] == dims[j] ){
					if( this.cap[i] >= caps[j] ){
						this.cap[i] = caps[j];
					}
					else
						throw new RuntimeException( "Caps zu gross" +
								" fuer interne Darstellung." );
				}
	}

	
	public DimType dimContents( int i ){ return usedDims[i]; }
	
	public int pathDim(){ return pathDim; }
	public int callDim(){ return callDim; }
	public int seqDim(){ return seqDim; }
	public int retDim(){ return retDim; }
	
	public boolean capDepth(){ return cap != null; }
	public int[] getCap(){ return cap; }
	
	public void setTaxonomy(Taxonomy taxonomy) {
		this.taxonomy = taxonomy;
	}
	
	public Taxonomy getTaxonomy() {
		return taxonomy;
	}

	public Weights getWeights() {
		return myWeights;
	}
	
	public boolean useWeights() {
		return exp > 1E-12;
	}
	
	public int getSeqLength(){ return seqPlength.length - 1; }
		
//	public void setSeqLength( int seqLength ) {
//		this.seqLength = seqLength;
//	}
	
	public HashSet<String> usedCalls(){ return usedCalls; } 
	
	
	//If the hierarchical variable "PATH" is used, we have to
	//translate file descriptors to filenames. In order to do so,
	//we have to log all calls that manipulate file descriptors in some way.
	//If the hierarchical variable "PATH" is used, logging these calls 
	//can't be switched off, they will be logged automatically.	
	public void setUsedCalls( HashSet<String> calls ){ 
	
		this.usedCalls = calls; 
		if( pathDim() > -1 ) {
		
			boolean redundant = false;
			
			for( String c : minCalls ){
				if( calls.contains(c) ) redundant = true; 
			}
			if(redundant) 
				System.out.println("Adding calls already in minCalls." +
						" Check if this is what you want.");
		
			this.usedCalls.addAll( minCalls );
		}
	}

	
	public int gethi( int dim ){
		return cap[dim];
	}


	public String toString(){
		
		String s = super.toString();
		s += cr + "# Dimensionen = " + Arrays.toString(usedDims);
		if( cap != null ) s += cr + "# " + Arrays.toString(cap);
		s += cr + "# L = " + getL(); 
		s += cr + "# Exp = " + exp + cr + "#" + cr;
		
		return s;		
	}

	
	/**
	 * Gives a String representation of the most important parameters,
	 * omits the details. To be used in RapidMiner.
	 * 
	 * @return a String representation of the most important parameters
	 */
	public String toNiceString(){
		
		String s = super.toNiceString();
		s += cr + "Hierarchical variables used: " + Arrays.toString(usedDims) + cr;
		s += "Depth of hierarchy for each variable: " + Arrays.toString(cap);
		return s;		
	}

	public String toFullString(){
		
		String s = toString();	
		s += cr + "# Verwendete Calls (" + usedCalls.size() + "): ";
		for( String call : usedCalls ){
			s += cr + "# " + call;
		}		
		return s;		
	}
	
	
	
	private int[][] calcMasks(){
	
		int[][] mask = new int[dimI][]; 
		
		for( int d = 0; d < dimI; d++ ){			
			mask[d] = new int[ plength[d].length ];		
			for( int i = 1; i < mask[d].length; i++ ){ 
				mask[d][i] = 0xFFFFFFFF << ( 32 - plength[d][i] );
				/* log( "Maske in dim " + d + " , i= "1 + i + " : "  );
				System.out.println( " 24 " + ((mask[d][i] & 0xff000000) >>> 24) );
				System.out.println( " 16 " + ((mask[d][i] & 0x00ff0000) >>> 16) );
				System.out.println( "  8 " + ((mask[d][i] & 0x0000ff00) >>>  8) );
				System.out.println( "  0 " + ((mask[d][i] & 0x000000ff)       ) ); */
			}	
		}		
		return mask;
	}
	
	
	private Taxonomy loadTaxonomy(){

		Taxonomy taxonomy = null;
		try{
			taxonomy = new Taxonomy("Taxonomy3");
		}catch(Exception e) {}	
		return taxonomy;	
	}

	
	private Weights loadWeights(){
		
		Weights myWeights = null;

		try{
			myWeights = new Weights("Freq.txt", exp);
		}catch(Exception e) {}		
		return myWeights;
	}
	
	
	private HashSet<String> loadAllCalls(){
		
		if( taxonomy == null ) taxonomy = loadTaxonomy();

		HashSet<String> allCalls = 
			new HashSet<String>( taxonomy.getAllCalls() );	

		return allCalls;	
	}
	
	
	public static HashSet<String> getMinCalls(){
		
		HashSet<String> minCalls = new HashSet<String>();	

		minCalls.add( "open" );
		minCalls.add( "close" );
		minCalls.add( "dup" );
		minCalls.add( "dup2" );
		minCalls.add( "dup3" );
		minCalls.add( "clone" );
		minCalls.add( "fork" );
		minCalls.add( "vfork" );
		minCalls.add( "execve" );
		minCalls.add( "fcntl" );
		minCalls.add( "fcntl64" );
		minCalls.add( "unshare" );
		
		// Nicht unbedingt erforderlich, eher um DateiUnbekannt
		// zu vermeiden: Diese Calls erzeugen Fildeskriptoren.
		minCalls.add( "socket" );
		minCalls.add( "pipe" );
		minCalls.add( "pipe2" );
		minCalls.add( "inotify_init" );
		minCalls.add( "inotify_init1" );
		minCalls.add( "accept" );
		minCalls.add( "accept4" );
		
		return minCalls;	
	}


	public String dimString() {		
		return "# Dimensionen = " + Arrays.toString(usedDims) + cr;
	}


	
}

