/**
 * @author Peter Fricke 
 */
package hitters.multi;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

/**
 * @author Peter Fricke
 *
 *	Speichert Informationen ueber einen einzelnen Prozess:
 *	 
 *		- Filedeskriptoren der geoeffneten Dateien.
 *		- Flags der Filedeskriptoren, die festlegen, ob
 *			die Datei waehrend eines execve geschlossen wird
 *		- Sequenzinformationen: Die letzten ausgefuehrten Calls.
 *		- Wenn ein Call blockt, steht er zweimal halb im Log, verarbeitet
 *			werden kann er erst, wenn er komplett bekannt ist, darum wird die
 *			unfertige Haelfte hier zwischengespeichert.
 *
 *
 */
public class ProcTable {

	SysParameter myParams;
	
	HashMap<Integer, String> fdTable = new HashMap<Integer, String>();
	HashMap<Integer, Boolean> fdFlag = new HashMap<Integer, Boolean>();
	String unfinished = "";	
	int[] roundBuffer;
	int pos = 0;
	int bLength = 0;
	

	public ProcTable( SysParameter myParams ){
		this.myParams = myParams;
		bLength = myParams.getSeqLength() + 1;
		roundBuffer = new int[ bLength ];
		for( int i = 0; i < bLength; i++ ){
			roundBuffer[i] = -1;	
		}
		fdTable.put( 0, "/stdin" );
		fdTable.put( 1, "/stdout" );
		fdTable.put( 2, "/stderr" );
		fdFlag.put( 0, false );
		fdFlag.put( 1, false );
		fdFlag.put( 2, false );
	}
	
	
	public String getFile( String fd ){
		String file = null;
		int fdInt = Integer.parseInt( fd );
		file = fdTable.get( fdInt );		
		if( file == null ) file = "\"/DateiUnbekannt\"";		
		return file;
	}

	
	public void putFile( int fd, String file, boolean cloExec ) {			
		if( file == null ) file = "\"/DateiUnbekannt\"";
		fdTable.put( fd, file );		
		fdFlag.put( fd, cloExec);
		return;
	}


	public String removeFile( String fd ) {			
			
		String file = null;
		int fdInt = Integer.parseInt( fd );
		file = fdTable.get( fdInt );
		fdTable.remove( fdInt );
		fdFlag.remove( fdInt );
		if( file == null ) file = "\"/DateiUnbekannt\"";		
		
		return file;
	}
	
	
	public void setCloseOnExec( String fd, boolean cloExec ) {			
		
		int fdInt = Integer.parseInt( fd );
		fdFlag.put(fdInt, cloExec);	
	}
	
	
	// Wenn CloExec gesetzt, wird die Datei waehrend eines execve
	// geschlossen.
	public void doCloseOnExec(){
		
		//Benutze Iterator wg. ConcurrentModification
		Set<Integer> keys = fdTable.keySet(); 
		Iterator<Integer> it = keys.iterator();
		int fd;
		
		while( it.hasNext() ){
			fd = it.next();
			if( fdFlag.get( fd ) ){
				it.remove();
				fdFlag.remove( fd );
			}
		}	
	}


	// Wenn ein Call blockt, steht er zweimal halb im Log, verarbeitet
	// werden kann er erst, wenn er komplett bekannt ist, darum wird die	
	// unfertige Haelfte hier zwischengespeichert.
	// Speichere die erste Haelfte mit "<unfinished ...>"
	// Baue beides zusammen, sobald zweite Haelfte 
	// mit "<...resumed> bekannt ist.
	public boolean isUnfinished(){ return unfinished.length() > 0; }
	
	
	public String mergeParts( String resumed ){ 

		String result, delim;

		//Sowas: 11017 <... stat64 resumed>
		String[] comps = 
			resumed.split( "<\\.\\.\\.\\s[a-z_][a-z_0-9]+\\sresumed>" );

		if( comps[1].trim().startsWith( ")" ) ) delim = "";
		else delim = " ";
		
		result = unfinished + delim + comps[1].trim();
		unfinished = "";
		//System.out.println( "result = " + result );
		return result; 
	}

	
	public void setUnfishedPart( String part ){ 

		String result;
		
		//Sowas: 11017 open("/etc/mtab", O_RDONLY <unfinished ...>
		String[] comps = 
			part.split( "<unfinished\\s\\.\\.\\.>" );
		result = comps[0];
		//System.out.println( "unf = " + result );
		unfinished = result.trim(); 
	}

	
	public void putCall( int call ){ 
		roundBuffer[ pos++ % bLength ] = call; 
	}
	
	
	// Juengste Vergangenheit zuerst
	public int[] getCalls(){
		
		int[] sequence = new int[bLength];
		int j = 0;
		for( int i = pos + bLength -1; i >= pos; i-- ){
			sequence[j++] = roundBuffer[ i % bLength ];
		}
		return sequence;
	}
	
	
	public String getSequence(){
		
		String part, seq = "";
		int[] intSequence = getCalls();
		if( intSequence.length == 0 ) seq = "*";
		
		for( int i : intSequence ){
			part = myParams.getTaxonomy().flat2call( i );
			if( part != null ) seq = seq + "/" + part;
		}
		
		return seq;
	}
	
	
	// Tiefe Kopie dieser ProcTable, damit die Calls fork und clone
	// dem neuen Prozess eine Kopie der Filedeskriptoren des alten 
	// Prozesses uebergeben koennen. Die unfinished / resume Werte werden
	// nicht kopiert, ebensowenig die Sequenzen.
	public ProcTable clone(){
		
		ProcTable clone = new ProcTable( myParams );
		clone.fdFlag = new HashMap<Integer, Boolean>( fdFlag );
		clone.fdTable = new HashMap<Integer, String>( fdTable );
		
		return clone;		
	}
	
	// Flache Kopie dieser ProcTable, damit der Systemcall clone
	// dem neuen Prozess und dem alten Prozess das gemeinsame 
	// Benutzen derselben Filedeskriptoren ermoeglichen kann. 
	// Die unfinished / resume Werte werden
	// nicht kopiert, ebensowenig die Sequenzen.
	public ProcTable shareFd(){
		
		ProcTable clone = new ProcTable( myParams );
		clone.fdFlag = fdFlag;
		clone.fdTable = fdTable;
		
		return clone;		
	}
}
