package hitters.multi;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;


/**
 * @author Peter Fricke
 * 
 */
public class Evaluation {

	public static final int OGM_GANESAN = 0;
	public static final int OGM_CORMODE = 1;
	public static final int BGM = 2;
	public static final int BGM2 = 3;
	
	/**
	 * Berechnet den Jaccard Koeffizienten: Kardinalitaet des Schnitts
	 * der Mengen durch Kardinalitaet der Vereinigung.
	 * 
	 * @param approx Menge der approximativen HHH
	 * @param exact Menge der exakten HHH
	 * @return Jaccard Koeffizient: Schnitt durch Vereinigung. 
	 */
	public static double jaccard( Set<Element> approx, Set<Element> exact ){

		// Seiteneffekte von retainAll unerwuenscht
		int cut = 0;
		for( Element elem : approx )
			if( exact.contains( elem ) ) cut++;

		int join = approx.size() + exact.size() - cut;

		return cut / (double) join;		
	}


	/**
	 * Berechnet den Dice Koeffizienten: Doppelte Kardinalitaet des Schnitts
	 * der Mengen durch Summe der Kardinalitaeten. 
	 * 
	 * @param approx Menge der approximativen HHH
	 * @param exact Menge der exakten HHH
	 * @return Dice Koeffizient: 2 * Schnitt durch Summe
	 */
	public static double dice( Set<Element> approx, Set<Element> exact ){

		int cut = 0;
		for( Element elem : approx )
			if( exact.contains( elem ) ) cut++;

		int sum = approx.size() + exact.size();

		return 2 * cut / (double) sum;		
	}



	public static double precision( Set<Element> approx, Set<Element> exact ){

		int count = 0;
		for( Element elem : approx )
			if( exact.contains( elem ) ) count++;				
		if( approx.size() == 0 ) return 0;
		else return count / (double) approx.size();		
	}



	public static double recall( Set<Element> approx, Set<Element> exact ){

		int count = 0;
		for( Element elem : exact )
			if( approx.contains( elem ) ) count++;

		return count / (double) exact.size();		
	}

	
	// Nicht benutzen, so ist das zu optimistisch.
	// Wenn eine Menge Obermenge der anderen ist, und die
	// nicht in beiden Mengen enthaltenen Elemente oberhalb liegen, sind
	// deren Suprema sie selbst und die Aehnlichkeit ist unberechtigt eins.
	// Siehe z.B. Testcase Eva1.
	public static double ogmGanesan( Set<Element> approx, Set<Element> exact ){

		double sim = 0.0;
		double res;

		for( Element elem : exact )
			sim += elemSimGanesan( elem, approx );

		for( Element elem : approx )
			sim += elemSimGanesan( elem, exact );

		res = sim / ( exact.size() + approx.size() );

		return res;		
	}

	
	// OGM, woertlich aus Cormode08. Mit Vorzeichenfehler,
	// Wertebereich der �hnlichkeit bemekenswert.
	// Beruecksichtigt das Level nicht, auf dem der Fehler passiert.
	// Es ist nicht ganz klar, ob das im Fall der HHH ein Nachteil sein muss,
	// wohl eher schon.
	public static double ogmCormode( Set<Element> approx, Set<Element> exact ){
		
		double term1 = 0;
		double term2 = 0;
		double res;
		
		for( Element elem : approx )
			term1 += Math.abs( depthLca( elem, exact ) - elem.getL() ); 

		for( Element elem : exact )
			term2 += Math.abs( depthLca( elem, approx ) - elem.getL() );			
		
		// Ich lasse den offensichtlichen Vorzeichenfehler
		// aus Cormode08, S. 39 mal nicht drin.
		res = 1.0 - term1 - term2; 

		return res;		
	}


	// Wenn eine Menge Obermenge der anderen ist, und die
	// nicht in beiden Mengen enthaltenen Elemente oberhalb 
	// liegen, sind deren Suprema sie selbst. Die �hnlichkeit der
	// Mengen w�re nach ogmGanesan eins.
	// Da die Mengen nicht gleich sind, sollte die �hnlichkeit
	// aber nicht eins sein. Das "balanced genealogy measure" 
	// versucht dies zu beruecksichtigen, indem  die Entfernungen 
	// BEIDER Partner zum Supremum beruecksichtigt werden.
	//
	// Berechnet Laenge des kuerzesten Weges von einem Knoten
	// zu seinem Match in der anderen Menge, wobei der Weg ueber
	// den lca gehen muss.
	//
	// Bei der Wahl des Matches reicht ein guter lca nicht aus:
	// Es muss unter allen Knoten mit der besten lca-Tiefe derjenige 
	// gew�hlt werden, der die geringste Tiefe hat. Blaetter wuerden
	// unnoetig lange Wege verursachen.
	// Bsp.: Fuer Knoten 6 aus Testfall Biglist ist der beste lca-Wert 1
	// der Knoten 17 mit Tiefe 1 ist besseres Match als der Knoten 8 
	// mit Tiefe 2, obwohl in beiden Faellen der lca auf Level 1 wohnt.
	// 
	public static double bgm( Set<Element> approx, Set<Element> exact ){
		
		double term1 = 0;
		double term2 = 0;
		double erg;
		double res;
		Element match, lca;
		
		for( Element elem : approx ){
			match = bestMatch( elem, exact );
			lca = elem.getLca( match );
			erg = elem.getL() + match.getL()  - 2 * lca.getL();
			//System.out.println( elem.toShortString() + "   " + match.toShortString() + "   " + lca.toShortString() + "   " + erg );
			term1 += erg;
		}
		
		for( Element elem : exact ){
			match = bestMatch( elem, approx );
			lca = elem.getLca( match );
			erg = elem.getL() + match.getL()  - 2 * lca.getL();
			//System.out.println( elem.toShortString() + "   " + match.toShortString() + "   " + lca.toShortString() + "   " + erg );
			term2 += erg; 			
		}

		res = 1.0 - term1 - term2; 
	
		return res;
	}

	
	// Variante des "balanced genealogy measure", das
	// die Entfernungen BEIDER Partner zum Supremum beruecksichtigt und
	// dabei nicht an Cormode08 angelegt ist, sondern an Ganesan,
	// so dass der Wertebereich etwas netter wird.
	// Siehe z.B. Testcase EVA1.
	public static double bgm2( Set<Element> approx, Set<Element> exact ){
		
		double sim = 0.0;
		double res;
		Element match, lca;
		
		for( Element elem : approx ){
			match = bestMatch( elem, exact );
			//System.out.println( elem.toShortString() + ", match = " + match.toShortString() );
			if( elem.getL() == 0 && match.getL() == 0 ){
				sim += 1;
				continue;
			}
			if( elem.getL() == 0 || match.getL() == 0 ){
				sim += 0.5;
				continue;
			}
			lca = elem.getLca( match );
			//System.out.println( elem.toShortString() + ", lca = " + lca.toShortString() + "   " + match.getL() );
			sim += 0.5 * lca.getL() * ( 1.0 / elem.getL() + 1.0 / match.getL() );
			//System.out.println( "sim = " + sim );
		}
		
		for( Element elem : exact ){
			match = bestMatch( elem, approx );
			//System.out.println( elem.toShortString() + ", match = " + match.toShortString() );
			if( elem.getL() == 0 && match.getL() == 0 ){
				sim += 1;
				continue;
			}
			if( elem.getL() == 0 || match.getL() == 0 ){
				sim += 0.5;
				continue;
			}
			lca = elem.getLca( match );
			//System.out.println( elem.toShortString() + ", lca = " + lca.toShortString() + "   " + match.getL() );
			sim += 0.5 * lca.getL() * ( 1.0 / elem.getL() + 1.0 / match.getL() ); 			
			//System.out.println( "sim = " + sim );
		}

		res = sim / ( exact.size() + approx.size() );

		return res;		
	}
	

	// Aehnlichkeitsmass fuer Datenstrukturen, fuer jedes Praefix, fuer das
	// mindestens eine Datenstruktur Hufigkeit != 0 schaetzt, wird
	// die Aehnlichkeit berechnet als Aehnlichkeit der geschaetzten Haeufigkeiten,
	// ( 2 * min / (min+max) ) anschlie�end wird ueber alle diese Praefixe gemittelt.
	public static double dsm(Map<Element, Integer> approxDs,
			Map<Element, Integer> exactDs) {
		
		
		
		//TODO: Durch Gesamtzahl N teilen!!!
		
		
		
		Integer other;
		int thisVal, otherVal, min, max;
		double sum = 0;
		Set<Element> union = new HashSet<Element>();
		for( Element e : approxDs.keySet() ) union.add(e);
		for( Element e : exactDs.keySet() ) union.add(e);

		for( Element e : approxDs.keySet() ){
			thisVal = approxDs.get( e );
			other = exactDs.get( e );
			if( other == null ) otherVal = 0;
			else otherVal = other;
			min = Math.min( thisVal, otherVal );
			max = Math.max( thisVal, otherVal );
			// Berechne: 2 * min / (min+max)
			// Aber: Doppelberechnung vermeiden fuer Elemente aus beiden Mengen
			sum += min / (double)(min + max);
			if( other == null ) sum += min / (min + max);
		}
		
		for( Element e : exactDs.keySet() ){
			thisVal = exactDs.get( e );
			other = approxDs.get( e );
			if( other == null ) otherVal = 0;
			else otherVal = other;
			min = Math.min( thisVal, otherVal );
			max = Math.max( thisVal, otherVal );
			// Berechne: 2 * min / (min+max)
			// Aber: Doppelberechnung vermeiden fuer Elemente aus beiden Mengen
			sum += min / (double)(min + max);
			if( other == null ) sum += min / (min + max);
		}		
	
		return sum / union.size();
	}
	
	
	public static String evaluate( Set<Element> approx, Set<Element> exact ){
		
		String result;
		
		result = "OGMGanesan = " + ogmGanesan(approx, exact);
		result += "\nOGMCormode = " + ogmCormode(approx, exact);
		result += "\nBgm = " + bgm(approx, exact);
		result += "\nBgm2 = " + bgm2(approx, exact);
		
		return result;		
	}
	
	
	public static String compactHierarch( Set<Element> approx, Set<Element> exact ){
		
		String result;
		
		result = ogmGanesan(approx, exact) + ";";
		result += ogmCormode(approx, exact) + ";";
		result += bgm(approx, exact) + ";";
		result += bgm2(approx, exact);
		
		return result;		
	}
	
	
	public static String compactFlat( Set<Element> approx, Set<Element> exact ){
		
		String result;
		
		result = precision(approx, exact) + ";";
		result += recall(approx, exact) + ";";
		result += jaccard(approx, exact) + ";";
		result += dice(approx, exact);
		
		return result;		
	}
	
	
	public static String compactAll( Set<Element> approx, Set<Element> exact ){
		return compactFlat(approx, exact) + ";" + compactHierarch(approx, exact);
	}

	
	private static double elemSimGanesan( Element e, Set<Element> other ){

		// Wurzel?
		// e ist dann Vorfahr, in Analogie zu anderen Vorfahrenbeziehungen
		// eins w�hlen, wird relativiert durch Symmetrie. NICHT GANZ!
		// Siehe eins drueber.
		if( e.getL() == 0 ) return 1;
		return (double)bestLca( e, other ).getL() / e.getL();
	}

	
	private static double depthLca( Element e, Set<Element> other ){
		Element lca = bestLca( e, other);
		return lca.getL();
	}
	
	
	private static Element bestLca( Element e, Set<Element> other ){
		return e.getLca( bestMatch( e, other ) );
	}
	
	
	private static Element bestMatch( Element e, Set<Element> other ){

		Element lca, bestMatch = null;		
		int bestDepth = -1;
		int bestMatchDepth = Integer.MAX_VALUE;
		if( other.size() == 0 ) throw new RuntimeException();
		
		for( Element cand : other ){
			lca = e.getLca( cand );
			if( lca.getL() > bestDepth ){
				bestDepth = lca.getL();
				bestMatchDepth = cand.getL();
				bestMatch = cand;
			}
			if( lca.getL() == bestDepth && cand.getL() < bestMatchDepth ){
				bestDepth = lca.getL();
				bestMatchDepth = cand.getL();
				bestMatch = cand;
			}
		}
		return bestMatch;
	}


	/**
	 * @param distanceMeasure
	 * @param sample
	 * @param hitter
	 * @return
	 */
	public static Double distance(int distanceMeasure, Set<Element> sample,
			Set<Element> hitter) {
		
		//	String[] distance = { "OGM Ganesan", "OGM Cormode", "BGM", "BGM2" };
		switch( distanceMeasure ){
		case OGM_GANESAN: return 1.0 - ogmGanesan(sample, hitter);
		case OGM_CORMODE: return 1.0 - ogmCormode(sample, hitter); 
		case BGM: return 1.0 - bgm(sample, hitter);
		case BGM2: return 1.0 - bgm2(sample, hitter); 
		default: throw new RuntimeException(" Kein gueltiges Distanzma� gew�hlt.");
		}
	}

}
