/**
 * 
 */
package hitters.multi;

import hitters.tools.Utils;

import java.util.Arrays;


/**
 * @author Peter Fricke
 * 
 * Ein Objekt vom Typ Element stellt entweder ein ggf. mehrdimensionales
 * Element aus dem Datenstrom dar oder ein (mehrdimensionales) Prfix.
 * 
 * Es gibt zwei Arten der internen Reprsentation der 
 * Prfixe in einer einzelnen Dimension: Eine Darstellung als
 * String und eine kompakte als int. Fr "mehrdimensionale Prfixe"
 * werden mehrere solcher Reprsentationen kombiniert. 
 * Die Hierarchien in den verschiedenen Dimensionen sind 
 * entweder Dateisystempfade und werden intern als Strings dargestellt,
 * bei vorab bekanntem Wertebereich (keine Pfade) knnen die Hierarchien
 * auch intern als int-Werte dargestellt werden, wobei die Bits 
 * flexibel auf die einzelne Hierarchiestufen verteilt werden knnen. 
 * 
 * Fuer die Systemcalls:
 * Beide Formen der internen Darstellen lassen sich mischen,
 * es kann also eine Dimension intern als int und eine andere 
 * als String dargestellt werden. Auslagern in eigene Klassen
 * wre hbscher, bewusst vermieden, um die vielen Objekterzeugungen
 * zu begrenzen.
 * 
 * Keine Verzeigerung, weil alles als Hashmap dargestellt wird.
 * Die Elemente sind der Schlssel, zugehriger Wert ist ein Tripel
 * aus Gewicht des Knotens, m-Wert und Delta. (Klasse MultiStringElement) 
 * Auch die Hufigkeiten der HHH-Knoten lassen sich in einer Hashmap 
 * ber den Schlssel Element mit ein Tripel als Wert darstellen: 
 * Er enthlt fmin, fmax und F (Klasse MultiHitterInfo).
 *  
 * 
 * Wenn die Darstellung ber Strings erfolgt, knnen die Dateinamen 
 * einschl. Pfad oder deren Prfixe dargestellt werden:
 * 
 * 		/usr/bin/bluetooth-properties mit Level 3 und den Prfixen
 * 		/usr/bin 
 * 		/usr
 * 		*
 *   
 * Level 0 wird durch einen Stern dargestellt.
 *   
 * Wenn die Darstellung ber ints erfolgt, muss der Wertebereich
 * bekannt sein. Die Festlegung der Tiefe der Hierarchie und der
 * Bits pro Hierarchiestufe erfolgt durch bergabe eines Arrays
 * an das Parameterobjekt. So definiert 
 * 
 * byte[][] plength = { {0,4,8,32}, {0,4,8,32} };
 *
 * zwei Dimensionen, dargestellt als int, mit jeweils
 * Hierarchietiefe drei, den ersten beiden Stufen werden 
 * vier Bit, der letzten Stufe 24 Bit zugeteilt.
 * 
 *   
 *   
 * Mehrdimensionale Elemente haben schrecklich viele Prfixe,
 * von denen der Algorithmus beim Compress sehr viele untersuchen 
 * muss. Dabei muss jeweils ein Element-Objekt erzeugt werden,  
 * per Hashing der zugehrige Wert gefunden werden, anschlieend 
 * wird das Objekt meist nicht mehr bentigt. 
 * Es werden also a) sehr viele Element-Objekte mit b) jeweil sehr wenig
 * Nutzlast und c) sehr geringer Lebensdauer erzeugt, der Overhead
 * fr die Objektorientierung ist enorm.
 * 
 * Zwei Anstze zur Begrenzung des Problems sind implementiert:
 * 
 * Die interne Darstellung als int verkleinert durch das Vermeiden von 
 * String-Objekten das Element-Objekt und fhrt zu erheblich besseren 
 * Laufzeiten.
 * (TODO Arrays vermeiden wo mglich.)
 * 
 * Die Methoden turnIntoParent, generalizeTo, setTo usw. reduzieren 
 * die Anzahl der erzeugten Element-Objekte, indem das 
 * vorhandene Objekt verndert wird, statt ein neues 
 * Objekt zu erzeugen. Die hufige Situation 
 * "Kind gibt Elter zurck, Kind wird nicht mehr bentigt" wird so 
 * vermieden. Diese Vorgehensweise erfordert das explizite Klonen,
 * falls eines der Objekte als Schlssel in einer Hashmap verwendet
 * werden soll, macht den Code also fehleranflliger.
 * 
 * Da andere Optimierungen ebenfalls dazu neigen, den Code fehleranflliger
 * oder weniger flexibel zu machen (festes Verdrahten der Dimensionen
 * wrde z.B Arrays sparen) verzichte ich darauf, wenn die Laufzeiten
 * fr einen praktischen Einsatz des Verfahrens nicht ausreichend gut sind,
 * ist eine bertragung des Codes nach C/++ sinvoller, als zu versuchen,
 * Java mit Gewalt in C zu verwandeln. Sowohl Laufzeit als auch Speicher-
 * bedarf sollten sich durch eine solche bertragung erheblich 
 * reduzieren lassen.
 * 
 *   
 *   
 * Equals und HashCode mssen implementiert sein, damit die
 * ganze Hasherei funktioniert, ggf. nochmal prfen, ob sich da
 * noch etwas verbesern lt.
 * 
 * 
 */
public class Element implements Comparable<Element>{

	protected Parameter myParams;
	protected final String[] eStr;
	protected final int[] 	 eInt;		

	// Gepuffert, bei nderungen nullsetzen
	private int hashCode = 0;  

	// Update bei nderungen!
	private final int[] level; 			

	// Update bei nderungen!
	private int L = 0;				

	public Element( final String[] eStr, final int[] eInt,
			final Parameter myParams ){

		this.myParams = myParams;
		this.eStr     = new String[ myParams.getDimS() ];
		this.eInt     = new int[ myParams.getDimI() ];
		this.level    = new int[ myParams.getDim() ];

		for( int i = 0; i < eStr.length; i++ ){
			if( eStr[i] != null ){
				if( eStr[i].length() > 0 ) this.eStr[i] = eStr[i];
				else this.eStr[i] = "*";
			}
		}

		if( eInt != null ){ 
			for( int i = 0; i < eInt.length; i++ ){
				this.eInt[i] = eInt[i];
			}	
		}

		updateLevel();  			
	}		

	public Element( final Parameter par ) {
		this( new String[par.getDimS()], new int[par.getDimI()], par );
	}


	public int getLevel( final int dim ){ return level[ dim ]; }


	public int getL(){ return L; }

	public int getDim(){ return myParams.getDim(); }

	public boolean hasParent( final int dim ){ 
		return ( level[ dim ] > 0 ); 
	}


	public boolean turnIntoParent( final int dim ){
		hashCode = 0;
		boolean has = hasParent( dim );
		if( has ){			

			if( dim < myParams.getDimS() ){
				int l = eStr[dim].lastIndexOf('/');

				// l  > 0: Elter ist laengstes Prfix
				if( l  > 0 ) eStr[dim] = eStr[dim].substring(0, l);	  

				// l == 0: Elter ist Wurzelverzeichnis
				else if( l == 0 ) eStr[dim] = "*";		

				// l  < 0: Kein Elter
				else eStr[dim] = null; 					
			}
			else{
				eInt[dim - myParams.getDimS()] &= 
					myParams.getMask( dim - myParams.getDimS(), level[dim] - 1 );
			}

			level[dim]--;
			L--;

		}
		return has;
	}		


	// Versucht, dieses Objekt auf label zu generalisieren.
	// Gibt true zurueck, wenn erfolgreich.
	// Wenn das ungeneralisierte Objekt noch benoetigt
	// wird, muss vorher geklont werden.
	public boolean generalizeTo( final int[] label ){

//		System.out.println( "this " + this );
//		System.out.println( "label " + Arrays.toString(label) );
//		System.out.println( "level " + Arrays.toString(level) );
//		System.out.println( "eStr= " + eStr[0] );
		
		//Erst prfen, dann ndern
		for( int i = 0; i < eStr.length + eInt.length; i++ )
			if( level[i] < label[i] ) return false;			

		hashCode = 0;
		String[] split;
		//int l = 0;
		L = 0;	

		for( int i = 0; i < eStr.length; i++ ){					
			if( (level[i] == 0) || (label[i] == 0) ) eStr[i] = "*";
			else{
				split = eStr[i].split("/");  // TODO Effizienz			
				if( split.length <= 1 )	eStr[i] = "*"; ///////////////////////// l -> 1
				else{
//					System.out.println( "Splitlength: " + split.length);
//					System.out.println( "split " + Arrays.toString(split) );
					eStr[i] = "";				
					for( int j = 1; j < label[i] + 1; j++ ) 
						eStr[i] = eStr[i] + "/" + split[j];
				}
			}
			level[i] = label[i];
			if( eStr[i].equals("*") && level[i] > 0 ) throw new RuntimeException( "level[i] > 0: " + level[i]);
			L += level[i];
		}

		for( int i = 0; i < eInt.length; i++ ){								
			eInt[i] &= myParams.getMask( i, label[i + myParams.getDimS()] );			
			level[ i + myParams.getDimS() ] = label[ i + myParams.getDimS() ];
			L += level[ i + myParams.getDimS() ];
		}		

		return true;
	}

	
	// Generalisiert dieses Objekt auf label in allen
	// Dimensionen, in denen es noch spezieller als label ist.
	// Anders als bei generalizeTo wird nicht abgebrochen, wenn
	// es in einer Dimension bereits genereller ist als Label.
	// Verwendung: Begrenzung der Hierarchietiefe beim Einlesen
	// der Daten.
	public void capHierarchy( final int[] label ){

		hashCode = 0;
		String[] split;
		//int l = 0;
		L = 0;			
		
		for( int i = 0; i < eInt.length + eStr.length; i++ )
			level[i] = Math.min( level[i], label[i] );
			
		for( int i = 0; i < eStr.length; i++ ){					
			if( level[i] == 0 ) eStr[i] = "*";
			else{
				split = eStr[i].split("/");  // TODO Effizienz			
				if( split.length <= 1 )	eStr[i] = "*";
				else{
					eStr[i] = "";				
					for( int j = 1; j < level[i] + 1; j++ ) 
						eStr[i] = eStr[i] + "/" + split[j];
				}
			}			
			L += level[i];
		}

		for( int i = 0; i < eInt.length; i++ ){								
			eInt[i] &= myParams.getMask( i, level[i + myParams.getDimS()] );			
			L += level[ i + myParams.getDimS() ];
		}		
	}

	
	public boolean isGenOf( final Element special ){

		for( int i = 0; i < eStr.length + eInt.length; i++ ) 
			if( special.getLevel(i) < level[i] ) return false;			

		Element dummy = special.clone();
		dummy.generalizeTo( level );		

		return dummy.equals( this );
	}


	// Versucht, dieses Objekt auf das Infimum
	// der Knoten a und b zu setzen.
	// Gibt true zurueck, wenn erfolgreich.
	// Wenn das ungeneralisierte Objekt noch benoetigt
	// wird, muss vorher geklont werden.
	public boolean setToGlbOf( final Element a, final Element b ) {
		hashCode = 0;
		boolean success = true;
		int ma, mb, ms;

		loop : for( int i = 0; i < myParams.getDimS(); i++ ){

			if( a.getLevel(i) <= b.getLevel(i) ){
				eStr[i] = b.eStr[i];
				if( ( ! a.eStr[i].startsWith("*") ) && ( ! b.eStr[i].startsWith(  a.eStr[i], 0 ) ) ){
					success = false;
					break loop;					
				}
			}else{
				eStr[i] = a.eStr[i];
				if( ( ! b.eStr[i].startsWith("*") ) && ( ! a.eStr[i].startsWith( b.eStr[i], 0 ) ) ){ 					
					success = false;
					break loop;					
				}
			}
		}

		loop2 : for( int i = 0; i < myParams.getDimI(); i++ ){

			ma = a.getLevel( myParams.getDimS() + i );
			mb = b.getLevel( myParams.getDimS() + i );
			ms = myParams.getMask(i, Math.min(ma,mb) );
			if( ma <= mb ) eInt[i] = b.eInt[i];
			else eInt[i] = a.eInt[i];
			if( ( b.eInt[i] & ms ) != ( a.eInt[i] & ms ) ) {								
				success = false;
				break loop2;					
			}
		}

		updateLevel();
		return success; 		
	}


	// Gibt das Supremum von diesem Objekt und dem 
	// Knoten other zurueck. Wird benoetigt fuer die
	// Aehnlichkeitsmasse, die hierarchische Informationen
	// beruecksichtigen.
	public Element getLca( Element other ){

		int[] label = new int[ myParams.getDim() ];
		Element e = other.clone();
		String[] split1;
		String[] split2;		
		int ma, mb, mx;
		int ms; // Maske		
		int counter = 0;

		for( int i = 0; i < myParams.getDimS(); i++ ){

			split1 = eStr[i].split("/"); // Erste Komponente ist leer
			split2 = other.eStr[i].split("/");

			ma = this.getLevel( i );
			mb = other.getLevel( i );
			mx = Math.min(ma,mb);			

			counter = 0;
			// Starte bei 1, weil erste Komponente leer
			inner: for( int j = 1; j < mx+1; j++ ){ 

				if( split1[j].equals(split2[j]) ){
					counter++;					
				}				
				else
					break inner;
			}
			label[i] = counter;
		}

		for( int i = 0; i < myParams.getDimI(); i++ ){

			ma = this.getLevel( myParams.getDimS() + i );
			mb = other.getLevel( myParams.getDimS() + i );
			mx = Math.min(ma,mb);			

			counter = 0;
			inner: for( int j = 0; j < mx; j++ ){ 
				ms = myParams.getMask(i, j+1 );
				if( ( this.eInt[i] & ms ) == ( other.eInt[i] & ms )  ){						
					counter++;					
				}				
				else
					break inner;
			}
			label[myParams.getDimS() + i] = counter;
		}
		e.generalizeTo( label );
		return e;
	}
	

	// Untersucht, ob dieses Objekt Nachfahr eines Elements
	// aus Label ist. Erlaubt Optimierung der Outputmethode,
	// weil die Menge der zu pruefenden HHH kleiner wird.
	public boolean isBelowLabel( final int[] label ){

		boolean ident = true;
		for( int i = 0; i < eStr.length + eInt.length; i++ ){
			if( level[i] < label[i] ) return false;
			if( level[i] != label[i] ) ident = false;
		}
		
		if(ident) return false;
		
		return true;
	}

	public void setTo( final Element value ) {
		hashCode = 0;		
		for( int i = 0; i < eStr.length; i++ ){
			eStr[i] = value.eStr[i];
			level[i] = value.level[i];
		}
		for( int i = 0; i < eInt.length; i++ ){
			eInt[i] = value.eInt[i];
			level[ i + myParams.getDimS() ] = value.level[ i + myParams.getDimS() ];
		}
		L = value.L;
	}


	public void setTo( final String[] s, final int[] newInt) {
		hashCode = 0;
		for( int i = 0; i < myParams.getDimS(); i++ ) eStr[i] = s[i];
		for( int i = 0; i < eInt.length; i++ ) eInt[i] = newInt[i];
		updateLevel();
	}


	public void setTo( final String[] s, final String[] eIntStr) {
		setTo( s, stringToInt( eIntStr ) );
	}


	public String get(final int dim) {		
		if( dim < myParams.getDimS() ) return eStr[dim];
		else return toString( eInt[ dim - myParams.getDimS() ], dim );		 
	}


	// TODO: Getestet: Je nachdem, wie oft der HashCode pro Objekt 
	// abgefragt wird, kann Puffern erheblich Laufzeit kosten (!), 
	// wenn alles fertig ist, nochmal testen.
	@Override		
	public int hashCode(){		
		if( hashCode == 0 ){
			int hash = 0;
			for( int i = 0; i < eStr.length ; i++ ){
				// Schneller als Math.pow
				//System.out.println( this );
				for( int j = 0; j < eStr[i].length(); j++ ) hash = hash * 31;  
				hash = hash + eStr[i].hashCode();		 				
			}
			for( int i = 0; i < eInt.length ; i++ ){				 
				hash = 31 * hash + eInt[i];//TODO: HashCode-Int: Ist das weise?			
			}
			hashCode = hash;
		}
		return hashCode;
	}


	public boolean equals( Object o ){
		Element m;
		boolean equal = true;
		if( ! (o instanceof Element) ) return false;
		else  m = (Element)o;
		if( eStr.length != m.eStr.length ) return false;
		for( int i = 0; i < eStr.length; i++ ){
			if( ! eStr[i].equals( m.eStr[i] ) ) equal = false; 
		}
		for( int i = 0; i < eInt.length; i++ ){
			if( eInt[i] != m.eInt[i] ) equal = false; 
		}
		return equal;
	}


	public int compareTo(Element m){ // Erleichtert das Testen
		int comp = 0;
		if( m.eStr.length != eStr.length ) 
			throw new RuntimeException( "Element:" +
					"Unterschiedl. Dim: " + m.eStr.length + " " + eStr.length);
		for( int i = 0; i < eStr.length; i++ ){ // Erste ungleiche Dim entscheidet
			comp = eStr[i].compareTo( m.eStr[i] );
			if( comp != 0 ) return comp;
		}
		for( int i = 0; i < eInt.length; i++ ){ // Erste ungleiche Dim entscheidet
			comp = eInt[i] -  m.eInt[i];
			if( comp != 0 ) return comp;
		}
		return comp;
	}


	public Element clone(){
		return new Element( eStr, eInt, myParams );
	}


	public String toString(){
		int res;
		String s = "( ";
		for( int i = 0; i < eStr.length; i++ ){
			s = s + eStr[i];
			if( i < myParams.getDim() -1 ) s = s + " (s) | ";
			else s = s + " (s)";
		}
		for( int i = 0; i < eInt.length; i++ ){			
			for( int j = 1; j < myParams.getMaskLength(i); j++ ){
				res = eInt[i] & myParams.getMask(i,j) & ~ myParams.getMask(i, j-1); 
				res = res >>> ( 32 - myParams.getPlength(i,j) );				
		s = s + "/" + res;
			}
			if( i < eInt.length -1 ) s = s + " (i) | "; 
			else s = s + " (i)";
		}
		s = s + " )";
		return s;
	}	


	// Kompakte Darstellung zur Beschriftung der Knoten in yed
	public String toShortString(){
		int res;
		String s = "";
		for( int i = 0; i < eStr.length; i++ ){
			if( eStr[i].equals("*") ) s = s + "*"; 
			else s = s + eStr[i].substring(1); // fuehrenden Slash streichen
			if( i < myParams.getDim() -1 ) s = s + " | ";
		}
		for( int i = 0; i < eInt.length; i++ ){			
			for( int j = 1; j < myParams.getMaskLength(i); j++ ){
				res = eInt[i] & myParams.getMask(i,j) & ~ myParams.getMask(i, j-1); 
				res = res >>> ( 32 - myParams.getPlength(i,j) );				
				s = s + "/" + res;
			}
			if( i < eInt.length -1 ) s = s + " | "; 			
		}		
		return s; 
	}


	public int[] stringToInt( String[] s ){
		return stringToInt( s, myParams );
	}


	public int[] stringToInt( String s0, String s1 ){

		String[] st = new String[2];
		st[0] = s0;
		st[1] = s1;		
		return stringToInt( st, myParams );
	}


	public static int[] stringToInt( String[] st, Parameter par ){

		String[] md;

		int width;
		long n;
		int[] eInt = new int[par.getDimI()];

		for( int k = 0; k < eInt.length ; k++ ){	
			md =  st[ k ].split("/");						
			for( int j = 1; j < md.length; j++ ){
				n = Long.parseLong( md[j] );				
				width = par.getWidth(k,j);				
				if( (n < 0) || ( n > Utils.exp( 2, width ) -1 ) )					
					throw new NumberFormatException("Angebener Wert " + n + 
							" nicht im zulssigen Bereich (Item: " +  
							Arrays.toString(st) + ").");
				n = n << ( 32 - par.getPlength(k, j) );
				eInt[k] += n;
			}
		}
		return eInt;
	}

	
	public void print( int num, int i ){
		System.out.println( toString(num, i) );
	}


	public String toString( int num, int dim ){
		String s = "";
		int res;
		for( int j = 1; j < myParams.getMaskLength( dim - myParams.getDimS() ); j++ ){
			res = num & myParams.getMask(dim - myParams.getDimS(), j) &
			~myParams.getMask(dim - myParams.getDimS(), j-1); 
			res = res >>> ( 32 - myParams.getPlength(dim - myParams.getDimS(),j) );				
				s = s + "/" + res;
		}
		return s;
	}


	public static String toString( int num, int dim, Parameter par ){
		String s = "";
		int res;
		for( int j = 1; j < par.getMaskLength( dim - par.getDimS() ); j++ ){
			res = num & par.getMask(dim - par.getDimS(), j) &
			~par.getMask(dim - par.getDimS(), j-1); 
			res = res >>> ( 32 - par.getPlength(dim - par.getDimS(),j) );				
				s = s + "/" + res;
		}
		return s;
	}

	
	
	//Service: Falls nur Strings als Reprsentation verwendet
	//werden, kann man sich das null fr die Ints auch sparen.
	public static Element createElement( String[] eStr,
			Parameter myParams ){
		return createElement( eStr, null, myParams );
	}


	public static Element createElement( String[] eStr, String[] eInt,
			Parameter myParams ){

		int[] res;

		if( eInt == null && myParams.getDimI() != 0 )
			throw new RuntimeException( "eInt == null && myParams.getDimI() != 0" );

		if( eStr == null && myParams.getDimS() != 0 )
			throw new RuntimeException( "eStr == null && myParams.getDimS() != 0 " );

		if( eInt != null ) res = stringToInt( eInt, myParams );
		else res = new int[ 0 ];

		if( eStr == null ) eStr = new String[ 0 ];

		return new Element( eStr, res, myParams );		
	}


	// VORSICHT! Sicherstellen, dass man das Level der Ints immer erkennen kann!
	// Alle Bits in Abschnitt = 0 bedeutet ANY
	//
	// UND IMMER SCHN AN DAS VORZEICHENBIT DENKEN
	//
	//
	private void updateLevel(){
		L = 0;		
		int k = 0;
		for( int i = 0; i < eStr.length; i++ ){
			level[i] = 0;
			k = 0;
			if( eStr[i] != null ){				
				k = eStr[i].indexOf( '/', k );
				while( k >= 0){
					k = eStr[i].indexOf( '/', k + 1 );
					level[i]++;
				}								
			}			
			L += level[i];			
		}

		for( int i = 0; i < eInt.length; i++ ){
			level[ i + myParams.getDimS() ] = 0;							
			for( int j = 0; j < myParams.getMaskLength(i); j++ ) 
				if( ( eInt[i] & ~myParams.getMask(i, j) ) != 0 ) 
					level[ i + myParams.getDimS() ]++;								

			L += level[ i + myParams.getDimS() ];
		}
	}

}
