/*
 *  RapidMiner
 *
 *  Copyright (C) 2001-2010 by Rapid-I and the contributors
 *
 *  Complete list of developers available at our web site:
 *
 *       http://rapid-i.com
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 */
package com.rapidminer.operator.io;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.List;

import com.rapidminer.datatable.DataTable;
import com.rapidminer.datatable.GnuPlotDataTableHandler;
import com.rapidminer.operator.Operator;
import com.rapidminer.operator.OperatorDescription;
import com.rapidminer.operator.OperatorException;
import com.rapidminer.operator.ports.DummyPortPairExtender;
import com.rapidminer.operator.ports.PortPairExtender;
import com.rapidminer.parameter.ParameterType;
import com.rapidminer.parameter.ParameterTypeFile;
import com.rapidminer.parameter.ParameterTypeString;
import com.rapidminer.tools.Tools;


/**
 * Writes the data generated by a ProcessLogOperator to a file in gnuplot
 * format.
 * 
 * @author Simon Fischer, Ingo Mierswa
 */
public class GNUPlotOperator extends Operator {


	/** The parameter name for &quot;The gnuplot file.&quot; */
	public static final String PARAMETER_OUTPUT_FILE = "output_file";

	/** The parameter name for &quot;The name of the process log operator which produced the data table.&quot; */
	public static final String PARAMETER_NAME = "name";

	/** The parameter name for &quot;The title of the plot.&quot; */
	public static final String PARAMETER_TITLE = "title";

	/** The parameter name for &quot;The values of the x-axis.&quot; */
	public static final String PARAMETER_X_AXIS = "x_axis";

	/** The parameter name for &quot;The values of the y-axis (for 3d plots).&quot; */
	public static final String PARAMETER_Y_AXIS = "y_axis";

	/** The parameter name for &quot;A whitespace separated list of values which should be plotted.&quot; */
	public static final String PARAMETER_VALUES = "values";

	/** The parameter name for &quot;Additional parameters for the gnuplot header.&quot; */
	public static final String PARAMETER_ADDITIONAL_PARAMETERS = "additional_parameters";

	private PortPairExtender dummyPorts = new DummyPortPairExtender("through", getInputPorts(), getOutputPorts());

	public GNUPlotOperator(OperatorDescription description) {
		super(description);

		dummyPorts.start();

		getTransformer().addRule(dummyPorts.makePassThroughRule());
	}

	@Override
	public void doWork() throws OperatorException {
		String dataTableName = getParameterAsString(PARAMETER_NAME);
		if (!getProcess().dataTableExists(dataTableName)) {
			getLogger().warning("Data table with name '" + dataTableName + "' does not exist.");
			return;
		}

		DataTable dataTable = getProcess().getDataTable(dataTableName);
		String[] valueNames = getParameterAsString(PARAMETER_VALUES).split(" ");
		int[] values = new int[valueNames.length];
		for (int i = 0; i < values.length; i++) {
			values[i] = dataTable.getColumnIndex(valueNames[i]);
			if (values[i] == -1) {
				getLogger().warning(getName() + ": No data column with name '" + valueNames[i] + "' exists.");
				return;
			}
		}

		String xAxisName = getParameterAsString(PARAMETER_X_AXIS);
		int xAxis = dataTable.getColumnIndex(xAxisName);
		if (xAxis == -1) {
			getLogger().warning("No data column with name '" + xAxisName + "' exists.");
			return;
		}
		String yAxisName = getParameterAsString(PARAMETER_Y_AXIS);
		int yAxis = -1;
		if (yAxisName != null) {
			yAxis = dataTable.getColumnIndex(yAxisName);
			if (yAxis == -1) {
				getLogger().warning("No data column with name '" + yAxisName + "' exists.");
				return;
			}
		}

		String additional = "";
		if (isParameterSet(PARAMETER_TITLE))
			additional += "set title \"" + getParameterAsString(PARAMETER_TITLE) + "\"" + Tools.getLineSeparator();
		if (isParameterSet(PARAMETER_ADDITIONAL_PARAMETERS))
			additional += getParameterAsString(PARAMETER_ADDITIONAL_PARAMETERS);

		File file = getParameterAsFile(PARAMETER_OUTPUT_FILE, true);
		getLogger().info("Creating gnuplot file '" + file + "'");
		PrintStream out = null;
		try {
			out = new PrintStream(new FileOutputStream(file));
			GnuPlotDataTableHandler handler = new GnuPlotDataTableHandler(dataTable);
			handler.writeGNUPlot(out, xAxis, yAxis, values, "linespoints", additional, null);
		} catch (IOException e) {
			getLogger().warning("Cannot create output file: " + e.getMessage());
			return;
		} finally {
			if (out != null) {
				out.close();
			}
		}

		dummyPorts.passDataThrough();
	}

	@Override
	public List<ParameterType> getParameterTypes() {
		List<ParameterType> types = super.getParameterTypes();
		types.add(new ParameterTypeFile(PARAMETER_OUTPUT_FILE, "The gnuplot file.", "gnu", false));
		types.add(new ParameterTypeString(PARAMETER_NAME, "The name of the process log operator which produced the data table.", false));
		types.add(new ParameterTypeString(PARAMETER_TITLE, "The title of the plot.", "Created by RapidMiner"));
		types.add(new ParameterTypeString(PARAMETER_X_AXIS, "The values of the x-axis.", false));
		ParameterType type = new ParameterTypeString(PARAMETER_Y_AXIS, "The values of the y-axis (for 3d plots).", true);
		type.setExpert(false);
		types.add(type);
		types.add(new ParameterTypeString(PARAMETER_VALUES, "A whitespace separated list of values which should be plotted.", false));
		types.add(new ParameterTypeString(PARAMETER_ADDITIONAL_PARAMETERS, "Additional parameters for the gnuplot header.", true));
		return types;
	}
}
