require 'results_object.rb'

class Counter < Array
  def initialize variation
    @variation = variation
    super @variation.variation_objects.size
    reset
  end

  attr_reader :current_digit, :variation

  def increase
    self[@current_digit] += 1
    previous_digit = @current_digit
    while self[@current_digit] >= @variation.variation_objects[@current_digit].size do
      self[@current_digit] = 0
      @current_digit -= 1
      self[@current_digit] += 1 unless @current_digit < 0
    end
    if @current_digit < previous_digit && @current_digit >= 0
      @current_digit = size-1
    end
    return @current_digit >= 0
  end


  def reset
    fill 0
    @current_digit = size-1
  end
end

class Variation
  # process: an instance of RMProcess
  # output_repository: a Hash with the fields :name, :local_directory and :remote_directory
  #   specifying where to create the process files
  # file_separator: the Variation composes the filename of the generated processes from the
  #   variated parameters and their values. The single parts of the filename will be separated
  #   by file_separator
  def initialize process, output_repository, file_separator = "/"
    @process = process
    @output_repository = output_repository
    @file_separator = file_separator
    @variation_objects = Array.new
    @namespace = ""
    
#    @base_variations = Array.new
  end

  # base_variations: variations this variations depends on. When create_process_files is called,
  #   that function is also called on each base_variation
#  attr_accessor :base_variations
  attr_accessor :variation_objects, :output_repository, :namespace
  attr_reader :process

  def num_variations
    num_variations = 1
    @variation_objects.each do |vo|
      num_variations *= vo.size
    end
    return num_variations
  end

  # creates the process_files. The filename of each process is constructed the same way as in
  # filenames(true).
  def create_process_files overwrite_existing = true
#    puts "Creating base variations..."
#    puts self.process.base_process
    @variation_objects.each do |vo|
      vo.base_variations.each{ |bv| bv.namespace = @namespace; bv.create_process_files overwrite_existing }
    end
#    puts "Done creating base variations."
    puts "Creating #{num_variations} processes..."

    counter = Counter.new self

    while counter.current_digit >= 0
      filename = filename_for_counter counter, :local_directory, true
      unless !overwrite_existing && File.exists?(filename)
        apply_values counter
        #puts "writing process to #{filename}"
        @process.write filename
      end

      counter.increase
    end
    puts "Done."
  end





  # returns an array of hashes. Each has contains a counter in key :counter and the
  # corresponding process filename in key :filename
  # prefix: one of :none, :repository_name or :local_directory
  # counter: used internally for the recursion
  # current_digit: used internally for the recursion
  def filenames prefix, with_suffix
    counter = Counter.new self

    filenames = Array.new
    while counter.current_digit >= 0
      filename = filename_for_counter counter, prefix, with_suffix
      filenames.push filename
      counter.increase
    end
    return filenames
  end

  # Searches all dependencies of this RMProcess induced by ProcessVariationValues.
  # Returns an array of filenames relative to the ouput_path of this Variation
  def dependencies
    dependency_list = Array.new
    @variation_objects.size.times do |i|
      vo = @variation_objects[i]
      if vo.respond_to? :base_path
        base_path = vo.base_path
        vo.values.each do |value|
          dependency_list.push "#{base_path}#{value}"
        end
      end
    end
    return dependency_list
  end

  # returns the filename for the given counter.
  # prefix: one of :none, :local_directory, :remote_directory, :repository name.
  #   the filenames are prefixed with the corresponding property of @output_repository
  # with_suffix: if false, the last .rmp suffix will be cut off
  def filename_for_counter counter, prefix, with_suffix = true
    if prefix == :none
      s = ""
    elsif prefix == :local_directory
      s = "#{@output_repository[:local_directory]}/"
    elsif prefix == :remote_directory
      s = "#{@output_repository[:remote_directory]}/"
    elsif prefix == :repository_name
      s = "//#{@output_repository[:name]}/"
    else
      raise "illegal prefix '#{prefix.to_s}'"
    end
    s += @namespace
    s += @process.filename

    counter.size.times do |i|
      s += "#{@file_separator}#{@variation_objects[i].to_s counter[i]}" unless @variation_objects[i].is_a? DynamicParameterValues
    end
    s += ".rmp"
    s.gsub! ' ', '_'

    4.times {s.chop!} unless with_suffix

    return s
  end

  def sync_processes_to_server server_info
    command  = "rsync -az --delete"
    command += " --rsh='ssh -p #{server_info[:port]}'" if server_info[:port]
    command += " #{@output_repository[:local_directory]}/"
    command += " #{server_info[:user]}@#{server_info[:server]}"
    command += ":#{@output_repository[:remote_directory]}"

    success = system(command)
    if !success
      throw "could not sync to the server. Command was:\n#{command}"
    end
  end







  def get_values_for_filename filename#, prefix, with_suffix
#    puts "Searching values for #{filename}..."
    values = Hash.new
    counter = Counter.new self
    while counter.current_digit >= 0
      cfilename = filename_for_counter(counter, :none, false)
      if filename.end_with? cfilename
        counter.size.times do |i|
          vo = @variation_objects[i]
          parameter = vo.parameter
          value = vo.value counter[i]
          values["#{vo.operator}.#{parameter}"] = value
          if parameter == "process_location"
            # recurse into base variations
            rec_values = nil
            vo.base_variations.each do |bv|
              rec_values = bv.get_values_for_filename(value)
              if rec_values
                values["#{vo.operator}.base_process"] = bv.process.base_process
                break
              end
            end
            rec_values.each_pair{|k,v| values["#{vo.operator}.#{k}"] = v} if rec_values
            if rec_values
            end
          end
        end
      end
      if values.size > 0 # correct counter found
        return values
      end
      counter.increase
    end
  end








  def get_results_object results_dir
    counter = Counter.new self

    results = ResultsObject.new self

    while counter.current_digit >= 0
      process_name = filename_for_counter counter, :none, false
      if File.exists? "#{results_dir}/#{process_name}"
        result_key = Hash.new
        counter.size.times do |i|
          vo = @variation_objects[i]
          result_key["#{vo.operator}.#{vo.parameter}"] = vo.value counter[i]
          if vo.parameter == "process_location"
            parsed_results = nil
            vo.base_variations.each do |bv|
              parsed_results = bv.get_values_for_filename(vo.value counter[i])#, :none, false)
              if parsed_results
                result_key["#{vo.operator}.base_process"] = File.basename bv.process.base_process, ".rmp"
                break
              end
            end
            if parsed_results
              parsed_results.each_pair do |parameter, value|
                result_key["#{vo.operator}.#{parameter}"] = value
              end
            end
          end
          i += 1
        end
        results[result_key] = Hash.new
        results[result_key][:process_name] = process_name

        d = Dir.new "#{results_dir}/#{process_name}"
        d.each do |f|
          if File.file? "#{d.path}/#{f}"
            i = 0
            rows = Array.new
            columns = Hash.new
            File.open("#{d.path}/#{f}", 'r') do |f1|
              while line = f1.gets
                if i == 0
                elsif i == 1 # second line, contains the header:
                  line = line.slice 1,line.size-1 # remove leading '#'
                  header = line.split ' '
                else
                  # a normal line
                  line = line.split ' '
                  line_hash = Hash.new
                  line.size.times{|j| line_hash[header[j]] = line[j]}
                  rows.push line_hash
                  header.size.times do |j|
                    columns[header[j]] ||= Array.new
                    columns[header[j]].push line[j]
                  end
                end
                i += 1
              end
            end
            results[result_key][File.basename f] = Hash.new
            results[result_key][File.basename f][:rows] = rows
            results[result_key][File.basename f][:columns] = columns
          end
        end
      end
      counter.increase
    end
    return results
  end

protected




  def apply_values counter
    counter.size.times do |i|
      @variation_objects[i].apply_value @process, counter[i], counter
      i += 1
    end
  end
end
