require "pp"

class ResultsObject < Hash
  def initialize variation
    @variation = variation
    super()
  end
  
  # filter_hash: a hash which maps parameter names to values or value arrays. Missing values
  #   are treated as wildcards
  # returns a list of keys matching the filter_hash
  def filter filter_hash, sort_list = filter_hash.keys
    matches = Array.new
    each_key do |key|
      match = true

      key.each_pair do |orig_key, orig_value|
        filter_value = filter_hash[orig_key]
        if filter_value.is_a? Array
          match = false unless filter_value.include?(orig_value.to_s) || filter_value.include?(orig_value)
        elsif filter_value != nil
          match = false unless filter_value == orig_value || filter_value == orig_value.to_s || filter_value == :all
        end
      end

      matches.push key if match

    end

    # sort results:
    matches.sort! do |key1,key2|
      return_value = 0
      sort_list.each do |parameter|
        # number comparison if parameter is a number; string comparison otherwise:
        unless key1[parameter] == key2[parameter]
          if key1[parameter].respond_to?(:to_f) && (key1[parameter].to_f != 0 || key2[parameter].to_f != 0)
            return_value = key1[parameter].to_f <=> key2[parameter].to_f unless return_value != 0
          else
            return_value = key1[parameter].to_s <=> key2[parameter].to_s unless return_value != 0
          end
        end
      end
      return_value
    end

    return matches
  end

  # Returns a list of all parameters used as keys.
  # Returns an empty list if this object is empty.
  # exclude_list: a single parameter or a list of parameters to be excluded from the result list.
  #   may also contain Regexp
  def all_parameters exclude_list = nil, key_list = self.keys
    exclude_list = [] unless exclude_list
    exclude_list = [exclude_list] unless exclude_list.is_a? Array
    return [] if size == 0
    result = key_list[0].keys.clone
    result.delete_if do |r|
      do_delete = false
      exclude_list.each do |p|
        do_delete = true if r == p
        do_delete = true if p.is_a?(Regexp) && p.match(r)
        do_delete = true if p == :redundant_process_locations && r.match(/\.process_location/) && result.include?( "#{r.match(/.*\./)[0]}base_process" )
      end
      do_delete
    end
    return result
  end

  # useful for results which are distributed over multiple files with only one row each.
  # Merges these values into one or more example tables with multiple rows.
  # First groups the results, so that the parameters in each group are identical besides x_parameter.
  # Then merges each group into one result, optionally sorting by sort_column.
  # The merged result is accessible via [result_file][:merged][x_parameter] in each value. This
  # implies of course, that all values which differ only in x_parameter share the same merged result.
  # x_parameter: the parameter which will be on the x-axis when plotting the merged result
  # result_file: the file whose values are merged
  # sort_column: a column in the result file, by which the merged table is sorted
  def merge_results x_parameter, result_file, sort_column = :x, remove_duplicates = true, filter_list = self.keys
    return self if size == 0
    parameters = all_parameters [x_parameter, :redundant_process_locations], filter_list
    puts "PARAMETERS:"
    PP.pp parameters
    puts "END PARAMETERS"
    groups = ResultsObject.group_keys filter_list, parameters
    merge_index = 0
    groups.each do |group|
      grouped_result = Hash.new
      grouped_result[:rows] = Array.new
      group[:keys].each do |group_key|
        values = self[group_key]
        if values
          if values[result_file]
            values[result_file][:rows].each do |row|
              grouped_result[:rows].push row.merge({:x => group_key[x_parameter]})
            end
          else
            puts "Warning: #{result_file} is missing"
          end
        end
        group_key["#{x_parameter}_merge_index"] = merge_index
      end
      self.rehash

      merge_index += 1
      # sort:
      grouped_result[:rows].sort!{|v1, v2| v1[sort_column] <=> v2[sort_column]} if sort_column

      # create columns:
      columns = Hash.new
      grouped_result[:rows].each do |row|
        row.each_pair do |column, value|
          columns[column] ||= []
          columns[column].push value
        end
      end
      grouped_result[:columns] = columns

      # add grouped results to each member of the group:
      group[:keys].each do |group_key|
        if self[group_key]
          self[group_key][result_file] ||= {}
          self[group_key][result_file][:merged] ||= {}
          self[group_key][result_file][:merged][x_parameter] = grouped_result
        end
      end
    end

    if remove_duplicates
      saved_merge_indices = []
      self.delete_if do |key, value|
        do_delete = false
        merge_index = key["#{x_parameter}_merge_index"]
        if merge_index
          if saved_merge_indices.include? merge_index
            do_delete = true
          else
            saved_merge_indices.push merge_index
          end
        end
        do_delete
      end
      self.keys.each{|key| key.delete x_parameter}
      self.rehash
    end


  end


  # partitions the keys, so that in each partition all parameters in parameter_list are constant.
  # That way you can e.g. group all results which have been generated by a specific classifier.
  # keys: a list of keys, as e.g. returned by filter(), or a group, as returned by this function
  # paramter_list: a list of parameter names in the form "operator_name.parameter_name",
  #   or a single parameter name
  def self.group_keys keys, parameter_list
    parameter_list = [parameter_list] unless parameter_list.is_a? Array
    keys = { :keys => keys, :parameters => {} } unless keys.is_a? Hash
    parameter = parameter_list[0]
    group_hash = Hash.new
    keys[:keys].each do |key|
      param_value = key[parameter]
      group_hash[param_value] ||= Hash.new
      group_hash[param_value][:keys] ||= Array.new
      group_hash[param_value][:keys].push key
      group_hash[param_value][:parameters] = keys[:parameters].clone
      group_hash[param_value][:parameters][parameter] = param_value
    end
    groups = group_hash.values
    
    if parameter_list.size > 1
      next_parameter_list = parameter_list.slice(1,parameter_list.size-1)
      new_groups = Array.new
      groups.each do |group|
        group_keys(group, next_parameter_list).each{|g| new_groups.push g}
      end
      groups = new_groups
    end

    return groups
  end

  protected

  def self.parse_filename filename
    re = /\/\w+\.\w+\.([-\/\w]+\.*\d+)\Z/
    results = {}
    while re.match filename
      m = re.match filename
      s = m[0]
      filename = filename.sub s, ''
      s = s[1,s.size-1]
      parts = s.split '.'
      parts[0].gsub! '_', ' '
      results["#{parts[0]}.#{parts[1]}"] = parts[2,parts.size-1].join('.')
    end
    results["base_process"] = filename
    return results
  end


end
