class ParameterValues
  def initialize operator, parameter, values = nil, parameter_prefix = nil
    @operator = operator
    @parameter = parameter

    @values = values || Array.new
    @parameter_prefix = parameter_prefix

    @base_variations = Array.new
  end

  attr_reader :values, :parameter, :operator

  # an XPath relative to the Operator, which leads to the XML-Element containing the parameters.
  # important e.g. for settings macros in the set_macros Operator. Here the XPath would be "list"
  attr_accessor :parameter_prefix, :base_variations


  
  def apply_value process, idx, counter
    value = @values[idx][0] if @values[idx].is_a? Array
    value ||= @values[idx]
    value = value.call counter if value.is_a? Proc
#    puts "applying #{to_s idx}"
    process.set_parameter(@operator, @parameter, value, @parameter_prefix)
  end

  def size
    @values.size
  end

  def value idx
    if @values[idx].is_a? Array
      return @values[idx][0]
    else
      return @values[idx]
    end
  end

  def value_name idx
    if @values[idx].is_a? Array
      return @values[idx][1]
    else
      return @values[idx].to_s
    end
  end

  def to_s idx = -1
    return "#{operator}.#{parameter} = [#{@values.join ', '}]" if idx == -1

    if @values[idx].is_a? Array
      s = @values[idx][1]
    else
      s = @values[idx]
    end
    return "#{operator}.#{parameter}.#{s}"
  end
end


# are not considered for the filename of the process
class DynamicParameterValues < ParameterValues
  def initialize operator, parameter, values = nil, parameter_prefix = nil
    @operator = operator
    @parameter = parameter

    @values = values || Array.new
    @parameter_prefix = parameter_prefix
  end

  def apply_value process, idx, counter
    value = @values[idx][0] if @values[idx].is_a? Array
    value ||= @values[idx]
    value = value.call counter if value.is_a? Proc
    #puts "applying #{to_s idx}"
    process.set_parameter(@operator, @parameter, value, @parameter_prefix)
  end

  def to_s idx = -1
    return "DynamicParameterValues"
  end
end


# a parameter which represents another process and thus induces dependencies on external process
# files.
class ProcessParameterValues < ParameterValues
  # base_path: specifies where to find the dependencies relative to the output_path of the
  #  Variation containing this ProcessParameterValues
  def initialize operator, parameter, values = nil, base_path = ""
    super operator, parameter, values
    @base_path = base_path
  end

  attr_accessor :base_path
end


# a set of ParameterValues, which are changed at the same time (compare "synchronized" parameter
# in RapidMiner's Loop Parameters operator).
# The contained ParameterValues must of course all have the same number of values.
#
# WARNING: currently, dependencies induced by ProcessParameterValues inside of
# SynchronizedParameterValues are not detected.
class SynchronizedParameterValues
  def initialize parameter_values = nil
    @parameter_values = parameter_values
    @parameter_values ||= Array.new
  end

  def operator
    return @parameter_values[0].operator
  end

  def parameter
    return @parameter_values[0].parameter
  end

  def value i
    return @parameter_values[0].value i
  end

  def base_variations
    bv = []
    @parameter_values.each{|pv| bv += pv.base_variations}
    return bv
  end

  # adds an instance of ParameterValues to this SynchronizedParameterValues.
  # If there is already an element in this object, values must have the same size as the previous
  # element.
  def add_values values
    size = values.size
    @parameter_values.each do |pv|
      raise "size does not match" if size != pv.size
    end
    @parameter_values.push values
  end

  # the number of different values in each ParameterValues
  def size
    return @parameter_values[0].size if @parameter_values.size > 0
    return 0
  end

  def apply_value process, idx, counter
    @parameter_values.each do |pv|
      pv.apply_value process, idx, counter
    end
  end

  # delegates to to_s() of the first ParameterValues
  def to_s idx = -1
    return @parameter_values[0].to_s idx if @parameter_values.size > 0
    return "empty SynchronizedParameterValues"
  end
end
