require 'find'

class JobFileCreator
  # remote_process_dir: the location of the process files on the server.
  #   The filenames returned from variation.filenames(false) must be
  #   relative to this directory during job execution
  # output_dir: the name of a job file for a process file is created from this scheme:
  #   output_dir/process_filename.job
  # pbs_job: a PBSJob template, where all variables besides process_file are set.
  #   process_file will be set automatically during create_job_files()
  def initialize remote_process_dir, output_dir, variation, pbs_job
    @remote_process_dir = remote_process_dir
    @output_dir = output_dir
    @variation = variation
    @pbs_job = pbs_job
  end

  attr_reader :remote_process_dir, :output_dir

  def variation
#    @variation.namespace = namespace.to_s
    return @variation
  end

  # creates the jobfiles in @output_dir
  #  log_file: write the process names (without prefix) into this file
  #  only_submitted: if true, files present in log_file are not created again
  #  include_list: jobs for processes listed in this list (without prefix or suffix) are
  #    created, without regard to only_submitted. In this case, these and only these files are
  #    created. If a file is in include_list, but not in the variation, an exception is raised.
  def create_job_files overwrite = false, state_list = nil, positive_states = nil
    process_files = @variation.filenames :none, false
    full_process_files = @variation.filenames :repository_name, false
    i = 0
    
    job_files = Array.new
    log_files = Array.new
    process_files.each do |rm_process|
      job_file_base = "#{rm_process}.job"
      job_file = "#{@output_dir}/#{job_file_base}"

      if !state_list || !state_list[rm_process] || positive_states.include?(state_list[rm_process])
        unless File.exists?(job_file) && !overwrite
          @pbs_job.process_file = full_process_files[i]
          @pbs_job.write_jobfile job_file
        end
        log_files.push "#{@pbs_job.log_directory}/#{rm_process}.error.log"
        job_files.push job_file_base
      end
      i += 1
    end

    unless job_files.empty?
      # create submit script
      s  = '#!/bin/sh'+"\n"

      log_files.each do |log_file|
        s += "mkdir -p '#{File.dirname(log_file)}'\n"
        s += "echo 'SUBMITTED' > '#{log_file}'\n"
      end

      job_files.each do |job_file|
        s += "qsub #{job_file}\n"
      end
      File.open("#{@output_dir}/submit_#{variation.namespace}#{File.basename(variation.process.base_process, ".rmp")}_#{Time.now.strftime("%Y-%m-%d_%H-%M-%S")}", "w") do |file|
        file.write s
        file.chmod 0740
      end
    end
  end

  def self.retrieve_results server_info, remote_directory, local_directory
    command  = "rsync -az"
    command += " --rsh='ssh -p #{server_info[:port]}'" if server_info[:port]
    command += " #{server_info[:user]}@#{server_info[:server]}"
    command += ":#{remote_directory}/"
    command += " #{local_directory}/"

    success = system(command)
    if !success
      throw "could not sync from the server. Command was:\n#{command}"
    end
  end

  # scans the given directory. Inside that directory subfolders named like the results from
  # @variation.filenames(:none). If inside those subfolders one of the files in indicator_files
  # is missing, the processing of the process named like the subfolder is considers erroneous.
  # Furthermore, if use_variation_filenames is true, directories returned by
  # @variation.filenames(:none) but are not a subdirectories of the given directory, are
  # added to the list of erroneous processes. Additionally subfolders not included in the variation
  # filenames are ignored
  def find_errors directory
    error_list = Hash.new

    process_names = @variation.filenames :none, false

    process_names.each do |process|
        error_log = "#{directory}/#{process}.error.log"
        if File.exists? error_log
          last_line = `tail -n2 #{error_log}`
          case
          when last_line.match(/(Process finished successfully|SUCCESS)/)
            state = :success
          when last_line.match(/Process not successful/)
            state = :error
          when last_line.match(/job killed/)
            state = :killed
          when last_line.match(/SUBMITTED/)
            state = :submitted
          when last_line.match(/RUNNING/)
            state = :running
          else
            state = :error
          end
        else
          state = :not_run
        end
        error_list[process] = state
    end
    return error_list
  end

  # replaces the log files of all processes which have finished successfully by a file containing
  # only the word "SUCCESS" to save disk space
  # directory: the log directory
  # error_list: an error list as returned by find_errors()
  def compact_logs directory, error_list
    error_list.each_pair do |filename, status|
      if status == :success
        error_log = "#{directory}/#{filename}.error.log"
        stdout_log = "#{directory}/#{filename}.stdout.log"
        File.open(error_log, "w"){|f| f.puts "SUCCESS"}
        File.open(stdout_log, "w"){|f| f.puts "SUCCESS"}
      elsif status == :error
        error_log = "#{directory}/#{filename}.error.log"
        stdout_log = "#{directory}/#{filename}.stdout.log"
        File.open(error_log, "w"){|f| f.puts "ERROR"}
        File.open(stdout_log, "w"){|f| f.puts "ERROR"}
      end
    end
  end


  # returns the names of all job files created by a call to create_job_files as Array
  def filenames
    result = Array.new
    process_files = @variation.filenames(false)
    process_files.each do |rm_process|
      job_file = "#{@output_dir}/#{rm_process}.job"
      result.push job_file
    end
    return result
  end

  # transfers the created job_files (i.e. the contents of @output_dir) to remote_directory
  # on the server specified by server_info
  # server_info: a Hash with the fields :user and :server and the optional field :port
  # remote_directory: the target directory on the server. Must already be existent
  def sync_jobfiles_to_server server_info, remote_directory
    command  = "rsync -az --delete"
    command += " --rsh='ssh -p #{server_info[:port]}'" if server_info[:port]
    command += " #{@output_dir}/"
    command += " #{server_info[:user]}@#{server_info[:server]}"
    command += ":#{remote_directory}"

    success = system(command)
    if !success
      throw "could not sync to the server. Command was:\n#{command}"
    end
  end

  def remote_process_filenames
    processes = @variation.filenames :none, true
    processes = prefix_array processes, "#{@remote_process_dir}/"
    return processes
  end
end
