package edu.udo.cs.mySVMdb.SVM;
import edu.udo.cs.mySVMdb.Optimizer.*;
import edu.udo.cs.mySVMdb.Container.*;
import edu.udo.cs.mySVMdb.Kernel.*;
import edu.udo.cs.mySVMdb.Util.*;
// import java.util.BitSet;
import java.lang.Integer;
import java.lang.Double;
import java.util.Random;
import java.lang.Math;

public abstract class SVM
{
    /**
     * Abstract base class for all SVMs
     * @author Stefan Rping
     * @version 1.0
     */
    

    protected Kernel the_kernel;
    protected JDBCDatabaseContainer the_container;
    protected int examples_total;
    protected int verbosity;
    protected int working_set_size;
    protected int parameters_working_set_size; // wss set in parameters
    protected int target_count;
    protected double convergence_epsilon;
    protected double is_zero;
    protected int shrink_const;
    protected double lambda_factor;
    protected int[] at_bound;
    protected double[] sum;
    protected boolean[] which_alpha;
    protected int[] working_set;
    protected double[] primal;
    protected double Cpos;
    protected double Cneg;
    protected double sum_alpha;
    protected double lambda_eq;
    protected double epsilon_pos;
    protected double epsilon_neg;
    protected int to_shrink;
    protected double feasible_epsilon;
    protected double lambda_WS;
    protected boolean quadraticLossPos;
    protected boolean quadraticLossNeg;
    protected double descend;
    boolean shrinked;

    MinHeap heap_min;
    MaxHeap heap_max;

    protected quadraticProblem qp;


    /**
     * class constructor
     */
    public SVM()
    {
    };
    
    
    /**
     * Init the SVM
     * @param Kernel new kernel function.
     * @param JDBCDatabaseContainer the data container
     * @exception Exception on any error
     */
    public void init(Kernel new_kernel, JDBCDatabaseContainer new_container)
    {
	String dummy;
	the_kernel = new_kernel;
	the_container = new_container;
	examples_total = the_container.count_examples();
	try{
	  verbosity = (new Integer(the_container.get_param("verbosity"))).intValue();
	}
	catch(Exception e){
	  verbosity = 3;
	};
	try{
	  working_set_size = (new Integer(the_container.get_param("working_set_size"))).intValue();
	  if(working_set_size < 2){
	      working_set_size = 2;
	  };
	}
	catch(Exception e){
	    working_set_size = 10; // !!! has to be identical to JDBCDatabaseContainer::prepareKisStatement
	};
	parameters_working_set_size = working_set_size;
	try{
	  is_zero = (new Double(the_container.get_param("is_zero"))).doubleValue();
	  if(is_zero <= 0){
	      is_zero = 1e-10;
	  };
	}
	catch(Exception e){
	  is_zero = 1e-10;
	};
	try{
	  convergence_epsilon = (new Double(the_container.get_param("convergence_epsilon"))).doubleValue();
	  if(convergence_epsilon <= 0){
	      convergence_epsilon = 1e-3;
	  };
	}
	catch(Exception e){
	  convergence_epsilon = 1e-3;
	};
	try{
	  shrink_const = (new Integer(the_container.get_param("shrink_const"))).intValue();
	  if(shrink_const <= 0){
	      shrink_const = 50;
	  };
	}
	catch(Exception e){
	  shrink_const = 50;
	};
	try{
	  descend = (new Double(the_container.get_param("descend"))).doubleValue();
	  if(descend < 0){
	      descend = 1e-15;
	  };
	}
	catch(Exception e){
	  descend = 1e-15;
	};
	quadraticLossPos = false;
	try{
	  dummy = the_container.get_param("quadraticLossPos");
	  if(dummy.equals("true")){
	      quadraticLossPos = true;
	  };
	}
	catch(Exception e){
	};
	quadraticLossNeg = false;
	try{
	  dummy = the_container.get_param("quadraticLossNeg");
	  if(dummy.equals("true")){
	      quadraticLossNeg = true;
	  };
	}
	catch(Exception e){
	};
	try{
	  Cpos = (new Double(the_container.get_param("C"))).doubleValue();
	  Cneg = Cpos;
	}
	catch(Exception e){
	    Cpos = 1.0;
	    Cneg = 1.0;
	};
	try{
	  Cpos = (new Double(the_container.get_param("Cpos"))).doubleValue();
	}
	catch(Exception e){
	};
	try{
	  Cneg = (new Double(the_container.get_param("Cneg"))).doubleValue();
	}
	catch(Exception e){
	};
	// better in subclass:
	try{
	  epsilon_pos = (new Double(the_container.get_param("epsilon"))).doubleValue();
	  epsilon_neg = epsilon_pos;
	}
	catch(Exception e){
	    epsilon_pos = 0.0;
	    epsilon_neg = 0.0;
	};
	try{
	  epsilon_pos = (new Double(the_container.get_param("epsilon_pos"))).doubleValue();
	}
	catch(Exception e){
	};
	try{
	  epsilon_neg = (new Double(the_container.get_param("epsilon_neg"))).doubleValue();
	}
	catch(Exception e){
	};
	try{
	    dummy = the_container.get_param("balance_cost");
	    if(dummy.equals("true")){
		Cpos *= ((double)the_container.count_pos_examples())/((double)the_container.count_examples());
		Cneg *= ((double)(the_container.count_examples()-the_container.count_pos_examples()))/((double)the_container.count_examples());
	    };
	}
	catch(Exception e){
	};

	//	System.out.println("Cpos "+Cpos);
	//	System.out.println("Cneg "+Cneg);
	//	System.out.println("epsilon_pos "+epsilon_pos);
	//	System.out.println("epsilon_neg "+epsilon_neg);

	lambda_factor = 1.0;
	lambda_eq=0;
	target_count=0;
	sum_alpha = 0;
	feasible_epsilon = convergence_epsilon;

	at_bound = new int[examples_total];
	sum = new double[examples_total];
	which_alpha = new boolean[examples_total];
	primal = new double[working_set_size];
    };
        

    /**
     * Train the SVM
     * @exception Exception on any error
     */
    public void train()
	throws Exception
    {
	target_count = 0;
	shrinked = false;
	init_optimizer();
	init_working_set();
	int iteration = 0;
	int max_iterations;
	try{
	  max_iterations = (new Integer(the_container.get_param("max_iterations"))).intValue();
	}
	catch(Exception e){
	    max_iterations=30000; 
	};
	boolean converged=false;
	//long time_train_loop = System.currentTimeMillis();
	//long time_dummy = 0;
	//long time_resetshrink = 0;
	M:while(iteration < max_iterations){
	    iteration++;
	    logln(4,"optimizer iteration "+iteration);
	    log(4,".");
	    optimize(); 
	    put_optimizer_values();
	    converged = convergence();
	    if(converged){
		logln(4,"");  // dots
		project_to_constraint();

		if(shrinked){
		    // check convergence for all alphas  
		    logln(2,"***** Checking convergence for all variables");
		    //		    time_resetshrink -= System.currentTimeMillis();
      		    reset_shrinked();
		    //		    time_resetshrink += System.currentTimeMillis();
		    converged = convergence();
		};
		
		if(converged){
		    logln(1,"*** Convergence");
		    break M;
		};
		
		// set variables free again
		shrink_const += 10;
		target_count = 0;
		for(int i=0;i<examples_total;i++){
		    at_bound[i]=0;
		};
	    };
	    shrink();
	    calculate_working_set();
	    update_working_set();
	};
	//time_train_loop = (System.currentTimeMillis() - time_train_loop)/1000;
	int i;
	if((iteration >= max_iterations) && (! converged)){
	  logln(1,"*** No convergence: Time up.");
	  if(shrinked){
	    // set sums for all variables for statistics
	    //time_resetshrink -= System.currentTimeMillis();
	    reset_shrinked();
	    //time_resetshrink += System.currentTimeMillis();
	  };
	};
	
	// calculate b
	double new_b=0;
	int new_b_count=0;
	double[] my_sum = sum;
	double[] my_y = the_container.get_ys();
	double[] my_alphas = the_container.get_alphas();
	for(i=0;i<examples_total;i++){
	  if((my_alphas[i]-Cneg < -is_zero) && 
	     (my_alphas[i] > is_zero)){
	    new_b +=  my_y[i] - my_sum[i]-epsilon_neg;
	    new_b_count++;
	  }
	  else if((my_alphas[i]+Cpos > is_zero) && 
		  (my_alphas[i] < -is_zero)){
	    new_b +=  my_y[i] - my_sum[i]+epsilon_pos;
	    new_b_count++;
	  };
	};
	
	if(new_b_count>0){
	  the_container.set_b(new_b/((double)new_b_count));
	}
	else{
	  // unlikely
	  for(i=0;i<examples_total;i++){
	    if((my_alphas[i]<is_zero) && 
	       (my_alphas[i]>-is_zero)) {
	      new_b += my_y[i] - my_sum[i];
	      new_b_count++;
	    };
	  };
	  if(new_b_count>0){
	    the_container.set_b(new_b/((double)new_b_count));
	  }
	  else{
	    // even unlikelier
	    for(i=0;i<examples_total;i++){
	      new_b += my_y[i] - my_sum[i];
	      new_b_count++;
	    };
	    the_container.set_b(new_b/((double)new_b_count));
	  };
	};
	
	if(verbosity>= 2){
	  logln(2,"Done training: "+iteration+" iterations.");
	  if(verbosity>= 3){
	    double now_target=0;
	    double now_target_dummy=0;
	    for(i=0;i<examples_total;i++){
	      now_target_dummy=sum[i]/2-the_container.get_y(i);
	      if(is_alpha_neg(i)){
		now_target_dummy+= epsilon_pos;
	      }
	      else{
		now_target_dummy-= epsilon_neg;
	      };
	      now_target+=the_container.get_alpha(i)*now_target_dummy;
	    };
	    logln(3,"Target function: "+now_target);
	  };
	};
	
	print_statistics();
	
	exit_optimizer();

	//	System.out.println("Time in resetshrink: "+(time_resetshrink/1000)+"s");
	//	System.out.println("Time in train loop: "+time_train_loop+"s");
    };
    

    /**
     * print statistics about result
     */
    protected void print_statistics()
	throws Exception
    {
      int dim = the_container.get_dim();
      int i,j;
      double alpha;
      double[] x;
      int svs=0;
      int bsv = 0;
      double mae=0;
      double mse = 0;
      int countpos = 0;
      int countneg = 0;
      double y;
      double prediction;
      double min_lambda = Double.MAX_VALUE;
      double b = the_container.get_b();
      for(i=0;i<examples_total;i++){
	  if(lambda(i) < min_lambda){
	      min_lambda = lambda(i);
	  };
	  y = the_container.get_y(i);
	  prediction = sum[i]+b;
	  mae += Math.abs(y-prediction);
	  mse += (y-prediction)*(y-prediction);
	  alpha = the_container.get_alpha(i);
	  if(y < prediction-epsilon_pos){
	      countpos++;
	  }
	  else if(y > prediction+epsilon_neg){
	      countneg++;
	  };
	  if(alpha != 0){
	      svs++;
	      if((alpha == Cpos) || (alpha == -Cneg)){
		  bsv++;
	      };
	  };
      };
      mae /= (double)examples_total;
      mse /= (double)examples_total;
      min_lambda = -min_lambda;
      logln(1,"Error on KKT is "+min_lambda);
      logln(1,svs+" SVs");
      logln(1,bsv+" BSVs");
      logln(1,"MAE "+mae);
      logln(1,"MSE "+mse);
      logln(1,countpos+" pos loss");
      logln(1,countneg+" neg loss");

      if(verbosity >= 2){
	  // print hyperplane
	  double[] w = new double[dim];
	  for(j=0;j<dim;j++) w[j] = 0;
	  for(i=0;i<examples_total;i++){
	      x = the_container.get_example(i);
	      alpha = the_container.get_alpha(i);
	      for(j=0;j<dim;j++){
		  w[j] += alpha*x[j];
	      };
	  };
	  double[] Exp = the_container.Exp;
	  double[] Dev = the_container.Dev;
	  if(Exp != null){
	      for(j=0;j<dim;j++){
		  if(Dev[j] != 0){
		      w[j] /= Dev[j];
		  };
		  if(0 != Dev[dim]){
		      w[j] *= Dev[dim];
		  };
		  b -= w[j]*Exp[j];
	      };
	      b += Exp[dim];
	  };
	  logln(2," ");
	  for(j=0;j<dim;j++){
	      logln(2,"w["+j+"] = "+w[j]);
	  };
	  logln(2,"b = "+b);
	  if(dim==1){
	      logln(2,"y = "+w[0]+"*x+"+b);
	  };
      };
    };


    /**
     * init the optimizer
     */
    protected void init_optimizer()
    {
      sum = new double[examples_total];
      at_bound = new int[examples_total];

      // init variables
      if(working_set_size>examples_total) working_set_size = examples_total;

      qp = new quadraticProblemSMO(is_zero/100,convergence_epsilon/100,
				   working_set_size*working_set_size);
      qp.set_n(working_set_size);
      
      which_alpha = new boolean[working_set_size];

      // reserve workspace for calculate_working_set
      working_set = new int[working_set_size];
      //      working_set_values = new double[working_set_size];

      heap_max = new MaxHeap(0);    
      heap_min = new MinHeap(0);    

      if(the_container.Dev != null){
	  double dev = the_container.Dev[the_container.get_dim()];
	  if(dev != 0){
	      epsilon_pos /= dev;
	      epsilon_neg /= dev;
	  };
      };
      
      int i;
      //      qp.l[i] = 0 done in svm_pattern_c::
      for(i=0;i<working_set_size;i++){
	  qp.l[i] = 0; //-is_zero; 
      };
      
      if(quadraticLossPos){
	Cpos = Double.MAX_VALUE;
      };
      if(quadraticLossNeg){ 
	Cneg = Double.MAX_VALUE;
      };

      lambda_WS = 0;
      to_shrink=0;
      
      qp.set_n(working_set_size);
    };


    /**
     * exit the optimizer
     */
    protected void exit_optimizer()
    {
	qp = null;
    };


    /**
     * shrink the variables
     */
    protected void shrink()
	throws Exception
    {
	// move shrinked examples to back
	if(to_shrink > examples_total/10){
	    int i;
	    int last_pos=examples_total;
	    if(last_pos - to_shrink > working_set_size){
		for(i=0;i<last_pos;i++){
		    if(at_bound[i] >= shrink_const){
			// shrink i
			sum_alpha += the_container.get_alpha(i);
			last_pos--;
			// keep the order of these swaps, kernel.swap depends on it!
			the_container.swap(i,last_pos);
			the_kernel.swap(i,last_pos);
			sum[i] = sum[last_pos];
			at_bound[i] = at_bound[last_pos];
			if(last_pos <= working_set_size){
			    break;
			};
		    };
		};
		to_shrink=0;
		shrinked = true;
		if(last_pos < examples_total){
		    examples_total = last_pos;
		    the_kernel.set_examples_size(examples_total);
		};
	    }
	    else{
		// cannot shrink because less than wss elements would be left
		// maybe other situation in shrink_const iterations?
		shrink_const *= 2;
	    };
	    logln(4,"shrinked to "+examples_total+" variables");
	};
    };

    
    /**
     * reset the shrinked variables
     */
    protected void reset_shrinked()
	throws Exception
    {
	int old_ex_tot=examples_total;
	double[] my_sum = sum;
	double[] alphas = the_container.get_alphas();
	target_count=0;
	examples_total = the_container.count_examples();
	the_kernel.set_examples_size(examples_total);
	// unshrink, recalculate sum for all variables 
	int i,j;
	// reset all sums
	double Kij;
	for(i=old_ex_tot;i<examples_total;i++){
	    my_sum[i] = 0;
	    at_bound[i] = 0;
	};
	double alpha;
	double[] kernel_row;
	int count=0;
	for(i=0;i<examples_total;i++){
	    alpha = alphas[i];
	    if(alpha != 0){
		count++;
		kernel_row = the_kernel.get_row(i);
		for(j=old_ex_tot;j<examples_total;j++){
		    my_sum[j]+=alpha*kernel_row[j];
		};
	    };
	};
	sum_alpha=0;
	shrinked = false;
	logln(5,"Resetting shrinked from "+old_ex_tot+" to "+examples_total);
    };
    
    
    /**
     * Project variables to constraints
     */
    protected void project_to_constraint()
	throws Exception
    {
      // project alphas to match the constraint
      double alpha_sum = sum_alpha;
      int SVcount=0;
      double alpha;
      double[] alphas = the_container.get_alphas();
      int i;
      for(i=0;i<examples_total;i++){
	alpha = alphas[i];
	alpha_sum += alpha;
	if(((alpha>is_zero) && (alpha-Cneg < -is_zero)) ||
	   ((alpha<-is_zero) && (alpha+Cpos > is_zero))){
	  SVcount++;
	};
      };
      if(SVcount > 0){
	// project
	alpha_sum /= (double)SVcount;
	for(i=0;i<examples_total;i++){
	  alpha = alphas[i];
	  if(((alpha>is_zero) && (alpha-Cneg < -is_zero)) ||
	     ((alpha<-is_zero) && (alpha+Cpos > is_zero))){
	    alphas[i] =  alpha-alpha_sum;
	  };
	};
      };
    }; 
    
    
    /**
     * Calculates the working set
     * @exception Exception on any error
     */
    protected void calculate_working_set()
      throws Exception
    {
      // reset WSS
      if(working_set_size < parameters_working_set_size){
	working_set_size = parameters_working_set_size;
	if(working_set_size>examples_total) working_set_size = examples_total;
      };

      heap_min.init(working_set_size/2);
      heap_max.init(working_set_size/2+working_set_size%2);

      int i=0;
      double sort_value;

      double the_nabla;
      boolean is_feasible;
      int atbound;
      int j;

      while(i<examples_total) {
	is_feasible = feasible(i);
	if(is_feasible){
	  the_nabla = nabla(i);
	  if(is_alpha_neg(i)){
	    sort_value = -the_nabla;  // - : maximum inconsistency approach
	  }
	  else{
	    sort_value = the_nabla;
	  };
	  // add to heaps
	  heap_min.add(sort_value,i);
	  heap_max.add(sort_value,i);
	};
	i++;
      };

      int[] new_ws = heap_min.get_values();
      working_set_size = 0;
      int pos;
      j = heap_min.size();
      for(i=0;i<j;i++){
	working_set[working_set_size] = new_ws[i];
	working_set_size++;
      };
      pos = working_set_size;
      new_ws = heap_max.get_values();
      j = heap_max.size();
      for(i=0;i<j;i++){
	working_set[working_set_size] = new_ws[i];
	working_set_size++;
      };
      if((! heap_min.empty()) &&
	 (! heap_max.empty())){
	if(heap_min.top_value() >= heap_max.top_value()){
	  // there could be the same values in the min- and maxheap,
	  // sort them out (this is very unlikely)
	  j=0;
	  i=0;
	  while(i<pos){
	    // working_set[i] also in max-heap?
	    j=pos;
	    while((j<working_set_size) && 
		  (working_set[j] != working_set[i])){
	      j++;
	    };
	    if(j<working_set_size){
	      // working_set[i] equals working_set[j]
	      // remove j from WS
	      working_set[j] = working_set[working_set_size-1];
	      working_set_size--;
	    }
	    else{
	      i++;
	    };
	  };
	};
      };

      if(target_count > 0){
	// convergence error on last iteration?
	// some more tests on WS
	// unlikely to happen, so speed isn't so important
	
	// are all variables at the bound?
	int pos_abs;
	boolean bounded_pos=true;
	boolean bounded_neg=true;
	pos=0;
	double alpha;
	double[] alphas = the_container.get_alphas();
	while((pos<working_set_size) && (bounded_pos || bounded_neg)){
	  pos_abs = working_set[pos];
	  alpha = alphas[pos_abs];
	  if(is_alpha_neg(pos_abs)){
	    if(alpha-Cneg < -is_zero){
	      bounded_pos = false;
	    };
	    if(alpha > is_zero){
	      bounded_neg = false;
	    };
	  }
	  else{
	    if(alpha+Cneg > is_zero){
	      bounded_neg = false;
	    };
	    if(alpha < -is_zero){
	      bounded_pos = false;
	    };
	  };
	  pos++;
	};
	if(bounded_pos){
	  // all alphas are at upper bound
	  // need alpha that can be moved upward
	  // use alpha with smallest lambda
	  double max_lambda = Double.MAX_VALUE;
	  int max_pos=examples_total;
	  for(pos_abs=0;pos_abs<examples_total;pos_abs++){
	    alpha = alphas[pos_abs];
	    if(is_alpha_neg(pos_abs)){
	      if(alpha-Cneg < -is_zero){
		if(lambda(pos_abs) < max_lambda){
		  max_lambda = lambda(pos_abs);
		  max_pos = pos_abs;
		};
	      };
	    }
	    else{
	      if(alpha < -is_zero){
		if(lambda(pos_abs) < max_lambda){
		  max_lambda = lambda(pos_abs);
		  max_pos = pos_abs;
		};
	      };
	    };
	  };
	  if(max_pos<examples_total){
	    if(working_set_size<parameters_working_set_size){
	      working_set_size++;
	    };
	    working_set[working_set_size-1] = max_pos;
	  };
	}
	else if(bounded_neg){
	  // all alphas are at lower bound
	  // need alpha that can be moved downward
	  // use alpha with smallest lambda
	  double max_lambda = Double.MAX_VALUE;
	  int max_pos=examples_total;
	  for(pos_abs=0;pos_abs<examples_total;pos_abs++){
	    alpha = alphas[pos_abs];
	    if(is_alpha_neg(pos_abs)){
	      if(alpha > is_zero){
		if(lambda(pos_abs) < max_lambda){
		  max_lambda = lambda(pos_abs);
		  max_pos = pos_abs;
		};
	      };
	    }
	    else{
	      if(alpha+Cneg > is_zero){
		if(lambda(pos_abs) < max_lambda){
		  max_lambda = lambda(pos_abs);
		  max_pos = pos_abs;
		};
	      };
	    };
	  };
	  if(max_pos<examples_total){
	    if(working_set_size<parameters_working_set_size){
	      working_set_size++;
	    };
	    working_set[working_set_size-1] = max_pos;
	  };
	};
      };

      if((working_set_size<parameters_working_set_size) &&
	 (working_set_size<examples_total)){
	// use full working set
	pos = (int)(Math.random()*(double)examples_total);
	int ok;
	while((working_set_size<parameters_working_set_size) &&
	      (working_set_size<examples_total)){
	  // add pos into WS if it isn't already
	  ok = 1;
	  for(i=0;i<working_set_size;i++){
	    if(working_set[i] == pos){
	      ok=0;
	      i = working_set_size;
	    };
	  };
	  if(1 == ok){
	    working_set[working_set_size] = pos;
	    working_set_size++;
	  };
	  pos = (pos+1)%examples_total;
	};
      };
      
      int ipos;
      for(ipos=0;ipos<working_set_size;ipos++){
	which_alpha[ipos] = is_alpha_neg(working_set[ipos]);
      };
    };
    

    /**
     * Updates the working set
     */
    protected void update_working_set()
	throws Exception
    {
      // setup subproblem
      int i,j;
      int pos_i, pos_j;
      double[] kernel_row;
      double sum_WS;
      double[] alphas = the_container.get_alphas();   
      double[] ys = the_container.get_ys();
      double[] my_sum = sum;
      boolean[] my_which_alpha = which_alpha;
      // the_kernel.prefetch(working_set,which_alpha);
      for(pos_i=0;pos_i<working_set_size;pos_i++){
	i = working_set[pos_i];
	
	// put row sort_i in hessian 
	kernel_row = the_kernel.get_row(i);

	sum_WS=0;
	//    for(pos_j=0;pos_j<working_set_size;pos_j++){
	for(pos_j=0;pos_j<pos_i;pos_j++){
	  j = working_set[pos_j];
	  // put all elements K(i,j) in hessian, where j in WS
	  if(((! my_which_alpha[pos_j]) && (! my_which_alpha[pos_i])) ||
	     ((my_which_alpha[pos_j]) && (my_which_alpha[pos_i]))){
	    // both i and j positive or negative
	    (qp.H)[pos_i*working_set_size+pos_j] = kernel_row[j];
	    (qp.H)[pos_j*working_set_size+pos_i] = kernel_row[j];
	  }
	  else{
	    // one of i and j positive, one negative
	    (qp.H)[pos_i*working_set_size+pos_j] = -kernel_row[j];
	    (qp.H)[pos_j*working_set_size+pos_i] = -kernel_row[j];
	  };
	};
	for(pos_j=0;pos_j<working_set_size;pos_j++){
	  j = working_set[pos_j];
	  sum_WS+=alphas[j]*kernel_row[j];
	};
	// set main diagonal 
	(qp.H)[pos_i*working_set_size+pos_i] = kernel_row[i];
	
	// linear and box constraints
	if(! my_which_alpha[pos_i]){
	  // alpha
	  (qp.A)[pos_i] = -1;
	  // lin(alpha) = y_i+eps-sum_{i not in WS} alpha_i K_{ij}
	  //            = y_i+eps-sum_i+sum_{i in WS}
	  (qp.c)[pos_i] = ys[i]+epsilon_pos-my_sum[i]+sum_WS;
	  primal[pos_i] = -alphas[i];
	  (qp.u)[pos_i] = Cpos;
	}
	else{
	  // alpha^*
	  (qp.A)[pos_i] = 1;
	  (qp.c)[pos_i] = -ys[i]+epsilon_neg+my_sum[i]-sum_WS;
	  primal[pos_i] = alphas[i];
	  (qp.u)[pos_i] = Cneg;
	};
      };
      if(quadraticLossNeg){
	for(pos_i=0;pos_i<working_set_size;pos_i++){
	  if(my_which_alpha[pos_i]){
	    (qp.H)[pos_i*(working_set_size+1)] += 1/Cneg;
	    (qp.u)[pos_i] = Double.MAX_VALUE;
	  };
	};
      };
      if(quadraticLossPos){
	for(pos_i=0;pos_i<working_set_size;pos_i++){
	  if(! my_which_alpha[pos_i]){
	    (qp.H)[pos_i*(working_set_size+1)] += 1/Cpos;
	    (qp.u)[pos_i] = Double.MAX_VALUE;
	  };
	};
      };
    };


    /**
     * Initialises the working set
     * @exception Exception on any error
     */
    protected void init_working_set()
      throws Exception
    {
      // calculate sum
      int i,j;
      
      project_to_constraint();
      // check bounds!
      if(the_container.initialised_alpha()){
	logln(2,"Initialising variables, this may take some time.");
	for(i=0; i<examples_total;i++){
	  sum[i] = 0;
	  at_bound[i] = 0;
	  for(j=0; j<examples_total;j++){
	    sum[i] += the_container.get_alpha(j)*the_kernel.calculate_K(i,j);
	  };
	};
      }
      else{
	// skip kernel calculation as all alphas = 0
	for(i=0; i<examples_total;i++){
	  sum[i] = 0;
	  at_bound[i] = 0;
	};    
      };
      
      if(the_container.initialised_alpha()){
	calculate_working_set();
      }
      else{
	// first working set is random
	j=0;
	i=0;

	while((i<working_set_size) && (j < examples_total)){
	  working_set[i] = j;
	  if(is_alpha_neg(j)){
	    which_alpha[i] = true;
	  }
	  else{
	    which_alpha[i] = false;
	  };
	  i++;
	  j++;
	};
      };   
      update_working_set();
    };

    
    /**
     * Calls the optimizer
     */
    protected abstract void optimize() throws Exception;
    
    
    /**
     * Stores the optimizer results
     */
    protected void put_optimizer_values()
	throws Exception
    {
      // update nabla, sum, examples.
      // sum[i] += (primal_j^*-primal_j-alpha_j^*+alpha_j)K(i,j)
      // check for |nabla| < is_zero (nabla <-> nabla*)
      //  cout<<"put_optimizer_values()"<<endl;
      int i=0; 
      int j=0;
      int pos_i;
      double the_new_alpha;
      double[] kernel_row;
      double alpha_diff;
      double my_sum[] = sum;
      double[] alphas = the_container.get_alphas();

      pos_i=working_set_size;
      while(pos_i>0){
	pos_i--;
	if(which_alpha[pos_i]){
	  the_new_alpha = primal[pos_i];
	}
	else{
	  the_new_alpha = -primal[pos_i];
	};
	// next three statements: keep this order!
	i = working_set[pos_i];
	alpha_diff = the_new_alpha-alphas[i];
	alphas[i] = the_new_alpha;
	
	if(alpha_diff != 0){
	  // update sum ( => nabla)
	  kernel_row = the_kernel.get_row(i);
	  for(j=examples_total-1;j>=0;j--){
	      my_sum[j] += alpha_diff*kernel_row[j];
	  };
	};
      };
    };
    
    
    /**
     * Checks if the optimization converged
     * @return boolean true optimzation if converged
     */
    protected boolean convergence()
	throws Exception
    {
      double the_lambda_eq = 0;
      int total = 0;
      double alpha_sum=0;
      double alpha=0;
      int i;
      boolean result = true;
      double[] alphas = the_container.get_alphas();
      // actual convergence-test
      total = 0; alpha_sum=0;
      
      for(i=0;i<examples_total;i++){
	alpha = alphas[i];
	alpha_sum += alpha;
	if((alpha>is_zero) && (alpha-Cneg < -is_zero)){
	  // alpha^* = - nabla
	  the_lambda_eq += -nabla(i); //all_ys[i]-epsilon_neg-sum[i];
	  total++;
	}
	else if((alpha<-is_zero) && (alpha+Cpos > is_zero)){
	  // alpha = nabla
	  the_lambda_eq += nabla(i); //all_ys[i]+epsilon_pos-sum[i];
	  total++;
	};
      };

      logln(4,"lambda_eq = "+(the_lambda_eq/total));
      if(total>0){
	lambda_eq = the_lambda_eq / total;
      }
      else{
	// keep WS lambda_eq
	lambda_eq = lambda_WS; //(lambda_eq+4*lambda_WS)/5;
	logln(4,"*** no SVs in convergence(), lambda_eq = "+lambda_eq+".");
      };

      if(target_count>2){
	// estimate lambda from WS
	if(target_count>20){
	  // desperate attempt to get good lambda!
	  lambda_eq = ((40-target_count)*lambda_eq + (target_count-20)*lambda_WS)/20;
	  logln(5,"Re-Re-calculated lambda from WS: "+lambda_eq);
	  if(target_count>40){
	    // really desperate, kick one example out!
	    i = working_set[target_count%working_set_size];
	    if(is_alpha_neg(i)){
	      lambda_eq = -nabla(i);
	    }
	    else{
	      lambda_eq = nabla(i);
	    };
	    logln(5,"set lambda_eq to nabla("+i+"): "+lambda_eq);
	  };
	}
	else{
	  lambda_eq = lambda_WS;
	  logln(5,"Re-calculated lambda_eq from WS: "+lambda_eq);
	};
      };

      // check linear constraint
      if(java.lang.Math.abs(alpha_sum+sum_alpha) > convergence_epsilon){
	// equality constraint violated
	logln(4,"No convergence: equality constraint violated: |"+(alpha_sum+sum_alpha)+"| >> 0");
	project_to_constraint();
	result = false;  
      };
      
      i=0;
      while((i<examples_total) && (result != false)){
	if(lambda(i)>=-convergence_epsilon){
	  i++;
	}
	else{
	    result = false;
	};
      };

      return result;
    };
    

    protected abstract double nabla(int i);


    /**
     * lagrangion multiplier of variable i
     * @param int variable index
     * @return lambda
     */
    protected double lambda(int i)
    {
      double alpha;
      double result;
      if(is_alpha_neg(i)){
	result = - java.lang.Math.abs(nabla(i)+lambda_eq);
      }
      else{
	result = - java.lang.Math.abs(nabla(i)-lambda_eq);
      };
      // default = not at bound
      
      alpha=the_container.get_alpha(i);
      
      if(alpha>is_zero){
	// alpha*
	if(alpha-Cneg >= - is_zero){
	  // upper bound active
	  result = -lambda_eq-nabla(i);
	};
      }
      else if(alpha >= -is_zero){
	// lower bound active
	if(is_alpha_neg(i)){
	  result = nabla(i) + lambda_eq;
	}
	else{
	  result = nabla(i)-lambda_eq;
	};
      }
      else if(alpha+Cpos <= is_zero){
	// upper bound active
	result = lambda_eq - nabla(i);
      };
      return result;
    };


    protected boolean feasible(int i)
	throws Exception
    {
      boolean is_feasible = true;
      double alpha = the_container.get_alpha(i);
      double the_lambda = lambda(i);
      
      if(alpha-Cneg >= - is_zero){
	// alpha* at upper bound
	if(the_lambda >= 0){
	  at_bound[i]++;
	  if(at_bound[i] == shrink_const) to_shrink++;
	}
	else{
	  at_bound[i] = 0;
	};
      }
      else if((alpha<=is_zero) && (alpha >= -is_zero)){
	// lower bound active
	if(the_lambda >= 0){
	  at_bound[i]++;
	  if(at_bound[i] == shrink_const) to_shrink++;
	}
	else{
	  at_bound[i] = 0;
	};
      }
      else if(alpha+Cpos <= is_zero){
	// alpha at upper bound
	if(the_lambda >= 0){
	  at_bound[i]++;
	  if(at_bound[i] == shrink_const) to_shrink++;
	}
	else{
	  at_bound[i] = 0;
	};
      }
      else{
	// not at bound
	at_bound[i] = 0;
      };
      if((the_lambda >= feasible_epsilon) || (at_bound[i] >= shrink_const)){
	is_feasible = false; 
      };
      return is_feasible;
    };


    protected abstract boolean is_alpha_neg(int i);


    /**
     * logs the outputs
     * @param int warning level
     * @param String Message test
     */
    protected void log(int level, String message)
    {
	if(level <= verbosity){
	    System.out.print(message);
	    System.out.flush();
	};
    };

    
    /**
     * log the output plus newline
     * @param int warning level
     * @param String Message test
     */
    protected void logln(int level, String message)
    {
	if(level <= verbosity){
	    System.out.println(message);
	};
    };

  
    /**
     * predict values on the testset with model
     */
    public void predict()
	throws Exception
    {
	int i;
	double prediction;
	double[] example;
	int size = the_container.count_test_examples();
	for(i=0;i<size;i++){
	    example = the_container.get_test_example(i);
	    prediction = predict(example);
	    the_container.set_test_y(i,prediction);
	};
	logln(4,"Prediction generated");
    };


    /**
     * predict a single example
     */
    protected double predict(double[] example)
	throws Exception
    {
	int i;
	double[] sv;
	double the_sum=the_container.get_b();
	double alpha;
	for(i=0;i<examples_total;i++){
	    alpha = the_container.get_alpha(i);
	    if(alpha != 0){
		sv = the_container.get_example(i);
		the_sum += alpha*the_kernel.calculate_K(sv,example);
	    };
	};
	if(the_container.Exp != null){
	    // unscale prediction
	    int dim = the_container.get_dim();
	    double exp = the_container.Exp[dim];
	    double dev = the_container.Dev[dim];
	    if(dev == 0){
		dev = 1;
	    };
	    the_sum = the_sum*dev+exp;
	};
	return the_sum;
    };


    /**
     * check internal variables
     */
    protected void check()
	throws Exception
    {
	double tsum;
	int i,j;
	double s=0;
	for(i=0; i<examples_total;i++){
	  s += the_container.get_alpha(i);
	  tsum = 0;
	  for(j=0; j<examples_total;j++){
	    tsum += the_container.get_alpha(j)*the_kernel.calculate_K(i,j);
	  };
	  if(Math.abs(tsum-sum[i]) > is_zero){
	      logln(1,"ERROR: sum["+i+"] off by "+(tsum-sum[i]));
	      throw(new Exception("ERROR: sum["+i+"] off by "+(tsum-sum[i])));
	  };
	};
	if(Math.abs(s+sum_alpha) > is_zero){
	    logln(1,"ERROR: sum_alpha is off by "+(s+sum_alpha));
	    throw(new Exception("ERROR: sum_alpha is off by "+(s+sum_alpha)));
	};
    };

};
