package edu.udo.cs.mySVMdb.Kernel;
import edu.udo.cs.mySVMdb.Container.JDBCDatabaseContainer;
import edu.udo.cs.mySVMdb.Util.Cache;
import java.util.HashMap;
import java.lang.Integer;
import java.sql.*;


public abstract class Kernel
{
    /**
     * Abstract base class for all kernels.
     * @author Stefan Rping
     * @version 1.0
     */

    /**
     * Text that executes the kernel select
     */
    String select_text;

    public String get_select_text(){ return select_text; };
    
    /**
     * Container for the examples, parameters etc.
     */
    protected JDBCDatabaseContainer the_examples;

    /**
     * dimension of the examples
     */
    protected int dim;

    //     public long kernel_time;
    //     public long kernel_time2;
    //     public long kernel_time3;
    //     public long kernel_time4;
    /**
     * Kernel cache
     */
    protected Cache kernel_cache;
    
    /**
     * Number of elements in cache
     */
    protected int kernel_cache_size;
    
    /**
     * Size of cache in MB
     */
    protected int cache_MB;

    /**
     * number of examples after shrinking
     */
    protected int examples_total;


    /**
     * prepared Statement to get K(i,j) from database
     */
    protected PreparedStatement Kij_query;


    /**
     * prepared Statement to get row i  from database
     */
    protected PreparedStatement Ki_query;


    /**
     * prepared Statement to get rows i1...ik  from database
     */
    protected PreparedStatement Kis_query;


    protected String[] keys;
    HashMap keyNumber;


    /**
     * Class constructor
     */
    public Kernel(){};


    /**
     * Class constructor
     * @param examples Container for the examples.
     */
    public Kernel(JDBCDatabaseContainer examples){
	init(examples);
    };


    /**
     * Output as String
     */
    public String toString(){
	return("abstract class");
    };


    /**
     * Init the kernel
     * @param examples Container for the examples.
     */
    public void init(JDBCDatabaseContainer examples){
	//	kernel_time=0;
	//	kernel_time2=0;
	//	kernel_time3=0;
	//	kernel_time4=0;
	the_examples = examples;
	examples_total = the_examples.count_examples();
	dim = the_examples.get_dim();
	int cacheMB;
	try{
	    cacheMB = (new Integer(the_examples.get_param("kernel_cache"))).intValue();
	}
	catch(Exception e){
	    cacheMB = 40;
	};
	init_kernel_cache(cacheMB);

	keys = examples.keys;
	// init keyNumber
	int nr = examples.count_examples();
	keyNumber = new HashMap(2*nr);
	Integer i_obj;
	for(int i=0;i<nr;i++){
	    i_obj = new Integer(i);
	    keyNumber.put(keys[i],i_obj);
	};
    };


    /**
     * Calculates kernel value of vectors x and y
     */
    public abstract double calculate_K(double[] x, double[] y);


    /**
     * Gets a kernel row
     */
    public double[] get_row(int i)
    {
	long my_time = System.currentTimeMillis();
	double[] result = null;
	result = ((double[])kernel_cache.get_element(i));
	if(result == null){
	    // get last cache element, don't assign new memory
	    result = (double[])kernel_cache.get_lru_element();
	    if(result == null){
		result = new double[examples_total];
	    };
	    calculate_K_row(result,i);
	    kernel_cache.put_element(i,result);
	};
	//	kernel_time += System.currentTimeMillis() - my_time;
	return result;
    };


    /**
     * Puts the requestet rows in cache, if possible
     * @param rows ids of the rows to be pre-fetched
     * @param which_alpha order to be kept consistent with rows
     */
    public void prefetch(int[] rows, boolean[] which_alpha)
	throws Exception
    {
	long my_time = System.currentTimeMillis();
	int to_be_fetched = rows.length; // = working_set_size
	//	System.out.println("wss: "+to_be_fetched);
	int i = 0;
	int already_cached = 0;
	int idummy;
	boolean bdummy;
	// see if any rows are already in cache
	for(i=0;i<to_be_fetched;i++){
	    if(kernel_cache.cached(rows[i])){
		kernel_cache.renew(rows[i]);
		if(i > already_cached){
		    // swap i and already_cached
		    idummy = rows[i];
		    rows[i] = rows[already_cached];
		    rows[already_cached] = idummy;
		    bdummy = which_alpha[i];
		    which_alpha[i] = which_alpha[already_cached];
		    which_alpha[already_cached] = bdummy;
		};
		already_cached++;		    
	    };
	};
	//	System.out.println("cached: "+already_cached);
	// caches rows are in pos 0..already_cached-1
	// don't fetch more rows than can be cached
	if(to_be_fetched > kernel_cache_size){
	    to_be_fetched = kernel_cache_size;
	};
	//	System.out.println("fetch: "+to_be_fetched);
	// fetch rows [already_cached..to_be_fetched[
	if(already_cached<to_be_fetched){
	    try{
		for(i=1;i<=already_cached;i++){
		    // put dummy in SQL-getrows-statement
		    //		    		    System.out.println(i+" -> "+keys[rows[already_cached]]);
		    Kis_query.setString(i,keys[rows[already_cached]]);
		};
		Object[] results = new Object[to_be_fetched];
		for(i=already_cached;i<to_be_fetched;i++){
		    // put i in SQL-getrows-statement
		    //		    		    System.out.println((i+1)+" -> "+keys[rows[i]]);
		    Kis_query.setString(i+1,keys[rows[i]]);
		    // prepare cache
		    results[i] = new double[examples_total];
		    kernel_cache.put_element(rows[i],(double[])(results[i]));
		};
		// fetch rows
		//		kernel_time4 -= System.currentTimeMillis();
		ResultSet rset = Kis_query.executeQuery();
		//		kernel_time4 += System.currentTimeMillis();
		// put in cache
		String the_x_key;
		String the_y_key;
		double the_val;
		i = already_cached;
		double[] result = (double[])results[i];
		while(rset.next()){
		    the_val = rset.getDouble(1);
		    the_x_key = rset.getString(2);
		    the_y_key = rset.getString(3);
		    while(! the_x_key.equals(keys[rows[i]])){
			i++;
			if(i >= to_be_fetched){
			    i = 0;
			};
			result = (double[])results[i];
		    };
		    result[((Integer)(keyNumber.get(the_y_key))).intValue()] = the_val;
		};    
		rset.close();
	    }
	    catch(Exception e){
		throw(new RuntimeException(e.getMessage()));
	    };
	};
	//	kernel_time3 += System.currentTimeMillis() - my_time;
    };


    /**
     * Inits the kernel cache.
     * @param Size of the cache in MB
     */
    public void init_kernel_cache(int size){
	cache_MB = size;
	// array of train_size doubles
	kernel_cache_size = size * 1048576 / 4 / examples_total;
	if(kernel_cache_size < 1){
	    kernel_cache_size = 1;
	};
	if(kernel_cache_size > the_examples.count_examples()){
	    kernel_cache_size = the_examples.count_examples();
	};
	//	System.out.println("Setting cache size to "+kernel_cache_size);
	kernel_cache = new Cache(kernel_cache_size,examples_total);
    };
  
  
    /**
     * Sets the number of examples to new value
     */
    public void set_examples_size(int new_examples_total)
	throws Exception
    {
	// number of rows that  fit into cache:
	int new_kernel_cache_size = cache_MB * 1048576 / 4 / new_examples_total;
	if(new_kernel_cache_size < 1){
	    new_kernel_cache_size = 1;
	};
	if(new_kernel_cache_size > new_examples_total){
	    new_kernel_cache_size = new_examples_total;
	};
	
	// kernel_cache = new Cache(kernel_cache_size);

	if(new_examples_total < examples_total){
	    // keep cache
	   kernel_cache.shrink(new_kernel_cache_size,new_examples_total);
	   the_examples.shrink(examples_total,new_examples_total);
	}
	else if(new_examples_total > examples_total){
	    kernel_cache.init(new_kernel_cache_size);
	    the_examples.shrink(examples_total,new_examples_total);
	};
	kernel_cache_size = new_kernel_cache_size;
	examples_total = new_examples_total;
    };


    /**
     * Calculate K(i,j) as SQL query
     */
    public double calculate_K(int i, int j)
    {
	double result;
	try{
	    Kij_query.setString(1,keys[i]);
	    Kij_query.setString(2,keys[j]);
	    ResultSet rset = Kij_query.executeQuery();
	    rset.next();
	    result = rset.getDouble(1);
	    rset.close();
	}
	catch(Exception e){
	    throw(new RuntimeException(e.getMessage()));
	};
	return result;
    };


    public double[] calculate_K_row(double[] result, int i)
    {
	try{

	    Ki_query.setString(1,keys[i]);

	    //	    kernel_time2 -= System.currentTimeMillis();
	    ResultSet rset = Ki_query.executeQuery();
	    //	    kernel_time2 += System.currentTimeMillis();
	    
	    String the_key;
	    double the_val;
	    //	kernel_time3 -= System.currentTimeMillis();
	    while(rset.next()){
		//		result[((Integer)(keyNumber.get(rset.getString(2)))).intValue()] =
		//    rset.getDouble(1);
		//	kernel_time4 -= System.currentTimeMillis();
		the_val = rset.getDouble(1);
		the_key = rset.getString(2);
		result[((Integer)(keyNumber.get(the_key))).intValue()] = the_val;
		//	kernel_time4 += System.currentTimeMillis();
	    };
	    //	kernel_time3 += System.currentTimeMillis();
	    rset.close();

	}
	catch(Exception e){
	    throw(new RuntimeException(e.getMessage()));
	};
	return result;
    };


    /**
     * swap two training examples
     * @param pos1
     * @param pos2
     */
    public void swap(int pos1, int pos2)
    {
	// called after container swap
	kernel_cache.swap(pos1,pos2);
	Integer i_obj1 = (Integer)keyNumber.get(keys[pos2]);
	Integer i_obj2 = (Integer)keyNumber.get(keys[pos1]);
	keyNumber.remove(keys[pos1]);
	keyNumber.remove(keys[pos2]);
	//	Integer i_obj1 = new Integer(pos1);
	//	Integer i_obj2 = new Integer(pos2);
	keyNumber.put(keys[pos1],i_obj1);
	keyNumber.put(keys[pos2],i_obj2);
    };
};
