package hitters.tools;

import hitters.multi.DimType;
import hitters.multi.Element;
import hitters.multi.MultiDatabase;
import hitters.multi.SysElement;
import hitters.multi.SysParameter;

import java.io.File;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import com.google.common.collect.HashMultiset;
import com.rapidminer.operator.UserError;


/**
 * 
 * Just a collection of some helper methods. In particular, this
 * class is used to create a code describing a subset of all 
 * system calls. This is used to create unique names for the files 
 * which are used to cache the Hierarchical Heavy Hitters in RapidMiner: 
 * The filenames must contain the parameters used to create the HHH 
 * which are stored in file.
 * 
 * @author Peter Fricke
 * @version $Id$
 * 
 */
public class Utils {

	
	//The code for caching must remain valid even if the taxonomy is changed.
	//Therefore, the order of the calls is fixed.
	//A change in the taxonomy (adding or removing calls) is handled 
	//like this: Removed calls remain in the String, new calls are 
	//added at the end of the String. This requires a change in the 
	//source code. I might implement a more flexible version later. 
	//For the moment, I just want to make sure I _never_ use an invalid cache.
	static{ 
		String fixedCalls = "_llseek;accept;accept4;access;brk;clock_gettime;clone;close;dup;dup2;dup3;execve;fcntl;fcntl64;fork;fstat;fstat64;futex;getdents;getdents64;gettimeofday;inotify_init;inotify_init1;ioctl;lseek;lstat;lstat64;mmap2;mprotect;munmap;nanosleep;open;pipe;pipe2;poll;read;readlink;recv;recvmsg;rt_sigaction;select;send;sendfile;sendmsg;sendto;socket;stat;stat64;time;uname;unshare;vfork;write;writev";
		String[] fixed = fixedCalls.split( ";" );
		list = Arrays.asList( fixed );
	}
	private static final List<String> list;
	
	
	/**
	 * Creates a code describing the calls in the given set. This is used
	 * to create unique names for the files which are used to 
	 * cache the Hierarchical Heavy Hitters in RapidMiner. 
	 * 
	 * It is guaranteed that the code remains valid even if the 
	 * taxonomy is changed.
	 * 
	 * @param calls
	 * @return a String that is a code describing the calls in the given set
	 */
	public static String createCallCode(HashSet<String> calls) throws UserError{

		//Check if taxonomy has changed without updating the
		//String "fixedCalls".
		for( String used : calls ){
			if( ! list.contains( used )  )
				throw new UserError(null, 127, "You have changed the taxonomy, but not the" +
						" way filenames for cachefiles are created in hitters.tools.Utils." +
				" Please switch off caching or check hitters.tools.Utils." );		
		} 								


		// Create the code, a simple binary representation 
		long code = 0;
		int i = 0;
		for( String single : list ){
			if( calls.contains( single ) ){
				code = code + ( 1L << i);				
			}
			i++;				
		}	
		//System.out.println( "i= " + i);
		return Long.toString(code);
	}
	
	
	
	/**
	 * Formats the double d to exactly six digits after the decimal point.
	 * 
	 * @param d
	 * @return d as String with exactly six digits after the decimal point
	 */
	public static String format( double d ){
		DecimalFormatSymbols formatInst = DecimalFormatSymbols.getInstance(Locale.ENGLISH);
		return new DecimalFormat("#.000000", formatInst ).format( d );		
	}


	/**
	 * Fast exponentiation for integers, faster than Math.pow().
	 * @param b base
	 * @param ex exponent
	 * @return b^exp
	 */
	public static long exp( int b, int ex ){
		if( ex == 0 ) return 1;
		long c = b;
		for( int i = 0; i < ex-1; i++ ) c = c * b;
		return c;
	}

	
	/**
	 * Returns a few paths that can be searched to locate resources living in the plugin file.
	 * @return paths that can be searched to locate resources in the plugin file 
	 */
	public static String[] resourceLocationsJar(){
		
		String locations = "/ /rsc/ /resources/rsc";
		return locations.split(" ");
	}
	
	/**
	 * Returns a few paths that can be searched to locate resources living in the file system.
	 * @return paths that can be searched to locate resources in the file system. 
	 */	
	public static String[] resourceLocationsFiles(){
		
		String locations = "/../resources/rsc/ /resources/rsc/ /../../resources/rsc/ /../../UBU/resources/rsc/ /../../HHH/resources/rsc/";		
		return locations.split(" ");
	}
	
	
	/**
	 * Counts the system calls in the given log file.
	 * @param filename - the name of the log file
	 * @return a HashMultiset containing the calls
	 */
	public static HashMultiset<String> getHist( String filename ){
		
		HashMultiset<String> set = HashMultiset.create(); 					
		DimType[] dims = { DimType.CALL };
		SysParameter par = new SysParameter( dims );		
		
		MultiDatabase d = new MultiDatabase( filename, par );
		d.openRead();
		Vector<Element> elements = d.readSystemCalls();
		d.closeRead();
		
		String s;
		for( Element e : elements ){
			s = ((SysElement)e).toShortString();
			s = s.split("/")[2];
			set.add( s );			
		}			
		return set;
	}

	
	/**
	 * Adds the system calls in every log file in the given directory.
	 * 
	 * @param path - the directory containing the log files
	 * @return a HashMultiset containing the calls
	 */
	public static HashMultiset<String> getGlobalHist( String path ){
		
		HashMultiset<String> set = HashMultiset.create(); 		
		File dir = new File(path);
		System.out.println( "Dir = " + dir );
		File[] files = dir.listFiles();
		
		for( File f : files ){
			System.out.println( "Get: " + f.getAbsolutePath() );
			set.addAll( getHist( f.getAbsolutePath() ) );
		}

		return set;
	}	

	
	public static String cleanPath2( String file ){
		if( file == null ) {
			//System.out.println( "Utils.cleanPath2: null" );
			return null;
		}
		if( ! file.startsWith("/") ) return null;
		if( file.equals("/") ) return "/";
		String  path = cleanPath( file );
		if( path.startsWith("/fs") ) return null;
		//if( path.startsWith("/proc") ) return null;
		//if( path.startsWith("/sys") ) return null;
		return path;
	}
	
	public static String cleanPath( String file ){
		
		while( file.endsWith( "/" ) ) 
			file = file.substring( 0, file.length() - 1 );
		
		while( file.startsWith( "../" ) || file.startsWith( "./" ) ){
			int i = file.indexOf("/");
			file = file.substring( i );
		}			
		
		
		// get rid of stuff like "/etc//////////sound" 
		int lev = 0;
		int k = file.indexOf( '/', 0 );
		int oldk = k;
		while( k >= 0){
			k = file.indexOf( '/', k + 1 );
			if( k == oldk + 1 ){ 
				if( k == 1 ) file = file.substring( k );
				else file = file.substring(0, k-1) + file.substring( k );
				k--;					
			}
			else{
				lev++;
				oldk = k;
			}
		}
		//get rid of /abc/../def
		String[] parts = file.split("/");
		int i = 0;
		for( String comp : parts ){
			if( comp.equals("..") ) i = Math.max( i-1, 1 );
			else{ 				
				if( ! comp.equals(".") ) parts[i++] = comp;
			}
		}
		file = "/";
		for( int j = 1; j < i; j++ ){
			if( j > 1 ) file += "/";
			file += parts[j]; 
		}
		return file;
	}	

	
}
