package hitters.multi;

import hitters.tools.LogService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Peter Fricke
 * 
 */
public class SysParser{

	public static final int MAXDEPTH = 20;
	
	//String und String mit trailing ...
	private String STRING = "(\"((?<=\\\\)\"|[^\"])*\")";
	private String STRINGTRAIL = "(" + STRING + "(\\.\\.\\.)?)";
	
	// Zahl, Dec oder Hex
	private String NUMBER = "((-?0x[0-9a-f]+)|(-?[0-9]+))";
	
	// Rueckgabewert: Integer, Hex (Adresse) oder Fragezeichen
	private String RETVAL = "(" + NUMBER + "|\\?)";
	private String ERRCO = "(\\s[A-Z_]+)?";
	private String OPTBR = "(\\([^\\(\\)]*\\))?";	
	private String OPTPART = "((" + ERRCO + "\\s" + OPTBR + ")?)";
	private String TAIL = "(?<=(\\s=\\s))" + RETVAL + OPTPART + "$";
	
	// Octal, fuehrende Null erlaubt, Bits z. B. fuer Dateirechte
	private String OCT = "[0-7]+";
	
	// Symbolische Darstellung der Bits (ver-odert)
	private String SYMBOL = "([A-Z_\\|]+)";
	
	// Veroderung der symbolischen Bitdarstellung mit octals
	private String EXT_SYMBOL = "(" + SYMBOL + "\\|" + OCT + ")|(" + SYMBOL + ")";   	
	
	// Key-Value Paare. Key klein, als Value alles moegliche erlaubt
	private String KEY = "[a-z_]+";
	private String KEYVAL = KEY + "=(" + EXT_SYMBOL + "|" + NUMBER + ")";
	
	// Komplexe Datentypen: U.a. in Array- und Mengenschreibweise
	// Gern auch mal geschachtelt.
	String FLAT_SET = "(\\{([^\\[\\]\\{\\}\\\"]|" + STRINGTRAIL + ")*\\})";
	String FLAT_ARRAY = "(\\[([^\\[\\]\\{\\}\\\"]|" + STRINGTRAIL + ")*\\])";

	String SET = "(\\{([^\\}\\{]|" + STRINGTRAIL + 
	"|" + FLAT_SET + "|" + FLAT_ARRAY + ")*\\})";			
	
	String ARRAY = "(\\[([^\\[\\]\\{\\}\\\"]|" + STRINGTRAIL +
	"|" + FLAT_SET + "|" + FLAT_ARRAY + ")*\\])";
	
	// Lange Arrays werden manchmal durch /* Kommentare */ ersetzt
	private String COM = "(/\\*[^\\*]*\\*/)";
	
	// Einzelner Parameter ist String, Zahl, Array, Menge oder Symbol
	private String SINGLE = "(" + STRINGTRAIL + "|" + NUMBER +
	"|" + SET + "|" + ARRAY + "|" + SYMBOL + ")"; 

	private String CLONE_VAL = "(" + NUMBER + "|" + SYMBOL + ")"; 
	private String CLONE_KEY = "(child_stack=|flags=|child_tidptr=|parent_tidptr=)";
	private String CLONE_SINGLE = "(" + CLONE_KEY + CLONE_VAL + ")|" + SET;			
	
	// Innerhalb eines Arrays wieder einzelne Werte, hier
	// ist zusaetzlich der Kommentar erlaubt
	private String EXT_SINGLE = "(" + STRINGTRAIL + "|" + NUMBER +
	"|" + SET + "|" + ARRAY + "|" + SYMBOL + "|" + COM + ")"; 

	// Innerhalb einer Menge wieder einzelne Werte, hier
	// ist zusaetzlich das Key-Value Paar erlaubt
	private String EXT_SINGLE_KV = STRINGTRAIL + "|" + NUMBER +
	"|" + SET + "|" + ARRAY + "|" + SYMBOL + "|" + KEYVAL; 

	private Pattern singleP = Pattern.compile( SINGLE );
	private Pattern extSingleP = Pattern.compile( EXT_SINGLE ); 
	private Pattern extSingleKVP = Pattern.compile( EXT_SINGLE_KV ); 
		
	private Pattern cloneP = Pattern.compile( CLONE_SINGLE );
	
	// Wenn er blockt, wird er nicht fertig...
	private Pattern unfinishedP = 
		Pattern.compile( "<unfinished\\s\\.\\.\\.>[\\s]*$" );
	
	// ... oder irgendwann schon.
	private Pattern resumedP = 
		Pattern.compile( "<\\.\\.\\.\\s[a-z_][a-z_0-9]+\\sresumed>" );			
	
	// Signal
	private Pattern sigP = Pattern.compile("^[0-9]+[\\s]+---[\\s]+");			
	
	// Mord
	private Pattern killP = Pattern.compile("^[0-9]+[\\s]+\\+\\+\\+[\\s]+");
	
	// ProzessID
	private Pattern pidP = Pattern.compile( "^[0-9]+" );
	
	// Rueckgabewert
	private Pattern tailP = Pattern.compile( TAIL );
	private Pattern retP = Pattern.compile(RETVAL);
	
	private SysParameter myParams;
	
	private HashMap<String, ProcTable> procTables = new HashMap<String, ProcTable>();	
	
	private String currentPid = null;
	private String currentReturnValue = null;
	
	int counter = 0;
	
	public SysParser( SysParameter myParams ){
		this.myParams = myParams;	
	}

	
	//TODO: Lineare Suche vermeiden, Hash des Calls
	//berechnen und per switch auf Kleingruppen verteilen.
	public SysElement createElement( String line ) {

		// Wenn Fehler auftreten, ueberspringe statt Abbbruch,
		// Ursache loggen. Nur im Produktionsbetrieb einschalten!

		try{							
		
		//Formatcheck
		if( line.startsWith("#") ){
			System.out.println("\n\n");
			return null; //Kommentare
		}
		Matcher m = pidP.matcher( line );
		if( ! m.find() ) throw new RuntimeException("Strace " +
				"bitte mit -f laufen lassen, Zeilen muessen PIDs " +
				"enthalten.");
		
		//Signale ignorieren
		m = sigP.matcher( line ); 
		if( m.find() ) return null;
		m = killP.matcher( line ); 
		if( m.find() ) return null;
		
		
		//Blockende Calls stehen jeweils halb im Log
		//und muessen repariert werden
		m = unfinishedP.matcher( line ); 
		if( m.find() ){
			storeUnfinished( line );
			return null; 
		}
		else{
			m = resumedP.matcher( line ); 
			if( m.find() ) line = mergeResumed( line );
			if( line == null ) return null;
		}

		
		//System.out.println( line );
		String call = getCallName( line );
		counter++; //Echte Zeilen ohne Sigs und unfinished
		if( ! isUsed( call ) ) return null;				
				
		currentPid = getPid( line );
		currentReturnValue = getReturnValue( line );
		// strace schreibt ganz selten mal kaputte Zeilen -> wegwerfen.
		if( currentReturnValue == null ) return null;
		
		updateSequence( currentPid, call ); 
	
		//int hash = call.hashCode();
		//int position = Math.abs(hash) % 30;
		
		if( call.equals("open") )  
			return getOpen( line );
		if( call.equals("read") )  
			return getRead( line );
		if( call.equals("write") ) 
			return getWrite( line );
		if( call.equals("mmap2") ) 
			return getMmap2( line );
		if( call.equals("munmap") ) 
			return getMunmap( line );
		if( call.equals("lseek") ) //Verwechselungsgefahr _llseek 
			return getLseek( line );
		if( call.equals("_llseek") ) 
			return get_llseek( line );
		if( call.equals("select") ) 
			return getSelect( line );
		if( call.equals("fstat64"))
			return getStat64( line, "fstat64" );		
		if( call.equals("stat64"))
			return getStat64( line, "stat64" );
		if( call.equals("lstat64"))
			return getStat64( line, "lstat64" );		
		if( call.equals("fstat"))
			return getStat64( line, "fstat" );		
		if( call.equals("stat"))
			return getStat64( line, "stat" );
		if( call.equals("lstat"))
			return getStat64( line, "lstat" );				
		if( call.equals("writev"))
			return getWritev( line );
		if( call.equals("access"))
			return getAccess( line );
		if( call.equals("fcntl64")) 
			return getFcntl64( line );
		if( call.equals("fcntl"))
			return getFcntl64( line );
		if( call.equals("dup"))
			return getDupfd( line, "dup" );		
		if( call.equals("dup2"))
			return getDupfd( line, "dup2" );		
		if( call.equals("dup3"))
			return getDupfd( line, "dup3" );		
		if( call.equals("close") ) //Keine Standardbehandlung! FD wird frei! 
			return getClose( line );
		if( call.equals("getdents"))
			return getDents( line );
		if( call.equals("getdents64")) //Exakt identisch mit getdents
			return getDents( line );		
		if( call.equals("clock_gettime"))
			return getClock_gettime( line );
		if( call.equals("mprotect"))
			return getMprotect( line );
		if( call.equals("futex"))
			return getFutex( line );
		if( call.equals("poll"))
			return getPoll( line );		
		if( call.equals("recvmsg"))
			return getRecvUni( line, "recvmsg" );
		if( call.equals("recv"))
			return getRecvUni( line, "recv" );
		if( call.equals("send"))
			return getSendUni( line, "send" );
		if( call.equals("sendmsg"))
			return getSendUni( line, "sendmsg" );
		if( call.equals("sendto"))
			return getSendUni( line, "sendto" );
		if( call.equals("rt_sigaction"))
			return getRt_sigaction( line );
		if( call.equals("pipe"))
			return getPipe( line, "pipe" );
		if( call.equals("pipe2"))
			return getPipe( line, "pipe2" );	
		if( call.equals("socket"))
			return getSocket( line );
		if( call.equals("inotify_init"))
			return getInotify_init( line );
		if( call.equals("inotify_init1"))
			return getInotify_init( line );
		if( call.equals("accept"))
			return getAccept( line, "accept" );
		if( call.equals("accept4"))
			return getAccept( line, "accept4" );
		if( call.equals("execve"))
			return getExecve( line );
		if( call.equals("clone"))
			return getClone( line );
		if( call.equals("fork"))
			return getFork( line, "fork" );
		if( call.equals("vfork"))
			return getFork( line, "vfork" );
		if( call.equals("unshare"))
			return getUnshare( line );
		if( call.equals("nanosleep"))
			return getNanosleep( line );

		
								
		// Standardbehandlung mit Pfad als fd
		if( call.equals("ioctl"))
			return getSimpleWFd(line, "/DEVICE/ioctl");
		
		
		// Standardbehandlung mit Pfad als String
		if( call.equals("readlink") ) 
			return getSimpleWPath( line, "/INFO/readlink" );

		
		// Standardbehandlung ohne Pfad
		if( call.equals("gettimeofday") ) 	
			return getSimple( "/INFO/gettimeofday" );
		if( call.equals("time"))			
			return getSimple( "/INFO/time" );
		if( call.equals("uname"))			
			return getSimple( "/INFO/uname" );
		if( call.equals("brk"))			
			return getSimple( "/PROCMEM/brk" );

		
				}
				catch( Exception e ){
					LogService.logF( "# Problem beim Parsen der Systemcalls: " +
							"line = " + line + " fuehrt zu Fehler:  " +
							e.toString() + " " );			
				}

		LogService.logF( line );
		return null;
	}


	public int getCounter(){ return counter; }
	
	
	private SysElement getOpen( String s ) {
		
		List<String> args = getPars( s );		
		boolean cloExec = false;
		int ret = Integer.parseInt( args.get(0) );
		String file = args.get(1);
		
		// CloExec-Flag wichtig wg. Seiteneffekten:
		// Wenn gesetzt, wird Filededeskriptor waehrend
		// execve geschlossen
		if( args.get(2).contains( "CLOEXEC" ) ) cloExec = true;
		
		int intFlag  = -1;
		if( args.get(2).contains( "O_RDONLY" ) ) intFlag = 1;
		if( args.get(2).contains( "O_WRONLY" ) ) intFlag = 2;
		if( args.get(2).contains( "O_RDWR"   ) ) intFlag = 3;

		if( ret > -1 ) putFile( getPid(s), ret, file, cloExec ); 

		if( intFlag < 0 ) 
			throw new RuntimeException("IntFlag nicht gefunden + s");

		return fixElement( file, "/FILESYS/open", intFlag + "" );
	}

	
	private SysElement getWritev( String s ) {

		List<String> args = getPars( s );
		
		String file = getFile( getPid(s), args.get(1) );
		int size = 1;
		
		if( Integer.parseInt( args.get(3) ) > 3 ) size = 2;
		
		return fixElement( file, "/FILESYS/writev", size + "" );
	}


	// Read und Write sind i.W. identisch
	private SysElement getReadWrite( String s, boolean write ) {

		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
		String call;
		
		int size = 1; // Klein		
		if( Integer.parseInt( args.get(3) ) > 1024 ) size = 3; // Gross
		else if( Integer.parseInt( args.get(3) ) > 128  ) size = 2; // Mittel
		
		if( write ) call = "/FILESYS/write";
		else call = "/FILESYS/read";
		
		return fixElement( file, call, size + "" );
	}
		

	private SysElement getRead(String s ) {			
		return getReadWrite( s, false );
	}


	private SysElement getWrite(String s ) {			
		return getReadWrite( s, true );
	}	

	
	private SysElement getClose( String s ) {			

		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
		
		removeFile( getPid(s), args.get(1) );

		return fixElement( file, "/FILESYS/close", "" );
	}

	
	// Variante von _llseek
	private SysElement getLseek( String s ) {

		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
		int amount = 2, seek = 0;
		
		// Kleine Verschiebung oder gro�er Sprung?
		if( Integer.parseInt( args.get(2) ) > 512 ) amount = 3;
		if( Integer.parseInt( args.get(2) ) < 0 ) amount = 1;		
		
		// Verschiebe rel. zu Dateianfang, aktueller Pos. oder Dateiende
		if( args.get(3).contains( "SEEK_SET" ) ) seek = 1;
		if( args.get(3).contains( "SEEK_CUR" ) ) seek = 2;
		if( args.get(3).contains( "SEEK_END" ) ) seek = 3;
		
		String code = seek + "/" + amount;		
		return fixElement( file, "/FILESYS/_llseek", code );
	}
	
	
	// Variante von lseek	
	private SysElement get_llseek( String s ) {

		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
		int amount = 2, seek = 0;
		
		// Kleine Verschiebung oder gro�er Sprung?
		if( Integer.parseInt( args.get(2) ) > 512 ) amount = 3;
		if( Integer.parseInt( args.get(2) ) < 0 ) amount = 1;		
		
		// Verschiebe rel. zu Dateianfang, aktueller Pos. oder Dateiende
		if( args.get(4).contains( "SEEK_SET" ) ) seek = 1;
		if( args.get(4).contains( "SEEK_CUR" ) ) seek = 2;
		if( args.get(4).contains( "SEEK_END" ) ) seek = 3;
		
		String code = seek + "/" + amount;
		return fixElement( file, "/FILESYS/_llseek", code );
	}

	
	private SysElement getFcntl64( String s ) {

		List<String> args = getPars( s );		
		String fd = args.get(1);
		String file = getFile( getPid(s), fd );
		int ret;
		
		//Default = notify
		int cat = 7, val = 0;		
		
		if( args.get(2).equals( "F_DUPFD" ) ){ cat = 1; val = 1; }
		if( args.get(2).equals( "F_DUPFD_CLOEXEC" ) ){ cat = 1; val = 2; }
		
		// Seiteneffekte: Filedeskriptor wurde kopiert
		if( cat == 1 ){
			ret = Integer.parseInt( args.get(0) );
			if( ret > -1 ){
				if( val == 2 )
					putFile( getPid(s), ret, file, true );
				else
					putFile( getPid(s), ret, file, false );
			}
		}
		
		if( args.get(2).equals( "F_GETFD" ) ){ cat = 2; val = 1; }
		if( args.get(2).equals( "F_SETFD" ) ){ 
			cat = 2; 
			val = 2; 
			// Seiteneffekte: CloExec-Flag ggf. gesetzt
			ret = Integer.parseInt( args.get(0) );
			if( ret > -1 ){
				if( args.get(3).contains("CLOEXEC") )
					setCloseOnExec( getPid(s), fd, true); 
				else
					setCloseOnExec( getPid(s), fd, false);
			}
		}
		
		if( args.get(2).equals( "F_GETFL" ) ){ cat = 3; val = 1; }
		if( args.get(2).equals( "F_SETFL" ) ){ cat = 3; val = 2; }
		
		if( args.get(2).equals( "F_GETLK" ) ){ cat = 4; val = 1; }
		if( args.get(2).equals( "F_SETLKW" ) ){ cat = 4; val = 2; }
		if( args.get(2).equals( "F_SETLK" ) ){ cat = 4; val = 3; }
		
		if( args.get(2).equals( "F_GETOWN" ) ){ cat = 5; val = 1; }
		if( args.get(2).equals( "F_SETOWN" ) ){ cat = 5; val = 2; }
		if( args.get(2).equals( "F_GETSIG" ) ){ cat = 5; val = 3; }
		if( args.get(2).equals( "F_SETSIG" ) ){ cat = 5; val = 4; }
				
		if( args.get(2).equals( "F_GETLEASE" ) ){ cat = 6; val = 1; }
		if( args.get(2).equals( "F_SETLEASE" ) ){ cat = 6; val = 2; }
				
		String code = cat + "/" + val;
		return fixElement( file, "/FILESYS/fcntl64", code );
	}

	
	//Fuer dup, dup2, dup3, Unterscheidung anhand von "name"
	//Wichtig ist, nicht nur den Call selbst in ein 
	//SysElement zu verwandeln, sondern auch die
	//Seiteneffekte auf Dateideskriptoren (FD) und FD-Flags
	//zu speichern, damit spaetere Calls die FD korrekt
	//aufloesen koennen.
	private SysElement getDupfd( String s, String name ) {

		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
		int ret = Integer.parseInt( args.get(0) );
		String code = "";
		
		if( name.equals("dup3") ){
			if( args.get(3).contains("CLOEXEC") ) code = "1";
			else code = "2";
		}
				
		if( ret > -1 ){
			if( code.equals("1") ) putFile( getPid(s), ret, file, true );
			else putFile( getPid(s), ret, file, false );
		}

		return fixElement( file, "/FILESYS/" + name, code );
	}

	
	//Fuer stat, lstat und fstat, Unterscheidung anhand von "name"
	private SysElement getStat64( String s, String name ) {

		List<String> args = getPars( s );
		List<String> struct = splitSet( args.get(2) );
		
		String file;
		if( name.contains( "fstat" ) ) 
			file = getFile( getPid(s), args.get(1) );
		else file = args.get(1);
		
		int mode = 1;//S_OTHER fuer reine Hex

		if( struct.size() > 0 ){
			if( struct.get(0).contains( "S_IFREG" ) ) mode = 2;
			if( struct.get(0).contains( "S_IFIFO" ) ) mode = 3;
			if( struct.get(0).contains( "S_IFCHR" ) ) mode = 4;
			if( struct.get(0).contains( "S_IFSOCK" ) ) mode = 5;
			if( struct.get(0).contains( "S_IFDIR" ) ) mode = 6;
			if( struct.get(0).contains( "S_IFLNK" ) ) mode = 7;
		}
		return fixElement( file, "/INFO/" + name, mode + "" );
	}

		
	private SysElement getAccess( String s ) {

		List<String> args = getPars( s );
		
		String file = args.get(1);
		int mode = 2;
		int read = 2, write = 2, exec = 2;

		// mode entweder F_OK oder bitweises OR von R_OK, W_OK, X_OK 
		if( args.get(2).contains( "F_OK" ) ) {
			mode = 1;
			read = 0;
			write = 0;
			exec = 0;
		}
		else{
			if( args.get(2).contains( "R_OK" ) ) read = 1;
			if( args.get(2).contains( "W_OK" ) ) write = 1;
			if( args.get(2).contains( "X_OK" ) ) exec= 1;
		}
		String code = mode + "/" + read + "/" + write + "/" + exec;
		return fixElement( file, "/INFO/access", code );
	}

	
	
	private SysElement getDents( String s ) {

		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
				
		int size = 1; //Default: Leeres Verzeichnis;		
		
		if( splitComment( args.get(2) ) > 0 ) size = 2;
		if( splitComment( args.get(2) ) > 10 ) size = 3;
		
		return fixElement( file, "/INFO/getdents", size + "" );
	}
	

	private SysElement getSelect( String s ) {

		List<String> args = getPars( s );		
		
		// 0 ist ANY, darum bedeutet 1, es wird 
		// gewartet, und 2 dass nicht gewartet wird.
		int read = 1, write = 1, exceptions = 1;
		
		if( args.get(2).equals("NULL") || args.get(2).equals("[]") )
			read = 2;
		
		if( args.get(3).equals("NULL") || args.get(3).equals("[]") )
			write = 2;
		
		if( args.get(4).equals("NULL") || args.get(4).equals("[]") )
			exceptions = 2;
				
		String code = read + "/" + write + "/" + exceptions;		
		return fixElement( "*", "/INFO/select", code );
	}

	
	private SysElement getPoll( String s ) {

		List<String> args = getPars( s );
		int time = 2, single = 0;

		if( Integer.parseInt( args.get( 3 ) ) > 0 ) time  = 1;
		else time = 2;
		
		if( Integer.parseInt( args.get( 2 ) ) == 1 ) single  = 1;
		else single = 2;
		
		String code = time + "/" + single;
		return fixElement( "*", "/INFO/poll", code );
	}
	
	
	private SysElement getInotify_init( String s ) {

		List<String> args = getPars( s );		
		String file = "*";
		int ret = Integer.parseInt( args.get(0) );		

		if( ret > -1 ){
			file = "\"/Inotify\"";
			if( args.size() > 1 && args.get(1).contains( "CLOEXEC" ) )
				putFile( getPid(s), ret, file, true );								
			else
				putFile( getPid(s), ret, file, false );
		}			

		return fixElement( file, "/INFO/inotify_init", "");
	}
	

	private SysElement getClock_gettime( String s ) {

		List<String> args = getPars( s );				
		int clockId = 0;		
		
		if( args.get(1).contains("CLOCK_REALTIME") ) clockId = 1;
		if( args.get(1).contains("CLOCK_MONOTONIC") ) clockId = 2;
		if( args.get(1).contains("CLOCK_PROCESS_CPUTIME_ID") ) clockId = 3;
		if( args.get(1).contains("CLOCK_THREAD_CPUTIME_ID") ) clockId = 4;
		
		return fixElement( "*", "/INFO/clock_gettime", clockId + "" );
	}
	
	
	//Fuer recv und recvmsg, Unterscheidung anhand von "name"
	private SysElement getRecvUni( String s, String name ) {

		List<String> args = getPars( s );
		//String file = getFile( getPid(s), args.get(1), fdTables );
		int flags = 2, position;
		
		// Flags bei recv an Position 4, bei recvmsg an 3
		if( name.contains("recvmsg") ) position = 3;
		else position = 4;
		
		// To PEEK or not to PEEK...
		// Kein Flag und PEEK sind die einzigen Flags, 
		// die auch tatsaechlich vorkommen.
		if( args.get( position ).contains( "MSG_PEEK" ) ) flags = 1;		
		
		return fixElement( "*", "/COMMUNICATION/" + name, flags + "" );
	}

	
	//Fuer send, sendto, Unterscheidung anhand von "name"
	private SysElement getSendUni( String s, String name ) {

		List<String> args = getPars( s );		
		//String file = getFile( getPid(s), args.get(1), fdTables );
		int flags = 2, position;
		
		// Flags bei recv an Position 4, bei recvmsg an 3
		if( name.contains("sendmsg") ) position = 3;
		else position = 4;
					
		// Kein Flag und MSG_NOSIGNAL sind die einzigen Flags, 
		// die auch tatsaechlich vorkommen.
		if( args.get( position ).contains( "MSG_NOSIGNAL" ) ) 
			flags = 1;		
		
		return fixElement( "*", "/COMMUNICATION/" + name, flags + "" );
	}
	

	private SysElement getRt_sigaction( String s ) {

		List<String> args = getPars( s );
				
		int action = 3;
		int sigCategory = 1;
		int sig = 0;

		if( args.get( 2 ).equals( "{SIG_DFL}" ) ) action = 1;
		if( args.get( 2 ).equals( "{SIG_IGN}" ) ) action = 2;
		
		if( args.get( 1 ).contains( "SIGCONT" ) ){ sigCategory = 2; sig = 0; }
		if( args.get( 1 ).contains( "SIGCHLD" ) ){ sigCategory = 3; sig = 0; }
		
		if( args.get( 1 ).contains( "SIGHUP" ) ){ sigCategory = 4; sig = 1; }
		if( args.get( 1 ).contains( "SIGINT" ) ){ sigCategory = 4; sig = 2; }
		if( args.get( 1 ).contains( "SIGKILL" ) ){ sigCategory = 4; sig = 3; }
		if( args.get( 1 ).contains( "SIGPIPE" ) ){ sigCategory = 4; sig = 4; }
		if( args.get( 1 ).contains( "SIGALRM" ) ){ sigCategory = 4; sig = 5; }
		if( args.get( 1 ).contains( "SIGTERM" ) ){ sigCategory = 4; sig = 6; }
		if( args.get( 1 ).contains( "SIGUSR1" ) ){ sigCategory = 4; sig = 7; }
		if( args.get( 1 ).contains( "SIGUSR2" ) ){ sigCategory = 4; sig = 8; }

		if( args.get( 1 ).contains( "SIGQUIT" ) ){ sigCategory = 5; sig = 1; }
		if( args.get( 1 ).contains( "SIGILL" ) ){ sigCategory = 5; sig = 2; }
		if( args.get( 1 ).contains( "SIGABRT" ) ){ sigCategory = 5; sig = 3; }
		if( args.get( 1 ).contains( "SIGFPE" ) ){ sigCategory = 5; sig = 4; }
		if( args.get( 1 ).contains( "SIGSEGV" ) ){ sigCategory = 5; sig = 5; }

		if( args.get( 1 ).contains( "SIGSTOP" ) ){ sigCategory = 6; sig = 1; }
		if( args.get( 1 ).contains( "SIGTSTP" ) ){ sigCategory = 6; sig = 2; }
		if( args.get( 1 ).contains( "SIGTTIN" ) ){ sigCategory = 6; sig = 3; }
		if( args.get( 1 ).contains( "SIGTTOU" ) ){ sigCategory = 6; sig = 4; }
		
		String code = action + "/" + sigCategory + "/" + sig;
		return fixElement( "*", "/COMMUNICATION/rt_sigaction", code );
	}

	
	//Fuer pipe, pipe2 Unterscheidung anhand von "name"
	private SysElement getPipe( String s, String name ) {

		List<String> args = getPars( s );		
		String fileW, fileR = "*";
		int ret = Integer.parseInt( args.get(0) );
		List<String> fileDescr = splitArray( args.get(1) );
		int fdR = Integer.parseInt( fileDescr.get(0) );
		int fdW = Integer.parseInt( fileDescr.get(1) );
		int cloExec = 0;
		if( name.equals( "pipe2" ) ){
			if( args.get(2).contains( "CLOEXEC" ) ) cloExec = 1;
			else cloExec = 2;			
		}
		
		if( ret > -1 ){
			fileR = "\"/RPipe\"";
			fileW = "\"/WPipe\"";
			if( cloExec == 1 ){
				putFile( getPid(s), fdR, fileR, true );
				putFile( getPid(s), fdW, fileW, true );				
			}
			else{
				putFile( getPid(s), fdR, fileR, false );
				putFile( getPid(s), fdW, fileW, false );				
			}
		}
		
		if( name.equals( "pipe2" ) )
			return fixElement( fileR, "/COMMUNICATION/pipe2", cloExec + "" );
		else
			return fixElement( fileR, "/COMMUNICATION/pipe", "");
	}

	
	private SysElement getSocket( String s ) {

		List<String> args = getPars( s );		
		String file = "*";
		int ret = Integer.parseInt( args.get(0) );
		String domain = args.get(1);
		int domainInt = 0;

		
		// Schoen inkonsistent, mal AF, mal PF, darum hier endsWith
		if( domain.endsWith( "UNIX" ) ) domainInt = 1;
		if( domain.endsWith( "LOCAL" ) ) domainInt = 1;// Wie Unix
		if( domain.endsWith( "INET" ) ) domainInt = 2;
		if( domain.endsWith( "INET6" ) ) domainInt = 3;
		if( domain.endsWith( "IPX" ) ) domainInt = 4;
		if( domain.endsWith( "NETLINK" ) ) domainInt = 5;
		if( domain.endsWith( "X25" ) ) domainInt = 6;
		if( domain.endsWith( "AX25" ) ) domainInt = 7;
		if( domain.endsWith( "ATMPVC" ) ) domainInt = 8;
		if( domain.endsWith( "APPLETALK" ) ) domainInt = 9;
		if( domain.endsWith( "PACKET" ) ) domainInt = 10;		
		if( domain.endsWith( "FILE" ) ) domainInt = 11;

		if( ret > -1 ){
			file = "\"/Socket\"";
			if( args.get(2).contains( "CLOEXEC" ) )
				putFile( getPid(s), ret, file, true );								
			else
				putFile( getPid(s), ret, file, false );
		}			

		return fixElement( file, "/COMMUNICATION/socket", domainInt + "");
	}


	
	private SysElement getAccept( String s, String name ) {

		List<String> args = getPars( s );		
		String file = "*";
		int ret = Integer.parseInt( args.get(0) );
		int cloExec = 0;
		
		if( ret > -1 ){
			file = "\"/ConnectedSocket\"";
			if( args.size() > 4 && args.get(4).contains( "CLOEXEC" ) ){
				putFile( getPid(s), ret, file, true );
				cloExec = 1;
			}
			else{
				putFile( getPid(s), ret, file, false );
				cloExec = 2;
			}
		}			
		if( name.equals( "accept" ) )
			return fixElement( file, "/COMMUNICATION/accept", "");
		else
			return fixElement( file, "/COMMUNICATION/accept4", cloExec + "" );
	}

	
	
	private SysElement getMmap2( String s ) {

		List<String> args = getPars( s );
		String file = "";
		int fd = Integer.parseInt( args.get(5) );
		if( fd > -1 ) file = getFile( getPid(s), args.get(5) );
		int share, addr, rwe = 1;
		
		//Flags: Private oder shared?
		if( args.get(4).contains( "MAP_PRIVATE" ) ) share = 2;
		else share = 1;
		
		// Genaue Position im Speicher gefordert?
		if( args.get( 1 ).contains( "NULL" ) ) addr = 1;
		else addr = 2;
		
		// Read, write, execute
		if( args.get(3).contains( "PROT_READ" ) ) rwe += 4;
		if( args.get(3).contains( "PROT_WRITE" ) ) rwe += 2;
		if( args.get(3).contains( "PROT_EXEC" ) ) rwe += 1;
		
		String code = share + "/" + addr + "/" + rwe;		
		return fixElement( file, "/PROCMEM/mmap2", code );
	}


	private SysElement getMunmap( String s ) {

		List<String> args = getPars( s );
		int size;
		
		if( Integer.parseInt( args.get(2) ) > 4096 ) size = 2; 
		else size = 1;
				
		return fixElement( "*", "/PROCMEM/munmap", size + "" );
	}	

	
		private SysElement getMprotect( String s ) {

		List<String> args = getPars( s );
		int cat = 2, prot = 0;
		
		if( args.get( 3 ).contains("PROT_NONE") ) cat = 1;
		else{
			if( args.get( 3 ).contains("PROT_READ") ) prot = 1;
			if( args.get( 3 ).contains("PROT_WRITE") ) prot = 2;
			if( args.get( 3 ).contains("PROT_EXEC") ) prot = 3;			
		}
				
		String code = cat + "/" + prot;
		return fixElement( "*", "/PROCMEM/mprotect", code );
	}


	private SysElement getFutex( String s ) {

		List<String> args = getPars( s );
		int op = 0, cat = 0;

		if( args.get( 2 ).contains("FUTEX_WAIT") ){ cat = 1; op = 1; }
		if( args.get( 2 ).contains("FUTEX_WAIT_PRIVATE") ){ cat = 1; op = 2; }	
		if( args.get( 2 ).contains("FUTEX_WAKE") ){ cat = 2; op = 1; }
		if( args.get( 2 ).contains("FUTEX_REQUEUE") ){ cat = 2; op = 2; }					
		if( args.get( 2 ).contains("FUTEX_CMP_REQUEUE") ){ cat = 2; op = 3; }
		if( args.get( 2 ).contains("FUTEX_CMP_REQUEUE_PRIVATE") ){ cat = 2; op = 4; }
		if( args.get( 2 ).contains("FUTEX_WAKE_PRIVATE") ){ cat = 2; op = 5; }				
		if( args.get( 2 ).contains("FUTEX_WAKE_OP_PRIVATE") ){ cat = 2; op = 6; }			
		
		String code = cat + "/" + op;
		return fixElement( "*", "/PROCMEM/futex", code );
	}		

	
	private SysElement getExecve( String s ) {

		List<String> args = getPars( s );
		String file = args.get(1);
		
		// Anzahl Argumente. Eigener Name immer, 1 = 0 Argumente
		List<String> argList = splitArray( args.get(2) );
		int argnum = argList.size(); 
		if( argnum > 1 ) argnum = 2;
		
		doCloseOnExec( getPid(s) );
		
		return fixElement( file, "/PROCMEM/execve", argnum + "" );
	}	


	private SysElement getClone( String s ) {

		List<String> args = getParsForClone( s );		
		//System.out.println( "CLONE: " + args );
		
		String oldPid = getPid( s );
		String newPid = args.get( 0 );

		int intNewPid = Integer.parseInt( newPid ); 
		int cloneFiles = 0;
		int stackNull = 0;
		int cloneThread = 0;
		
		// Seiteneffekte: Wenn CLONE_FILES nicht gesetzt ist, werden FDs
		// von Elter zu Kind kopiert. Ist CLONE_FILES gesetzt, teilen
		// Elter und Kind die Deskriptoren und sehen die Aenderungen, 
		// die der andere vornimmt. Ist CLONE_FILES gesetzt, wird also
		// gerade *nicht* geklont.
		// Fuer eine genauere Beschreibung siehe Kommentar 
		// zu copyProcTable.
		if( args.get(2).contains( "CLONE_FILES" ) ){
			if( intNewPid > -1 ) shareProcTable( oldPid, newPid );
			cloneFiles = 1;
		}
		else{
			if( intNewPid > -1 ) copyProcTable( oldPid, newPid );
			cloneFiles = 2;
		}
		
		String[] comps = args.get(1).split("=");
		if( comps[1].trim().equals("0") )
			stackNull = 1;
		else 
			stackNull = 2;
		
		if( args.get(2).contains( "CLONE_THREAD" ) )			
			cloneThread = 1;
		else			
			cloneThread = 2;
		
		String code = cloneFiles + "/" + stackNull + "/" + cloneThread;
		return fixElement( "*", "/PROCMEM/clone", code + "" );
	}	

	
	// fork und vfork
	private SysElement getFork( String s, String name ) {

		List<String> args = getPars( s );		
		
		String oldPid = getPid( s );
		String newPid = args.get( 0 );

		int intNewPid = Integer.parseInt( newPid ); 
		if( intNewPid > -1 ) copyProcTable( oldPid, newPid );
		
		return fixElement( "*", "/PROCMEM/" + name, "" );
	}	


	private SysElement getUnshare( String s ) {

		List<String> args = getPars( s );					

		String pid = getPid( s );
		int ret = Integer.parseInt( args.get(0) );
		int files = 0;
		int fs = 0;
		int newns = 0;
		
		if( args.get( 1 ).contains( "CLONE_FILES" ) ) files = 1;
		else files = 2;
		
		if( args.get( 1 ).contains( "CLONE_FS" ) ) fs = 1;
		else fs = 2;
		
		if( args.get( 1 ).contains( "CLONE_NEWNS" ) ) newns = 1;
		else newns = 2;
		
		//Seiteneffekte: Ggf. werden nun Dateideskriptoren nicht
		//mehr gemeinsam benutzt, sondern es eird auf Kopien gearbeitet.
		//Vgl. getClone()
		if( files == 1 && ret > -1 ){
			// 2 x pid, arbeite nun auf eigener Kopie
			//System.out.println( "IN: ret = " + ret + "  pid=" + pid );
			copyProcTable( pid, pid );
		}
		
		String code = files + "/" + fs + "/" + newns;		
		return fixElement( "*", "/PROCMEM/unshare", code );
	}	

	
	private SysElement getNanosleep( String s ) {

		List<String> args = getPars( s );		
		
		int time = 0;
		List<String> timespec = splitSet( args.get(1) ); 
		int secs = Integer.parseInt( timespec.get(0) );
		int nano = Integer.parseInt( timespec.get(1) );
		
		// Alles ueber 20 ms ist lang
		if( secs > 0 || nano > 20000000 ) time = 1;
		else time = 2;
		
		return fixElement( "*", "/PROCMEM/nanosleep", time + "" );
	}	

		
	//Vereinfachte Behandlung fuer folgenden Fall:
	//Einziger benutzter Wert ist der als String angegebene 
	//Pfad. Der Pfad muss erstes Argument sein.
	private SysElement getSimpleWPath( String s, String code ) {

		List<String> args = getPars( s );
		String file = args.get(1);
		
		return fixElement( file, code, "" );
	}


	
	//Vereinfachte Behandlung fuer folgenden Fall:
	//Einziger benutzter Wert ist der als Filedeskriptor angegebene 
	//Pfad. Der Deskriptor muss erstes Argument sein.
	private SysElement getSimpleWFd( String s, String code ) {		
		
		List<String> args = getPars( s );
		String file = getFile( getPid(s), args.get(1) );
		
		return fixElement( file, code, "" );
	}

	//Vereinfachte Behandlung fuer folgenden Fall:
	//Es werden keine Parameter verwendet.	
	private SysElement getSimple( String code ) {
		
		return fixElement( "", code, "" );
	}
	
	
	// Lies Rueckgabewert und speichere ihn an Position 0,
	// Lies Parameter aus dem Teil zwischen den Klammern und 
	// speichere ihn an Positionen 1 ... n.
	// Rueckgabewert "?" wird zu -2.
	public List<String> getPars( String line ){

		List<String> pars = new ArrayList<String>();
		Matcher m;					
		int start, end;

		// Rueckgabewert an nullte Position
		pars.add( getReturnValue( line ) );
		
		int pos = line.lastIndexOf("=");				
		line = line.substring(0, pos + 1);
		end = line.lastIndexOf( ")" );
		start = line.indexOf( "(" );
		line = line.substring( start, end + 1 );
		
		m = singleP.matcher( line );			
		while( m.find() ){
			pars.add( m.group() );				
		}
		return pars;		
	}

	
	// Clone haelt sich nicht ans Format der uebrigen Calls,
	// um die anderen nicht auszubremsen, gibt es eine
	// eigene Methode zum Auslesen der Parameter.
	public List<String> getParsForClone( String line ){

		List<String> pars = new ArrayList<String>();
		Matcher m;					
		int start, end;

		// Rueckgabewert an nullte Position
		pars.add( getReturnValue( line ) );
		
		int pos = line.lastIndexOf("=");				
		line = line.substring(0, pos + 1);
		end = line.lastIndexOf( ")" );
		start = line.indexOf( "(" );
		line = line.substring( start, end + 1 );
		
		m = cloneP.matcher( line );			
		while( m.find() ){
			pars.add( m.group() );				
		}
		return pars;		
	}


	
	public String getReturnValue( String line ){
		
		String ret = null;
		//int pos = line.lastIndexOf("=");
		Matcher m = tailP.matcher( line );
		if( ! m.find() ) return null;
		m = retP.matcher( m.group() );

		if( m.find() ){
			ret = m.group();
			// Manche Calls haben keinen Rueckgabewert
			// Codiere als -2, denn -1 ist schon vergeben
			// fuer Fehler
			if( ret.equals("?") ) ret = "-2";
		} 
		else return null;
		
		return ret.trim();
	}
	
	
	// Wenn ein Parameter/Ergebnis eines Calls ein Array ist,
	// laesst es sich weiter zerlegen
	public List<String> splitArray( String line ){

		List<String> list = new ArrayList<String>();		
		
		Matcher m = extSingleP.matcher( line.substring(1) ); 

		while( m.find() ){
			list.add( m.group() );				
		}
		return list;
	}
		

	// Wenn ein Parameter/Ergebnis eines Calls eine Menge ist,
	// laesst sie sich weiter zerlegen
	public List<String> splitSet( String line ){

		List<String> list = new ArrayList<String>();		
		
		Matcher m = extSingleKVP.matcher( line.substring(1) ); 

		while( m.find() ){
			list.add( m.group() );				
		}
		return list;
	}
	
	
	// Wenn ein Parameter/Ergebnis eines Calls ein Kommentar
	// ist (etwa so: /* 19 entries */), laesst sich die Zahl extrahieren.
	public int splitComment( String line ){

		Matcher m = extSingleKVP.matcher( line ); 
		Integer res = null;
		int val = 0;
		
		if( m.find() ){
			res = Integer.parseInt( m.group() );				
		}
		
		if( res != null ) val = res;
		
		return val;
	}

	
	public String getPid( String s ){
		Matcher m = pidP.matcher( s );
		m.find();
		String pidString = m.group();
		//int pid = Integer.parseInt( pidString );
		//return pid;
		return pidString;
	}
	
	
	public String getFile( String pid, String fd ) {			
					
		String file = null;
		
		ProcTable pt = procTables.get( pid );
		
		if( pt != null )
			file = pt.getFile( fd );		

		if( file == null ) file = "\"/DateiUnbekannt\"";
		
		return file;
	}

	
	public void putFile( String pid, int fd, String file, boolean cloExec ) {			
		
		ProcTable pt = procTables.get( pid );

		if( file == null ) file = "\"/DateiUnbekannt\"";						

		if( pt == null ){
			pt = new ProcTable( myParams );
			procTables.put( pid, pt );
		}

		pt.putFile(fd, file, cloExec);						
	}
	
	
//	public void putFile( String pid, int fd, String file ) {
//		putFile( pid, fd, file, false );		
//	}
	
	
	public String removeFile( String pid, String fd ) {			
			
		String file = null;
		ProcTable pt = procTables.get( pid );

		if( pt != null ){			
			file = pt.getFile(fd);
			pt.removeFile(fd);			
		}
		if( file == null ) file = "\"/DateiUnbekannt\"";
		
		return file;
	}

	
	public void setCloseOnExec( String pid, String fd, boolean cloExec ) {			
				
		ProcTable pt = procTables.get( pid );		
		if( pt != null ) pt.setCloseOnExec(fd, cloExec);					
	}



	/* copyProcTable und shareProcTable sind Hilfsmethoden fuer den
	 * Systemcall clone(). Clone() aehnelt fork(), der neue 
	 * Prozess / Thread erbt u.a. die Dateideskriptoren des Elters.
	 * Die genaue Bedeutung von "erben" ist abhaengig vom Flag CLONE_FILES:
	 * Entweder die Deskriptoren werden kopiert und sind dann unabhaengig
	 * voneinander. Oder sie werden gemeinsam von Elter und Kind 
	 * benutzt, die Aenderungen (Oeffnen, Schliessen, Flags aendern) 
	 * sind dann jeweils fuer den anderen sichtbar.
	 * 
	 * If CLONE_FILES is set, the calling process and the child process share 
	 * the same file descriptor table. Any file descriptor created by the 
	 * calling process or by the child process is also valid in the other process.
	 * Similarly, if one of the processes closes a file descriptor, or changes its 
	 * associated  flags  (using  the fcntl(2) F_SETFD operation), the other 
	 * process is also affected. If  CLONE_FILES is not set, the child process
	 * inherits a copy of all file descriptors opened in the calling 
	 * process at the time of clone().  (The duplicated file 
	 * descriptors in the child refer to the same open  file 
	 * descriptions (see open(2)) as the corresponding file 
	 * descriptors in the calling process.)  Subsequent operations 
	 * that open or close file descriptors, or change file descriptor 
	 * flags, performed by either the  calling process or the child 
	 * process do not affect the other process.
	 */
	public void copyProcTable( String oldPid, String newPid ) {			

		ProcTable pt = procTables.get( oldPid );
		if( pt == null ) procTables.put( newPid, null );
		else procTables.put( newPid, pt.clone() ); //FD kopieren							
	}
	

	public void shareProcTable( String oldPid, String newPid ) {			
		
		ProcTable pt = procTables.get( oldPid );
		procTables.put( newPid, pt.shareFd() ); // FD teilen
	}

	
	public void doCloseOnExec( String pid ) {			
				
		ProcTable pt = procTables.get( pid );		
		if( pt != null ) pt.doCloseOnExec();					
	}

	

	public String mergeResumed( String line ) {			
		
		String mergedLine = null;
		String pid = getPid( line );
		ProcTable pt = procTables.get( pid );

		// Wenn resumed die erste Zeile eines bislang
		// unbekannten Prozesses ist, gib null zurueck,
		// die Zeile wird dann ignoriert.
		if( pt != null ) mergedLine = pt.mergeParts( line );
		
		return mergedLine;
	}


	public void storeUnfinished( String line ) {			
		
		String pid = getPid( line );
		ProcTable pt = procTables.get( pid );

		// Wenn unfinished die erste Zeile eines bislang
		// unbekannten Prozesses ist, neue ProcTable erzeugen
		if( pt == null ){
			pt = new ProcTable( myParams );
			procTables.put( pid, pt );
		}

		pt.setUnfishedPart( line );
	}

	
	public void updateSequence( String pid, String call ) {			
		
		ProcTable pt = procTables.get( pid );
		int flat = call2flat(call);
		
		// Wenn es etwas zu speichern gibt und ProcTable
		// nicht existiert, neue ProcTable erzeugen
		if( flat > 0 ){
			if( pt == null ){
				pt = new ProcTable( myParams );
				procTables.put( pid, pt );
			}
			pt.putCall( flat );
		}
	}
	
	public String getSequence( String line ) {			
		
		String sequence = null;
		String pid = getPid( line );
		ProcTable pt = procTables.get( pid );

		if( pt != null ) sequence = pt.getSequence();
		
		return sequence;
	}

			
	public SysElement fixElement( String file, String call, String code ){		
		
		SysElement sysEl;
		
		//Init String
		String[] eStr = new String[ myParams.getDimS() ];
		
		if( myParams.pathDim() > -1 ){
			
			if( file == null || file.equals("") ) file = "*";
			if( file.length() > 1 ) file = cleanPath(file);
			
			eStr[ myParams.pathDim() ] = file;
		}
		
		//Init int
		String[] eInt = new String[ myParams.getDimI() ];
		
		for(int i = 0; i < myParams.getDimI(); i++ ){

			eInt[i] = "";

			switch( myParams.dimContents( i + myParams.getDimS() ) ){

			case CALL: 

				eInt[i] = getCode( call ) + "/" + code; 
				if( eInt[i] == null )
					throw new RuntimeException( "Kann den Code fuer " + eInt[0] +
					" nicht in der Taxonomie finden.");
				break;


			case SEQUENCE: 
								
				int[] callSeq = null;
				
				ProcTable pt = procTables.get( currentPid );
				if( pt != null ) callSeq = pt.getCalls();
				
				if( callSeq == null || callSeq.length == 1 ){
					eInt[i] = "*";
				}
				else{
					for( int k = 1; k < callSeq.length; k++ ){
						if( callSeq[k] > 0 ) eInt[i] += "/" + callSeq[k];  	
					}
				}								
				break;

				
			case RETURN:
				
				if( currentReturnValue.equals( "-1" ) )
					eInt[i] = "/2";
				else 
					eInt[i] = "/1";
					
				break;

			default: throw new RuntimeException("default");

			}

		}
				
		int[] res = Element.stringToInt( eInt, myParams );		
		sysEl = new SysElement( eStr, res, myParams );
		
		if( sysEl != null ) sysEl.capHierarchy( myParams.getCap() );
		
		return sysEl;
	}

	
	public SysElement parseLine( String line ){					
		
		SysElement sysEl;		
		String[] comps = line.split("\\|");
		String file;
		String clean;
		
		//Init String
		String[] eStr = new String[ myParams.getDimS() ];
		
		if( myParams.pathDim() > -1 ){
			file = comps[ myParams.pathDim() ].trim();
			if( file == null || file.equals("") ) file = "*";
			if( file.length() > 1 ) file = cleanPath(file);
			
			eStr[ myParams.pathDim() ] = file;
		}
		
		//Init int
		String[] eInt = new String[ myParams.getDimI() ];
		
		for(int i = 0; i < myParams.getDimI(); i++ ){

			eInt[i] = "";

			switch( myParams.dimContents( i + myParams.getDimS() ) ){

			case CALL: 
				
				clean = comps[ myParams.callDim() ].trim();
				
				eInt[i] = getFullCode( clean ); 
				if( eInt[i] == null )
					throw new RuntimeException( "Kann den Code fuer " + eInt[0] +
					" nicht in der Taxonomie finden.");
				break;


			case SEQUENCE: 
						
				//TODO: Leere Sequenz
				
				clean = comps[ myParams.seqDim() ].trim();
				String[] singleCall = clean.split("/");
				
				if( singleCall.length == 1 ){
					eInt[i] = "*";
				}
				else{
					for( int k = 1; k < singleCall.length; k++ ){
						eInt[i] += "/" + myParams.getTaxonomy().call2flat(singleCall[k]);
					}
				}					
				break;

				
			case RETURN:
				
				clean = comps[ myParams.retDim() ].trim();
				
				if( clean.equals( "/ERROR" ) ) eInt[i] = "/2";
				if( clean.equals( "/NO_ERROR" ) ) eInt[i] = "/1";
				if( clean.equals( "*" ) ) eInt[i] = "/0";
				if( eInt[i].equals("") ) throw new RuntimeException( "Kann " +
						"Rueckgabewert nicht aufloesen: " + clean );	
				
				break;

			default: throw new RuntimeException("default");

			}

		}
				
		int[] res = Element.stringToInt( eInt, myParams );		
		sysEl = new SysElement( eStr, res, myParams );
		
		if( sysEl != null ) sysEl.capHierarchy( myParams.getCap() );
		
		return sysEl;
	}

	
	
	public String getCode( String s ){
		return myParams.getTaxonomy().getCode( s );
	}
	
	
	public String getFullCode( String s ){
		int[] array = myParams.getTaxonomy().pathlabel2code( s );
		String code = "";
		for( int i = 1; i < array.length; i++ ){
			code = code + "/" + array[i];
		}		
		return code;
	}
	
	
	public int call2flat( String call ){
		return myParams.getTaxonomy().call2flat(call);
	}
	
	
	public String getCallName( String s ){
		int start = s.indexOf( " " );
		int stop = s.indexOf( "(" );
		if( start+1 > stop ) 
			System.out.println("Fehler in getCallName: Zeile = " + s + 
					", start = " + start + " stop= " + stop);
		return s.substring(start+1, stop).trim();
	}
	
	public boolean isUsed( String call ){
		return myParams.usedCalls().contains( call );
		
	}
	
	// Nach Posix ist der Pfad "/etc//////////sound" legal.
	// Firefox benutzt das, aus welchen Gruenden auch immer.
	// Bei mir trennt "/" die Hierarchieebenen, also muss das raus.
	public String cleanPath( String file ){
		
		if( file == null ) file = "/DateiUnbekannt";
		if( file.startsWith( "\"" ) ){
			file = file.substring( 1 );
			if( file.endsWith( "\"" ) ) 
				file = file.substring( 0, file.length() - 1 );
		}
		while( file.endsWith( "/" ) ) 
			file = file.substring( 0, file.length() - 1 );
		
		// Pfade vom Typ "/etc//////////sound" normieren
		int lev = 0;
		int k = file.indexOf( '/', 0 );
		int oldk = k;
		while( k >= 0){
			k = file.indexOf( '/', k + 1 );
			if( k == oldk + 1 ){ // Ueberschuessige slashs raus
				if( k == 1 ) file = file.substring( k );
				else file = file.substring(0, k-1) + file.substring( k );
				k--;					
			}
			else{
				lev++;
				oldk = k;
			}
		}
		return file;
	}

}
