package hitters.multi;

import hitters.tools.LogService;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.collect.HashMultiset;


/*

  	@author Peter Fricke


	Exakter Algorithmus zum Finden der Hierarchischen Heavy Hitters.
	Anzahl ist unabhaengig von N durch A/phi beschraenkt, A Laenge der
	laengsten Antikette.

	Optimierung: log N gespart durch Aggregation per Hashen statt
	Sortieren, sofern die Daten in den Speicher passen. 
	Zusaetzlich (myParams.optimizeOutput): Hitter, die
	nicht unter p liegen, muessen nicht getestet werden. Hitter h,
	fr die ein Hitter zwischen	p und h liegt, wegen Transitivitt
	ebenfalls nicht. 
	Ansonsten exakter Algorithmus (Overlap case, 
	Offline Algorithmus) wie beschrieben in Cormode04:   

 	@INPROCEEDINGS{1007588,
  		author = {Cormode, Graham and Korn, Flip and Muthukrishnan,
		S. and Srivastava, Divesh},
  		title = {Diamond in the rough: finding Hierarchical 
  		Heavy Hitters in multi-dimensional data},
  		booktitle = {SIGMOD '04: Proceedings of the 2004 ACM SIGMOD 
  		international conference on Management of data},
  		year = {2004},
  		pages = {155--166},
  		address = {New York, NY, USA},
  		publisher = {ACM},
  		doi = {http://doi.acm.org/10.1145/1007568.1007588},  
  		isbn = {1-58113-859-8},
  		location = {Paris, France}
  	}


 	Relativ kompakte Speicherung des Stroms durch Hashen der 
 	Elemente und Speicherung der Anzahl der Sichtungen.


 */
public class Exact extends AbstractHHH{

	// Kopie des Stroms, aggregiert durch Zaehlen der verschiedenen Elemente
	private HashMultiset<Element> streamCopy = HashMultiset.create();

	// Zum Speichern der wahren F- und f-Werte fuers Plotten,
	// nur falls in outputSet() log = true, nicht im normalen 
	// Betrieb einsetzen.
	HashMap<Element, Integer> trueF; 
	HashMultiset<Element> fCounter;

	private boolean logsUptodate = false;	

	public Exact(Parameter myParams) {
		super(myParams);
	}


	public void insert(Element e, int count, boolean increase){

		if( increase = false ) 
			throw new RuntimeException( "Exact: insert: " +
			"increase = false nicht sinnvoll. ");

		logsUptodate = false;		

		streamCopy.add( e, count );
		N = N + count;
	}


	// Standardaufruf ohne loggen
	public HashMap<Element, MultiHitterInfo> outputSet(double phi) {
		return this.outputSet(phi, false);
	}


	// Optionales loggen aller f und F-Werte
	public HashMap<Element, MultiHitterInfo> outputSet(double phi, boolean log) {				

		//Diskontierte Hufigkeit der Prfixe EINES Labels
		HashMultiset<Element> list = HashMultiset.create( N );		

		//Absolute Hufigkeit der Prfixe EINES Labels
		HashMultiset<Element> f = HashMultiset.create( N );	
		HashMap<Element, MultiHitterInfo> hInfoSet = 
			new HashMap<Element, MultiHitterInfo>();

		// Reduzierte HHH-Menge 
		Set<Element> usedHitters = new HashSet<Element>();
		
		//int[] label = new int[ par.getDim() ];
		List<int[]> labelList;
		boolean block;

		Element p = new Element( par );				

		if( log ){
			//Diskontierte Hufigkeit der Prfixe ALLER Label
			trueF = new HashMap<Element, Integer>();

			//Absolute Hufigkeit der Prfixe ALLLER Label
			fCounter = HashMultiset.create();
		}

		//Fr alle Ebenen
		for( int lv = par.getL(); lv >= 0; lv-- ){

			labelList = calcLabels(lv);
			for( int[] label : labelList ){

				list.clear();
				f.clear();

				LogService.log( LogService.EXACT, 
						" *** Exact Label: ["+Arrays.toString(label)+"] ***");


				// Optional: Optimierung durch Verkleinern der 
				// HHH-Menge, die ueberprueft werden muss
				if( par.optimizeOutput() )
					usedHitters = trimHitters( hInfoSet.keySet(), label);				
				else 
					usedHitters = hInfoSet.keySet();


				// Strom generalisierter Elemente								
				for( Element e : streamCopy.elementSet() ){
					LogService.log( LogService.EXACT, "Berabeite nun " + e);
					p.setTo( e );						

					if( p.generalizeTo( label ) == true ){ // Wenn generalisierbar, tu es
						LogService.log( LogService.EXACT, "Exact Absolut: Addiere " + 
								streamCopy.count(e) + " zu f fr p="+ p + " wegen " + e );						
						if( log ) fCounter.add( p.clone(), streamCopy.count( e ) );
						f.add( p.clone(), streamCopy.count( e ) );
						block = false;
						for( Element hitter : usedHitters ){
							if( hitter.isGenOf( e ) && p.isGenOf( hitter ) ) 
								block = true;
						}
						if( ! block ){
							list.add( p.clone(), streamCopy.count( e ) );
							LogService.log( LogService.EXACT, "Exact Discounted: Addiere " + 
									streamCopy.count(e) + " zu F fr p="+ p + " wegen " + e );
						}
					}
				}// Strom durchlaufen

				for( Element e : list.elementSet() ){							
					if( log ) trueF.put( e.clone(), list.count( e ) );
					if( list.count( e ) >= phi * N - 1E-9){						 

						MultiHitterInfo h = 
							new MultiHitterInfo( f.count(e), f.count(e), list.count(e) );																		
						hInfoSet.put( e, h );
						LogService.log( LogService.EXACT, e + 
								" mit h=" + h + 
								" wegen phi * N =" + (phi * N) );						 
					}						
				}
			}//Label beendet

			LogService.log( LogService.EXACT, "\nEbene beendet: " + lv );
		}

		LogService.log( LogService.EXACT, "Ergebnis: ");
		for( Element h : hInfoSet.keySet() ){		
			LogService.log( LogService.EXACT, h.toShortString() + " " + hInfoSet.get(h) );			
		}

		logsUptodate = log;

		return hInfoSet;
	}		


	public int getN() { 		
		return N; 
	}


	// Strom wird gespeichert (verschiedene Elemente gespeichert, jeweils gezaehlt)
	public int getTupelCount() { return streamCopy.elementSet().size(); }		


	// Kein Compress -> MaxTupelCount == TupelCount 
	public int getMaxTupelCount() { return getTupelCount(); }	


	public HashMap<Element, Integer> dumpF( double phi ){ 

		this.outputSet( phi, true );

		return trueF; 
	}

	
	public HashMap<Element, Integer> dumpf(){ 

		HashMap<Element, Integer> truef = 
			new HashMap<Element, Integer>();

		//Output ausfuehren, phi egal
		if( ! logsUptodate ) this.outputSet(1.0, true); 

		//Umkopieren fuer einheitliche Schnittstelle		
		for( Element e : fCounter.elementSet() ) 
			truef.put( e, fCounter.count(e) ); 

		return truef; 
	}

	
	public int instantiateDelta(Element e){ return 0; }

}
