/**
 * 
 */
package hitters.multi;

import hitters.tools.LogService;
import hitters.tools.Utils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;


/**
 * @author Peter Fricke
 * 
 *  Abstrakte Klasse, die die approximativen Algorithmen zur Berechnung
 *  der Hierarchischen Heavy Hitters umfasst und die gemeinsame Funktionalitaet
 *  von Partial Ancestry und Full Ancestry enthaelt.
 *  Die drei abstrakten Methoden specializedInsert, checkChildren
 *  und updateNextParentSet umfassen die einzigen Punkte, in denen
 *  sich Full Ancestry und Partial Ancestry unterscheiden,
 *  die Klassen FullAncHHH und PartAncHHH implementieren diese und
 *  erhalten die gesamte uebrige Funktionalitaet von AbstractComplexHHH.
 *
 *
 *  
 *  Optimierungsmglichkeiten Speicherbedarf fr Onlinevariante: 
 *  Strings abschaffen, Objekte weitgehend abschaffen, Hashen von Hand usw.
 *  Letztlich lohnt das nicht, weil man in Java den Objektoverhead und den 
 *  Overhead fr die virtuelle Maschine nie los wird. Wenn das Verfahren 
 *  tatschlich online eingesetzt werden soll, ist es sinnvoller, alles 
 *  in C/++ zu implementieren und die Java-Variante als Referenz 
 *  und zum Testen verwenden. Da eine lauffhige Java-Version vorliegt, 
 *  sollte der Aufwand im Vergleich zur Java-Implementierung erheblich 
 *  geringer sein: Die Fehler im Cormode-Papier sind identifiziert, 
 *  die sonstigen Subtleties ebenfalls, der Java-Code kann als Referenz 
 *  verwendet werden, die erneute aufwndige Generierung von Testfllen 
 *  entfllt und der neue Code kann gegen den Java-Code 
 *  auf Korrektheit getestet werden.
 *  
 *  
 *  
 *  
 */
public abstract class AbstractComplexHHH extends AbstractHHH {

	final double epsilon;
	final int w;
	int bcurrent   = 1;	

	private final boolean OPTIMIZE = false;	

	// NICHT benutzen, siehe Kommentare Compress und instantiateDelta
	private final int 	  QUICK    = 1;	
	private final int 	  STANDARD = 3; 
	private final int	  PARENT   = 4;
	// Naiv: Benutze bcurrent - 1 wie in LossyCounting
	private final int	  SILLY	   = 2; 

	private final int 	  deltaMode  = STANDARD;//SILLY;

	// Datenstruktur, implementiert als HashMap je Level. 
	// Zugriff auf die Vorfahren bers Prfix,
	// Zugriff auf Nachkommen nicht erforderlich. 
	private final ArrayList< HashMap<Element, MultiStringElement> > levelSets;
	private final ArrayList< HashMap<Element, QueueElement> > levelQueues;

	Element tst;	

	protected Element aTemp;
	private MultiStringElement mTemp = new MultiStringElement(0, 0, 0);

	private int[] label;
	private int[] depth;
	private int[] pos;
	private int[] hi;
	
//	private boolean[][] reachable = 
//		new boolean[ par.gethi(0) + 1 ][ par.gethi(1) + 1 ];
	BitSet bits = new BitSet( par.getH() );
	
	// Nur zum Loggen
	public int reach = 0;
	public int unreach = 0;
	public int[] counter = new int[ par.getL() + 1 ];
	public int[] success = new int[ par.getL() + 1 ];
	public int[] sum     = new int[ par.getL() + 1 ];

	// Speichern der geschaetzten absoluten und diskontierten Haeufgkeiten
	// aller Praefixe zum Loggen und Plotten,
	// nicht im Produktionsbetrieb verwenden, ein wenig zu teuer.
	HashMap<Element, Integer> fLog = new HashMap<Element, Integer>();
	HashMap<Element, Integer> estFLog = new HashMap<Element, Integer>();
	boolean logsUptodate = false;


	public AbstractComplexHHH( double epsilon, Parameter p ){	
		super( p );

		this.epsilon = epsilon;
		if( epsilon > 0.0000001 ) w = (int)Math.ceil( 1.0 / epsilon );
		else w = Integer.MAX_VALUE;			

		label = new int[ par.getDim() ];	
		depth = new int[4];
		pos = new int[4];
		hi = new int[4];
		for( int dim = 0; dim < 4; dim++ ){
			if( dim < par.getDim() ) hi[dim] = par.gethi(dim);
			else hi[dim] = 0;
		}
		
		levelSets = 
			new ArrayList< HashMap<Element, MultiStringElement> >( par.getL() + 1 );
		levelQueues = 
			new ArrayList< HashMap<Element, QueueElement> >( par.getL() + 1 );

		for( int i = 0; i <= par.getL(); i++ ){
			levelSets.add( i, new HashMap<Element, MultiStringElement>() );					
			levelQueues.add( i, new HashMap<Element, QueueElement>() );
		}							
	}	


	public void insert(Element e, int count, boolean increase) {		
		insertWork(e, count, true, increase);				
	}


	/* (non-Javadoc)
	 * 
	 * 
	 * Teure Methode, Leistung wird auch von InstantiateDelta 
	 * beeinflusst. 
	 * 
	 * Ziel von InstantiateDelta ist es, durch Befragen der Vorfahren
	 * Informationen darber zu erhalten, welcher Delta-Wert dem einzufgenden
	 * Knoten zugewiesen werden muss. Dabei soll ein besserer
	 * Wert als der Standardwert bcurrent - 1 gefunden werden, um mit dieser
	 * besseren, engeren Schranke den Knoten bei Bedarf schneller wieder 
	 * lschen zu knnen.
	 * Das Befragen ALLER Vorfahren, zu denen ein nicht blockierter Pfad 
	 * existiert, ist zulssig und fhrt zu einer engen Schranke, ist aber 
	 * auch teuer. 
	 * Da das Befragen entfernter Vorfahren und das Testen dieser auf Existenz
	 * eines unblockierten Pfades offenbar nur relativ selten zu wesentlich
	 * engeren Schranken fhrt (TODO: Grndlicher Testen), gibt es die Option
	 * QUICK, die nur Eltern und Groeltern befragt. Die Zeitersparnis ist 
	 * wegen der sehr hufigen Ausfhrung der Methode erheblich, der 
	 * zustzliche Platzbedarf durch die weniger enge Schranke scheint 
	 * gering zu sein.
	 * 
	 * UPDATE: Nach weiten Tests scheinbar NICHT KORREKT, auch ferne 
	 * Vorfahren wichtig, QUICK KEINE GUTE OPTION.
	 *  
	 * Nach noch mehr Tests sieht es wieder umgekehrt aus, Quick und Silly sparen
	 * erheblich Laufzeit bei nur etwas schlechterer Approxgte.
	 *  
	 * Die Befragung der Vorfahren ist nicht effizient, gilt die Aussage 
	 * auch noch bei einer besseren Umsetzung? 
	 * 
	 */
	public void compress() {

		HashMap<Element, MultiStringElement> lSet;		
		HashMap<Element, QueueElement>		 parentQueue = null;
		HashMap<Element, QueueElement>		 grandPQueue = null;

		int parcount = 0, maxM;
		boolean inDomain;			

		MultiStringElement mse;
		QueueElement tmp;
		Element e, p = aTemp.clone();		

		// Fuer FullAnc duerfen nur kinderlose Knoten
		// geloescht werden. 
		// childrenOK == true fuer PartAnc immer
		// childrenOK == true fuer FullAnc gdw. e kinderlos
		boolean childrenOK;

		// FullAnc merkt sich, welche Knoten auf der naechsthoeheren
		// Ebene Eltern sind, um kinderlose Knoten erkennen
		// zu koennen. PartAnc tut nichts.
		HashSet<Element> parentSet = new HashSet<Element>();		

		logsUptodate = false;

		LogService.logF( "Complex. N, Tupelcount: " + N + " " + getTupelCount() );

		if( LogService.shouldLog( LogService.COMPR_NICE ) ){ 
			LogService.log(LogService.COMPR_NICE,"\nDatenstruktur vor compress" +
					" (bcurrent = " + bcurrent + "):");
			List<Map.Entry<Element, MultiStringElement>> dump = dumpTrie();
			for( Map.Entry<Element, MultiStringElement> entry : dump ){	 
				LogService.log( LogService.COMPR_NICE, entry );
			}
			LogService.log(LogService.COMPR_NICE, "\n");
		}

		for( int l = par.getL(); l >= 0; l-- )
			if( levelQueues.get( l ).size() > 0 ) 
				throw new RuntimeException("PQ nicht leer");


		//Fr alle Ebenen
		for( int l = par.getL(); l >= 0; l-- ){

			//Fr alle Knoten aus der Ebene	
			lSet = levelSets.get(l);
			Iterator<Element> it = lSet.keySet().iterator();			

			if( l >= 1) parentQueue = levelQueues.get( l - 1 );
			if( l >= 2 ) grandPQueue = levelQueues.get( l - 2 );

			while( it.hasNext() ){
				e = it.next();
				mse = lSet.get( e );

				// Fuer FullAnc duerfen nur kinderlose Knoten
				// geloescht werden. 
				// childrenOK == true fuer PartAnc immer
				// childrenOK == true fuer FullAnc gdw. e kinderlos
				childrenOK = this.checkChildren( e, parentSet );

				if( childrenOK && Math.abs(mse.g) + mse.delta <= bcurrent ){

					if( LogService.shouldLog( LogService.COMPR_NICE ) ) 
						LogService.log(LogService.COMPR_NICE, "Compress -> " +
								"Bereite remove vor fuer " + e + 
								" wegen g = " + mse.g + ", delta = " +
								mse.delta + ", bcurrent " + bcurrent );					

					int limit = (int) Utils.exp( 2, par.getDim() );

					jloop : for( int j = 1; j < limit; j++ ){
						parcount = 0;
						inDomain = true;
						p.setTo(e);
						for( int i = 0; i < par.getDim(); i++ ){
							if( (j & ( 1 << (par.getDim()-i-1) ) ) != 0 ){
								parcount++;
								if( ! p.turnIntoParent( i ) ) {
									inDomain = false;
									continue jloop;
								}								 
							}							
						}
						if( inDomain ){ //Hashmap whrend while-Schleife nicht verndern														
							int fac = 2 * (parcount % 2) - 1;													
							if( LogService.shouldLog( LogService.COMPR_NICE ) ) 
								LogService.log( LogService.COMPR_NICE, "Compress " +
										"-> Insert fuer " + p + " mit "+ mse.g * fac );
							if( parcount == 1 ){ 	

								// Hier ruht MBUG, wenn false verschwinden keine Werte mehr
								if( par.mBug() == false ){
									maxM = Math.max( Math.abs(mse.g) + mse.delta, mse.m );				
								}
								else maxM = Math.abs(mse.g) + mse.delta;

								if( ! parentQueue.containsKey( p ) ){
									tmp = new QueueElement( mse.g * fac, true, maxM );
									parentQueue.put( p.clone(), tmp );
								}else{
									tmp = parentQueue.get( p );
									tmp.increase = true;
									tmp.g += (mse.g * fac);
									tmp.maxM = Math.max( tmp.maxM, maxM );									
								}
							}
							else{
								if( ! grandPQueue.containsKey( p ) ){
									tmp = new QueueElement( mse.g * fac, false, 0 );
									grandPQueue.put( p.clone(), tmp );
								}else{
									tmp = grandPQueue.get( p );									
									tmp.g += (mse.g * fac);																		
								}								
							}
						}
					}
					if( LogService.shouldLog( LogService.COMPR_UGLY ) ) 
						LogService.log(LogService.COMPR_UGLY, "Compress " +
								"-> Entferne " + e  );
					it.remove();
					tupelCount--;
				}												
			}
			//Iteration beendet, darf HashSet nun ndern

			//Optimize:
			//Man kann das Einfgen von Knoten, die
			//auf dem nchsten Level wieder gelscht werden, verhindern
			//und gleich das Ergebnis der Lschung auf dem bernchsten 
			//Level verwenden. Man darf dabei das teure instantiateDelta nicht
			//doppelt berechnen, kann aber das hier berechnete delta
			//wieder verwenden (insertWork bernimmt delta).
			//Abdeckung durch Testflle erfordert noch grere Flle,
			//-> noch ein paar erzeugen und Optimierung optional machen.
			//Zum Testen erstmal nur fr 2 Dimensionen.
			// Ergebnis: Bringt im Profiler und bei sehr kleinen 
			// Datenstzen etwas, bei groen Datenstzen macht  
			// die just-in-time Complilierung offenbar alles zunichte.
			//
			// Darum vorerst nicht weiter verfolgt, mbug in diesem
			// Teil nicht behoben, darum auskommentiert.

			if( l > 0 ){				
				for( Element q : parentQueue.keySet() ){
					tmp = parentQueue.get( q );
					if( LogService.shouldLog( LogService.COMPR_UGLY ) ) 
						LogService.log( LogService.COMPR_UGLY, "level " +
								l + " Compress -> Insert (Queue) " + q +
								" mit "+ tmp.g +" und increase = " + 
								tmp.increase + " und MaxM " + tmp.maxM +
								" Level(q)=" + q.getL() );

					if( ! OPTIMIZE ){ // Standardvariante
						// false: Kein compress und Insert darf den Zhler N nicht erhhen
						insertWork( q, tmp.g, false, false ); 						
						if( tmp.increase ){
							MultiStringElement parent = levelSets.get( l - 1 ).get( q );						
							parent.m = Math.max( parent.m, tmp.maxM );
						}				
					}
					else{	// Optimierte Variante 	

						throw new RuntimeException( "MBUG (Verschwinden von Werten)" +
						" in Optimize nicht behandelt " );
					}//Else optimierte Variante
				}//Element q : parentQueue.keySet()
			}// if l > 0
			parentQueue.clear();

			// Verwaltung der Information, welche Knoten Eltern sind.
			// Verwendet nur von FullAnc.
			parentSet.clear();

			// FullAnc merkt sich, welche Knoten auf der naechsten
			// Ebene Eltern sind, PartAnc tut nichts.
			for( Element el : lSet.keySet() )
				this.updateNextParentSet( el, parentSet );

		}

		if( LogService.shouldLog( LogService.COMPR_NICE ) ){ 
			LogService.log(LogService.COMPR_NICE,"\nDatenstruktur nach compress" +
					" (bcurrent = " + bcurrent + " und wird nun erhoeht):");
			List<Map.Entry<Element, MultiStringElement>> dump = this.dumpTrie();
			for( Map.Entry<Element, MultiStringElement> entry : dump ){
				LogService.log( LogService.COMPR_NICE, entry );
			}
			LogService.log(LogService.COMPR_NICE, "\n");
		}
	}


	public void output(double phi) {
		HashMap<Element, MultiHitterInfo> hInfoSet = outputSet( phi );

		for( Element h : hInfoSet.keySet() ){		
			System.out.println( h + " " + hInfoSet.get(h) );			
		}
	}


	// Standardvariante ohne Loggen der disk. und abs. Haeufigkeiten
	// aller Praefixe.
	@Override
	public HashMap<Element, MultiHitterInfo> outputSet(double phi) {
		return outputSet( phi, false );
	}


	/*  
	 * Ausgabe der Menge der HHH.
	 * 
 	 * Fuer die Berechnung der diskontierten Haeufigkeit 
 	 * von Elementen eines Labels wird eine reduzierte HHH-Menge
 	 * verwendet: Es sind nur solche HHH von Interesse, die a) Nachfahren
	 * von Elementen des Labels sind und b) undominiert
 	 * von anderen HHH sind. Wird berechnet ueber trimHitters().
	 * 
	 * Laufzeit ist dann nicht mehr kubisch in der Groesse von P,
	 * sondern quadratisch in der Groesse der (kleineren) 
	 * Menge usedHitters. Da beide Mengen sehr klein sind,
  	 * ist der Effekt allerdings nicht allzu gross.
  	 * 
	 */	
	public HashMap<Element, MultiHitterInfo> outputSet(double phi, boolean log) {

		if( N < 1 ) return simpleSet();
		
		// f und F: Geschaetzte absolute und diskontierte Haeufigkeiten 
		// der Praefixe eines Labels, werden nach Abarbeitung des 
		// Labels geleert.
		HashMap<Element, Integer> F = new HashMap<Element, Integer>();
		HashMap<Element, Integer> f = new HashMap<Element, Integer>();

		// Bisher gefundene HHH auf tieferen Ebenen als der aktuellen
		HashMap<Element, MultiHitterInfo> hInfoSet = 
			new HashMap<Element, MultiHitterInfo>();		
		
		// Bisher gefundene HHH auf der aktuellen Ebene
		HashMap<Element, MultiHitterInfo> tempHitterSet =
			new HashMap<Element, MultiHitterInfo>();

		// Elemente der aktuellen Ebene
		HashMap<Element, MultiStringElement> lSet;

		// Reduzierte HHH-Menge: Fuer die Berechnung der 
		// diskontierten Haeufigkeit von Elementen eines Labels
		// sind nur solche HHH von Interesse, die a) Nachfahren
		// von Elementen des Labels sind und b) undominiert
		// von anderen HHH sind. Wird berechnet ueber trimHitters().
		Set<Element> usedHitters = new HashSet<Element>();
				
		List<int[]> labelList;
		Element p = aTemp.clone();
		Element meet = aTemp.clone();
		MultiStringElement m;
		boolean block;
		int delta, intF;

		if( log ){
			fLog.clear();
			estFLog.clear();
		}

		if( LogService.shouldLog( LogService.DUMPTRIE ) ){ 
			LogService.log( LogService.DUMPTRIE, "Zustand der Datenstruktur vor Output: " );		
			for( Entry<Element, MultiStringElement> entr : dumpTrie() ){			 
				LogService.log( LogService.DUMPTRIE, entr.getKey().toShortString() +
						" " + entr.getValue().toShortString() );					
			}
		}

		if( LogService.shouldLog( LogService.OUTPUTLOG ) ){ 
			LogService.log(LogService.OUTPUTLOG, "Output: " + new Date() );
			LogService.log(LogService.OUTPUTLOG, "bcurrent= " + bcurrent + " N=" + N);
		}


		//Fr alle Ebenen
		for( int lv = maxLevel(); lv >= 0; lv-- ){

			labelList = calcLabels(lv);
			for( int[] label : labelList ){

				if( LogService.shouldLog( LogService.OUTPUTLOG ) ) 
					LogService.log(LogService.OUTPUTLOG, "\n *** " +
							"Label: " + Arrays.toString(label) + "  ***");

				// Optimierung durch Verkleinern der 
				// HHH-Menge, die ueberprueft werden muss
				usedHitters = trimHitters( hInfoSet.keySet(), label );
				
				//Fr alle e. 
				for( int j = maxLevel(); j >= lv; j-- ){ 

					lSet = levelSets.get(j);
					for( Element e : lSet.keySet() ){

						p.setTo( e );						
						if( p.generalizeTo( label ) == true ){ // Wenn generalisierbar, tu es							

							m = lSet.get( e );
							if( ! f.containsKey( p ) ) f.put( p.clone(), m.g);
							else f.put( p , f.get( p ) + m.g);

							if( log ){
								if( ! fLog.containsKey( p ) ) fLog.put( p.clone(), m.g);
								else fLog.put( p , fLog.get( p ) + m.g);														
							}
							block = false;
							for( Element h : usedHitters ){ 

								//Hier ruht bProb alias B1, Bug im Paper S.29, 
								//korrekt im Pseudocode
								if( par.bProb() ){
									if(  h.isGenOf( e ) ) block = true;									
								}
								else{
									if( p.isGenOf(h) && h.isGenOf( e ) ) block = true;	
								}
							}
							if( ! block ){
								if( ! F.containsKey( p ) ) F.put( p.clone(), m.g );
								else F.put( p , F.get( p ) + m.g);
								if( log ){
									if( ! estFLog.containsKey( p ) ) estFLog.put( p.clone(), m.g);
									else estFLog.put( p , estFLog.get( p ) + m.g);
								}

								if( LogService.shouldLog( LogService.OUTPUTLOG ) ) 
									LogService.log(LogService.OUTPUTLOG, "Addiere g = "
											+ m.g + " zu F (F nun " + F.get( p ) + ") fr" +
											" p = " + p + "\nwegen e = " + e );					
							}
						}						
					}
				}				


				// Fr alle h aus P
				for( Element h : usedHitters ){
					p.setTo( h );					
					if( p.generalizeTo( label ) ){
						block = false;
						for( Element q : usedHitters ){
							if( ( ! h.equals(q) ) && p.isGenOf(q) 
									&& q.isGenOf( h ) ) block = true;													
						}
						if( ! block ){	
							m = null;
							m = trieGet( h );							
							if( m != null ) delta = m.delta;	
							else delta = instantiateDelta( h ); 
							if( ! F.containsKey( p ) ) F.put( p.clone(), delta);
							else F.put( p , F.get( p ) + delta);
							if( LogService.shouldLog( LogService.OUTPUTLOG ) ) 
								LogService.log(LogService.OUTPUTLOG, "Addiere " +
										"Delta-h = " + delta + 
										" zu F (F nun " + F.get( p ) + ") fr p = " + p + 
										"\nwegen h = "+ h );
						}
					}
				}
				
				// Fr alle h, h'=k aus usedHitters
				// Pseudocode Zeilen 13 - 16
				// Weil hier mit der Menge usedHitters statt P gearbeitet
				// wird, muessen einige Dinge (Zeile 15), die im Pseudocode
				// geprueft werden, nicht erneut geprueft werden:
				
				// Die Pruefung (Z.15), ob kein q in P existiert, entfaellt,
				// weil usedHitters nur undominierte HHH enthaelt.
				
				// Die im Pseudocode (Z.15) im Originalpaper fehlende Pruefung,
				// ob das Hitterpaar Nachfahr von p ist entfaellt, weil usedHitters
				// nur Nachfahren von Elementen enthaelt, die am Label wohnen.
				// Wenn das Infimum eines Paares solcher Nachfahren Nachfahr
				// eines bestimmten Elementes p ist, ist auch das Paar Nachfahr
				// von p.
				
				// Laufzeit ist dann nicht mehr kubisch in der Groesse von P,
				// sondern quadratisch in der Groesse der (kleineren) 
				// Menge usedHitters. Da beide Mengen sehr klein sind,
				// ist der Effekt allerdings nicht allzu gross.
				for( Element h : usedHitters ){    
					for( Element k : usedHitters ){								
						// Paare nicht doppelt, und nur ungleiche Elemente
						// Infimum muss existieren
						if( ( h.compareTo(k) < 0 ) && meet.setToGlbOf(h, k) ){
							p.setTo( meet );												
							if( p.generalizeTo( label ) ){									
									MultiStringElement e = trieGet( meet ); 									
									
								 	// Wenn meet in Datenstruktur ist alles gut
									if( e != null) delta = e.delta;
									// Sonst berechne das Delta, das er htte, 
									//wenn er in der Datenstruktur wre											
									else delta = instantiateDelta( meet );
									
									if( ! F.containsKey( p ) ) F.put( p.clone(), delta );
									else F.put( p , F.get( p ) + delta);		
									if( LogService.shouldLog( LogService.OUTPUTLOG ) )
										LogService.log(LogService.OUTPUTLOG, 
												"Addiere glb-Delta = " + 
												delta +	" zu F (F nun " +
												F.get( p ) + ") fr p = " + 
												p + "\nwegen h = " + h + 
												" und k = " + k + 
												"\nund meet = " + meet );									
							}
						}
					}		
				}
				
				// Nicht alle p aus Level(l) mssen in der Datenstruktur sein,
				// wenn sie nicht vorhanden sind, muss man sich einen
				// Delta-Wert fr sie ausdenken...
				for( Element e : f.keySet() ){
					//if( e.getL() == l ){
					if( F.get(e) == null ) intF = 0;
					else intF = F.get(e); 
					if( log ) estFLog.put( e.clone(), intF );
					//}

					if( ( f.get( e ) > 0 ) && ( e.getL() == lv ) ){
						m = trieGet( e );
						if( m != null ) delta = m.delta;
						else delta = instantiateDelta( e ); 
						if( F.get(e) == null ) intF = 0;
						else intF = F.get(e); 					
						if( intF + delta >= phi * N - 1E-9 ) {
							MultiHitterInfo h = 
								new MultiHitterInfo( f.get(e) - delta, 
										f.get(e) + delta, intF );						
							tempHitterSet.put( e, h );						
							if( LogService.shouldLog( LogService.OUTPUTLOG ) ) 
								LogService.log(LogService.OUTPUTLOG, "Fge zu" +
										" HHH hinzu: " + e + " mit" + h +
										" wegen phi * N = " + (phi * N) + 
										" delta=" + delta );
						}				
					}
				}
				f.clear();
				F.clear();

			}//Ende fr alle Label aus level(l)

		if( LogService.shouldLog( LogService.OUTPUTLOG ) ) 
			LogService.log(LogService.OUTPUTLOG, "\nEbene beendet: " + lv);			

		hInfoSet.putAll( tempHitterSet );
		tempHitterSet.clear();		

		}// Ende fr alle Ebenen

		logsUptodate = logsUptodate || log;

		if( LogService.shouldLog( LogService.OUTPUTLOG ) ) 
			LogService.log(LogService.OUTPUTLOG,  "Output beendet: " + new Date() );

		if( LogService.shouldLog( LogService.OUTPUTHHH ) ){
			LogService.log(LogService.OUTPUTHHH, "Output beendet: ");
			for( Element h : hInfoSet.keySet() ){		
				LogService.log(LogService.OUTPUTHHH, h.toShortString() +
						" " + hInfoSet.get(h).toShortString() );			
			}
		}
		return hInfoSet;
	}


	/*
	 * Berechnung des Delta-Wertes anhand der m-Werte fr einen Knoten,
	 * der nicht in der Datenstruktur vorhanden ist, dessen Delta-Wert aber zur Berechnung
	 * der F-Werte bentigt wird. Dieser Fall tritt zum Bleistift auf, wenn 
	 * die glb zweier HHH aus Hp nicht in der Datenstruktur ist.
	 * 
	 * Wird ebenfalls benutzt, um den Delta- und m-Wert neu einzufgender
	 * Knoten zu bestimmen.
	 * 
	 * Die Berechnung erfolgt durch simuliertes Einfgen.
	 *  
	 * 
	 */
	public int instantiateDelta(Element e){
		int delta  = bcurrent - 1;
		int level  = e.getL();		
//		int level0 = e.getLevel( 0 );
//		int level1 = e.getLevel( 1 );

		HashMap<Element, MultiStringElement> set = levelSets.get( level );				

		if( aTemp == null ) aTemp = e.clone();
		mTemp = set.get( e );

		if( mTemp != null )	return mTemp.delta;
		else{

			switch( deltaMode ){

			case SILLY:

				// Tu nichts, benutze bcurrent - 1.
				// Korrekt, aber unntig lose Schranke,
				// erschwert Lschungen unntig.

				break;

			case PARENT:

				for( int i = 0; i < par.getDim(); i++ ){
					aTemp.setTo( e );
					if( aTemp.turnIntoParent( i ) ){		//Eltern
						mTemp = levelSets.get( level - 1 ).get( aTemp );
						if( mTemp != null )	delta = Math.min( delta, mTemp.m );														
					}
				}				
				break;

			case QUICK: 						

				for( int i = 0; i < par.getDim(); i++ ){
					aTemp.setTo( e );
					if( aTemp.turnIntoParent( i ) ){		//Eltern
						mTemp = levelSets.get( level - 1 ).get( aTemp );
						if( mTemp != null )	delta = Math.min(delta, mTemp.m);														
						for( int j = 0; j < par.getDim(); j++ ){
							//Groeltern, doppelt strt nicht
							if( aTemp.turnIntoParent( j ) ){
								mTemp = levelSets.get( level - 2 ).get( aTemp );
								if( mTemp != null )	delta = Math.min( delta, mTemp.m );																						
							}					
						}	
					}					
				}								
				break;

				
			case STANDARD: 

				bits.clear();

				for( int dim = 0; dim < 4; dim++ ){
					if( dim < par.getDim() ) depth[dim] = e.getLevel( dim );
					else depth[dim] = 0;
				}				

				for( int dim = 0; dim < 4; dim++ ){
					if( depth[dim] > 0 ){
						copy(depth, pos);
						pos[dim]--;
						setReachable( pos );					
					}
				}

				for( pos[3] = depth[3]; pos[3] >= 0; pos[3]-- ){
					for( pos[2] = depth[2]; pos[2] >= 0; pos[2]-- ){
						for( pos[1] = depth[1]; pos[1] >= 0; pos[1]-- ){
							for( pos[0] = depth[0]; pos[0] >= 0; pos[0]-- ){

								if( isReachable(pos) ){

									//Projektion
									for( int dim = 0; dim < par.getDim(); dim++ ){
										label[dim] = pos[dim];
									}
									aTemp.setTo( e );
									aTemp.generalizeTo( label );						
									mTemp = trieGet( aTemp );

									if( mTemp != null ){
										delta = Math.min(delta, mTemp.m);								
									}				
									else{								
										for( int dim = 0; dim < 4; dim++ ){
											if( pos[dim] > 0 ){										
												pos[dim]--;
												setReachable( pos );
												pos[dim]++;
											}
										}
									}
								}//End  if( isReachable(pos) )
								
							}
						}
					}
				}
				break;

			default: throw new RuntimeException("deltaMode " +
			"fr InstantiateDelta nicht angegeben.");
			}

		}
		return delta;
	}


	public void setN(int N) {
		this.N = N;
		bcurrent = (int)Math.ceil( N/((double)w) );
	}


	public double getEpsilon() { return epsilon; }


	public int getW() { return w; }


	public HashMap<Element, Integer> dumpF(double phi) {	
		outputSet(phi, true);
		return estFLog;
	}


	public HashMap<Element, Integer> dumpf() {	
		if( ! logsUptodate )
			outputSet(1.0, true);
		return fLog;
	}


	public MultiStringElement trieGet( Element e ){		
		MultiStringElement m = levelSets.get( e.getL() ).get( e );
		return m;
	}


	public void triePut( Element e, MultiStringElement m ){		
		levelSets.get( e.getL() ).put( e, m );		
	}


	/* (non-Javadoc)
	 * 
	 * setTest:
	 * Erweiterte Einfge-Methode zum Erleichtern des Testens,
	 * Datenstrukturen lassen sich so direkt in den gewnschten 
	 * Zustand bringen und das Verhalten des Algorithmus lt 
	 * sich leichter testen.
	 * 
	 * Nicht ungefhrlich, weil sich so unsinnige Zustnde basteln 
	 * lassen, die auf natrlichem Weg nicht erreichbar sind 
	 * (insbesondere muss N korrekt gesetzt werden).  
	 *  
	 */
	public void setTest(Element e, MultiStringElement m) {
		if( aTemp == null ) aTemp = e.clone();
		levelSets.get( e.getL() ).put( e, m );		
	}		


	public void insertWithoutCompress(Element e, int count, boolean increase) {		
		insertWork( e, count, false, increase);
	}


	/* Ausgabe der Datenstruktur zu Testzwecken */
	public List<Map.Entry<Element, MultiStringElement>> dumpTrie() {
		ArrayList<Map.Entry<Element, MultiStringElement>> res = 
			new ArrayList<Map.Entry<Element, MultiStringElement>>();
		for( HashMap<Element, MultiStringElement>  map : levelSets ){
			res.addAll( map.entrySet() );			
		}		
		Collections.sort( res, new NodeComparator() );
		return res;
	}


	public ArrayList< HashMap<Element, MultiStringElement> > dumpLevelSets(){
		return levelSets;
	}


	/* Ausgabe eines einzelnen Knotens zu Testzwecken */
	public MultiStringElement dumpElement( Element a ) {		
		MultiStringElement elem = levelSets.get( a.getL() ).get( a );
		return elem;
	}		

	/* 
	 * 
	 * Erledigt die eigentliche Arbeit fr die vielen Insert-Methoden.
	 * 
	 * Erste Fassung: Lose Schranke bcurrent fr m und delta.
	 * Zweite Fassung: Enge Schranken implementiert. 
	 * 
	 * Fllt etwas lnger aus als im Pseudocode, weil "for p in ancestors"
	 * bedeutet, dass alle Vorgnger, zu denen es einen nicht durch einen
	 * anderen Vorgnger blockierten Pfad gibt, nach ihrem m-Wert befragt
	 * werden drfen und dann das Minimum gebildet werden kann. 
	 * Siehe auch MBUG.
	 * 
	 */
	protected void insertWork( Element e, int count, boolean compress, boolean increase ){
		insertWork( e, count, compress, increase, -1 );
	}


	private void insertWork( Element e, int count, boolean compress, boolean increase, int delta ) {

		if( LogService.shouldLog( LogService.NORMAL ) )
			LogService.log(LogService.NORMAL, "insert " + e +
					" count=" + count);

		if( aTemp == null ) aTemp = e.clone();		
		mTemp = this.trieGet( e );

		logsUptodate = false;
		
		// Items koennen unterschiedlich gewichtet werden,
		// z.B. um analog zu TFIDF seltene Items besser zu beruecksichtigen.
		// Siehe Erklaerung in Klasse Parameter.
		//if( myParams.useWeights() ){
			// Hier muessten die counts angepasst werden,
			// Umstellung der counts auf double.
			// Aenderungen: Hier, in MultiStringElement muessen
			// g, m und delta zu double werden, in MultiHitterinfo
			// fmin, fmax und F, in allen benutzenden Klassen 
			// ebenso. In Exact , Plotter usw. jeweils
			// die Hasherei von int zu double usw.
		//}

		if( mTemp != null ) mTemp.g += count;
		else{		

			specializedInsert( e, count, delta );

			tupelCount++;			
		}

		// Die Angabe bcurrent = ceil(epsilon * N) im Paper ist ungenau,
		// der genaue Zeitpunkt der Erhoehung von bcurrent, der 
		// Ausfuehrung von compress und einer mglichen Ausfuehrung 
		// von output sind entscheidend.
		//
		// - compress muss zwischen den Fenstern ausgefhrt werden. Dabei
		//   behlt bcurrent whrend compress den alten Wert, denn Knoten,
		//   die whrend compress erzeugt werden, erhalten so eine engere 
		//   obere Schranke des Fehlers delta = bcurrent - 1. Der Deltawert 
		//   ist korrekt, weil Knoten, die whrend compress	erzeugt werden, 
		//   wegen des ebenenweisen Vorgehens nicht whrend desselben compress
		//   gelscht wurden.
		// - Sofort nach compress und bereits vor Verarbeitung des nchsten
		//   Items wird bcurrent erhht. Eine eventuell zwischen den 
		//   Fenstern aufgerufene Outputmethode arbeitet bereits auf
		//   dem hheren bcurrent (selbst wenn kein einziges neues Item 
		//   eingefgt wurde), weil Knoten, die whrend Output erzeugt 
		//   werden, im gerade vorangegangenen compress gelscht worden 
		//   sein knnten. Da ihr Delta auf bcurrent - 1 gesetzt wird,
		//	 muss bcurrent vor output erhht worden sein, sonst wre 
		//   Delta keine korrekte obere Schranke des Fehlers.
		// - Der Test auf if( Math.floor( N/((double)w) ) + 1 > bcurrent ) 
		//   stellt genau das sicher, er identifiziert das letzte Item des 
		//   alten Fensters, fhrt nach dem Einfgen compress mit dem 
		//   alten bcurrent aus und erhht dann sofort bcurrent.
		// - WICHTIG: Sofern das Gewicht (count) des letzten Items des altens 
		//   Fensters so gross war, dass ein oder mehrere Fenster 
		//   bersprungen werden, bleibt die Methode korrekt.				
		if( increase ){
			N += count; // Nicht erhhen, wenn Knoten wegen compress eingefgt

			if( LogService.shouldLog( LogService.NORMAL ) ) 
				LogService.log(LogService.NORMAL, "bcurrent " +
						"alt= " + bcurrent + " N=" + N + 
						" count = "+ count);
			if( Math.floor( N/((double)w) ) + 1 > bcurrent ){

				if( tupelCount > maxTupel ) maxTupel = tupelCount;
		
				if( compress ) compress();

				// Inkrement um mehr als 1 unkritisch
				bcurrent = (int)Math.floor( N/((double)w) ) + 1;
				if( LogService.shouldLog( LogService.NORMAL ) ) 
					LogService.log(LogService.NORMAL, "bcurrent " +
							"neu= " + bcurrent + " N=" + N + 
							" count = "+ count);
			}
		}
	}

	// Die drei abstrakten Methoden specializedInsert, checkChildren
	// und updateNextParentSet umfassen die einzigen Punkte, in denen
	// sich Full Ancestry und Partial Ancestry unterscheiden,
	// die Klassen FullAncHHH und PartAncHHH implementieren diese und
	// erhalten die gesamte uebrige Funktionalitaet von AbstractComplexHHH.
	protected abstract void specializedInsert(Element e, int count, int delta);


	protected abstract boolean checkChildren( Element e, 
			HashSet<Element> currentParentSet  );


	protected abstract void updateNextParentSet( Element e,
			HashSet<Element> nextParentSet );

	
	private HashMap<Element, MultiHitterInfo> simpleSet(){

		HashMap<Element, MultiHitterInfo> hitterSet =
			new HashMap<Element, MultiHitterInfo>();
		
		String[] iStr = new String[ par.getDimI() ];
		for( int i = 0; i < iStr.length; i++ ) iStr[i] = "*";
		String[] sStr = new String[ par.getDimS() ];
		for( int i = 0; i < sStr.length; i++ ) sStr[i] = "*";

		Element e;
		if( par instanceof SysParameter ){
			int[] intA = Element.stringToInt(iStr, par);
			e = new SysElement( sStr, intA, (SysParameter)par );
		}
		else{
			e = Element.createElement( sStr, iStr, par );
		}
		MultiHitterInfo m = new MultiHitterInfo(0,0,0);
		hitterSet.put( e, m );
		
		return hitterSet;
	}

	
	// Gibt das maximale tatschlich verwendete Level an,
	// wenn z.B. alle Pfade kuerzer als die maximale Pfadlaenge sind,
	// kann maxLevel kleiner sein als myParams.getL().
	private int maxLevel(){
		int maxLevel = 0;	
		HashMap<Element, MultiStringElement> lSet;
		for( int j = 0; j <= par.getL(); j++ ){
			lSet = levelSets.get(j);
			if( lSet.size() > 0 ) maxLevel = j;
		}
		return maxLevel;		
	}

	
	//Kopieren schneller als clone()
	private void copy( int[] a, int[] b ){
		for( int i = 0; i < a.length; i++ )
			b[i] = a[i];
	}
	
	
	public void setReachable( int[] pos ){
		int index = 0;
		int fac = 1;
		for( int i = 0; i < pos.length; i++ ){
			for( int j = 0; j < i; j++ ){
				fac *= (hi[j]+1);
			}
			index += (pos[i] * fac);
		}
		bits.set(index);
	}
	
	
	public boolean isReachable( int[] pos ){
		int index = 0;
		int fac = 1;
		for( int i = 0; i < pos.length; i++ ){
			for( int j = 0; j < i; j++ ){
				fac *= (hi[j]+1);
			}
			index += (pos[i] * fac);
		}
		return bits.get(index);
	}
	
}
