package hitters;

import hitters.multi.Element;
import hitters.multi.Evaluation;
import hitters.multi.Exact;
import hitters.multi.MultiHitterInfo;
import hitters.multi.Parameter;
import hitters.multi.PartAncHHH;

import java.util.Map;
import java.util.Set;

/**
 * @author Peter Fricke 
 *
 */
public class Beispiel {

        
        // Eingabe der Algorithmen sind Objekte vom Typ Element. 
        // Elemente stellen sowohl Items aus dem Strom 
        // als auch Praefixe dar. Elemente werden erzeugt,
        // indem man ein Array s erzeugt, dessen Laenge
        // der gewuenschten Dimension des Elements entspricht.
        // Z. Zt. ist nur Dimension 2 zulaessig. 
        // Das Array s wird befuellt und erzeugt dann ueber
		// Element e = Element.createElement( s, parameter )
		// ein Element (siehe auch Kommentar in Element).
		//		
		// Ausgabe der Algorithmen ist eine Map.
		// Schluessel sind die HHH-Prfixe, Wert sind 
		// Tupel (fmin, fmax, F). fmin und fmax sind
		// obere und untere Schranke der wahren absoluten Haeufigkeit.
		// F ist Schaetzung der diskontierten Haeufigkeit.       
        
        public static void main(String[] args) {


                // "/" trennt Hierarchieebenen
                String [][] stream = { {"/a", "/1" }, // Jede Zeile ist ein Item
                                { "/b", "/1" },
                                { "/a", "/2" },
                                { "/b", "/2" },
                                { "/a", "/1" },
                                { "/a", "/1" },
                                { "/a", "/1" },
                                { "/b", "/1" },
                                { "/b", "/2" },
                                { "/a", "/1" }, // Fenstergroesse 10, Compress
                                { "/a", "/1" },
                                { "/a", "/2" },
                                { "/a", "/2" } };
                
                // Dimensionen 2, noch fest verdrahtet
                Parameter par = new Parameter( 2 );
                
                double epsilon = 0.1;
                double phi = 0.35;
                double precision;
                double recall;
                double dice;

                //Partial Ancestry
                PartAncHHH myPA = new PartAncHHH( epsilon, par );
                
                //Exakter Algorithmus
                Exact myExact = new Exact( par );
                Element e;
                
                System.out.println( "\n\nEingabe: 13 Items, " +
                		"Epsilon = 0.1, Phi = 0.35\n");
                
                // Items einzeln einfuegen
                for( String[] s : stream ){                                                
                        
                        e = Element.createElement( s, par );
                        System.out.println( e.toShortString() );
                        
                        // Einfuegen mit Gewicht 1
                        myExact.insert( e, 1 );
                        myPA.insert( e.clone(), 1);
                }
                        
                // Ausgegeben wird eine Map, die HHH sind die Schluessel,
                // die Werte sind Tripel (fmin, fmax, F). fmin und fmax sind
                // obere und untere Schranke der wahren absoluten Haeufigkeit.
                // F ist Schaetzung der diskontierten Haeufigkeit. 
                // F darf die wahre diskontierte Haeufigkeit hoechstens
                // um Delta unterschaetzen, aber beliebig ueberschaetzen, 
                // so dass vereinzelt F > fmax auftritt. 
                // Hershberger et. al. (Space Complexity of Hierarchical 
                // Heavy Hitters in Multi-Dimensional Data Streams)
                // zeigen, dass Fehlerschranken fuer F fiese 
                // untere Schranken fuer den Speicherbedarf verursachen. 
                Map<Element, MultiHitterInfo> approx = myPA.outputSet( phi );
                Map<Element, MultiHitterInfo> exact = myExact.outputSet( phi );
                
                //Braucht man nur die HHH, verwendet man nur die Schluessel
                Set<Element> approxSet = approx.keySet();
                Set<Element> exactSet = exact.keySet();
                
                precision = Evaluation.precision( approxSet, exactSet );
                recall = Evaluation.recall(approxSet, exactSet);
                dice = Evaluation.dice(approxSet, exactSet);
                
                System.out.println( "\nExakte Loesung:" );
                for( Element elem : exactSet ){ 
                        System.out.println( elem.toShortString() );
                }
                
                System.out.println( "\nApproximationsloesung:" );
                for( Element elem : approxSet ){ 
                        System.out.print( elem.toShortString() );
                        System.out.println( "\t\t" + approx.get( elem ) );
                }
                
                System.out.println( "\n\nPrecision = " + precision );
                System.out.println( "Recall = " + recall );
                System.out.println( "Dice = " + dice );                                                
                
        }

}
