package com.rapidminer.example.table.struct.tree;

import java.util.ArrayList;


/**
 * <p>
 * This is an abstract implementation of the KTreeNode interface which provides
 * the basic handling of tree related stuff, i.e. storing a list of children,
 * accessing them, calculating the depth, etc.
 * </p>
 * <p>
 * This implementation is derived from Martin's original <code>Tree</code> code.
 * </p>
 * 
 * @author Martin Had, Christian Bockermann
 *
 */
public abstract class AbstractKTreeNode implements KTreeNode{

	/* Each node has a specific label */
	protected String label = "";
	
	/* This node's parent or null, if this is the tree root */
	protected KTreeNode parent;
	
	/* The children of a node a stored in this array list */
	ArrayList<KTreeNode> children;

	/* This is a "cache" of the depth to avoid recursive calculations */
	protected int depth = -1;
	
	protected int depthBelow = -1;
	
	/**
	 * Create a new KTreeNode instance by initializing the set of children.
	 * The parent is set to null, i.e. this is currently a root node.
	 */
	public AbstractKTreeNode(){
		parent = null;
		children = new ArrayList<KTreeNode>();
		depth = 0;
	}

	
	/**
	 * Create a new instance with the given node as the parent of this node.
	 * 
	 * @param parent
	 */
	public AbstractKTreeNode( KTreeNode parentNode ){
		this();
		this.parent = parentNode;
		depth = parent.getDepth() + 1;
	}
	
	
	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#getParent()
	 */
	public KTreeNode getParent() {
		return parent;
	}
	
	public void setParent( KTreeNode newParent ){
		parent = newParent;
		depth = parent.getDepth() + 1;
	}
	
	public NodeList getProductionList(int cs) { 
		return null;
	}
	
	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#getDepth()
	 */
	public int getDepth() {
		//
		// a "short-cut" for calculation of the depth. if the depth has not been
		// set before it will be "-1" by default and this method will start computing
		// it by traversing the tree upwards. The computed depth is then cached. 
		//
		if( depth >= 0 )
			return depth;
		
		if( parent != null ){
			depth = parent.getDepth() + 1;
		} else
			depth = 0;
		
		return depth;
	}
	
	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#getLabel()
	 */
	public String getLabel() {
		return label;
	}
	
	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#setLabel(java.lang.String)
	 */
	public void setLabel(String label) {
		this.label = label;
	}

	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#addChild(com.rapidminer.operator.learner.kernel.tree.KTreeNode)
	 */
	public void addChild(KTreeNode child) {
		child.setParent( this );
		children.add( child );
		depthBelow = Math.max( depthBelow, child.getDepthBelow() );
	}

	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#children()
	 */
	public ArrayList<KTreeNode> children() {
		return children;
	}

	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#getChild(int)
	 */
	public KTreeNode getChild(int i) {
		return children.get( i );
	}

	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#isLeaf()
	 */
	public boolean isLeaf() {
		return children().isEmpty();
	}
	
	
	//
	//
	// No the unimplemented methods follow...
	//
	//
	
	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#getDepthBelow()
	 */
	public int getDepthBelow(){
	    if( children.size() == 0 )
	        return 0;
	    
	    if( depthBelow <= 0 ){
	        depthBelow = computeDepthBelow();
	        //System.out.println("Computing depthBelow for " + this.label + " = " + depthBelow);
	    }
	    
	    return depthBelow;
	}

	protected int computeDepthBelow(){
	    int below = 0;
	    for( KTreeNode ch : children() )
	        below = Math.max( below, ch.getDepthBelow() );
	        
	    if( children().size() > 0 )
	        return 1 + below;
	    
	    return below;
	}
	
	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#setKernelValue(java.lang.Double)
	 */
	public abstract void setKernelValue(Double kernelValue);

	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#getKernelValue()
	 */
	public abstract Double getKernelValue();

	/**
	 * @see com.rapidminer.operator.learner.kernel.tree.KTreeNode#toString2()
	 */
	public abstract String toString2();
	
	
	public String printTree(){
	    StringBuffer s = new StringBuffer();
	    s.append( this.getLabel() );
	    s.append( " " );
	    
	    if( ! isLeaf() ){
	        
	        s.append( "[ ");
	        for( KTreeNode ch : children() )
	            s.append( ch.printTree() );
	        
	        s.append( " ]" );
	    }
	    return s.toString();
	}
	
	public abstract String getProduction();
}