/*
 *  RapidMiner
 *
 *  Copyright (C) 2001-2009 by Rapid-I and the contributors
 *
 *  Complete list of developers available at our web site:
 *
 *       http://rapid-i.com
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 */
package com.rapidminer.tools.math.similarity;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import com.rapidminer.example.ExampleSet;
import com.rapidminer.operator.IOContainer;
import com.rapidminer.operator.OperatorException;
import com.rapidminer.parameter.ParameterHandler;
import com.rapidminer.parameter.ParameterType;
import com.rapidminer.parameter.ParameterTypeCategory;
import com.rapidminer.parameter.UndefinedParameterError;
import com.rapidminer.parameter.conditions.AndCondition;
import com.rapidminer.parameter.conditions.EqualTypeCondition;
import com.rapidminer.tools.math.kernels.Kernel;
import com.rapidminer.tools.math.similarity.divergences.GeneralizedIDivergence;
import com.rapidminer.tools.math.similarity.divergences.ItakuraSaitoDistance;
import com.rapidminer.tools.math.similarity.divergences.KLDivergence;
import com.rapidminer.tools.math.similarity.divergences.LogarithmicLoss;
import com.rapidminer.tools.math.similarity.divergences.LogisticLoss;
import com.rapidminer.tools.math.similarity.divergences.MahalanobisDistance;
import com.rapidminer.tools.math.similarity.divergences.SquaredEuclideanDistance;
import com.rapidminer.tools.math.similarity.divergences.SquaredLoss;
import com.rapidminer.tools.math.similarity.mixed.MixedEuclideanDistance;
import com.rapidminer.tools.math.similarity.nominal.DiceNominalSimilarity;
import com.rapidminer.tools.math.similarity.nominal.JaccardNominalSimilarity;
import com.rapidminer.tools.math.similarity.nominal.KulczynskiNominalSimilarity;
import com.rapidminer.tools.math.similarity.nominal.NominalDistance;
import com.rapidminer.tools.math.similarity.nominal.RogersTanimotoNominalSimilarity;
import com.rapidminer.tools.math.similarity.nominal.RussellRaoNominalSimilarity;
import com.rapidminer.tools.math.similarity.nominal.SimpleMatchingNominalSimilarity;
import com.rapidminer.tools.math.similarity.numerical.CamberraNumericalDistance;
import com.rapidminer.tools.math.similarity.numerical.ChebychevNumericalDistance;
import com.rapidminer.tools.math.similarity.numerical.CorrelationSimilarity;
import com.rapidminer.tools.math.similarity.numerical.CosineSimilarity;
import com.rapidminer.tools.math.similarity.numerical.DTWDistance;
import com.rapidminer.tools.math.similarity.numerical.DiceNumericalSimilarity;
import com.rapidminer.tools.math.similarity.numerical.EuclideanDistance;
import com.rapidminer.tools.math.similarity.numerical.InnerProductSimilarity;
import com.rapidminer.tools.math.similarity.numerical.JaccardNumericalSimilarity;
import com.rapidminer.tools.math.similarity.numerical.KernelEuclideanDistance;
import com.rapidminer.tools.math.similarity.numerical.ManhattanDistance;
import com.rapidminer.tools.math.similarity.numerical.MaxProductSimilarity;
import com.rapidminer.tools.math.similarity.numerical.OverlapNumericalSimilarity;

/**
 * This is a convinient class for using the distanceMeasures. It offers methods
 * for integrating the measure classes into operators.
 * 
 * @author Sebastian Land
 */
public class DistanceMeasures {
	
	public static final String PARAMETER_MEASURE_TYPES = "measure_types";
	public static final String PARAMETER_NOMINAL_MEASURE = "nominal_measure";
	public static final String PARAMETER_NUMERICAL_MEASURE = "numerical_measure";
	public static final String PARAMETER_MIXED_MEASURE = "mixed_measure";
	public static final String PARAMETER_DIVERGENCE = "divergence";
	
	private static final String[] MEASURE_TYPES = new String[] {
		"MixedMeasures",
		"NominalMeasures",
		"NumericalMeasures",
		"BregmanDivergences"
	};
	
	public static final int MIXED_MEASURES_TYPE = 0;
	public static final int NOMINAL_MEASURES_TYPE = 1;
	public static final int NUMERICAL_MEASURES_TYPE = 2;
	public static final int DIVERGENCES_TYPE = 3;

	private static String[] NOMINAL_MEASURES = new String[] {
		"NominalDistance",
		"DiceSimilarity",
		"JaccardSimilarity",
		"KulczynskiSimilarity",
		"RogersTanimotoSimilarity",
		"RussellRaoSimilarity",
		"SimpleMatchingSimilarity"
	};
	private static Class[] NOMINAL_MEASURE_CLASSES = new Class[] {
		NominalDistance.class,
		DiceNominalSimilarity.class,
		JaccardNominalSimilarity.class,
		KulczynskiNominalSimilarity.class,
		RogersTanimotoNominalSimilarity.class,
		RussellRaoNominalSimilarity.class,
		SimpleMatchingNominalSimilarity.class
	};
		
	private static String[] MIXED_MEASURES = new String[] {
		"MixedEuclideanDistance"
	};
	private static Class[] MIXED_MEASURE_CLASSES = new Class[] {
		MixedEuclideanDistance.class
	};
	
	/* If this changes, the parameter dependencies might need to be updated */
	private static String[] NUMERICAL_MEASURES = new String[] {
		"EuclideanDistance",
		"CamberraDistance",
		"ChebychevDistance",
		"CorrelationSimilarity",
		"CosineSimilarity",
		"DiceSimilarity",
		"DynamicTimeWarpingDistance",
		"InnerProductSimilarity",
		"JaccardSimilarity",
		"KernelEuclideanDistance",
		"ManhattanDistance",
		"MaxProductSimilarity",
		"OverlapSimilarity"
	};
	private static Class[] NUMERICAL_MEASURE_CLASSES = new Class[] {
		EuclideanDistance.class,
		CamberraNumericalDistance.class,
		ChebychevNumericalDistance.class,
		CorrelationSimilarity.class,
		CosineSimilarity.class,
		DiceNumericalSimilarity.class,
		DTWDistance.class,
		InnerProductSimilarity.class,
		JaccardNumericalSimilarity.class,
		KernelEuclideanDistance.class,
		ManhattanDistance.class,
		MaxProductSimilarity.class,
		OverlapNumericalSimilarity.class
    };
    
	private static String[] DIVERGENCES = new String[] {
		"GeneralizedIDivergence",
		"ItakuraSaitoDistance",
		"KLDivergence",
		"LogarithmicLoss",
		"LogisticLoss",
		"MahalanobisDistance",
		"SquaredEuclideanDistance",
		"SquaredLoss",
	};
		
	private static Class[] DIVERGENCE_CLASSES = new Class[] {
    	GeneralizedIDivergence.class,
    	ItakuraSaitoDistance.class,
    	KLDivergence.class,
    	LogarithmicLoss.class,
    	LogisticLoss.class,
    	MahalanobisDistance.class,
    	SquaredEuclideanDistance.class,
    	SquaredLoss.class,
    };
    
	private static String[][] MEASURE_ARRAYS = new String[][] {
		MIXED_MEASURES,
		NOMINAL_MEASURES,
		NUMERICAL_MEASURES,
		DIVERGENCES
	};
	
	private static Class[][] MEASURE_CLASS_ARRAYS = new Class[][] {
		MIXED_MEASURE_CLASSES,
		NOMINAL_MEASURE_CLASSES,
		NUMERICAL_MEASURE_CLASSES,
		DIVERGENCE_CLASSES
	};
	
	/**
	 * This method allows registering distance or similarity measures defined in plugins.
	 * There are four different types of measures: Mixed Measures coping with examples containing
	 * nominal and numerical values. Numerical and Nominal Measures work only on their respective type of 
	 * attribute. Divergences are a less restricted mathematical concept than distances but might be used
	 * for some algorithms not needing this restrictions. This type has to be specified using the first parameter.
	 * @param measureType  The type is available as static property of class
	 * @param measureName  The name of the measure to register
	 * @param measureClass The class of the measure, which needs to extend DistanceMeasure
	 */
	public static void registerMeasure(int measureType, String measureName, Class<? extends DistanceMeasure> measureClass) {
//		System.out.println( "Pre Register Measure " + measureType +  " = " + MEASURE_CLASS_ARRAYS[measureType].length ); 
//		System.out.println( "Pre Register Measure div direkt"  +  " = " + DIVERGENCE_CLASSES.length );
		
		String[] newTypeNames = new String[MEASURE_ARRAYS[measureType].length + 1];
		System.arraycopy(MEASURE_ARRAYS[measureType], 0, newTypeNames, 0, MEASURE_ARRAYS[measureType].length);
		newTypeNames[newTypeNames.length - 1] = measureName;
		MEASURE_ARRAYS[measureType] = newTypeNames;

		Class[] newTypeClasses = new Class[MEASURE_CLASS_ARRAYS[measureType].length + 1];
		System.arraycopy(MEASURE_CLASS_ARRAYS[measureType], 0, newTypeClasses, 0, MEASURE_CLASS_ARRAYS[measureType].length);
		newTypeClasses[newTypeClasses.length - 1] = measureClass;
		MEASURE_CLASS_ARRAYS[measureType] = newTypeClasses;
//		System.out.println( "Post Register Measure " + measureType +  " = " + MEASURE_CLASS_ARRAYS[measureType].length );
//		System.out.println( "Post Register Measure div direkt"  +  " = " + DIVERGENCE_CLASSES.length );
	}
	
	public static DistanceMeasure createMeasure(ParameterHandler parameterHandler, ExampleSet exampleSet, IOContainer ioContainer) throws UndefinedParameterError, OperatorException {
		int measureType = parameterHandler.getParameterAsInt(PARAMETER_MEASURE_TYPES);
		//System.out.println( "Create Measure " + measureType +  " = " + MEASURE_CLASS_ARRAYS[measureType].length ); 
		//System.out.println( "Create Measure div direkt"  +  " = " + DIVERGENCE_CLASSES.length ); 

		switch (measureType) {
			case MIXED_MEASURES_TYPE: 
				return createMixedMeasure(parameterHandler, exampleSet, ioContainer);
			case NOMINAL_MEASURES_TYPE: 
				return createNominalMeasure(parameterHandler, exampleSet, ioContainer);
			case NUMERICAL_MEASURES_TYPE: 
				return createNumericalMeasure(parameterHandler, exampleSet, ioContainer);
			case DIVERGENCES_TYPE: 
				return createDivergence(parameterHandler, exampleSet, ioContainer);			
		}
		return null;
	}
	
	public static DistanceMeasure createNumericalMeasure(ParameterHandler parameterHandler, ExampleSet exampleSet, IOContainer ioContainer) throws UndefinedParameterError, OperatorException {
		Class measureClass = MEASURE_CLASS_ARRAYS[NUMERICAL_MEASURES_TYPE][parameterHandler.getParameterAsInt(PARAMETER_NUMERICAL_MEASURE)];
		return createMeasureFromClass(measureClass, parameterHandler, exampleSet, ioContainer);
	}

	public static DistanceMeasure createNominalMeasure(ParameterHandler parameterHandler, ExampleSet exampleSet, IOContainer ioContainer) throws UndefinedParameterError, OperatorException {
		Class measureClass = MEASURE_CLASS_ARRAYS[NOMINAL_MEASURES_TYPE][parameterHandler.getParameterAsInt(PARAMETER_NOMINAL_MEASURE)];
		return createMeasureFromClass(measureClass, parameterHandler, exampleSet, ioContainer);
	}

	public static DistanceMeasure createMixedMeasure(ParameterHandler parameterHandler, ExampleSet exampleSet, IOContainer ioContainer) throws UndefinedParameterError, OperatorException {
		Class measureClass = MEASURE_CLASS_ARRAYS[MIXED_MEASURES_TYPE][parameterHandler.getParameterAsInt(PARAMETER_MIXED_MEASURE)];
		return createMeasureFromClass(measureClass, parameterHandler, exampleSet, ioContainer);
	}

	public static DistanceMeasure createDivergence(ParameterHandler parameterHandler, ExampleSet exampleSet, IOContainer ioContainer) throws UndefinedParameterError, OperatorException {
		Class measureClass = MEASURE_CLASS_ARRAYS[DIVERGENCES_TYPE][parameterHandler.getParameterAsInt(PARAMETER_DIVERGENCE)];
		return createMeasureFromClass(measureClass, parameterHandler, exampleSet, ioContainer);
	}

	
	private static DistanceMeasure createMeasureFromClass(Class measureClass, ParameterHandler parameterHandler, ExampleSet exampleSet, IOContainer ioContainer) throws OperatorException {
		try {
			DistanceMeasure measure = (DistanceMeasure) measureClass.newInstance();
			measure.init(exampleSet, parameterHandler, ioContainer);
			return measure;
		} catch (InstantiationException e) {
			throw new OperatorException("Could not instanciate distance measure " + measureClass);
		} catch (IllegalAccessException e) {
			throw new OperatorException("Could not instanciate distance measure " + measureClass);
		}
	}
	
	/**
	 * This method adds a parameter to chose a distance measure as parameter
	 */
	public static List<ParameterType> getParameterTypes(ParameterHandler parameterHandler) {
		List<ParameterType> list = new LinkedList<ParameterType>();
		list.add(new ParameterTypeCategory(PARAMETER_MEASURE_TYPES, "The measure type", MEASURE_TYPES, 0));
		ParameterType type = new ParameterTypeCategory(PARAMETER_MIXED_MEASURE, "Select measure", MEASURE_ARRAYS[MIXED_MEASURES_TYPE], 0);
		type.registerDependencyCondition(new EqualTypeCondition(parameterHandler, PARAMETER_MEASURE_TYPES, true, 0));
		list.add(type);
		type = new ParameterTypeCategory(PARAMETER_NOMINAL_MEASURE, "Select measure", MEASURE_ARRAYS[NOMINAL_MEASURES_TYPE], 0);
		type.registerDependencyCondition(new EqualTypeCondition(parameterHandler, PARAMETER_MEASURE_TYPES, true, 1));
		list.add(type);
		type = new ParameterTypeCategory(PARAMETER_NUMERICAL_MEASURE, "Select measure", MEASURE_ARRAYS[NUMERICAL_MEASURES_TYPE], 0);
		type.registerDependencyCondition(new EqualTypeCondition(parameterHandler, PARAMETER_MEASURE_TYPES, true, 2));
		list.add(type);
		type = new ParameterTypeCategory(PARAMETER_DIVERGENCE, "Select divergence", MEASURE_ARRAYS[DIVERGENCES_TYPE], 0);
		type.registerDependencyCondition(new EqualTypeCondition(parameterHandler, PARAMETER_MEASURE_TYPES, true, 3));
		list.add(type);
		list.addAll(registerDependency(Kernel.getParameters(parameterHandler), 9, parameterHandler));
	
		return list;
	}
	
	/**
	 * This method provides the parameters to chose only from numerical measures
	 */
	public static List<ParameterType> getParameterTypesForNumericals(ParameterHandler handler) {
		List<ParameterType> list = new LinkedList<ParameterType>();
		ParameterType type = new ParameterTypeCategory(PARAMETER_NUMERICAL_MEASURE, "Select measure", MEASURE_ARRAYS[NUMERICAL_MEASURES_TYPE], 0);
		list.add(type);
		return list;
	}
	 
	private static Collection<ParameterType> registerDependency(Collection<ParameterType> sourceTypeList, int selectedValue, ParameterHandler handler) {
		for (ParameterType type: sourceTypeList) {
			type.registerDependencyCondition(new AndCondition(
					new EqualTypeCondition(handler, PARAMETER_NUMERICAL_MEASURE, false, selectedValue),
					new EqualTypeCondition(handler, PARAMETER_MEASURE_TYPES, false, NUMERICAL_MEASURES_TYPE)));
		}
		return sourceTypeList;
	}

}
