/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.schemamatching;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Concept;

/**
 * Abstract superclass of those schema matching methods in the
 * MiningMart environment that use only pairs of strings to do
 * the matching.
 * 
 * @author Timm Euler
 * @version $Id: NameAndTypeBasedSchemaMatcher.java,v 1.4 2006/09/27 15:00:03 euler Exp $
 */
public abstract class NameAndTypeBasedSchemaMatcher extends MmSchemaMatcher {

	/**
	 * @see edu.udo.cs.miningmart.schemamatching.MmSchemaMatcher#getDirectSimilarity(edu.udo.cs.miningmart.m4.BaseAttribute, edu.udo.cs.miningmart.m4.BaseAttribute)
	 */
	public double getDirectSimilarity(BaseAttribute ba1, BaseAttribute ba2) throws SchemaMatchException {
		double nameSimilarity = this.getNameSimilarity(ba1.getName(), ba2.getName());
		double typeSimilarity = this.getDatatypeSimilarity(ba1, ba2);
		
		// give a bonus for type similarity:
		if (typeSimilarity < THRESHOLD) {
			nameSimilarity *= 0.75;
		}
		
		return nameSimilarity;
	}
	
	public  double getStringSimilarity(String str1, String str2)
	throws SchemaMatchException {
		if (str1 == null || str2 == null)
			return 0d;
		if (str1.equalsIgnoreCase(str2))
			return 1d;
		else
			return 0d;
	}
	
	/**
	 * @see edu.udo.cs.miningmart.schemamatching.MmSchemaMatcher#getDirectSimilarity(edu.udo.cs.miningmart.m4.Concept, edu.udo.cs.miningmart.m4.Concept)
	 */
	public double getDirectSimilarity(Concept con1, Concept con2) throws SchemaMatchException {
		return (this.getNameSimilarity(con1.getName(), con2.getName()));
	}

	/**
	 * Returns the double value that this matcher considers as
	 * the similarity between ba1 and ba2, taking the similarity
	 * of their respective concepts into accounts as well.
	 *  
	 * @param ba1 One BaseAttribute
	 * @param ba2 Another BaseAttribute
	 * @return a similarity value
	 * @throws SchemaMatchException
	 */
	public double getSimilarityInvolvingConcepts(BaseAttribute ba1,	BaseAttribute ba2) throws SchemaMatchException {
		try {
			Concept con1 = ba1.getConcept();
			if (con1 == null) {
				throw new SchemaMatchException("BaseAttribute '" + ba1.getName() +
						"' does not have a concept!");
			}
			Concept con2 = ba2.getConcept();
			if (con2 == null) {
				throw new SchemaMatchException("BaseAttribute '" + ba2.getName() +
						"' does not have a concept!");
			}
			return (this.getNameSimilarity(ba1.getName(), ba2.getName()) + 
					this.getNameSimilarity(con1.getName(), con2.getName()))
					/ 2;
		}
		catch (M4Exception m4e) {
			throw new SchemaMatchException("Error accessing concept of BA: " + m4e.getMessage());
		}
	}	
}
/*
 * $Log: NameAndTypeBasedSchemaMatcher.java,v $
 * Revision 1.4  2006/09/27 15:00:03  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:17  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:16  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 13:57:28  euler
 * Changed hierarchy of matchers, added Editdistance matcher.
 *
 */
