/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.schemamatching;

import java.sql.SQLException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import edu.udo.cs.miningmart.exception.DbConnectionClosed;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Case;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.Relation;

/**
 * Provides methods to match sets of concepts to sets of concepts.
 * @author euler
 */
public class DataModelMatcher {

	private MmSchemaMatcher theMatcherToUse;
	
	public DataModelMatcher(MmSchemaMatcher useThisMatcher) {
		this.theMatcherToUse = useThisMatcher;
	}
	
	/**
	 * Adds concepts to the given case that represent
	 * the results of possible joins in the given case. Joins
	 * are possible between concepts if they are linked by a 
	 * relationship. Only two-way joins are considered. The
	 * added concepts are part of the case, but are also listed
	 * in the returned collection.
	 * 
	 * @param someCase the Case
	 * @return a collection of added concepts
	 */
	public static Collection<Concept> addJoinsToCase(Case someCase) throws M4Exception {
		if (someCase == null) {
			return null;
		}
		Collection<Concept> theNewConcepts = new Vector<Concept>();
		Iterator relIt = someCase.getAllRelations().iterator();
		while (relIt.hasNext()) {
			Relation oneRel = (Relation) relIt.next();
			Columnset joinCs = oneRel.getResultOfJoin();
			if (joinCs != null) {
				Concept joinConcept = someCase.createConceptFromColumnset(joinCs);
				if (joinConcept != null)
					theNewConcepts.add(joinConcept);
			}
		}
		return theNewConcepts;
	}
	
	/**
	 * Tries to map as many elements of the conceptual model as possible
	 * to the target model. Starts with the relationships, continues with
	 * the concepts. Returns maps of concepts.
	 * 
	 * @param conceptualModel the conceptual data model
	 * @param targetModel the target data model
	 * @param theMatcherToUse instance of MmSchemaMatcher to be used for finding
	 * the similarities
	 * @return a collection with mapped concepts
	 * @throws SchemaMatchException
	 */
	public Collection<MatchingResult<Concept>> getDataModelMatching(
			Collection<Concept> conceptualModel, 
			Collection<Concept> targetModel)
	throws SchemaMatchException {
		if (conceptualModel == null || targetModel == null)
			return null;
		
		try {
			// first step is to match the "stars", ie the concepts that are involved
			// in several relations:
			Collection<Concept> conceptualStars = this.findStarsInDataModel(conceptualModel);
			Collection<Concept> targetStars = this.findStarsInDataModel(targetModel);
			
			Collection<MatchingResult<Concept>> conceptsMatchedBasedOnStars = 
				this.findMatchingsBasedOnStars(
						conceptualStars, 
						targetStars);
			
			// second step is to match all other relations:
			Collection<Relation> conceptualRelations = this.findRelationsInDataModel(conceptualModel);
			Collection<Relation> targetRelations = this.findRelationsInDataModel(targetModel);
			
			conceptualRelations = 
				this.removeRelationsAlreadyMatched(
						conceptsMatchedBasedOnStars, 
						conceptualRelations, true);
			targetRelations = 
				this.removeRelationsAlreadyMatched(
						conceptsMatchedBasedOnStars,
						targetRelations, false);
			
			MatchingResult[][] relationSimilarityMatrix = 
				this.theMatcherToUse.getSimilarityMatrix(conceptualRelations, targetRelations);
			Collection<MatchingResult<Relation>> matchedRelations = 
				this.theMatcherToUse.getSimilarMatchingsGreedy(relationSimilarityMatrix, true);
			
			// now combine the previous matchings to one collection
			// of concept mappings:
			Collection<MatchingResult<Concept>> allConceptMappings = new Vector<MatchingResult<Concept>>();
			allConceptMappings.addAll(conceptsMatchedBasedOnStars);
			allConceptMappings.addAll(this.getConceptMappingsFromRelationMappings(matchedRelations));
			
			// third step is to determine which concepts have not been matched
			// yet, then to match them:
			Collection<Concept> remainingConceptualConcepts = 
				this.removeConceptsAlreadyMatched(
						conceptualModel, allConceptMappings, true);
			Collection<Concept> remainingTargetConcepts = 
				this.removeConceptsAlreadyMatched(
						targetModel, allConceptMappings, false);
			
			MatchingResult[][] conceptSimilarityMatrix =
				this.theMatcherToUse.getSimilarityMatrix(remainingConceptualConcepts, remainingTargetConcepts);
			Collection<MatchingResult<Concept>> matchedConcepts = 
				this.theMatcherToUse.getSimilarMatchingsGreedy(conceptSimilarityMatrix, true);
			
			// return all mappings found between concepts:
			allConceptMappings.addAll(matchedConcepts);
			
			// debug:
			if (allConceptMappings.size() > Math.min(conceptualModel.size(), targetModel.size())) {
				throw new SchemaMatchException("DataModelMatcher.getDataModelMatching(): got more mappings than single objects!");
			}
			return allConceptMappings;
		}
		catch (M4Exception m4e) {
			throw new SchemaMatchException("M4 error when computing data model similarity: " + m4e.getMessage());
		}
	}
	
	public boolean isSimilarityGoodEnough(double similarity) {
		return (similarity >= this.theMatcherToUse.THRESHOLD); 
	}
	
	/*
	 * Computes an overall similarity of the two given data models,
	 * based on the mappings between them that were found.
	 * 
	 * @param conceptualModel one given data model
	 * @param targetModel second given data model
	 * @param conceptMappings mappings between concepts of data models
	 * @return a similarity value between 0 and 1
	 *
	public static double getDataModelSimilarity(
			Collection<Concept> conceptualModel, 
			Collection<Concept> targetModel,
			Collection<MatchingResult<Concept>> conceptMappings) 
	throws SchemaMatchException {
		if (conceptualModel == null || targetModel == null || conceptMappings == null)
			return 0d;
		
		// very simple approach; compute the sum of similarities of each concept
		// mapping, divided by the number of concepts that could 
		// have been matched:
		int numberOfPossibleMatches = Math.min(conceptualModel.size(), targetModel.size());
		if (conceptMappings.size() > numberOfPossibleMatches) {
			// this would possibly result in a similarity value larger than 1
			throw new SchemaMatchException("DataModelMatcher.getDataModelSimilarity: got more mappings than concepts in conceptual model!");
		}
		double sum = 0d;
		for (MatchingResult<Concept> singleMapping : conceptMappings) {
			sum += singleMapping.getSimilarity();
		}
		return sum / numberOfPossibleMatches;
	}
	*/
	
	// matches the From-Concepts and the To-Concepts of the relations
	// in the given mappings. Computes the concept similarity using the
	// given matcher.
	private Collection<MatchingResult<Concept>> getConceptMappingsFromRelationMappings(
				Collection<MatchingResult<Relation>> relationMaps)
	throws SchemaMatchException {
		Collection<MatchingResult<Concept>> ret = new Vector<MatchingResult<Concept>>();
		if (relationMaps == null)
			return ret;
		try {
			for (MatchingResult<Relation> relMR : relationMaps) {				
				MatchingResult<Concept> fromConceptMR = new MatchingResult<Concept>();
				Relation firstRel = relMR.getObjectOfFirstSchema();
				Relation secondRel = relMR.getObjectOfSecondSchema();
				// first compare FromConcept with FromConcept, and ToConcept with ToConcept:
				Concept from1 = firstRel.getTheFromConcept();
				Concept from2 = secondRel.getTheFromConcept();
				fromConceptMR.setObjectOfFirstSchema(from1);
				fromConceptMR.setObjectOfSecondSchema(from2);
				fromConceptMR.setSimilarity(this.theMatcherToUse.getSimilarity(from1, from2));
				MatchingResult<Concept> toConceptMR = new MatchingResult<Concept>();
				Concept to1 = firstRel.getTheToConcept();
				Concept to2 = secondRel.getTheToConcept();
				toConceptMR.setObjectOfFirstSchema(to1);
				toConceptMR.setObjectOfSecondSchema(to2);
				toConceptMR.setSimilarity(this.theMatcherToUse.getSimilarity(to1, to2));
				// in the case of many-to-many-relationships the From/To Direction
				// is irrelevant, so the other way round might give a better similarity:
				if (firstRel.isManyToManyRelation() && secondRel.isManyToManyRelation()) {
					double averageSimilarityNow = (fromConceptMR.getSimilarity() + toConceptMR.getSimilarity()) / 2;
					double similarityFrom1To2 = this.theMatcherToUse.getSimilarity(from1, to2);
					double similarityFrom2To1 = this.theMatcherToUse.getSimilarity(to1, from2);
					// check if the average similarity is higher when From-Concept
					// is matched with To-Concept:
					if (((similarityFrom1To2 + similarityFrom2To1) / 2) > averageSimilarityNow) {
						fromConceptMR.setObjectOfFirstSchema(from1);
						fromConceptMR.setObjectOfSecondSchema(to2);
						fromConceptMR.setSimilarity(similarityFrom1To2);
						toConceptMR.setObjectOfFirstSchema(to1);
						toConceptMR.setObjectOfSecondSchema(from2);
						toConceptMR.setSimilarity(similarityFrom2To1);
					}
				}
				ret.add(fromConceptMR);
				ret.add(toConceptMR);
			}
			return ret;
		}
		catch (M4Exception m4e) {
			throw new SchemaMatchException("M4 error when matching concepts of matched relations: " + m4e.getMessage());
		}
	}
	
	// returns all concepts from the given data model that have not
	// been matched previously, as indicated by the given collections
	// of previous matchings
	private Collection<Concept> removeConceptsAlreadyMatched(
				Collection<Concept> givenDataModel,
				Collection<MatchingResult<Concept>> conceptMatchings,
				boolean useConceptualLevelFromMatchings) {
		
		Collection<Concept> ret = new Vector<Concept>();
		if (givenDataModel == null || conceptMatchings == null) 
			return ret;
		
		for (Concept oneConcept : givenDataModel) {
			boolean oneConceptIsMatched = false;
			for (MatchingResult<Concept> mr : conceptMatchings) {
				Concept matched = 
					(useConceptualLevelFromMatchings ? mr.getObjectOfFirstSchema() : mr.getObjectOfSecondSchema());
				if (matched.equals(oneConcept)) 
					oneConceptIsMatched = true;
			}
			if ( ! oneConceptIsMatched)
				ret.add(oneConcept);
		}
		return ret;
	}
	
	// removes from the given collection of relations all relations 
	// that occur among the concepts already matched
	private Collection<Relation> removeRelationsAlreadyMatched(
				Collection<MatchingResult<Concept>> matchedConcepts, 
				Collection<Relation> theRelations,
				boolean useFirstSchemaOfMapping) 
	throws SchemaMatchException {
		
		Collection<Relation> ret = theRelations;
		if (matchedConcepts == null)
			return ret;
		
		// the matched concepts contain a mapping from concepts of the 
		// conceptual model, among which the relations are sought, to
		// the target model concepts
		Collection<Concept> matchedConceptsOfOneModel = new Vector<Concept>();
		for (MatchingResult<Concept> map : matchedConcepts) {
			Concept matched = (useFirstSchemaOfMapping ? map.getObjectOfFirstSchema() : map.getObjectOfSecondSchema());
			matchedConceptsOfOneModel.add(matched);
		}
		
		// now we have only the conceptual model concepts in a collection;
		// look for any relations among them:
		try {
			for (Concept oneConcept : matchedConceptsOfOneModel) {
				Iterator relIt = oneConcept.getTheFromRelationships().iterator();
				while (relIt.hasNext()) {
					Relation rel = (Relation) relIt.next();
					if (matchedConceptsOfOneModel.contains(rel.getTheToConcept())) {
						ret.remove(rel);
					}
				}
				relIt = oneConcept.getTheToRelationships().iterator();
				while (relIt.hasNext()) {
					Relation rel = (Relation) relIt.next();
					if (matchedConceptsOfOneModel.contains(rel.getTheFromConcept())) {
						ret.remove(rel);
					}
				}
			}
		}
		catch (M4Exception m4e) {
			throw new SchemaMatchException("M4 error caught when removing star-matched relations from other relations: " + m4e.getMessage());
		}
		return ret;
	}
	
	private Collection<MatchingResult<Concept>> findMatchingsBasedOnStars(
			Collection<Concept> starsOfConceptualModel,
			Collection<Concept> starsOfTargetModel)
	throws SchemaMatchException {

		if (starsOfConceptualModel == null || starsOfTargetModel == null)
			return null;
		Collection<MatchingResult<Concept>> conceptsMatchedBasedOnStars = new HashSet<MatchingResult<Concept>>();
		// must compute matrix by hand so that star similarity is used for
		// matrix cells, rather than concept similarity:
		MatchingResult[][] simMatrix = new MatchingResult[starsOfConceptualModel.size()][starsOfTargetModel.size()];
		int row = 0;
		Iterator conceptStarsIt = starsOfConceptualModel.iterator();
		while (conceptStarsIt.hasNext()) {
			Concept conceptualStar = (Concept) conceptStarsIt.next();
			Iterator targetStarsIt = starsOfTargetModel.iterator();
			int col = 0;
			while (targetStarsIt.hasNext()) {
				Concept targetStar = (Concept) targetStarsIt.next();
				// get the best match:
				double sim = this.getStarSimilarity(conceptualStar, targetStar);
				MatchingResult<Concept> mr = new MatchingResult<Concept>();
				mr.setObjectOfFirstSchema(conceptualStar);
				mr.setObjectOfSecondSchema(targetStar);
				mr.setSimilarity(sim);
				simMatrix[row][col] = mr;
				col++;
			}
			row++;
		}
		conceptsMatchedBasedOnStars = this.theMatcherToUse.getSimilarMatchingsGreedy(simMatrix, true);
		return conceptsMatchedBasedOnStars;
	}
	
	// returns all relations that hold between any of the given concepts
	private Collection<Relation> findRelationsInDataModel(
				Collection<Concept> dataModel) 
	throws M4Exception {
		if (dataModel == null) 
			return null;
		HashSet<Relation> ret = new HashSet<Relation>();
		Iterator conceptIt = dataModel.iterator();
		while (conceptIt.hasNext()) {
			Concept oneConcept = (Concept) conceptIt.next();
			Iterator relIt = oneConcept.getTheFromRelationships().iterator();
			while (relIt.hasNext()) {
				Relation myRel = (Relation) relIt.next();
				if (dataModel.contains(myRel.getTheToConcept()))
					ret.add(myRel); // adds nothing twice
			}
			relIt = oneConcept.getTheToRelationships().iterator();
			while (relIt.hasNext()) {
				Relation myRel = (Relation) relIt.next();
				if (dataModel.contains(myRel.getTheFromConcept()))
					ret.add(myRel); // adds nothing twice
			}
		}
		return ret;
	}
	
	// a "star" is a concept that is involved (as From- or To-Concept) in
	// more than one relation. All concepts that take part in the star
	// must be part of the given data model!
	private Collection<Concept> findStarsInDataModel(
				Collection<Concept> dataModel)
	throws M4Exception {
		if (dataModel == null) 
			return null;
		HashSet<Concept> ret = new HashSet<Concept>();
		Iterator conceptIt = dataModel.iterator();
		while (conceptIt.hasNext()) {
			Concept oneConcept = (Concept) conceptIt.next();
			int noOfRelsThatCountForStar = 0;
			Iterator relsIt = oneConcept.getTheFromRelationships().iterator();
			while (relsIt.hasNext()) {
				Relation fromRel = (Relation) relsIt.next();
				if (dataModel.contains(fromRel.getTheToConcept()))
					noOfRelsThatCountForStar++;
			}
			relsIt = oneConcept.getTheToRelationships().iterator();
			while (relsIt.hasNext()) {
				Relation toRel = (Relation) relsIt.next();
				if (dataModel.contains(toRel.getTheFromConcept()))
					noOfRelsThatCountForStar++;
			}
			if (noOfRelsThatCountForStar > 1)
				ret.add(oneConcept); // adds nothing twice
		}
		return ret;
	}
	
	// finds a similarity value that involves the neighbours
	// of the concepts in terms of their relations
	// (so the concepts are supposed to be the center of "stars")
	/*
	private double getStarSimilarity(
			Concept one, 
			Concept two)
	throws SchemaMatchException {
		try {
			// we compute the number of relationships that can
			// be aligned:
			int firstNoOfRels = one.getTheFromRelationships().size() + one.getTheToRelationships().size();
			int secondNoOfRels = two.getTheFromRelationships().size() + two.getTheToRelationships().size();
			if (firstNoOfRels <= 1 || secondNoOfRels <= 1) {
				return 0d; // no stars
			}
			int noOfPossibleAlignments = Math.min(firstNoOfRels, secondNoOfRels);
			int noOfActualAlignments = 0;
			Collection<MatchingResult<Concept>> aligned = this.getStarAlignment(one, two);
			
			if (aligned != null) {
				noOfActualAlignments = aligned.size();
			}
			return (double) noOfActualAlignments / (double) noOfPossibleAlignments;
		}
		catch (M4Exception m4e) {
			throw new SchemaMatchException("M4 error computing star similarity between concepts '"
					+ one.getName() + "' and '" + two.getName() + "': " + m4e.getMessage());
		}
	}
	*/
	private double getStarSimilarity(
			Concept one, 
			Concept two) 
	throws SchemaMatchException {
		if (one == null || two == null)
			return 0d;
		try {
			// check if the concepts are stars indeed:
			int firstNoOfRels = one.getTheFromRelationships().size() + one.getTheToRelationships().size();
			int secondNoOfRels = two.getTheFromRelationships().size() + two.getTheToRelationships().size();
			if (firstNoOfRels <= 1 || secondNoOfRels <= 1) {
				return 0d; // no stars
			}
			
			// build a combined matrix of all concepts at the other end of the
			// relations:
			Collection<Concept> conceptsReachedFromFirst = new Vector<Concept>();
			Collection<Concept> conceptsReachedFromSecond = new Vector<Concept>();
			
			// collect "ends of first star":
			Iterator relItOne = one.getTheFromRelationships().iterator();
			while (relItOne.hasNext()) {
				Relation oneRel = (Relation) relItOne.next();
				Concept oneTo = oneRel.getTheToConcept();
				conceptsReachedFromFirst.add(oneTo);
			}
			relItOne = one.getTheToRelationships().iterator();
			while (relItOne.hasNext()) {
				Relation oneRel = (Relation) relItOne.next();
				Concept oneTo = oneRel.getTheFromConcept();
				conceptsReachedFromFirst.add(oneTo);
			}
			// collect "ends of second star":
			relItOne = two.getTheFromRelationships().iterator();
			while (relItOne.hasNext()) {
				Relation oneRel = (Relation) relItOne.next();
				Concept oneTo = oneRel.getTheToConcept();
				conceptsReachedFromSecond.add(oneTo);
			}
			relItOne = two.getTheToRelationships().iterator();
			while (relItOne.hasNext()) {
				Relation oneRel = (Relation) relItOne.next();
				Concept oneTo = oneRel.getTheFromConcept();
				conceptsReachedFromSecond.add(oneTo);
			}
			// get similarity matrix:
			MatchingResult[][] matrix = 
				this.theMatcherToUse.getSimilarityMatrix(
						conceptsReachedFromFirst, 
						conceptsReachedFromSecond);
			// return best-matching concepts:
			Collection maps = this.theMatcherToUse.getSimilarMatchingsGreedy(matrix, true);
			return this.theMatcherToUse.getGlobalSimilarity(conceptsReachedFromFirst, conceptsReachedFromSecond, maps);
		}
		catch (M4Exception m4e) {
			throw new SchemaMatchException("M4 error computing star similarity between concepts '"
					+ one.getName() + "' and '" + two.getName() + "': " + m4e.getMessage());
		}
	}
	
	
	/**
	 * Creates a MiningMart case whose concepts and relations
	 * represent the given tables/views and the foreign key links
	 * between them as declared in the business database. The case
	 * is NOT set as the current case in the given M4 interface.
	 * 
	 * @param tableNames a collection of names of tables/views
	 * @param theM4Interface the M4 interface that provides access 
	 * to the database
	 * @return a Case object; it should be treated as temporary
	 * @throws M4Exception
	 */
	public static Case readSchemaToMatch(
			Collection<String> tableNames,
			M4Interface theM4Interface,
			String nameForTheCase) 
	throws M4Exception {
		try {
			if (tableNames != null) {				
				boolean setAsCurrentCase = false;
				if (nameForTheCase == null)
					nameForTheCase = "__TemporaryCaseWithSchemaToMatch";
				Case newCase = theM4Interface.createCase(nameForTheCase, setAsCurrentCase);
				
				// iterate through names of tables/views:
				boolean conceptsCreated = false;
				for (String tableName : tableNames) {					
					if (tableName.indexOf("$") == -1) {
						if ( ! conceptExistsInCase(newCase, tableName)) {								
							if ( ! checkForManyToManyRelation(tableName, tableNames, newCase, theM4Interface)) {
								newCase.createConceptAndRelationsFromTables(tableName);
							}
							conceptsCreated = true;
						}
					}
				}
				if (conceptsCreated) {
					return newCase;
				}
				else {
					return null;
				}
			}
			return null;
		}
		catch (M4Exception e) {
			throw new M4Exception("M4 error when trying to create a temporary Case representing a part of the business schema: " + e.getMessage());
		}
	}	

	/*
	 * returns TRUE iff a concept with the given name (ignoring case) exists
	 * in the given case
	 */
	private static boolean conceptExistsInCase(Case theCase, String nameOfConcept) throws M4Exception {
		Iterator namesIt = theCase.getAllConceptNames().iterator();
		while (namesIt.hasNext()) {
			String oneConceptName = (String) namesIt.next();
			if (oneConceptName.equalsIgnoreCase(nameOfConcept)) {
				return true;
			}
		}
		return false;
	}
	
	/*
	 * Returns TRUE if the given db object can be treated as a cross table,
	 * OR if a cross table refers to it.
	 * In that case all needed concepts, columnsets and many-to-many relations
	 * are created! Otherwise nothing is done and false is returned.
	 */
	private static boolean checkForManyToManyRelation(
			String dbTableName,
			Collection<String> allAllowedTableNames,
			Case theCase,
			M4Interface theM4Interface) 
	throws M4Exception {
		try {
			// the given db object may be a cross table itself:
			if (theM4Interface.getM4db().isCrossTable(dbTableName)) {				
				Collection createdRelations = theCase.createManyToManyRelationsFromCrossTable(dbTableName, allAllowedTableNames);				
				return (createdRelations != null && ( ! createdRelations.isEmpty()));
			}
			else {
				// but it may also be referred to by some cross table:
				Collection crossTablesReferring = theM4Interface.getM4db().getCrossTablesReferringTo(dbTableName);
				if (crossTablesReferring == null || crossTablesReferring.isEmpty())
					return false;
				Iterator crossIt = crossTablesReferring.iterator();
				boolean anythingCreated = false;
				while (crossIt.hasNext()) {
					String crossTable = (String) crossIt.next();
					if (allAllowedTableNames == null || allAllowedTableNames.contains(crossTable)) {
						// does not create if the relation already exists:
						Collection createdRels = theCase.createManyToManyRelationsFromCrossTable(crossTable, allAllowedTableNames);
						anythingCreated = (createdRels != null && ( ! createdRels.isEmpty()));
					}
				}
				return anythingCreated;
			}
		}
		catch (SQLException sqle) {
			 throw new M4Exception("SQL error occurred when trying to see if '" +
			 		dbTableName + "' is a cross table: " + sqle.getMessage());			 
		}
		catch (DbConnectionClosed dbc) {
			 throw new M4Exception("Suddenly closed connection to DB when trying to see if '" +
			 		dbTableName + "' is a cross table: " + dbc.getMessage());				
		}
	}	
}