/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

/**
 * <p>M4 Compiler</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: University Dortmund</p>
 */

import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4CompilerWarning;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Feature;
import edu.udo.cs.miningmart.m4.Value;

/**
 * Neue Version des Windowing-Operators ohne Stored Procedure.
 * Alte Version in Datei WindowingOld.java.
 * 
 * @author Timm Euler
 * @version $Id: Windowing.java,v 1.6 2006/09/27 14:59:56 euler Exp $
 */
public class Windowing extends SingleCSOperator {

    // Parameter names:
    private static final String TIMEBASEATTRIB   = "TimeBaseAttrib";
    private static final String VALUEBASEATTRIB  = "ValueBaseAttrib";
    private static final String WINDOWSIZE       = "WindowSize";
    private static final String OUTTIMESTARTBA   = "OutputTimeStartBA";
    private static final String OUTTIMEENDBA     = "OutputTimeEndBA";

    private String nameOfTempView = null;
    
    /**
     * {@inheritDoc}
     */
    public String getTypeOfNewColumnSet() {
    	return Columnset.CS_TYPE_VIEW;
    }

    /**
     * {@inheritDoc}
     */
    public String generateSQLDefinition(String selectPart) 
    throws M4CompilerError, M4CompilerWarning {
    	
    	// create the final view which reads from the intermediate view:
		String sqlDef = "(SELECT " + 
						selectPart + 
		                " FROM " + 
		                nameOfTempView +
		                " group by windownumber)";
		return sqlDef;
    }

	/**
	 * This method is overridden because this operator uses a mapping to fill
	 * the output BaseAttributes.
	 * 
	 * @see edu.udo.cs.miningmart.m4.operator.ConceptOperator#generateColumns(Columnset)
	 */
    protected String generateColumns(Columnset csForOutputConcept) throws M4CompilerError {

    	// create an intermediate view:
    	try {
    		nameOfTempView = 
    			this.createTemporaryView(getInputConcept().getCurrentColumnSet().getSchemaPlusName(), getTimeBA(), getValueBA());
    	}
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Operator Windowing: could not create intermediate view, M4Exception caught: " + m4e.getMessage());
    	}
    	if (nameOfTempView == null)
    		throw new M4CompilerError("Operator Windowing: could not create intermediate view!");
    	
    	// create the select string that selects from the intermediate view:
		String columnExpr = ""; // to be returned
		BaseAttribute theInputTimeBA = this.getTimeBA();
		BaseAttribute theInputValueBA = this.getValueBA();

		try	{
			// go through the Features of the output concept:
			Collection outFeatures = this.getOutputConcept().getFeatures();
			if (outFeatures == null || outFeatures.isEmpty()) {
				throw new M4CompilerError("Operator Windowing: No Features found in the output concept!");
			}
			Iterator it = outFeatures.iterator();
		    while (it.hasNext()) {
				Feature outF = (Feature) it.next();
				if ( ! (outF instanceof BaseAttribute)) {
					throw new M4CompilerError("Windowing ('" + this.getStep().getName() +
							"'): cannot handle MultiColumnFeatures like '" +
							outF.getName() + "!");
				}
				// check if it's one of the Time attributes:
				if (outF.correspondsTo(this.getOutStartTimeBA()) ||
					outF.correspondsTo(this.getOutEndTimeBA())) {
				    // yes - copy column from input time attrib:
					columnExpr = this.createTimeColumn(theInputTimeBA, (BaseAttribute) outF, csForOutputConcept, columnExpr);
				}			
				else {
					// no - outF must be a windowed Attribute, so create 
					// special column:					
					columnExpr = this.createWindowedColumn(theInputValueBA, (BaseAttribute) outF, csForOutputConcept, columnExpr);
				}			
			} // end of loop through the output features
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator Pivotize: M4 Exception caught when generating metadata: " +
			                          m4e.getMessage());
		}	
		if (columnExpr.length() > 2) {
			columnExpr = columnExpr.substring(0, columnExpr.length() - 2);
		}
		else {
			throw new M4CompilerError("Operator Pivotize: No columns created for output concept!");
		}
		return columnExpr;		             
    }
    
	/**
	 * This method is never called.
	 * 
	 * @see miningmart.compiler.operator.ConceptOperator#mustCopyFeature(String)
	 */
    public boolean mustCopyFeature(String nameOfFeature) {
    	return false;
    }

	private String createWindowedColumn( BaseAttribute inputValueBA,
										 BaseAttribute windowedBA,
	                                     Columnset csForOutputConcept, 
	                                     String columnExpr) 
	throws M4CompilerError {
		
		Column inputCol;
		String when = null;
		try {
			inputCol = inputValueBA.getCurrentColumn();
			if (inputCol == null) { 
				throw new M4CompilerError("Operator Windowing: Could not find column for input value BA '" + 
				                          inputValueBA.getName() + "!");
			}
			when = getTimeBA().getCurrentColumn().getSQLDefinition();
			when += " - (" + this.getWindowNumExpr() + " * " + this.getWinSize() + ")";
			when += " = " + this.getWindowPosition(windowedBA);
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator Windowing: M4 Exception caught when accessing column " +
			                          "of input or windowed attribute: " + m4e.getMessage());
		}
		
	    String colDef = "MAX(CASE WHEN " +
	                    	when + 
	                    	" THEN " +
	                    	nameOfTempView + "." + inputCol.getName() +
	                    	" ELSE null END)";
	               
	    // create Metadata for the new Column:
	    Column outputColumn;
        try {
            outputColumn = inputCol.copyColToCS(csForOutputConcept);
            this.getStep().addToTrash(outputColumn);
            outputColumn.setBaseAttribute(windowedBA);
            outputColumn.setSQLDefinition(windowedBA.getName());
            outputColumn.setName(windowedBA.getName());
	    
            return columnExpr + colDef + " AS " + windowedBA.getName() + ", "; 
        }
        catch (M4Exception m4e) {
            throw new M4CompilerError("Operator Windowing: M4 exception caught when creating column for " +
                                      "output BaseAttribute '" + windowedBA.getName() + 
                                      "': " + m4e.getMessage());
        } 
	}    

	private String createTimeColumn( BaseAttribute inputTimeBA,
									 BaseAttribute outputTimeBA,
	                                 Columnset csForOutputConcept, 
	                                 String columnExpr) 
	throws M4CompilerError {
		
		// check input column:
		Column outTimeCol, inputCol;
		try {
			inputCol = inputTimeBA.getCurrentColumn();
			if (inputCol == null) { 
				throw new M4CompilerError("Operator Windowing: Could not find column for input value BA '" + 
				                          inputTimeBA.getName() + "!");
			}
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator Windowing: M4 Exception caught when accessing column " +
			                          "of input or windowed attribute: " + m4e.getMessage());
		}

		// create definition for output time column:
		boolean isStart = false;
		if (outputTimeBA.correspondsTo(this.getOutStartTimeBA()))
			isStart = true;
		else if (outputTimeBA.correspondsTo(this.getOutEndTimeBA()))
			isStart = false;
		else throw new M4CompilerError("Operator Windowing, Step '" + 
				this.getStep().getName() +
				"': output time attribute could not be matched!");
		String colName = (isStart ? "start_index" : "end_index");
		String colDef = "MAX(" + colName + ")";
	               
	    // create Metadata for the new Column:
	    Column outputColumn;
        try {
            outputColumn = inputCol.copyColToCS(csForOutputConcept);
            this.getStep().addToTrash(outputColumn);
            outputColumn.setBaseAttribute(outputTimeBA);
            outputColumn.setSQLDefinition(outputTimeBA.getName());
            outputColumn.setName(outputTimeBA.getName());
	    
            return columnExpr + colDef + " AS " + outputTimeBA.getName() + ", "; 
        }
        catch (M4Exception m4e) {
            throw new M4CompilerError("Operator Windowing: M4 exception caught when creating column for " +
                                      "output BaseAttribute '" + outputTimeBA.getName() + 
                                      "': " + m4e.getMessage());
        } 
	}    
	
	// Returns name of intermediate view from which final view reads
	private String createTemporaryView(
			String inputCsName, 
			BaseAttribute inputTimeBA, 
			BaseAttribute inputValueBA) 
	throws M4CompilerError {
		String tempViewName = null;
		if (inputTimeBA == null || inputValueBA == null)
			return null;
		try {
			Column inputTimeCol = inputTimeBA.getCurrentColumn();
			Column inputValueCol = inputValueBA.getCurrentColumn();
			if (inputTimeCol == null || inputValueCol == null)
				return null;
			
			String n = this.getWinSize();
			if (n == null)
				throw new M4CompilerError("Operator Windowing, Step '" + this.getStep().getName() + "': Window size is invalid!");
			
			String viewDef = "(SELECT " + inputTimeCol.getSQLDefinition() +
			                              ", " +
			                              inputValueCol.getSQLDefinition() +
			                              ", " +
			                              this.getWindowNumExpr() + " AS windownumber, " +
			                              "(" + this.getWindowNumExpr() + " * " + n + ") AS start_index, " + 
			                              "(" + this.getWindowNumExpr() + " * " + n + " + " + n + " - 1) AS end_index " +
			                 "FROM " + inputCsName + ")";
			tempViewName = inputCsName + "_TMP";
			// remove name of schema from name of temp view:
			if (tempViewName.indexOf('.') > -1) {
				tempViewName = tempViewName.substring(tempViewName.lastIndexOf('.') + 1);
			}
			String viewCreation = "CREATE OR REPLACE VIEW " + tempViewName + " AS " + viewDef;
			this.getM4Db().executeBusinessSqlWrite(viewCreation);
			this.getM4Db().addViewToTrash(tempViewName, this.getM4Db().getBusinessSchemaName(), this.getStep().getId());
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator Windowing (Step '" + 
					this.getStep().getName() + 
					"': M4 Exception caught trying to create temporary view: " +
					m4e.getMessage());
		}
		catch (SQLException sqle) {
			throw new M4CompilerError("Operator Windowing (Step '" + 
					this.getStep().getName() + 
					"': SQL Exception caught trying to create temporary view in business data schema: " +
					sqle.getMessage());
		}
		return tempViewName;
	}
	
	// returns the index of the windowed output attribute, ie
	// its position in the window (between 0 and (windowsize-1)).
	private int getWindowPosition(BaseAttribute windowedBA) 
	throws M4CompilerError {
		String baName = windowedBA.getName();
		int separatorPos = baName.lastIndexOf('_');
		if (separatorPos < 0)
			throw new M4CompilerError("Operator Windowing (Step '" + 
					this.getStep().getName() + 
					"': could not determine position in window of output attribute " + baName + "!");
		int winpos = -1;
		String suffix = baName.substring(separatorPos + 1);
		try {
			winpos = Integer.parseInt(suffix);
		}
		catch (NumberFormatException nfe) {
			throw new M4CompilerError("Operator Windowing (Step '" + 
					this.getStep().getName() + 
					"': could not determine position in window of output attribute " + baName + "!");			
		}
		return winpos - 1;
	}
	
	private static String windowNumExpr = null;
	
	// returns an sql expression that computes the windownumber for the 
	// current row 
	private String getWindowNumExpr() throws M4CompilerError {
		if (windowNumExpr == null) {
			try {
				windowNumExpr = 
					this.getM4Db().getFloorExpression() + 
					"(" + 
						getTimeBA().getCurrentColumn().getSQLDefinition() + 
						" / " + this.getWinSize() + 
					")";
			}
			catch (M4Exception m4e) {
				throw new M4CompilerError("Operator Windowing, Step '" + this.getStep().getName() + "': M4 Exception caught when computing floor expression: " + m4e.getMessage());
			}
		}
		return windowNumExpr;
	}
	
    private BaseAttribute getTimeBA() throws M4CompilerError {
    	return (BaseAttribute) this.getSingleParameter(TIMEBASEATTRIB);
    }
	
    private BaseAttribute getValueBA() throws M4CompilerError {
    	return (BaseAttribute) this.getSingleParameter(VALUEBASEATTRIB);
    }
	
    private String getWinSize() throws M4CompilerError {
    	Value v = (Value) this.getSingleParameter(WINDOWSIZE);
    	if (v == null)
    		return null;
    	String s = v.getValue();
    	// check that it is an integer:
    	try {
    		Integer.parseInt(s);
    	}
    	catch (NumberFormatException nfe) {
    		return null;
    	}
    	return s;
    }
	
    private BaseAttribute getOutStartTimeBA() throws M4CompilerError {
    	return (BaseAttribute) this.getSingleParameter(OUTTIMESTARTBA);
    }
	
    private BaseAttribute getOutEndTimeBA() throws M4CompilerError {
    	return (BaseAttribute) this.getSingleParameter(OUTTIMEENDBA);
    }
}

/*
 * Historie
 * --------
 *
 * $Log: Windowing.java,v $
 * Revision 1.6  2006/09/27 14:59:56  euler
 * New version 1.1
 *
 * Revision 1.5  2006/04/11 14:10:10  euler
 * Updated license text.
 *
 * Revision 1.4  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.3  2006/03/30 16:07:13  scholz
 * fixed author tags for release
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:20  hakenjos
 * Initial version!
 *
 */
