/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4CompilerWarning;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Feature;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * This abstract class is the superclass for all operators that create a single
 * ColumnSet for the output concept.
 *
 * @see edu.udo.cs.miningmart.m4.core.operator.MultipleCSOperator
 * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator
 * 
 * @author Timm Euler
 * @version $Id: SingleCSOperator.java,v 1.6 2006/09/27 14:59:55 euler Exp $
 */
public abstract class SingleCSOperator extends ConceptOperator
{
    /**
     * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#generateColumnSetsForOp
     */
    public Columnset[] generateColumnSetsForOp(boolean lazy)
    throws M4CompilerError, M4CompilerWarning
    {
        // Since only one ColumnSet is created, the "lazy" parameter can be ignored.
        return new Columnset[] { createSingleColumnSet(-1) };
    }

    /**
     * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#generateSQLDefinition
     */
    public String generateSQLDefinition(String selectPart, int index) 
    throws M4CompilerError, M4CompilerWarning {
        // ignore index, as this operator creates only one ColumnSet
        return generateSQLDefinition(selectPart);
    }

    /**
     * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#getTypeOfNewColumnSet
     */
    public String getTypeOfNewColumnSet(int index)
    {
        // ignore index, as this operator creates only one ColumnSet
        return getTypeOfNewColumnSet();
    }
    
    /**
     * The standard handling of extra output features is printing out 
     * a warning. Subclasses may overwrite this method.
     */
    protected String handleExtraOutputFeature(Feature outputFeature, Columnset csForOuputConcept) 
    throws M4CompilerError {
        this.doPrint(Print.OPERATOR, "Output Concept '" + getOutputConcept().getName() + "': skipped feature '" +
                outputFeature.getName() + "' because no corresponding input feature was found.");
        return "";
    }

	/**
	 * For a <code>SingleConcept</code> operator the index is always -1,
	 * so this method can be defined without an argument.
	 * 
	 * @see ConceptOperator#getNewCSName(int)
	 */
	public String getNewCSName() throws M4CompilerError {
		return this.getNewCSName(-1);
	}

    /**
     * This method must return the entry String for the M4 table "Columnset_t",
     * indicating whether the new Columnset is a view or a table.
     *
     * @return The type of the new Columnset, either "V" or "T".
     * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#getTypeOfNewColumnSet
     */
    public abstract String getTypeOfNewColumnSet();

    /**
     * This method must return the SQL definition of the newly created ColumnSet.
     *
     * @param selectPart The SQL string between "SELECT" and "FROM" for the
     *        SQL definition to be returned.
     * @return The SQL definition for the new ColumnSet.
     */
    public abstract String generateSQLDefinition(String selectPart) 
    throws M4CompilerError, M4CompilerWarning;
}
/*
 * Historie
 * --------
 *
 * $Log: SingleCSOperator.java,v $
 * Revision 1.6  2006/09/27 14:59:55  euler
 * New version 1.1
 *
 * Revision 1.5  2006/06/13 13:30:42  euler
 * Updates
 *
 * Revision 1.4  2006/04/11 14:10:10  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:20  hakenjos
 * Initial version!
 *
 */
