/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.Operator;
import edu.udo.cs.miningmart.m4.Parameter;
import edu.udo.cs.miningmart.m4.ParameterObject;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.Value;

/**
 * @author euler
 *
 * This operator reverses the operation that constructed a feature
 * in another step. It is a normal feature construction operator,
 * only its method depends on the way the original feature was constructed.
 * This operator is useful when the target values of a prediction task are
 * changed during data preparation. This operator reverses those changes,
 * and can therefore be used for deploying the values predicted by the learned
 * model. For example, predicting the sales values of a shop with an SVM
 * may require to scale the target values for training to [0..1]. Thus the
 * SVM predicts only values between 0 and 1 when applied on unseen data. 
 * This operator then scales the values back to the original range. 
 */
public class ReverseFeatureConstruction extends FeatureConstruction {

	private int loopNumberUsedInStepToBeReversed = 0;
	
	/**
	 * @see edu.udo.cs.miningmart.operator.FeatureConstruction#generateSQL(edu.udo.cs.miningmart.m4.Column)
	 */
	public String generateSQL(Column targetColumn) throws M4CompilerError {
		// first check that there is something to be reversed:
		Step reversedStep = null;
		try {
		    reversedStep = this.getStep().getReversedStep();
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "':\nM4 problem accessing reversed step: " + m4e.getMessage());
		}
		if (reversedStep == null) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "':\ncould not find the step whose feature construction is to be reversed! No compilation possible.");
		}
		
		// second find the original attribute (the one with the result of
		// the operation that is to be reversed now):
		Column originalOutputColumn = this.getOriginalColumn(reversedStep);
		
		// third do the reversing, depending on the operator to be reversed:
		Operator operatorToBeReversed = reversedStep.getTheOperator();		
		String sql = null;
		
		if (operatorToBeReversed.getName().equalsIgnoreCase("LinearScaling")) {
			sql = this.reverseLinearScaling(reversedStep, targetColumn);
		}
		else if (operatorToBeReversed.getName().equalsIgnoreCase("LogScaling")) {
			sql = this.reverseLogScaling(reversedStep,  targetColumn);			
		}
		else if (operatorToBeReversed.getName().indexOf("rouping") > -1) {
			String sqlToBeReversed = originalOutputColumn.getSQLDefinition();
			if (sqlToBeReversed == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
						this.getStep().getName() + "':\nColumn to be reversed (from step '" + reversedStep.getName() + "') does not have an SQL definition!\nPlease compile that step first!");
			}
			sql = this.reverseGrouping(sqlToBeReversed, targetColumn);
		}
		else {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "':\nUnknown operator in step '" + 
					reversedStep.getName() + "', cannot reverse it!");
		}
		return sql;
	}
	
	private Column getOriginalColumn(Step reversedStep) throws M4CompilerError {
		Value origAttrParam = (Value) this.getSingleParameter("OriginalOutputAttribute");
		BaseAttribute origAttr = null;
		// The parameter giving the original name is optional, because
		// sometimes the original name can be found easily!
		try {
			if (origAttrParam != null) {
				String origAttrName = origAttrParam.getValue();
				Parameter inputConcParam = reversedStep.getParameterTuple("TheInputConcept", 0);
				if (inputConcParam != null) {
					Concept inputConc = (Concept) inputConcParam.getTheParameterObject();
					if (inputConc != null) {
						origAttr = inputConc.getBaseAttribute(origAttrName);
						this.loopNumberUsedInStepToBeReversed = this.getLoopNumber(reversedStep, origAttr);
						if (this.loopNumberUsedInStepToBeReversed == -1) {
							throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
									this.getStep().getName() + "':\nCannot determine which input attribute of step '" +
									reversedStep.getName() + 
									"'\nis to be reversed (error with loop identification)!\nPlease specify the attribute name in parameter 'OriginalAttribute' of step '" +
									this.getStep().getName() + "'!");							
						}
					}
				}
			}
			else {
				// try to find the original attribute. If not possible
				// throw message to user!
				// see if there is only one output BA, if so, it must be the one:
				if (reversedStep.getLoopCount() > 0) {
					throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
							this.getStep().getName() + "':\nCannot determine which input attribute of step '" +
							reversedStep.getName() + 
							"'\nis to be reversed, since the step has loops!\nPlease specify the attribute name in parameter 'OriginalAttribute' of step '" +
							this.getStep().getName() + "'!");
				}
				Parameter outBaParam = reversedStep.getParameterTuple("TheOutputAttribute", 0);
				if (outBaParam != null) {
					origAttr = (BaseAttribute) outBaParam.getTheParameterObject();
				}
				this.loopNumberUsedInStepToBeReversed = 0;
			}
			if (origAttr == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
						this.getStep().getName() + "':\nCannot determine which input attribute of step '" +
						reversedStep.getName() + 
						"'\nis to be reversed! Please specify the attribute name in parameter 'OriginalAttribute' of step '" +
						this.getStep().getName() + "'!");				
			}
			Column origCol = origAttr.getCurrentColumn();
			if (origCol == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
						this.getStep().getName() + "':\nCannot determine which input attribute of step '" +
						reversedStep.getName() + 
						"'\nis to be reversed! Please specify the attribute name in parameter 'OriginalAttribute' of step '" +
						this.getStep().getName() + "'!");				
			}
			return origCol;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': M4 error trying to access attributes of reversed step: " + m4e.getMessage());
		}
	}
	
	private int getLoopNumber(Step reversedStep, BaseAttribute theOutputAttr) 
	throws M4CompilerError {
		try {
			if (reversedStep.getLoopCount() == 0) {
				return 0;
			}
			else {
				for (int loop = 1; loop <= reversedStep.getLoopCount(); loop++) {
					Parameter outBaParam = reversedStep.getParameterTuple("TheOutputAttribute", loop);
					if (outBaParam != null) {
						BaseAttribute attr = (BaseAttribute) outBaParam.getTheParameterObject();
						if (attr.equals(theOutputAttr)) {
							return loop;
						}
					}
				}
				return -1;
			}
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': M4 error trying to access original output attribute: " + m4e.getMessage());
		}
	}
	
	private String reverseLinearScaling(Step stepToReverse, Column targetColumn) 
	throws M4CompilerError {
		try {
			// find the original scaling parameters:
			String minRange = this.getValueOfParameter(stepToReverse, "NewRangeMin", this.loopNumberUsedInStepToBeReversed);
			String maxRange = this.getValueOfParameter(stepToReverse, "NewRangeMax", this.loopNumberUsedInStepToBeReversed);
		
			// find the original actual data minimum and maximum:
			ParameterObject origTarget = this.getParameterObject(stepToReverse, "TheTargetAttribute", this.loopNumberUsedInStepToBeReversed);
			if ( ! (origTarget instanceof BaseAttribute)) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
						this.getStep().getName() + "': could not access parameter 'TheTargetAttribute' of step '" + stepToReverse.getName() + "'!");
			}
			Column origColumn = ((BaseAttribute) origTarget).getCurrentColumn();
			if (origColumn == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
						this.getStep().getName() + "': could not access column of 'TheTargetAttribute' of step '" + stepToReverse.getName() + "'!");
			}
		
			String actMin = origColumn.readOrComputeMinimum();
			String actMax = origColumn.readOrComputeMaximum();
		
			// double check:
			if (minRange == null || maxRange == null || actMin == null || actMax == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
						this.getStep().getName() + "': some scaling parameter of the original step '" + stepToReverse + "' is missing!");
			}
		
			// set up the "dual" formula:
			String sql = "(((" + targetColumn.getSQLDefinition() + " - " + minRange + 
							") / (" + maxRange + " - " + minRange + ")) * (" +
							actMax + " - " + actMin + ") + " + actMin + ")";
			return sql;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': M4 error trying to set up reverse scaling formula: " + m4e.getMessage());
		}
	}

	private String reverseLogScaling(Step stepToReverse, Column targetCol) 
	throws M4CompilerError {
		try {
			String logBase = this.getValueOfParameter(stepToReverse, "LogBase", this.loopNumberUsedInStepToBeReversed);
			return this.getM4Db().getPowerExpression(logBase, targetCol.getSQLDefinition());
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': M4 error trying to reverse log scaling: " + m4e.getMessage());
		}
	}
	
	private String reverseGrouping(String toReverse, Column targetColumn) 
	throws M4CompilerError {
		// Since there are many possible ways by which the original value mapping
		// was created, it's easiest to parse the original SQL definition here.
		// Its form: (CASE WHEN oldColumnName IN ('Val1', 'Val2', 'Val3') THEN 'Label1' WHEN oldColumnName IN ('Val4') THEN 'Label2' ELSE ('DefaultLabel') END)
		
		toReverse = this.removeOuterBrackets(toReverse);
		toReverse = this.removeCaseAndEndCommand(toReverse);
		
		Map originalMapping = new HashMap();
		while (toReverse.startsWith("WHEN")) {
			toReverse = this.removeAndProcessFirstWhen(toReverse, originalMapping);
		}
		// default values are ignored, since it is unknown what they could be
		// mapped back to, but the sql command is checked for wellformedness:
		this.removeElse(toReverse);
	
		// now use the found mapping to reverse it:
		String sql = "(CASE ";
		Collection entries = originalMapping.entrySet();
		if (entries.isEmpty()) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': could not find any original grouping, nothing to reverse!");
		}
		Iterator entryIt = entries.iterator();
		while (entryIt.hasNext()) {
			Map.Entry oneEntry = (Map.Entry) entryIt.next();
			String key = (String) oneEntry.getKey();
			String object = (String) oneEntry.getValue();
			sql += "WHEN " + targetColumn.getSQLDefinition() + " = '" + 
					object + "' THEN '" + key + "' ";  
		}
		sql += " ELSE NULL END)";
		
		return sql;
	}
	
	private String removeOuterBrackets(String sql) throws M4CompilerError {
		while (sql.startsWith("(")) {
			if ( ! sql.endsWith(")")) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
				this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (closing bracket missing)!");
			}
			sql = sql.substring(1, sql.length() - 1);
		}
		return sql.trim();
	}
	
	private String removeCaseAndEndCommand(String command) throws M4CompilerError {
		if ( ! command.toUpperCase().startsWith("CASE")) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (CASE command missing)!");
		}
		if ( ! command.toUpperCase().endsWith("END")) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (END command missing)!");
		}
		return command.substring(4, command.length()-3).trim();
	}
	
	private String removeAndProcessFirstWhen(String whenCommand, Map theMap)
	throws M4CompilerError {
		if ( ! whenCommand.toUpperCase().startsWith("WHEN")) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (WHEN command missing)!");
		}
		whenCommand = whenCommand.substring(4).trim();
		int valuesIndex = whenCommand.toUpperCase().indexOf("IN (") + 4;
		if (valuesIndex == 3) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (values missing)!");
		}
		int valuesEndIndex = whenCommand.indexOf(")", valuesIndex);
		if (valuesEndIndex == -1) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (values' closing bracket missing)!");
		}
		String valuesList = whenCommand.substring(valuesIndex, valuesEndIndex);
		whenCommand = whenCommand.substring(valuesEndIndex + 1).trim();
		if ( ! whenCommand.toUpperCase().startsWith("THEN")) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
				this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (THEN command missing)!");
		}
		whenCommand = whenCommand.substring(4).trim();
		int labelEndIndex = whenCommand.indexOf(" ") + 1;
		if (labelEndIndex == 0) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (label after THEN missing)!");
		}
		String label = whenCommand.substring(0, labelEndIndex).trim();
		
		this.putToMap(theMap, valuesList, label);
		
		return whenCommand.substring(labelEndIndex).trim();
	}
	
	private void putToMap(Map theMap, String keys, String object) 
	throws M4CompilerError {
		StringTokenizer st = new StringTokenizer(keys);
		if (st.hasMoreTokens()) {
			String aKey = st.nextToken();
			theMap.put(this.deQuote(aKey), this.deQuote(object));
		}
		if (st.hasMoreTokens()) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': the original step performed a many-to-one grouping, which is irreversible!");
		}
	}
	
	private void removeElse(String elseRest) throws M4CompilerError {
		if ( ! elseRest.toUpperCase().startsWith("ELSE")) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': given SQL string with grouping definition is not well-formed (ELSE command missing)!");
		}
	}
	
	private String deQuote(String quoted) {
		quoted = quoted.trim();
		if (quoted.startsWith("'") && quoted.endsWith("'")) {
			return quoted.substring(1, quoted.length() - 1);
		}
		return quoted;
	}
	
	private String getValueOfParameter(Step whichStep, String nameOfParameter, int loopNr) 
	throws M4CompilerError {
		ParameterObject parObj = this.getParameterObject(whichStep, nameOfParameter, loopNr);
			
		if ( ! (parObj instanceof Value)) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
				this.getStep().getName() + "': could not access parameter '" + 
				nameOfParameter + "' of step '" + whichStep.getName() + "'!");
		}
		String value = ((Value) parObj).getValue();
		return value;
	}

	private ParameterObject getParameterObject(Step whichStep, String nameOfParameter, int loopNr) 
	throws M4CompilerError {
		try {
			Parameter par = whichStep.getParameterTuple(nameOfParameter, loopNr);
			if (par == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': could not access parameter '" + 
					nameOfParameter + "' of step '" + whichStep.getName() + "'!");
			}
			ParameterObject parObj = par.getTheParameterObject();
			if (parObj == null) {
				throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': could not access parameter '" + 
					nameOfParameter + "' of step '" + whichStep.getName() + "'!");
			}
			return parObj;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator '" + this.getName() + "' in Step '" +
					this.getStep().getName() + "': M4 error trying to access parameter '" + 
					nameOfParameter + "' of step '" + whichStep.getName() + "'!");
		}
	}
}
