/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.m4.Column;


/**
 * The abstract superclass of all operators that replace a missing value with a
 * new value. It implements the abstract
 * method generateColumnForOp() inherited from FeatureConstruction. All its
 * subclasses only have to implement the abstract generateValueForOp() method
 * specified here.
 * 
 * @author Timm Euler, Martin Scholz
 * @version $Id: MissingValues.java,v 1.5 2006/09/27 14:59:57 euler Exp $
 */
public abstract class MissingValues
    extends FeatureConstruction {
    	
    //~ Methods ...............................................................

    /**
     * Method to update the Column structure for missing values. - Generate the common
     * sql-string to replace only NULL-values - call the subclass-specific
     * method to generate the value to be filled for NULL-values
     *
     * @param theTargetAttributeColumn the Column object of the column with the missing values
     * @return the sql definition of the new column for TheOutputAttribute
     */
    public String generateSQL(Column theTargetAttributeColumn) throws M4CompilerError
    {
        String replacingValue = this.generateValueForOp(theTargetAttributeColumn);
        
        String sql = "(CASE WHEN " + theTargetAttributeColumn.getSQLDefinition() +
                     " is NULL THEN " + replacingValue +
                     " ELSE " + theTargetAttributeColumn.getSQLDefinition() +
                     " END)";
        return sql;
    }


    /**
     * The abstract method to be implemented by the subclasses.
     *
     * @param columnWithMissingValues The Column which has some missing values.
     * @return A String with an SQL definition which is used to replace the missing values
     *         of the given column. It must include quotes if necessary.
     * @throws M4CompilerError A simple exception object with an error message.
     */
    public abstract String generateValueForOp(Column columnWithMissingValues)
                                       throws M4CompilerError;

}
/*
 * Historie
 * --------
 * 
 * $Log: MissingValues.java,v $
 * Revision 1.5  2006/09/27 14:59:57  euler
 * New version 1.1
 *
 * Revision 1.4  2006/04/11 14:10:11  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/30 16:07:12  scholz
 * fixed author tags for release
 *
 * Revision 1.1  2006/01/03 09:54:22  hakenjos
 * Initial version!
 *
 */


/*
 * Old Historie
 * ------------
 *
 * Revision 1.17  2003/06/05 09:52:02  euler
 * Organised imports.
 *
 * Revision 1.16  2002/11/21 09:36:47  euler
 * Added SVMForClassification.
 *
 * Revision 1.15  2002/11/11 11:51:03  euler
 * Updated Javadoc comments.
 *
 * Revision 1.14  2002/10/25 13:43:26  euler
 * Changed Sql Definitions that FeatureConstruction
 * operators create. Adapted the Join operators.
 *
 * Revision 1.13  2002/10/08 18:07:56  scholz
 * Prepared code for parallel execution of multiple compile-Threads.
 * Calls to static fields like DB.m4Db were removed. Now each
 * Thread has its own DB object, reachable via
 * CompilerAccessLogic or Case.
 * The methods getCase() and getM4Db() were added to M4Object.
 * The static methods of Parameter now need an additional
 * parameter of type DB.
 * All direct calls from Operators to these Parameter methods were
 * removed.
 * All old load() and print() routines were removed.
 * The static calls to Print were removed. Now CompilerAccessLogic
 * references a valid Print object for the current Thread. This is
 * reachable via Case. The methods doPrint for messages and
 * Exceptions were added to M4Object.
 * The Print mechanism is not fully functional, yet.
 * A getStatus method was added to the Interface. It is not
 * functional yet for multiple Threads.
 *
 * Status: Compiles.
 *
 * Revision 1.12  2002/08/05 10:35:47  euler
 * Restructured the operator hierarchy: introduction of SingleCSOperator
 * and MultipleCSOperator. Changed this class accordingly.
 *
 * Revision 1.11  2002/06/11 11:32:36  euler
 * Added operator DeleteRecordsWithMissingValues.
 *
 * Revision 1.10  2002/06/06 07:55:40  euler
 * Fixed some bugs.
 *
 * Revision 1.9  2002/06/03 12:02:37  euler
 * Restructured FeatureConstruction.
 *
 * Revision 1.8  2002/05/31 15:11:59  euler
 * Changed link from column to BA to 1:n.
 *
 * Revision 1.7  2002/05/23 11:34:59  euler
 * A few tests worked, others still to be done.
 *
 * Revision 1.6  2002/05/16 13:54:42  euler
 * First version that compiles.
 *
 * Revision 1.5  2002/05/15 10:36:43  euler
 * First version that compiles.
 *
 * Revision 1.4  2002/05/14 12:44:19  bauschul
 * Compile-able, Code reformatet
 *
 * Revision 1.3  2002/05/10 14:03:45  euler
 * First M4 extensions, does not yet compile
 *
 * Revision 1.2  2002/05/10 13:23:21  euler
 * First M4 extensions, does not yet compile
 *
 * Revision 1.1  2002/04/30 13:00:28  wiese
 * compileable version
 *
 * Revision 1.6  2002/04/19 15:23:12  wiese
 * Initialversion nach der uebergabe
 *
 */
