/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.util.Iterator;
import java.util.Vector;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.RelationalDatatypes;
import edu.udo.cs.miningmart.m4.Value;

/**
 * This operator merges two attributes. It is intended for sparsely
 * populated attributes (with many NULL values, say at least 95 per cent).
 * The target attribute is merged with the 'AttributeToMerge'.
 * Whereever one attribute has a value while the other has NULL, the
 * value appears in the output attribute. Whereever both attributes
 * have a value, either the target attribute wins, or a new
 * value is introduced that represents this combination of values. This depends
 * on the entry in parameter 'ClashResolvation'. The attributes to be merged must
 * be CATEGORIAL; numeric attributes have to be discretised first.
 *
 * @author Timm Euler
 * @version $Id: MergeAttributes.java,v 1.3 2006/09/27 14:59:55 euler Exp $
 */
public class MergeAttributes extends FeatureConstruction {

	// Parameter names for this Operator
	public static final String PARAMETER_MERGE_ATTRIB = "AttributeToMerge";
	public static final String PARAMETER_CLASH_RESOLVE = "ClashResolvation";

	private BaseAttribute mergeAttrib;
	
    public String generateSQL(Column targetAttributeColumn) throws M4CompilerError {
    	String colDef = "(CASE ";
    	colDef += this.getConditionForPrioritisation();
    	if ( ! this.usePriorities()) {
    		colDef += this.getConditionForValueCombi(targetAttributeColumn);
    	}
    	colDef += " ELSE " + this.getTheTargetAttribute().getName() + " END)";
    	return colDef;
    }
    
    private String getConditionForPrioritisation() throws M4CompilerError {
    	String targetBaName = this.getTheTargetAttribute().getName();
    	String mergeBaName = this.getAttributeToMerge().getName();
    	String sql = "WHEN " + targetBaName + " IS NULL AND " + mergeBaName + " IS NOT NULL THEN " + mergeBaName +
		             " WHEN " + mergeBaName + " IS NULL THEN " + targetBaName;
    	return sql;
    }
    
    private String getConditionForValueCombi(Column targetCol) 
    throws M4CompilerError {
    	String[][] allCombis = this.getValueCombinations(targetCol);
    	String targetBaName = this.getTheTargetAttribute().getName();
    	String mergeBaName = this.getAttributeToMerge().getName();
    	String sql = "";
    	for (int i = 0; i < allCombis.length; i++) {
			sql += " WHEN " + targetBaName + " = '" + allCombis[0][i] +
			            "' AND " + mergeBaName + " = '" + allCombis[1][i] +
						"' THEN '" + allCombis[2][i] + "'";			
		}
    	return sql;
    }
    
    private BaseAttribute getAttributeToMerge() throws M4CompilerError {
    	if (this.mergeAttrib == null) {
    	    this.mergeAttrib = (BaseAttribute) this.getSingleParameter(PARAMETER_MERGE_ATTRIB);    	   
    	}
    	return this.mergeAttrib;
    }
    
    private boolean usePriorities() throws M4CompilerError {
    	Value v = (Value) this.getSingleParameter(PARAMETER_CLASH_RESOLVE, this.getCurrentLoopNumber());
    	if (v == null || v.getValue() == null || v.getValue().equals("")) {
    		throw new M4CompilerError("Operator 'MergeAttributes', step '" + this.getStep().getName() + 
    				"': could not read parameter '" + PARAMETER_CLASH_RESOLVE + "'!");
    	}
    	String resolveMethod = v.getValue();
    	return resolveMethod.equalsIgnoreCase("priority");
    }    
    
    // returned array has in each row the single values and a combined value
    private String[][] getValueCombinations(Column targetCol) throws M4CompilerError {
    	try {
    		this.getAttributeToMerge();
    		int noOfColumns = 3; // no of columns of array to be returned
    		Vector valueLists = new Vector();
    		int noOfRows = 1;
    		
    		// read occurring values of the two attributes to be merged:
    		String[] vals = this.getM4Db().getDistinctElementsWithoutNull(targetCol);
    		Vector valueList = new Vector();
    		for (int j = 0; j < vals.length; j++) {
				valueList.add(vals[j]);
			}
    		noOfRows = noOfRows * vals.length;
    		valueLists.add(valueList);
    		vals = this.getM4Db().getDistinctElementsWithoutNull(this.mergeAttrib.getCurrentColumn());
    		valueList = new Vector();
    		for (int j = 0; j < vals.length; j++) {
				valueList.add(vals[j]);
			}
    		noOfRows = noOfRows * vals.length;
    		valueLists.add(valueList);
    		
    		// compute all combinations of values of the attribs to be merged:
    		Vector allCombis = this.getCombinations(valueLists);
    		
    		// create array to be returned:    	
    		String[][] ret = new String[noOfColumns][noOfRows];
    		int currentRow = 0;
    		Iterator it = allCombis.iterator();
    		while (it.hasNext()) {
				Vector oneCombination = (Vector) it.next();
				Iterator it2 = oneCombination.iterator();
				int currentColumn = 0;
				String combi = "";
				while (it2.hasNext()) {
					String aValue = (String) it2.next();
					ret[currentColumn][currentRow] = aValue;
					combi += aValue + "_";
					currentColumn++;
				}
				combi = combi.substring(0, combi.length() - 1);
				ret[currentColumn][currentRow] = combi;
				currentRow++;
			}
    		return ret;
    	}
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Operator 'MergeAttributes': M4 error when accessing occurring values of attributes to merge: " + m4e.getMessage());
    	}
    }
    
    private Vector getCombinations(Vector listsOfValues) {
    	if (listsOfValues == null || listsOfValues.isEmpty()) {
    		Vector emptyList = new Vector();
    		Vector emptyCombi = new Vector();
    		emptyCombi.add(emptyList);
    		return emptyCombi;
    	}
    	Vector firstList = (Vector) listsOfValues.remove(0);
    	Vector allCombis = new Vector();
		Vector combisSoFar = this.getCombinations(listsOfValues);
    	Iterator it = firstList.iterator();
    	while (it.hasNext()) {
			String aValue = (String) it.next();
			Iterator it2 = combisSoFar.iterator();
			while (it2.hasNext()) {
				Vector aCombi = (Vector) it2.next();
				aCombi.add(0, aValue);
				allCombis.add(new Vector(aCombi));
				aCombi.remove(0);
			}
		}
    	return allCombis;
    }
}
/*
 * Historie
 * --------
 * 
 * $Log: MergeAttributes.java,v $
 * Revision 1.3  2006/09/27 14:59:55  euler
 * New version 1.1
 *
 * Revision 1.2  2006/05/28 17:14:30  euler
 * First working version.
 *
 * Revision 1.1  2006/05/26 15:04:05  euler
 * Initial version, not complete yet.
 *
 * Revision 1.4  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/03/17 17:06:39  euler
 * *** empty log message ***
 *
 */
