/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;
import java.sql.SQLException;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.EstimatedStatistics;
import edu.udo.cs.miningmart.m4.ForeignKeyLink;
import edu.udo.cs.miningmart.m4.Relation;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.Value;
import edu.udo.cs.miningmart.m4.core.ForeignKey;

/**
 * @author Martin Scholz 
 */
public class MaterializeRelation extends ExecutableOperator {

	/**
	 * @see ExecutableOperator#createStatement(boolean)
	 */
	public void createStatement(boolean lazy) throws M4CompilerError {
		try {
			// Example SQL statement:
			// ALTER TABLE CS_12345 ADD 
			// CONSTRAINT CS_12345_FK
		    // FOREIGN KEY (CS_ABC_COLNAME)
		    //   REFERENCES CS_54564_T (CS_54564_PK)

			String tableName = this.getTableName();
			
			this.getM4Db().dropBusinessTable(tableName); // try to drop the table first if it exists
			String sql = "CREATE TABLE " + tableName + " AS ( SELECT * FROM " + this.getCrossTableName() + " )";			
			this.executeBusinessSqlWrite(sql);
		
			Relation rel = this.getRelation();
			sql = this.keyToStatement(tableName, (ForeignKey) rel.getFromKey());
			this.executeBusinessSqlWrite(sql);
			
			sql = this.keyToStatement(tableName, (ForeignKey) rel.getToKey());
			this.executeBusinessSqlWrite(sql);
			
			this.getM4Db().addTableToTrash(tableName, this.getSchema(), this.getStep().getId());
		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"Operator 'MaterializeRelation': M4Exception caught during createStatement:\n"
				+ e.getMessage());
		}
		catch (SQLException e) {
			throw new M4CompilerError(
				"Operator 'MaterializeRelation': SQLException caught during createStatement:\n"
				+ e.getMessage());
		}
	}
	
	private String keyToStatement(String tableName, ForeignKey fk) throws M4Exception {
		// simplified: assumes single column keys
		ForeignKeyLink fkl = fk.getFirstColumnLink(); 
		String links = fkl.getForeignKeyColumn().getName();
		
		String linkedToTab = fkl.getForeignKeyColumn().getColumnset().getSchemaPlusName();
		String linkedToCol = fkl.getForeignKeyColumn().getName();
		
		return "ALTER TABLE " + tableName + " ADD CONSTRAINT " + tableName
		       + "_FK FOREIGN KEY (" + links + ") REFERENCES "
			   + linkedToTab + "(" + linkedToCol + ")";		
	}

	/** Not required. Dummy method. */
	public void compileStatement() {}
	
	/** Not required. Dummy method. */
	public EstimatedStatistics estimateStatistics(Step theStep) throws M4Exception {
    	return null;
    }
	
	/**
	 * Gets the table name.
	 * 
	 * @return a Concept
	 */
	public String getTableName() throws M4CompilerError {
		Value value = (Value) this.getSingleParameter("TableName");
		
		if (value != null && value.getValue().trim().length() > 0) {
			return value.getValue().trim();
		}
		else return this.getCrossTableName() + "_T";
	}
	
	/**
	 * Gets the Relation.
	 * @return a Relation
	 */
	protected Relation getRelation() throws M4CompilerError {
		return (Relation) this.getSingleParameter("TheRelation");
	}

	/** @return the business schema of the cross-table as a String */
	protected String getSchema() throws M4CompilerError, M4Exception {
		return ((Relation) this.getSingleParameter("TheRelation")).getCrossLinkColumnSet().getSchema();
	}
	
	/**
	 * Only n:m relations are currently supported.
	 * @return the schemaPlusName String of the relation's cross-table 
	 */
	protected String getCrossTableName() throws M4CompilerError {
		try {
			Columnset cs = this.getRelation().getCrossLinkColumnSet();
			return cs.getSchemaPlusName();
		}
		catch (M4Exception e) {
			throw new M4CompilerError("MaterializeRelation: Could not read cross-table name. Only n:m relations are supported.");
		}	
	}
	
}
