/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;
import java.sql.SQLException;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.Value;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * @author Martin Scholz
 * @version $Id: Materialize.java,v 1.6 2006/09/27 14:59:56 euler Exp $
 */
public class Materialize extends SingleCSOperator {

	/**
	 * @see ExecutableOperator#createStatement(boolean)
	 */
	public String generateSQLDefinition(String selectPart)
		throws M4CompilerError
	{
		Concept concept = this.getInputConcept();
		edu.udo.cs.miningmart.m4.Columnset cs;
		try {
			if (concept == null || (cs = concept.getCurrentColumnSet()) == null) {
				throw new M4CompilerError("Operator 'Materialize': Found no input Concept/Columnset!");
			}
			
			String tableName = this.getNewCSName();			
			this.getM4Db().dropBusinessTable(tableName); // try to drop the table first if it exists

			// inform the user how long they have to wait:
			long noOfRowsToMaterialise = cs.getStatisticsAll();
			this.doPrint(Print.OPERATOR, "Materialisation of " + noOfRowsToMaterialise + " tuples starts now.");
			
			// start materialising:
			String sql = "CREATE TABLE " + tableName + " AS " + cs.getCompleteSQLQuery();			
			this.executeBusinessSqlWrite(sql);
			this.getM4Db().addTableToTrash(tableName, cs.getSchema(), this.getStep().getId());
			
			// ready:
			return tableName;
		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"Operator 'Materialize': M4Exception caught during createStatement:\n"
				+ e.getMessage());
		}
		catch (SQLException e) {
			throw new M4CompilerError(
				"Operator 'Materialize': SQLException caught during createStatement:\n"
				+ e.getMessage());
		}
	}

	/** Inherited abstract method. */
    public String getTypeOfNewColumnSet() {
    	return Columnset.CS_TYPE_TABLE;	// output is a table
    }
    
	/** Inherited abstract method. */
	protected boolean mustCopyFeature(String nameOfFeature) {
		return true; // copy all features from the input to the output concept
	}

	/** Overrides the standard naming method to use user specified name instead. */
	public String getNewCSName(int index) throws M4CompilerError {
		String tableName = this.getTableName();
		if (tableName == null) {
			return super.getNewCSName(index);
		}
		else return tableName + this.getInputConcept().getCurrentSuffix();
	}

	/**
	 * Gets the table name.
	 * 
	 * @return Returns a Concept
	 */
	public String getTableName() throws M4CompilerError {
		Value value = (Value) this.getSingleParameter("TableName");
		if (value != null) {
			String s = value.getValue();
			if (s != null && s.trim().length() > 0) {
				return s.trim();
			}
		}
		return null;
	}   
}
/*
 * Historie
 * --------
 * 
 * $Log: Materialize.java,v $
 * Revision 1.6  2006/09/27 14:59:56  euler
 * New version 1.1
 *
 * Revision 1.5  2006/08/10 14:38:02  euler
 * New mechanism for reversing steps
 *
 * Revision 1.4  2006/04/11 14:10:10  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.1  2006/01/03 09:54:20  hakenjos
 * Initial version!
 *
 */
