/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.util.Collection;
import java.util.Iterator;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Feature;

/**
 * This class is the super class for all Feature Selection operators.
 * Subclasses must implement the method <code>Feature[] computeListOfAttributes</code>
 * which delivers the selected features as an array of Feature objects.
 * 
 * @author Timm Euler
 * @version $Id: FeatureSelection.java,v 1.5 2006/09/27 14:59:56 euler Exp $
 */
public abstract class FeatureSelection extends SingleCSOperator {

	protected Collection mySelectedFeatures = null;
	private Collection outputFeatures = null;
	
	/**
	 * Abstract method to be implemented by all subclasses. 
	 * 
	 * @param theSuperset An array of Feature objects to which the selection 
	 * 	 	  is to be applied
	 * @return the selected Features as a <code>Collection</code>
	 */
	abstract protected Collection computeListOfAttributes(Feature[] theSuperset) throws M4CompilerError;
	
	/**
	 * @see edu.udo.cs.miningmart.m4.core.Operator#mustCopyFeature
	 */
    protected boolean mustCopyFeature(String nameOfFeature) throws M4CompilerError 
    {
		if (mySelectedFeatures == null)
		{	mySelectedFeatures = this.computeListOfAttributes(this.getTheAttributes());  }
		
		// Two kinds of features must be copied:
			
		// Case 1: the feature is one of the selected ones (thus also one of TheAttributes)
		if (this.occursIn(nameOfFeature, mySelectedFeatures))
		{   return true;  }		
	
		// Case 2: the feature is in TheOutputConcept, but NOT one of TheAttributes
		try {
			if (outputFeatures == null)
			{  outputFeatures = this.getOutputConcept().getFeatures();  }
		}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  }    		
		
		Iterator it = outputFeatures.iterator();
		while (it.hasNext())
		{
			if (((Feature) it.next()).getName().equalsIgnoreCase(nameOfFeature) 
				&& 
			   ( ! this.occursIn(nameOfFeature, this.getTheAttributes())))
			{   return true;  }
		}	
		
		return false;
    }

    /**
     * @see edu.udo.cs.miningmart.m4.core.operator.ConceptOperator#getTypeOfNewColumnSet
     */
    public String getTypeOfNewColumnSet()
    {  return Columnset.CS_TYPE_VIEW;  }

    /**
     * @see edu.udo.cs.miningmart.m4.core.operator.SingleCSOperator#generateSQLDefinition
     */
    public String generateSQLDefinition(String selectPart) throws M4CompilerError
    {
    	try {
	        String viewDef = "(select " + selectPart +
		                     " from " + getInputConcept().getCurrentColumnSet().getSchemaPlusName() + ")";
	        return viewDef;
    	}
   		catch (M4Exception m4e)
   		{   throw new M4CompilerError("M4 interface error in " + this.getName() + ": " + m4e.getMessage());  }    		
    } // end public Columnset generateSQLDefinition
    
    /**
     * Getter method for the parameter "TheAttributes". This parameter specifies
     * the set of features from which some should be selected.
     * 
     * @return an array of Feature objects
     */
    protected Feature[] getTheAttributes() throws M4CompilerError
    {
    	return (Feature[]) this.getParameter("TheAttributes");
    }
    
    // returns true if featureName occurs in theFeatures, false otherwise
    protected boolean occursIn(String featureName, Collection theFeatures)
    {
    	Iterator it = theFeatures.iterator();
    	while (it.hasNext())
		{
			if (((Feature) it.next()).getName().equalsIgnoreCase(featureName))
				return true;
		}
		return false;
    }
    
    // returns true if featureName occurs in theArray, false otherwise
    protected boolean occursIn(String featureName, Feature[] theArray)
    {
		for (int i = 0; i < theArray.length; i++)
		{
			if (theArray[i].getName().equalsIgnoreCase(featureName))
				return true;
		}
		return false;
    }
}
/*
 * Historie
 * --------
 *
 * $Log: FeatureSelection.java,v $
 * Revision 1.5  2006/09/27 14:59:56  euler
 * New version 1.1
 *
 * Revision 1.4  2006/04/11 14:10:11  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/30 16:07:13  scholz
 * fixed author tags for release
 *
 * Revision 1.1  2006/01/03 09:54:21  hakenjos
 * Initial version!
 *
 */

/*
 * Old Historie
 * ------------
 *
 * Revision 1.19  2003/03/06 14:09:08  euler
 * Small changes.
 *
 * Revision 1.18  2002/11/26 17:57:47  euler
 * Updates on Feature Selection and SVMs.
 *
 * Revision 1.17  2002/11/08 15:11:56  euler
 * Added parameter "TheAttributes".
 *
 * Revision 1.16  2002/11/07 09:50:25  euler
 * Bugfix: must fully qualify name in
 * "FROM" clause of view definition.
 *
 * Revision 1.15  2002/11/07 09:49:06  euler
 * Bugfix: must not compute list of attributes
 * more than once.
 *
 * Revision 1.14  2002/11/04 10:01:52  euler
 * New Javadoc comments.
 *
 * Revision 1.13  2002/10/08 18:07:56  scholz
 * Prepared code for parallel execution of multiple compile-Threads.
 * Calls to static fields like DB.m4Db were removed. Now each
 * Thread has its own DB object, reachable via
 * CompilerAccessLogic or Case.
 * The methods getCase() and getM4Db() were added to M4Object.
 * The static methods of Parameter now need an additional
 * parameter of type DB.
 * All direct calls from Operators to these Parameter methods were
 * removed.
 * All old load() and print() routines were removed.
 * The static calls to Print were removed. Now CompilerAccessLogic
 * references a valid Print object for the current Thread. This is
 * reachable via Case. The methods doPrint for messages and
 * Exceptions were added to M4Object.
 * The Print mechanism is not fully functional, yet.
 * A getStatus method was added to the Interface. It is not
 * functional yet for multiple Threads.
 *
 * Status: Compiles.
 *
 * Revision 1.12  2002/10/01 14:22:30  euler
 * New Feature Selection class.
 *
 * Revision 1.11  2002/08/30 12:33:34  scholz
 * view definitions no longer inline in sql statements
 *
 * Revision 1.10  2002/08/05 10:35:47  euler
 * Restructured the operator hierarchy: introduction of SingleCSOperator
 * and MultipleCSOperator. Changed this class accordingly.
 *
 * Revision 1.9  2002/06/13 16:30:05  euler
 * Changed naming of new Columnset;
 * uses "inputConcept.getSuffix()" now.
 *
 * Revision 1.8  2002/05/31 12:35:01  euler
 * *** empty log message ***
 *
 * Revision 1.7  2002/05/23 11:34:59  euler
 * A few tests worked, others still to be done.
 *
 * Revision 1.6  2002/05/15 13:05:02  euler
 * First version that compiles.
 *
 * Revision 1.5  2002/05/15 10:36:42  euler
 * First version that compiles.
 *
 * Revision 1.4  2002/05/14 12:56:03  euler
 * First M4 extensions, does not yet compile
 *
 * Revision 1.3  2002/05/10 14:03:38  euler
 * First M4 extensions, does not yet compile
 *
 * Revision 1.2  2002/05/10 13:23:21  euler
 * First M4 extensions, does not yet compile
 *
 * Revision 1.1  2002/04/30 13:00:27  wiese
 * compileable version
 *
 * Revision 1.6  2002/04/19 15:23:12  wiese
 * Initialversion nach der uebergabe
 *
 */


