/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.ConceptualDatatypes;
import edu.udo.cs.miningmart.m4.Feature;
import edu.udo.cs.miningmart.m4.ForeignKey;
import edu.udo.cs.miningmart.m4.ForeignKeyLink;
import edu.udo.cs.miningmart.m4.Relation;
import edu.udo.cs.miningmart.m4.RelationalDatatypes;
import edu.udo.cs.miningmart.m4.Value;

/**
 * @author Timm Euler
 * @version $Id: FeatureConstructionByRelation.java,v 1.7 2006/09/27 14:59:57 euler Exp $
 */
public class FeatureConstructionByRelation extends SingleCSOperator
{
	public static final String PARAMETER_AGGREGATION  = "AggregationOperator";
	public static final String PARAMETER_RELATION = "TheRelation";
	public static final String PARAMETER_FROMCONCEPT = "TheFromConcept";
	public static final String PARAMETER_TARGET_ATTR = "TheTargetAttribute";
	public static final String PARAMETER_OUTPUT_ATTR = "TheOutputAttribute";
	public static final String AGGREGATION_VALUEOF = "VALUE_OF";

	private boolean isManyToMany;
	private Relation myRelation;
	private Vector myToKeys;
	private String tempViewName = null;

    /**
     * Implementation of the abstract method from <code>SingleCSOperator</code>.
     */
    public String generateSQLDefinition(String selectPart) throws M4CompilerError {
    	
	    String ret = "(select " + selectPart +
		             " from " + this.getFromPart() +
	    	         " where " + this.getJoinCondition() + ")";
	    return ret;
    }

    /**
     * Overwrites the super class method so that the newly constructed Feature
     * can be handled.
     */
    protected String handleExtraOutputFeature(Feature outF, Columnset csForOutputConcept) 
    throws M4CompilerError {
    	try {
    		// create temporary view if it has not been created yet:
    		if (this.tempViewName == null) {
    			this.tempViewName = this.createIntermediateView();
    		}
    		
    		// loop through the target attributes to see if any of them matches
    		// the given output feature outF:
    		int endLoop = this.getStep().getLoopCount();
    		if (endLoop == 0) endLoop = 1;
    		String sql = null;
    		for (int loopnr = 0; loopnr < endLoop; loopnr++) {				
    			BaseAttribute outBA = (BaseAttribute) this.getSingleParameter(PARAMETER_OUTPUT_ATTR, loopnr);
    			if (outF instanceof BaseAttribute && ((BaseAttribute) outF).equals(outBA)) {    				
    				sql = "TMPCOL_" + loopnr;
    				Column newCol = (Column) this.getM4Db().createNewInstance(Column.class);
    				newCol.setName(outBA.getName());
    				newCol.setSQLDefinition(sql);
    				newCol.setBaseAttribute(outBA);
    				newCol.setColumnDataTypeName(RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER);
    				newCol.setColumnset(csForOutputConcept);
    			}
    		}
    		if (sql == null)
    			return "";
    		else 
    			return sql + ", ";
    	}
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
    				this.getStep().getName() +"': M4 error accessing target or output attribute: " + m4e.getMessage());
    	}
    }
    
    // creates a view to be joined with the FromConcept to give
    // the output of this operator. Returns the view name
    private String createIntermediateView() throws M4CompilerError {
    	String viewName = "CS_" + this.getStep().getId() + "_TMP";
    	String sql = "CREATE OR REPLACE VIEW " + viewName + " AS (";
    	sql += "SELECT " + this.getPrimKeysOfToConcept() + ", ";
		int endLoop = this.getStep().getLoopCount();
		if (endLoop == 0) endLoop = 1;
		for (int loopnr = 0; loopnr < endLoop; loopnr++) {
			String op = ((Value) this.getSingleParameter(PARAMETER_AGGREGATION, loopnr)).getValue();
			if (op.equalsIgnoreCase(AGGREGATION_VALUEOF)) {
				op = "MIN"; // the minimum of a set of equal values is that value
			}
			sql += op + "(CASE WHEN " + this.getTargetCol(loopnr) + 
					" = " + this.getMostCommonValueOfTargetAttribute(loopnr) + 
					" THEN " + this.getTargetCol(loopnr) + 
					" ELSE NULL END) AS TMPCOL_" + loopnr + ", ";
		}
		sql = sql.substring(0, sql.length() - 2);
		sql += " FROM " + this.getListOfColumnSets() + " WHERE " + this.getCondition() + ")";
		try {
			this.getM4Db().executeBusinessSqlWrite(sql);
			this.getM4Db().addViewToTrash(viewName, this.getM4Db().getBusinessSchemaName(), this.getStep().getId());
			return viewName;
		}
		catch (SQLException sqle) {
			throw new M4CompilerError("Step '" + this.getStep().getName() + "': SQL error creating an internal temporary view:" + sqle.getMessage());
		}
    }
    
    private String getJoinCondition() throws M4CompilerError {
    	try {
    		Concept to = this.getMyRelation().getTheToConcept();
    		String toCsName = to.getCurrentColumnSet().getSchemaPlusName();
    		String joinCondition = "";
    		Iterator it = this.myToKeys.iterator();
    		while (it.hasNext()) {
				String columnName = (String) it.next();
				joinCondition += toCsName + "." + columnName + " = " + this.tempViewName + "." + columnName + " AND ";
			}
    		joinCondition = joinCondition.substring(0, joinCondition.length() - 5);
    		return joinCondition;
    	}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
					this.getStep().getName() +"': M4 error trying to join columnsets: " + m4e.getMessage());
		}
    }
    
    private String getFromPart() throws M4CompilerError {
    	try {
			String list = "";
			Concept from = this.getMyRelation().getTheToConcept();
			list += from.getCurrentColumnSet().getSchemaPlusName() + ", ";
			list += this.tempViewName;
			return list;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
					this.getStep().getName() +"': M4 error trying to join columnsets: " + m4e.getMessage());
		}
    }
    
	/**
	 * Gets the relation.
	 * @return Returns a Relation
	 */
	protected Relation getMyRelation() throws M4CompilerError {
		if (this.myRelation == null) {
			this.myRelation = (Relation) this.getSingleParameter(PARAMETER_RELATION);
			if (this.myRelation == null) {
				throw new M4CompilerError("Operator 'FeatureConstructionByRelation', Step '" +
						this.getStep().getName() + "': no input relation found!");
			}
			this.isManyToMany = this.myRelation.isManyToManyRelation();
		}
		return this.myRelation;
	}

	/**
	 * Creates the WHERE part of the view definition.
	 */
	protected String getCondition() throws M4CompilerError {
		String condition = "";
		try {
			ForeignKey fromFk = this.getMyRelation().getFromKey();
			if (fromFk == null) {
				throw new M4CompilerError("Step '" + this.getStep().getName() + "': the relation given by parameter 'TheRelation' has no keys; please compile the step that creates it first!");
			}
			condition += this.getWhereLink(fromFk);
			if (this.isManyToMany) {
				condition += " AND ";
				ForeignKey toFk = this.getMyRelation().getToKey();
				condition += this.getWhereLink(toFk);
			}

			condition += " GROUP BY " + this.getPrimKeysOfToConcept();
			return condition;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
					this.getStep().getName() +"': M4 error accessing the relation keys: " + m4e.getMessage());
		}
	}
	
	private String getPrimKeysOfToConcept() throws M4CompilerError {
		try {
			String ret = "";
			ForeignKey link = (this.isManyToMany ? this.getMyRelation().getToKey() : this.getMyRelation().getFromKey());
			if (link == null) {
				throw new M4CompilerError("Step '" + this.getStep().getName() + "': the relation given by parameter 'TheRelation' has no keys; please compile the step that creates it first!");
			}
			Iterator linkIt = link.getAllColumnLinks().iterator();
			this.myToKeys = new Vector(); // needed later, in 'getJoinCondition'
			while (linkIt.hasNext()) {
				ForeignKeyLink columnPair = (ForeignKeyLink) linkIt.next();
				Column toColumn = columnPair.getPrimaryKeyColumn();
				ret += toColumn.getSQLPlusLocation() + ", ";
				this.myToKeys.add(toColumn.getSQLDefinition());
			}
			ret = ret.substring(0, ret.length() - 2);
			return ret;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
					this.getStep().getName() +"': M4 error accessing keys of ToConcept of given relation: " + m4e.getMessage());
		}
	}

	/**
	 * Creates the FROM part of the view definition.
	 */
	protected String getListOfColumnSets() throws M4CompilerError {
		try {
			String listOfColumnSets = "";
			Concept from = this.getMyRelation().getTheFromConcept();
			listOfColumnSets += from.getCurrentColumnSet().getSchemaPlusName() + ", ";
			if (this.isManyToMany) {
				listOfColumnSets += this.getMyRelation().getCrossLinkColumnSet().getSchemaPlusName() + ", ";
			}
			Concept to = this.getMyRelation().getTheToConcept();
			listOfColumnSets += to.getCurrentColumnSet().getSchemaPlusName();
			return listOfColumnSets;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
					this.getStep().getName() +"': M4 error accessing the relation: " + m4e.getMessage());
		}
	}

    /**
     * Must implement this method from ConceptOperator().
     */
    public String getTypeOfNewColumnSet()
    {  return Columnset.CS_TYPE_VIEW;  }

    protected boolean mustCopyFeature(String nameOfFeature)
    {  return true;  }

	/**
	 * This method overrides the super class method because the
	 * input concept is given by the ToConcept of the relation.
	 */
    public Concept getInputConcept() throws M4CompilerError {
		try {
			return this.getMyRelation().getTheToConcept();
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
					this.getStep().getName() +"': M4 error accessing the relation's ToConcept: " + m4e.getMessage());
		}
    }

    private String getMostCommonValueOfTargetAttribute(int loopnr) throws M4CompilerError {
    	try {
    		Concept from = this.getMyRelation().getTheFromConcept();    		
    		String targetCol = this.getTargetCol(loopnr);
    		String toCS = null;
    		ForeignKey fk = this.getMyRelation().getFromKey();
    		if (this.isManyToMany) {
    			toCS = this.getMyRelation().getCrossLinkColumnSet().getSchemaPlusName();
    		}
    		else {
    			toCS = this.getMyRelation().getTheToConcept().getCurrentColumnSet().getSchemaPlusName();
    		}
    		String selectCounts = "SELECT " + targetCol + ", COUNT(" + targetCol +
				") FROM " + toCS + ", " + from.getCurrentColumnSet().getSchemaPlusName() +
				" WHERE " + this.getWhereLink(fk) + " GROUP BY " + targetCol;
    		String mostFrequentValue = null;
    		int highestFrequency = -1;
    		ResultSet rs = this.executeBusinessSqlRead(selectCounts);
    		while (rs.next()) {
				int freq = rs.getInt(2);
				if (freq > highestFrequency) {
					highestFrequency = freq;
					mostFrequentValue = rs.getString(1);
				}
			}
    		if (mostFrequentValue == null) {
    			throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
						this.getStep().getName() +"': could not determine a frequent value of target attribute!");
    		}

    		BaseAttribute target = (BaseAttribute) this.getSingleParameter(PARAMETER_TARGET_ATTR, loopnr);
    		if (target.getConceptualDataTypeName().equals(ConceptualDatatypes.CDT_NOMINAL)) {
    			mostFrequentValue = DB.quote(mostFrequentValue);    			
    		}
    		return mostFrequentValue;
    	}
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
				this.getStep().getName() +"': M4 error trying to determine most frequent value of target attribute: " + m4e.getMessage());
    	}	
    	catch (SQLException sqle) {
    		throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
				this.getStep().getName() +"': SQL error trying to determine most frequent value of target attribute: " + sqle.getMessage());
    	}			 
    }
    
    private String getTargetCol(int loopnr) throws M4CompilerError {
    	try {
    		BaseAttribute target = (BaseAttribute) this.getSingleParameter(PARAMETER_TARGET_ATTR, loopnr);
    		String targetCol = target.getCurrentColumn().getSQLPlusLocation();
    		return targetCol;
    	}
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
    				this.getStep().getName() +"': M4 error trying to access target attribute: " + m4e.getMessage());
        }	
    }
    
    // returns a string that links the columns of the given key,
    // for a where clause in an sql statement
    private String getWhereLink(ForeignKey fk) throws M4CompilerError {
    	String where = "";
    	try {
    		Collection colLinks = fk.getAllColumnLinks();
			if (colLinks == null) {
				throw new M4CompilerError("Step '" + this.getStep().getName() + "': the relation given by parameter 'TheRelation' has no keys; please compile the step that creates it first!");
			}
    		
    		Iterator it = fk.getAllColumnLinks().iterator();
    		while (it.hasNext()) {
    			ForeignKeyLink myColKey = (ForeignKeyLink) it.next();
    			where += myColKey.getForeignKeyColumn().getSQLPlusLocation() + " = ";
    			where += myColKey.getPrimaryKeyColumn().getSQLPlusLocation() + " AND ";
    		}
    		where = where.substring(0, where.length() - 5);
    		return where;
    	}
    	catch (M4Exception m4e) {
    		throw new M4CompilerError("Operator 'FeatureConstructionByRelation, Step '" +
    				this.getStep().getName() +"': M4 error trying to access foreign key '" +
					fk.getName() + "'!");    		
    	}
    }    
}
/*
 * Historie
 * --------
 * 
 * $Log: FeatureConstructionByRelation.java,v $
 * Revision 1.7  2006/09/27 14:59:57  euler
 * New version 1.1
 *
 * Revision 1.6  2006/06/20 11:30:54  euler
 * Uses MIN for VALUE_OF (no aggregation)
 *
 * Revision 1.5  2006/06/20 09:49:01  euler
 * Bugfix
 *
 * Revision 1.4  2006/06/20 09:35:03  euler
 * Bugfix
 *
 * Revision 1.3  2006/06/20 07:51:42  euler
 * Bugfixes
 *
 * Revision 1.2  2006/06/19 15:37:13  euler
 * Bugfixes
 *
 * Revision 1.1  2006/06/16 16:26:32  euler
 * New operator "FeatureConstructionByRelation"
 *
 * Revision 1.7  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.6  2006/04/06 16:31:11  euler
 * Prepended license remark.
 *
 * Revision 1.5  2006/03/31 15:22:20  euler
 * *** empty log message ***
 *
 * Revision 1.4  2006/03/30 16:07:13  scholz
 * fixed author tags for release
 *
 * Revision 1.3  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.2  2006/01/06 16:28:50  euler
 * Bugfixes
 *
 * Revision 1.1  2006/01/03 09:54:22  hakenjos
 * Initial version!
 *
 */
