/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.ForeignKey;
import edu.udo.cs.miningmart.m4.Relation;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * @author Martin Scholz 
 */
public class CreateOneToManyRelation extends ExecutableOperator {
	
	/**
	 * Prerequisites:
	 * <ul>
	 * <li> Currently supports only a single Columnset per Concept.</li>
	 * <li> The From-Concept connected by a Relation should be materialized in advance.</li>
	 * <li> If no primary key exists in the database for the ToConcept the operator tries to
	 *      create one. This may fail for views.</li>  
	 * </ul>
	 * 
	 * @see ExecutableOperator#createStatement(boolean)
	 */
	public void createStatement(boolean lazy) throws M4CompilerError {
		final Step step = this.getStep();
		final Relation rel = this.getRelation();

		try {	
			// Identify target Columnsets:
			final Columnset fromCs = this.getFromConcept().getCurrentColumnSet();
			final Columnset toCs = this.getToConcept().getCurrentColumnSet();
			
			// Make sure the target columnset has a primary key:
			Collection keysTarget = this.getM4Db().getPrimaryKeysFromDbSchema(toCs);
			if (keysTarget == null || keysTarget.isEmpty()) {
				Collection keysB = CreateManyToManyRelation.baKeyCollectionToStringCollection(this.getPrimaryKeyToConcept());

				// Create target table primary key on demand:
				String constraintName =
					this.getM4Db().getBusinessDbCore().createPrimaryKeyConstraint(toCs.getName(), keysB, null);
				this.getM4Db().addPkConstraintToTrash(toCs.getName(), constraintName, toCs.getSchema(), this.getStep().getId());
			
			}		
			// Create foreign key constraint in database:
			this.createForeignKey(fromCs, this.getForeignKeyFromConcept(), toCs, this.getPrimaryKeyToConcept(), "_FK");
			
			// Create M4 foreign key:
			ForeignKey fk;
			{
				fk = (ForeignKey) this.getM4Db().createNewInstance(edu.udo.cs.miningmart.m4.core.ForeignKey.class);
				fk.setName("FK_" + step.getId());
				fk.setPrimaryKeyColumnset(toCs);
				fk.setForeignKeyColumnset(fromCs);
			
				CreateManyToManyRelation.fillForeignKey(fk,
						this.getForeignKeyFromConcept().iterator(), fromCs,
						this.getPrimaryKeyToConcept().iterator());

				step.addToTrash(fk);
			}
			
			// Register key at the conceptual level:
			rel.setFromKey(fk);
		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"Operator 'MaterializeRelation': M4Exception caught during createStatement:\n"
				+ e.getMessage());
		}
		catch (SQLException e) {
			throw new M4CompilerError(
				"Operator 'MaterializeRelation': SQLException caught during createStatement:\n"
				+ e.getMessage());
		}
	}
		
	private void createForeignKey(Columnset fromCs, Collection fromFks, Columnset toCs, Collection toPks, String fkNameSuffix)
		throws M4Exception, M4CompilerError
	{
		Collection fromKeyAttribs = CreateManyToManyRelation.baKeyCollectionToStringCollection(fromFks);
		Collection toKeyAttribs = CreateManyToManyRelation.baKeyCollectionToStringCollection(toPks);

		try {
			String tableName = fromCs.getName(); 
			String constraintName = fromCs.getName() + fkNameSuffix;
			
			this.getM4Db().getBusinessDbCore().createForeignKeyConstraint(
					fromCs.getName(), fromKeyAttribs,
					toCs.getName(), toKeyAttribs,
					constraintName);
			this.getM4Db().addFkConstraintToTrash(tableName, constraintName, fromCs.getSchema(), this.getStep().getId());
		}
		catch (SQLException e) {
			this.doPrint(Print.WARNING,
					"Operator CreateOneToManyRealtion failed to create foreign key:\n"
					+ e.getMessage());
			// throw new M4CompilerError(
			//		"Operator CreateOneToManyRealtion failed to create foreign key constraint between tables.\n"
			//		+ e.getMessage());
		}
	}
	
	// ------------------------------------------------------------
	
	/** Not required. Dummy method. */
	public void compileStatement() {}

	// ------------------------------------------------------------
	//               Getter for operator parameters:
	// ------------------------------------------------------------
	
	/**
	 * @return the Concept &quot;TheFromConcept&quot;
	 */
	protected Concept getFromConcept() throws M4CompilerError {
		return (Concept) this.getSingleParameter("TheFromConcept");		
	}

	/**
	 * @return the Concept &quot;TheToConcept&quot;
	 */
	protected Concept getToConcept() throws M4CompilerError {
		return (Concept) this.getSingleParameter("TheToConcept");		
	}

	/**
	 * @return the output Relation
	 */
	protected Relation getRelation() throws M4CompilerError {
		return (Relation) this.getSingleParameter("TheRelation");
	}

	
	/**
	 * @return <code>Collection</code> of Key Baseattributes of Concept B
	 */
	protected Collection getPrimaryKeyToConcept() throws M4CompilerError {
		BaseAttribute[] keys = (BaseAttribute[]) this.getParameter("ToConceptKeys");
		return Arrays.asList(keys);
	}
	
	/**
	 * @return <code>Collection</code> of Key Baseattributes part of the Cross Concept
	 * that point to Concept B 
	 */
	protected Collection getForeignKeyFromConcept() throws M4CompilerError {
		BaseAttribute[] keys = (BaseAttribute[]) this.getParameter("FromConceptKeys");
		return Arrays.asList(keys);
	}

}
