/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.ForeignKey;
import edu.udo.cs.miningmart.m4.Relation;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.Value;

/**
 * @author Martin Scholz 
 */
public class CreateManyToManyRelation extends ExecutableOperator {
	
	/**
	 * Prerequisites:
	 * <ul>
	 * <li> Currently supports only a single Columnset per Concept.</li>
	 * <li> Concepts that are connected by a Relation should be materialized in advance.</li>
	 * <li> If there are no primary keys for the From- and ToConcepts in the
	 * database the operator tries to create them. This may fail for views.</li>  
	 * </ul>
	 * 
	 * @see ExecutableOperator#createStatement(boolean)
	 */
	public void createStatement(boolean lazy) throws M4CompilerError {
		final Step step = this.getStep();
		final Concept crossConcept = this.getCrossConcept();
		final Relation rel = this.getRelation();

		try {
			final String nameCrossTable = this.getTableName();
			
			// Try to drop the table first if it exists:
			this.getM4Db().dropBusinessTable(nameCrossTable);

			// Materialize the table without foreign key constraints:
			final Columnset crossTableCs =
				this.materializeCrossTable(crossConcept, nameCrossTable);
		
			// Identify target Columnsets:
			final Columnset targetCsA = this.getFromConcept().getCurrentColumnSet();
			final Columnset targetCsB = this.getToConcept().getCurrentColumnSet();
			
			// Make sure the target views have primary keys:
			{
				Collection pksA = this.getM4Db().getPrimaryKeysFromDbSchema(targetCsA);
				if (pksA == null || pksA.isEmpty()) {
					Collection keysA = baKeyCollectionToStringCollection(this.getKeysConceptA());
					this.createTargetTablePrimaryKeys(targetCsA, keysA);
				}
			}{
				Collection pksB = this.getM4Db().getPrimaryKeysFromDbSchema(targetCsB);
				if (pksB == null || pksB.isEmpty()) {
					Collection keysB = baKeyCollectionToStringCollection(this.getKeysConceptB());
					this.createTargetTablePrimaryKeys(targetCsB, keysB);
				}
			}
			
			// Add foreign key constraints to cross table:
			this.createCrossTableForeignKeys(crossTableCs, this.getKeysOfCrossToConceptA(), targetCsA, this.getKeysConceptA(), "_FKA");
			this.createCrossTableForeignKeys(crossTableCs, this.getKeysOfCrossToConceptB(), targetCsB, this.getKeysConceptB(), "_FKB");
			
			// Create M4 Key objects:
			final ForeignKey fromFK, toFK;
			{
				fromFK = (ForeignKey) this.getM4Db().createNewInstance(edu.udo.cs.miningmart.m4.core.ForeignKey.class);
				fromFK.setName("FKFR_" + step.getId());
				fromFK.setPrimaryKeyColumnset(targetCsA);
				fromFK.setForeignKeyColumnset(crossTableCs);
			
				this.fillForeignKey(fromFK, this.getKeysOfCrossToConceptA().iterator(), crossTableCs,
										this.getKeysConceptA().iterator());
			} {		
				toFK = (ForeignKey) this.getM4Db().createNewInstance(edu.udo.cs.miningmart.m4.core.ForeignKey.class);
				toFK.setName("FKTO_" + this.getStep().getId());
				toFK.setPrimaryKeyColumnset(targetCsB);
				toFK.setForeignKeyColumnset(crossTableCs);

				this.fillForeignKey(toFK, this.getKeysOfCrossToConceptB().iterator(), crossTableCs,
									  this.getKeysConceptB().iterator());

				step.addToTrash(toFK);
				step.addToTrash(fromFK);
			}
			
			// Register new n:m relation at the conceptional level: 
			rel.setCrossLinkColumnSet(crossTableCs);
			rel.setM2mKeys(fromFK, toFK);

		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"Operator 'MaterializeRelation': M4Exception caught during createStatement:\n"
				+ e.getMessage());
		}
		catch (SQLException e) {
			throw new M4CompilerError(
				"Operator 'MaterializeRelation': SQLException caught during createStatement:\n"
				+ e.getMessage());
		}
	}
	
	private Columnset materializeCrossTable(Concept crossConcept, String nameCrossTable)
		throws M4Exception, M4CompilerError, SQLException
	{
		Columnset ccs = crossConcept.getCurrentColumnSet();
		Vector copyCols = new Vector(); // remember columns to copy to new columnset
		
		// ------------
		String sqlPrefix = "CREATE TABLE " + nameCrossTable + " AS ( SELECT ";
		String sqlSuffix = " FROM " + ccs.getSchemaPlusName() + " )";
		StringBuffer keySqlDefs = new StringBuffer();
		{
			Vector crossTableBaKeys = new Vector(this.getKeysOfCrossToConceptA());
			crossTableBaKeys.addAll(this.getKeysConceptB());
			Iterator it = crossTableBaKeys.iterator();
			while (it.hasNext()) {
				BaseAttribute ba = (BaseAttribute) it.next();
				Column col = ba.getCurrentColumn();
				copyCols.add(col);
				String sqlDef =	col.getSQLDefinition();
				if (sqlDef != null && sqlDef.trim().length() > 0 && ! col.getName().equals(sqlDef)) {
					keySqlDefs.append(sqlDef + " ");
				}
				keySqlDefs.append(col.getName() + (it.hasNext() ? ", " : ""));
			}
		}
		String sql = sqlPrefix + keySqlDefs.toString() + sqlSuffix;
		this.executeBusinessSqlWrite(sql);
		this.getM4Db().addTableToTrash(nameCrossTable, ccs.getSchema(), this.getStep().getId());
		// ------------
		
		// Create output columnset:
		Columnset outputCs = (Columnset) this.getM4Db().createNewInstance(edu.udo.cs.miningmart.m4.core.Columnset.class);
		outputCs.setType(Columnset.CS_TYPE_TABLE);
		outputCs.setName(nameCrossTable);
		outputCs.setSchema(ccs.getSchema());
		
		final Step step = this.getStep();
		step.addToTrash(outputCs);
		
		// Copy columns to new cross columnset, but
		// remove SQL definitions and base attributes:
		Iterator it = copyCols.iterator();
		while (it.hasNext()) {
			((Column) it.next()).copyColToCS(outputCs);;
		}
		it = outputCs.getColumns().iterator();
		while (it.hasNext()) {
			Column col = (Column) it.next();
			col.setSQLDefinition(null);
			col.setBaseAttribute(null);
			step.addToTrash(col);
		}

		return outputCs;
	}

	private void createTargetTablePrimaryKeys(Columnset cs, Collection pkAttributeNames)
		throws M4CompilerError, SQLException
	{
		String constraintName =
			this.getM4Db().getBusinessDbCore().createPrimaryKeyConstraint(cs.getName(), pkAttributeNames, null);
		this.getM4Db().addPkConstraintToTrash(cs.getName(), constraintName, cs.getSchema(), this.getStep().getId());
	}
	
	private void createCrossTableForeignKeys(Columnset crossCs, Collection crossKeys, Columnset targetCs, Collection conPk, String fkNameSuffix)
		throws M4Exception, M4CompilerError
	{
		Collection crossKeyAttribs = baKeyCollectionToStringCollection(crossKeys);
		Collection conKeyAttribs = baKeyCollectionToStringCollection(conPk);

		try {
			String tableName = crossCs.getName(); 
			String constraintName = crossCs.getName() + fkNameSuffix;
			
			this.getM4Db().getBusinessDbCore().createForeignKeyConstraint(
					crossCs.getName(), crossKeyAttribs,
					targetCs.getName(), conKeyAttribs,
					constraintName);
			this.getM4Db().addFkConstraintToTrash(tableName, constraintName, crossCs.getSchema(), this.getStep().getId());
		}
		catch (SQLException e) {
			throw new M4CompilerError(
					"Operator CreateManyToManyRealtion failed to create cross table foreign key constraint.\n"
					+ e.getMessage());
		}
	}

	static Vector baKeyCollectionToStringCollection(Collection baKeys) throws M4Exception {
		Iterator it = baKeys.iterator();
		Vector nameCollection = new Vector();
		while (it.hasNext()) {
			BaseAttribute keyBa = (BaseAttribute) it.next();
			Column keyCol = keyBa.getCurrentColumn();
			if (keyCol != null && keyCol.getName() != null && keyCol.getName().trim().length() > 0) {
				nameCollection.add(keyCol.getName());
			}
		}
		return nameCollection;
	}

	static void fillForeignKey(ForeignKey fk, Iterator fkKeysIt, Columnset crossTableCs, Iterator pkKeysIt)
		throws M4Exception
	{
		while (fkKeysIt.hasNext() && pkKeysIt.hasNext()) {
			BaseAttribute fkKey = (BaseAttribute) fkKeysIt.next(); 
			BaseAttribute pkKey = (BaseAttribute) pkKeysIt.next();
			Column fkColumn = fkKey.getCurrentColumn();
			Column pkColumn = pkKey.getCurrentColumn();
			if ( fkColumn != null && pkColumn != null) {
				Column crossTableColumn = crossTableCs.getColumn(fkColumn.getName());
				fk.addColumnLink(crossTableColumn, pkColumn);
			}
		}		
	}
	
	// ------------------------------------------------------------
	
	/** Not required. Dummy method. */
	public void compileStatement() {}

	// ------------------------------------------------------------
	//               Getter for operator parameters:
	// ------------------------------------------------------------
	
	/**
	 * @return the Concept &quot;TheFromConcept&quot;
	 */
	public Concept getFromConcept() throws M4CompilerError {
		return (Concept) this.getSingleParameter("TheFromConcept");		
	}

	/**
	 * @return the Concept &quot;TheToConcept&quot;
	 */
	public Concept getToConcept() throws M4CompilerError {
		return (Concept) this.getSingleParameter("TheToConcept");		
	}

	/**
	 * @return the Concept representing the cross table at the relational level
	 */
	public Concept getCrossConcept() throws M4CompilerError {
		return (Concept) this.getSingleParameter("TheCrossTable");		
	}

	/**
	 * @return the output Relation
	 */
	protected Relation getRelation() throws M4CompilerError {
		return (Relation) this.getSingleParameter("TheRelation");
	}

	/**
	 * @return the target table name for the materialized cross table.
	 */
	public String getTableName() throws M4CompilerError, M4Exception {
		Value value = (Value) this.getSingleParameter("NameForCrossTable");
		
		if (value != null && value.getValue().trim().length() > 0) {
			return value.getValue().trim();
		}
		else {
			Columnset cs =  this.getCrossConcept().getCurrentColumnSet();
			return (cs == null ? this.getCrossConcept().getName() : cs.getName()) + "_CT";
		}
	}

	/**
	 * @return <code>Collection</code> of Key Baseattributes of Concept A
	 */
	public Collection getKeysConceptA() throws M4CompilerError {
		BaseAttribute[] keys = (BaseAttribute[]) this.getParameter("FromConceptKeys");
		return Arrays.asList(keys);
	}
	
	/**
	 * @return <code>Collection</code> of Key Baseattributes of Concept B
	 */
	public Collection getKeysConceptB() throws M4CompilerError {
		BaseAttribute[] keys = (BaseAttribute[]) this.getParameter("ToConceptKeys");
		return Arrays.asList(keys);
	}

	/**
	 * @return <code>Collection</code> of Key Baseattributes part of the Cross Concept
	 * that point to Concept A 
	 */
	public Collection getKeysOfCrossToConceptA() throws M4CompilerError {
		BaseAttribute[] keys = (BaseAttribute[]) this.getParameter("CrossToFromConceptKeys");
		return Arrays.asList(keys);
	}
	
	/**
	 * @return <code>Collection</code> of Key Baseattributes part of the Cross Concept
	 * that point to Concept B 
	 */
	public Collection getKeysOfCrossToConceptB() throws M4CompilerError {
		BaseAttribute[] keys = (BaseAttribute[]) this.getParameter("CrossToToConceptKeys");
		return Arrays.asList(keys);
	}

}
