/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.operator;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.Vector;

import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.Feature;
import edu.udo.cs.miningmart.m4.RelationalDatatypes;
import edu.udo.cs.miningmart.m4.Value;

/**
 * An operator that uses a Java method programmed by the user to
 * create values of the OutputAttribute.
 * 
 * @author Timm Euler
 * @version $Id: AttributeDerivation.java,v 1.5 2006/09/27 14:59:56 euler Exp $
 */
public class AttributeDerivation extends SingleCSOperator {
	
	private String[] columnNames;
	private String[] columnTypes;
	private int[] columnWidths;
	public static final String PARAM_CLASSNAME = "ClassName";
	public static final String PARAM_OUTPUT_ATTR = "TheOutputAttribute";
	public static final String PARAM_TARGET_ATTR = "TheTargetAttributes";
	
	public static final String externalMethodName = "deriveAttribute";
	public static final Class[] externalMethodParameters = new Class[] { String[].class, String[].class, String[][].class };
	
	/**
	 * @see SingleCSOperator#getTypeOfNewColumnSet()
	 */
	public String getTypeOfNewColumnSet() {
		return Columnset.CS_TYPE_TABLE;
	}
	
	/**
	 * @see SingleCSOperator#generateSQLDefinition(String)
	 */
	public String generateSQLDefinition(String selectPart) throws M4CompilerError {
		
		Concept concept = this.getInputConcept();		
		Columnset cs;
		
		try {		
			if (concept == null || (cs = concept.getCurrentColumnSet()) == null) {
				throw new M4CompilerError("Operator 'AttributeDerivation': Found no input Concept/Columnset!");
			}

			int startLoop = 0;
    		int endLoop = this.getStep().getLoopCount();
    		if (endLoop > 0) startLoop = 1;
    		
    		// 1. access the input data:
    		String[][] inputData = this.getDataFromInputConcept(cs);    		
    		
    		// 2. get values of the added attributes for table:
    		Vector allNewValues = new Vector();
    		for (int loopnr = startLoop; loopnr <= endLoop; loopnr++) {			
    			BaseAttribute[] targetBAs = (BaseAttribute[]) this.getParameter(PARAM_TARGET_ATTR, loopnr);
    			String[] newValuesForThisLoop = this.callExternalMethod(targetBAs, loopnr, inputData);
    			allNewValues.add(newValuesForThisLoop);
    		}

    		// 3. add values to dataset:
    		String[][] resultData = this.addAttributesToDataSet(inputData, allNewValues);
    		
    		// 4. create table for output, connect it to output CS, and add it to trash:
    		String tablename = this.createOutputTable(cs);
    		
    		// 5. fill table:
    		this.fillTable(tablename, resultData);
    		
			return tablename;
		}
		catch (M4Exception e) {
			throw new M4CompilerError(
				"Operator 'AttributeDerivation': M4Exception caught during generateSQLDefinition:\n"
				+ e.getMessage());
		}
	}
	
	/**
	 * @see ConceptOperator#mustCopyFeature(String)
	 */
	protected boolean mustCopyFeature(String nameOfFeature) throws M4CompilerError {
		return true;
	}	

	// returns the name of the table it creates
	private String createOutputTable(Columnset inputCs) throws M4CompilerError {
		// get the name of the output table:
		String tablename = this.getNewCSName();
		
		// get the column names for the table, adding the column
		// names for the output attributes:
		int noOfLoops = (this.getStep().getLoopCount() > 0 ? this.getStep().getLoopCount() : 1);
		int noOfOriginalColumns = this.columnNames.length;
		String[] allColumnNames = new String[noOfOriginalColumns + noOfLoops];
		String[] allColumnTypes = new String[noOfOriginalColumns + noOfLoops];
		int[] allColumnWidths = new int[noOfOriginalColumns + noOfLoops];
		try {
			for (int i = 0; i < noOfOriginalColumns; i++) {
				allColumnNames[i] = this.columnNames[i];
				allColumnTypes[i] = inputCs.getColumn(this.columnNames[i]).getColumnDataTypeName();
				allColumnWidths[i] = this.columnWidths[i];
			}
			int startLoop = (noOfLoops > 1 ? 1 : 0);
			int endLoop = (noOfLoops > 1 ? noOfLoops : 0);
			int counter = 1;
			for (int i = startLoop; i <= endLoop; i++) {
				BaseAttribute outBa = (BaseAttribute) this.getSingleParameter(PARAM_OUTPUT_ATTR, i);
				BaseAttribute targetBa = (BaseAttribute) this.getSingleParameter(PARAM_TARGET_ATTR, i);
				allColumnNames[noOfOriginalColumns + counter - 1] = outBa.getName();
				allColumnTypes[noOfOriginalColumns + counter - 1] = outBa.getCurrentColumn().getColumnDataTypeName();
				allColumnWidths[noOfOriginalColumns + counter - 1] = 100; // default
				// if there is a target attribute, take its width rather than the default:
				if (targetBa != null) {
					int targetIndex = -1;
					for (int j = 0; j < this.columnNames.length; j++) {
						if (targetBa.getCurrentColumn().getName().equalsIgnoreCase(this.columnNames[j])) {
							targetIndex = j;
						}
					}
					if (targetIndex > -1) {
						allColumnWidths[noOfOriginalColumns + counter - 1] = allColumnWidths[targetIndex];
					}
				}
				counter++;
			}
			this.columnTypes = allColumnTypes;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("M4 error preparing creation of output table: " + m4e.getMessage());
		}
		
		// now create the table:
		String sql = "CREATE TABLE " + tablename + " (";
		for (int i = 0; i < allColumnTypes.length; i++) {
			String type = this.getM4Db().getNameOfVarcharDatatype(allColumnWidths[i]);
			if (allColumnTypes[i].equals(RelationalDatatypes.RELATIONAL_DATATYPE_NUMBER))
				type = this.getM4Db().getNameOfNumericDatatype();
			if (allColumnTypes[i].equals(RelationalDatatypes.RELATIONAL_DATATYPE_DATE))
				type = this.getM4Db().getNameOfVarcharDatatype(100); // DB-Type 'DATE' is more complex, can be done in the future
			sql += allColumnNames[i] + " " + type + ", ";
		}
		sql = sql.substring(0, sql.length() - 2) + ")";
		try {
			this.executeBusinessSqlWrite(sql);
			this.getM4Db().commitBusinessTransactions();
			this.getM4Db().addTableToTrash(tablename, inputCs.getSchema(), this.getStep().getId());
		}
		catch (SQLException s) {
			throw new M4CompilerError("SQL error creating output table: " + s.getMessage());
		}
		return this.getNewCSName();
	}
	
	private String[] callExternalMethod(BaseAttribute[] targetAttribs, int loop, String[][] dataset) 
	throws M4CompilerError {
		String pref = "Operator 'AttributeDerivation' in Step '" + this.getStep().getName() + "': ";
		try {
			String[] targetColNames = null;
			if (targetAttribs != null) { 
				targetColNames = new String[targetAttribs.length];
				for (int i = 0; i < targetColNames.length; i++) {
					Column targetCol = targetAttribs[i].getCurrentColumn();
					if (targetCol != null) 
						targetColNames[i] = targetCol.getName();
					else targetColNames[i] = null;
				}
			}
			String classname = this.getClassName(loop);
			Class theClassToUse = Class.forName(classname);
			if (theClassToUse == null) {
				throw new M4CompilerError(pref + "Class.forName() returned NULL for class with external method!");
			}
			Class[] constructorParameters = new Class[0];
			Constructor defaultConstr = theClassToUse.getConstructor(constructorParameters);
			Object[] constructorCallParams = new Object[0];
			Object theObjectOfTheClassToUse = defaultConstr.newInstance(constructorCallParams);
			if (theObjectOfTheClassToUse == null) {
				throw new M4CompilerError(pref + "could not construct an object of given class!");
			}
			if ( ! (theObjectOfTheClassToUse instanceof AttrDerivInterface)) {
				throw new M4CompilerError(pref + "the external class must implement the interface 'AttrDerivInterface'!");
			}
			Method theMethod = theClassToUse.getMethod(externalMethodName, externalMethodParameters);
			if (theMethod == null) {
				throw new M4CompilerError(pref + "got NULL for external method!");
			}
			Object[] arguments = new Object[] { this.columnNames, targetColNames, dataset }; 
			Object result = theMethod.invoke(theObjectOfTheClassToUse, arguments);
			if ( ! (result instanceof String[])) {
				throw new M4CompilerError(pref + "External method returned wrong result type, expected String[]!");
			}
			return (String[]) result;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError(pref + "M4 error preparing the call to the external method: " + m4e.getMessage());
		}
		catch (ClassNotFoundException cnfe) {
			throw new M4CompilerError(pref + "Class not found with external method: " + cnfe.getMessage());
		}
		catch (NoSuchMethodException nsme) {
			throw new M4CompilerError(pref + "Method not found in external class: " + nsme.getMessage());
		}
		catch (IllegalAccessException iae) {
			throw new M4CompilerError(pref + "Illegal access to method in external class: " + iae.getMessage());
		}
		catch (InvocationTargetException ite) {
			throw new M4CompilerError("Invocation target exception accessing method in external class: " + ite.getMessage());
		}
		catch (InstantiationException ine) {
			throw new M4CompilerError("Instantiation exception using default constructor in external class: " + ine.getMessage());
		}
	}
	
	private String[][] getDataFromInputConcept(Columnset inputCs) 
	throws M4CompilerError {
		if (inputCs == null) {
			throw new M4CompilerError("Operator 'AttributeDerivation': input columnset not found!");
		}
		try {
			long s = inputCs.getStatisticsAll();
			int size = 0;
			try {
				size = (int) s;
			}
			catch (ClassCastException c) {
				throw new M4CompilerError("Operator 'AttributeDerivation': input data set is too large!");
			}
			String query = inputCs.getCompleteSQLQuery();
			ResultSet rs = this.executeBusinessSqlRead(query);
			ResultSetMetaData rsmd = rs.getMetaData();
			int noOfCols = rsmd.getColumnCount();
			this.columnNames = new String[noOfCols];
			this.columnWidths = new int[noOfCols];
			for (int i = 1; i <= noOfCols; i++) {
				this.columnNames[i - 1] = rsmd.getColumnName(i);
				this.columnWidths[i - 1] = rsmd.getColumnDisplaySize(i);
			}
			String[][] data = new String[noOfCols][size];
			int row = 0;
			while (rs.next()) {
				for (int i = 1; i <= noOfCols; i++) {
					data[i - 1][row] = rs.getString(i);
				}
				row++;
			}
			rs.close();
			return data;
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'AttributeDerivation': M4 error reading input data: " + m4e.getMessage());
		}
		catch (SQLException sqle) {
			throw new M4CompilerError("Operator 'AttributeDerivation': SQL error reading input data: " + sqle.getMessage());
		}
	}

	private void fillTable(String tablename, String[][] dataToFillIn) throws M4CompilerError {
		if (dataToFillIn == null || dataToFillIn.length == 0) {
			throw new M4CompilerError("Operator 'AttributeDerivation': result data set is empty!");
		}
		String prefix = "INSERT INTO " + tablename + " VALUES (";
		try {
			for (int i = 0; i < dataToFillIn[0].length; i++) {
				String query = prefix;
				for (int j = 0; j < dataToFillIn.length; j++) {
					String dataItem = dataToFillIn[j][i];
					if (this.columnTypes[j].equals(RelationalDatatypes.RELATIONAL_DATATYPE_STRING) ||
						this.columnTypes[j].equals(RelationalDatatypes.RELATIONAL_DATATYPE_DATE)) {
						dataItem = "'" + dataItem + "'";
					}
					query += dataItem + ", ";
				}
				query = query.substring(0, query.length() - 2) + ")";
				this.executeBusinessSqlWrite(query);
			}
			this.getM4Db().commitBusinessTransactions();
		}
		catch (SQLException sqle) {
			throw new M4CompilerError("Operator 'AttributeDerivation': SQL error writing result data: " + sqle.getMessage());
		}
	}
	
    /**
     * Overwrites the super class method so that the newly constructed Feature
     * can be handled.
     */
    protected String handleExtraOutputFeature(Feature outF, Columnset csForOutputConcept) 
    throws M4CompilerError {

    	try {
    		int endLoop = this.getStep().getLoopCount();
    		if (endLoop == 0) endLoop = 1;
    		String sql = null;
    		for (int loopnr = 0; loopnr < endLoop; loopnr++) {				
    			BaseAttribute outBA = (BaseAttribute) this.getSingleParameter(PARAM_OUTPUT_ATTR, loopnr);
    			if (outF instanceof BaseAttribute && ((BaseAttribute) outF).equals(outBA)) {    				
    				sql = outBA.getName();
    				Column newCol = (Column) this.getM4Db().createNewInstance(Column.class);
    				newCol.setName(outBA.getName());
    				newCol.setSQLDefinition(sql);
    				newCol.setBaseAttribute(outBA);
    				newCol.setColumnDataTypeName(this.getM4Db().getM4RelTypeForConceptualType(outBA.getConceptualDataTypeName()));
    				newCol.setColumnset(csForOutputConcept);
    			}
    		}
    		if (sql == null)
    			return "";
    		else 
    			return sql + ", ";
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("Operator 'AttributeDerivation': M4 error: " + m4e.getMessage());
		}
    }
    
	private String[][] addAttributesToDataSet(String[][] dataset, Vector arraysWithNewValues) 
	throws M4CompilerError {
		if (dataset == null)
			return null;
		if (arraysWithNewValues == null)
			return dataset;
		
		int noOfCols = dataset.length + arraysWithNewValues.size();
		int noOfRows = dataset[0].length;
		String[][] ret = new String[noOfCols][noOfRows];
		for (int i = 0; i < noOfCols; i++) {
			String[] newValues = null;
			if (i >= dataset.length) {
				newValues = (String[]) arraysWithNewValues.get(i - dataset.length);
				if (dataset[0].length != newValues.length) {
					throw new M4CompilerError("Operator 'AttributeDerivation': too few or too many values created!");
				}
			}
			for (int k = 0; k < noOfRows; k++) {
				if (i < dataset.length) {
					ret[i][k] = dataset[i][k];
				}
				else {
					ret[i][k] = newValues[k];
				}
			}
		}
		return ret;
	}
	
	private String getClassName(int loopNr) throws M4CompilerError {
		Value v = (Value) this.getSingleParameter(PARAM_CLASSNAME, loopNr);
		if (v == null) {
			throw new M4CompilerError("Could not find parameter '" +
					PARAM_CLASSNAME + "' in loop number " + loopNr + "!");
		}
		return v.getValue();
	}
}
/*
 * Historie
 * --------
 *
 *
 */
	
