/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.utils;

import java.sql.ResultSet;
import java.sql.SQLException;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.core.M4Data;

/**
 * @author Martin Scholz
 * @version $Id: M4InfoEntry.java,v 1.5 2006/09/27 14:59:55 euler Exp $
 */
public class M4InfoEntry {

	private final String dbAttribute;	
	private final String getter;
	private final String setter;
	private final Class theObjectClass;

	private boolean notNull    = false;
	
	/**
	 * @param dbAttribute the name of the attribute in the database
	 * @param getter the name of the getter for the attribute
	 * @param setter the name of the setter for the attribute
	 * @param objectClass the class of the object returned by the getter and
	 *        expected by the setter
	 */
	public M4InfoEntry(String dbAttribute, String getter, String setter, Class objectClass)
	{
		this.dbAttribute = dbAttribute;
		this.getter = getter;
		this.setter = setter;
		this.theObjectClass = objectClass;
	}

	/**
	 * @param dbAttribute the name of the attribute in the database
	 * @param getter the name of the getter for the attribute
	 * @param setter the name of the setter for the attribute
	 * @param objectClass the class of the object returned by the getter and
	 *        expected by the setter
	 * @param constraint a <code>String</code> constant defined in the
	 *        <code>M4Table</code> interface specifying the constraints
	 *        of the attribute
	 */
	public M4InfoEntry(String dbAttribute, String getter, String setter, Class objectClass, String constraint)
	{
		this(dbAttribute, getter, setter, objectClass);
		if (constraint == null)
			return;
		
		else if (M4Table.NOT_NULL.equals(constraint)) {
			this.notNull = true;
		}
	}

	/**
	 * Gets the dbAttribute.
	 * @return Returns a String
	 */
	public String getDbAttribute() {
		return dbAttribute;
	}

	/**
	 * Gets the getter.
	 * @return Returns a String
	 */
	public String getGetter() {
		return getter;
	}

	/**
	 * Gets the setter.
	 * @return Returns a String
	 */
	public String getSetter() {
		return setter;
	}

	/**
	 * Gets the theObjectClass.
	 * @return Returns a Class
	 */
	public Class getTheObjectClass() {
		return theObjectClass;
	}

	public void activateSetter(M4Data m4d, ResultSet rs, DB db)
		throws SQLException, M4Exception
	{
		final String attributeName = this.getDbAttribute();
		final Class type = this.getTheObjectClass();
		final Object object;

		if (! edu.udo.cs.miningmart.m4.M4Object.class.isAssignableFrom(type)) {
			if (type.equals(long.class)) {
				object = new Long(rs.getLong(attributeName));
			}
			else if (type.equals(int.class)) {
				object = new Integer(rs.getInt(attributeName));
			}
			else if (type.equals(short.class)) {
				object = new Short(rs.getShort(attributeName));
			}
			else if (type.equals(double.class)) {
				object = new Double(rs.getDouble(attributeName));
			}
			else if (type.equals(String.class)) {
				object = rs.getString(attributeName);
			}			
			else if (type.equals(boolean.class)) {
				String s = (rs.getString(attributeName)).trim().toUpperCase();
				boolean val = (s.startsWith("Y") || s.equals("IN") || s.equals("TRUE"));
				object = new Boolean(val);
			}
			else if (type.equals(Long.class)) {
				long num = rs.getLong(attributeName);
				object = (rs.wasNull() ? null : new Long(num));
			}
			else if (type.equals(Integer.class)) {
				int num = rs.getInt(attributeName);
				object = (rs.wasNull() ? null : new Integer(num));
			}
			else if (type.equals(Double.class)) {
				double num = rs.getDouble(attributeName);
				object = (rs.wasNull() ? null : new Double(num));
			}			
			else {
				throw new M4Exception(
					"Found unsupported Class in M4InfoEntry.activateSetter(M4Table, ResultSet, DB): "
					+ m4d.getClass().getName());
			}
		}
		else {
			long id = rs.getLong(attributeName);
			object = db.getM4Object(id, type);
		}
		m4d.genericSetter(this.getSetter(), type, object);					
	}

	/**
	 * Gets the <code>NOT NULL</code> property
	 * @return Returns <code>true</code> iff this attribute
	 * is not allowed to be <code>null</code> in the database
	 */
	public boolean isNotNull() {
		return notNull;
	}

}
/*
 * Historie
 * --------
 * 
 * $Log: M4InfoEntry.java,v $
 * Revision 1.5  2006/09/27 14:59:55  euler
 * New version 1.1
 *
 * Revision 1.4  2006/04/11 14:10:10  euler
 * Updated license text.
 *
 * Revision 1.3  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.2  2006/03/30 16:07:16  scholz
 * fixed author tags for release
 *
 * Revision 1.1  2006/01/03 09:54:04  hakenjos
 * Initial version!
 *
 */
