/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.utils;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.core.M4Object;

/**
 * This abstract class embeds all the necessary communication between two
 * <code>M4Objects</code> A and B, where A and B have each a single reference 
 * to each other. A is referred to as the &quot;first object&quot; and B as 
 * the &quot;second object&quot;.
 * 
 * All that is necessary to enable the communication is to create a subclass
 * S implementing the abstract methods by calling the original methods of the
 * classes.
 * 
 * Then the offered methods of an &quot;S object&quot; can be called from the
 * corresponding method of the communicating objects.
 * 
 * @author Timm Euler
 * @version $Id: InterM4ObjectToObject.java,v 1.4 2006/09/27 14:59:55 euler Exp $
 */
public abstract class InterM4ObjectToObject
{
	/**
	 * This method should call the getter in the first object which
	 * returns the reference to the second object.
	 * 
	 * @param first the first object
	 * @return a reference to the second object
	 * */		
	abstract M4Object getFirstSingleRef(M4Object first) throws M4Exception;
	
	/**
	 * This method should call the getter in the second object which
	 * returns the reference to the first object.
	 * 
	 * @param second the second object
	 * @return a reference to the first object
	 * */		
	abstract M4Object getSecondSingleRef(M4Object second) throws M4Exception;

	/**
	 * This method should call the primitive setter for the first object,
	 * setting the reference to the second object.
	 *
	 * @param first the first object
	 * @param second the second object
	 */		
	abstract void setFirstSingleRefPrimitive(M4Object first, M4Object second) throws M4Exception;

	/**
	 * This method should call the primitive setter for the second object,
	 * setting the reference to the first object.
	 *
	 * @param second the second object
	 * @param first the first object
	 */		
	abstract void setSecondSingleRefPrimitive(M4Object second, M4Object first) throws M4Exception;
	
	// ----------------------------------------------------------------

	/**
	 * 
	 * This method makes sure that the given M4Objects do NOT refer
	 * to each other. The order of objects (first and second) is important
	 * and must be consistent with the use of the other methods in this class!
	 * 
 	 * @param first the first object
	 * @param second the second object
	 */
	public void removeReferences(M4Object first, M4Object second) throws M4Exception 
	{
		
    	// if one of first or second is null, that's ok, but not both:
    	if (first == null && second == null)
    	{   return;   }
    	
    	// step 1: remove reference from first to second:
    	if (first != null)
    	{ 
    		this.setFirstSingleRefPrimitive(first, null);
    	}
    	
    	// step 2: remove reference from second to first:
    	if (second != null)
    	{
    		this.setSecondSingleRefPrimitive(second, null);
    	}
	}

	/**
	 * This method makes sure that each of the given M4Objects refers
	 * to the other one. The order of objects (first and second) is important
	 * and must be consistent with the use of the other methods in this class!
	 * 
 	 * @param first the first object
	 * @param second the second object
	 */
    public void setReciprocalReferences(M4Object first, M4Object second)
		throws M4Exception
    {
    	// if one of first or second is null, that's ok, but not both:
    	if (first == null && second == null)
    	{   return;   }
    	
    	// step 1: set reference from first to second:
    	if (first != null)
    	{ 
    		M4Object oldReference = this.getFirstSingleRef(first);
    		
    		// nothing to do if nothing changes:
    		if (oldReference == second)
    		{   return;   }
    		
    		if (oldReference != null)
			{   this.setSecondSingleRefPrimitive(oldReference, null);   }
			
			this.setFirstSingleRefPrimitive(first, second);
    	}
    	
    	// step 2: set reference from second to first:
    	if (second != null)
    	{
    		M4Object oldReference = this.getSecondSingleRef(second);
    		
    		// nothing to do if nothing changes:
    		if (oldReference == first)
    		{   return;   }
    		
    		if (oldReference != null)
			{   this.setFirstSingleRefPrimitive(oldReference, null);   }
			
			this.setSecondSingleRefPrimitive(second, first);
    	}
    }
}
/*
 * Historie
 * --------
 * 
 * $Log: InterM4ObjectToObject.java,v $
 * Revision 1.4  2006/09/27 14:59:55  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:10  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:10  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:04  hakenjos
 * Initial version!
 *
 */
