/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.core;

import java.io.IOException;
import java.io.Writer;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import javax.swing.JOptionPane;

import edu.udo.cs.miningmart.compiler.CompilerAccessLogic;
import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.DbConnectionClosed;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.exception.ParameterError;
import edu.udo.cs.miningmart.exception.ParameterNotFoundException;
import edu.udo.cs.miningmart.exception.UserError;
import edu.udo.cs.miningmart.exception.XmlException;
import edu.udo.cs.miningmart.m4.utils.HasCrossReferences;
import edu.udo.cs.miningmart.m4.utils.InterM4Communicator;
import edu.udo.cs.miningmart.m4.utils.InterM4StepParameter;
import edu.udo.cs.miningmart.m4.utils.M4Info;
import edu.udo.cs.miningmart.m4.utils.M4InfoEntry;
import edu.udo.cs.miningmart.m4.utils.M4Xml;
import edu.udo.cs.miningmart.m4.utils.Print;
import edu.udo.cs.miningmart.m4.utils.XmlInfo;
import edu.udo.cs.miningmart.operator.ConceptOperator;
import edu.udo.cs.miningmart.operator.ExecutableOperator;
import edu.udo.cs.miningmart.operator.Pivotize;
import edu.udo.cs.miningmart.operator.SpecifiedStatistics;

/**
 * This class represents an M4 Step.
 * 
 * Apart from the tuple-based load mechanism there is a second one,
 * loading and aggregating tuples as specified by the entries in the
 * <code>OP_PARAM_T</code> table.
 * 
 * The first mechanism uses the methods <code>getParameters()</code>,
 * <code>addParameterTuple(Parameter)</code>, and
 * <code>removeParameterTuple(Parameter)</code>.
 * 
 * The latter uses the methods <code>getParameterDictionary()</code>,
 * <code>getParameterFromDict(String, int)</code>,
 * <code>getSingleParameter(String)</code>,
 * <code>getSingleParameter(String, int)</code>,
 * <code>getTheParameter(String)</code>, and
 * <code>getTheParameter(String, int)</code>.
 * Adding or removing parameters is not possible with the second mechanism,
 * for it is a support for compiler access to aggregated parameter arrays
 * only.
 * 
 * @author Timm Euler
 * @version $Id: Step.java,v 1.71 2006/10/01 19:14:22 euler Exp $
 */
public class Step 
     extends GraphicalM4Object 
	 implements edu.udo.cs.miningmart.m4.Step, XmlInfo, HasCrossReferences {
	
	/** name of the corresponding m4 table */
	public static final String M4_TABLE_NAME = "step_t";

	/** db level id of this object is realized by this attribute */
	public static final String ATTRIB_STEP_ID = "st_id";

	/** db level name of this object is realized by this attribute */
	public static final String ATTRIB_STEP_NAME = "st_name";
	
	/** db level reference to the embedding chain by this attribute */
	public static final String ATTRIB_CHAIN_ID = "st_chid";

	/** db level reference to the embedding case by this attribute */
	public static final String ATTRIB_CASE_ID = "st_caid";
	
	/** db level reference to the step's operator */
	public static final String ATTRIB_OPERATOR_ID = "st_opid";

	/** db level: number of loops attribute */
	public static final String ATTRIB_NUM_OF_LOOPS = "st_loopnr";
	
	/** db level: step number attribute */
	public static final String ATTRIB_STEP_NR = "st_nr";
		
	/** db level: operator name attribute */
	public static final String ATTRIB_MULTISTEP_COND = "st_multistepcond";

	/** Helper classes instance for inter M4 communication */
	public static InterM4Communicator step2par = new InterM4StepParameter();


	/** Cache for getM4Info() */
	public static M4Info m4Info = null;

	/** @see M4Table.getM4TableName() */
	public String getM4TableName() {
		return M4_TABLE_NAME;	
	}

	/** @see M4Table.getIdAttributeName() */
	public String getIdAttributeName() {
		return ATTRIB_STEP_ID;
	}

	/** @see M4Table.getM4Info() */
	public M4Info getM4Info() {
		if (m4Info == null) {
			M4InfoEntry[] m4i = {
			    // The NOT_NULL for the foreign keys to Case and Operator are not in the DB, but they are useful
			    // to enforce integrity! As a result storing Steps without a Case or Operator means deleting them
			    // from the DB and Cache!

				new M4InfoEntry(ATTRIB_STEP_ID,        "getId",                 "setId",                 long.class,   NOT_NULL),
				new M4InfoEntry(ATTRIB_STEP_NAME,      "getName",               "setName",               String.class, NOT_NULL),
				new M4InfoEntry(ATTRIB_STEP_NR,        "getNumber",             "setNumber",             long.class),
				new M4InfoEntry(ATTRIB_NUM_OF_LOOPS,   "getLoopCount",          "setLoopCount",          int.class),
				new M4InfoEntry(ATTRIB_MULTISTEP_COND, "getMultiStepCondition", "setMultiStepCondition", String.class),
				new M4InfoEntry(ATTRIB_CASE_ID,        "getTheCase",            "primitiveSetCase",      edu.udo.cs.miningmart.m4.Case.class,   NOT_NULL),
				new M4InfoEntry(ATTRIB_CHAIN_ID,       "getTheChain",           "primitiveSetChain",     edu.udo.cs.miningmart.m4.Chain.class),
				new M4InfoEntry(ATTRIB_OPERATOR_ID,    "getTheOperator",        "setTheOperator",        edu.udo.cs.miningmart.m4.Operator.class, NOT_NULL)
			};
			m4Info = new M4Info(m4i);
		}
		return m4Info;
	}

	/** Cache for getXmlInfo() */
	private static M4Info xmlInfo = null;

	/** @see XmlInfo.getXmlInfo() */
	public M4Info getXmlInfo() {
		if (xmlInfo == null) {
			M4InfoEntry[] m4i = {				
				new M4InfoEntry("Name",               "getName",               "setName",               String.class),
				new M4InfoEntry("Number",             "getNumber",             "setNumber",             long.class),
				new M4InfoEntry("LoopCount",          "getLoopCount",          "setLoopCount",          int.class),
				new M4InfoEntry("MultiStepCondition", "getMultiStepCondition", "setMultiStepCondition", String.class),
				new M4InfoEntry("Case",               "getTheCase",            "setTheCase",            edu.udo.cs.miningmart.m4.Case.class),
				new M4InfoEntry("Chain",              "getTheChain",           "setTheChain",           edu.udo.cs.miningmart.m4.Chain.class),
				new M4InfoEntry("Operator",           "getTheOperator",        "setTheOperator",        edu.udo.cs.miningmart.m4.Operator.class),
				new M4InfoEntry("Successors",         "getSuccessors",         "setSuccessors",         Collection.class),
				new M4InfoEntry("Docu",               "getDocumentation",      "setDocumentation",      String.class)
			};
			xmlInfo = new M4Info(m4i);
		}
		return xmlInfo;
	}


	// ***** Database constants for the table STEPSEQUENCE_T *****

	/** The name of the corresponding M4 table. */
	public static final String M4_TABLE_NAME_STEPSEQ = "stepsequence_t";

	/** db level: name of the attribute holding the step dependency's id */
	public static final String ATTRIB_STEPSEQ_DEPENDENCY_ID = "sts_id";
	
	/** db level: name of the attribute holding the from step's id */
	public static final String ATTRIB_STEPSEQ_FROM_STEP = "sts_stid";

	/** db level: name of the attribute holding the dependent step's id */
	public static final String ATTRIB_STEPSEQ_TO_STEP = "sts_successorstid";

	/* The names of the database table for database trash objects and its columns */
	public static final String DB_TRASH_TABLENAME = "dbtrash_t";
	public static final String ATTRIB_DBT_OBJTYPE = "ObjType";
	public static final String ATTRIB_DBT_OBJNAME = "ObjName";
	public static final String ATTRIB_DBT_SCHEMA  = "SchemaName";
	public static final String ATTRIB_DBT_STEPID  = "StepId";
	
	/* The constants used in that table for the object types of database objects */
	public static final String DBT_TYPE_FUNCTION = "F";
	public static final String DBT_TYPE_VIEW     = "V";
	public static final String DBT_TYPE_TABLE    = "T";
	public static final String DBT_TYPE_INDEX    = "I";
	public static final String DBT_TYPE_PRIMARY_KEY = "PK";
	public static final String DBT_TYPE_FOREIGN_KEY = "FK";

	/* The names of the database table for links to reversing steps, and its columns */
	public static final String REVSTEP_TABLENAME   = "revstep_t";
	public static final String ATTRIB_REV_ID       = "rev_id";
	public static final String ATTRIB_REV_ORIGSTEP = "rev_orgstid";
	public static final String ATTRIB_REV_REVSTEP  = "rev_revstid";
	
	/* Tag for exporting link to reversing steps */
	private static final String EXPORT_TAG_REVERSING_STEPS = "ReversedByStep";
	
    // ***** Step Variables from M4-DB *****
    private Case     myCase;
    private Chain    myChain;
    private long     myNr;
    private int      loopCount;
    private String   multiStepCondition;
    private Operator myOperator;
    
    // this step's objects listed in the M4 trash table:
    private final HashSet myM4Trash = new HashSet();
    private boolean allM4TrashLoaded = false;
	
	// the parameters associated to this step in tuple-wise form:
	private final Collection myParameterTuples = new Vector();
	private boolean allParameterTuplesLoaded = false;

	// This information is stored in a separate table:
	private final Vector dependentSteps = new Vector();
	private boolean allDependentStepsLoaded = false;

	private boolean dependentStepsDirty = false;

	// DB Trash related fields:
	private final Vector myDbTrash = new Vector();
	private boolean dbTrashDirty = false;
	private boolean allDbTrashLoaded = false;
	
	// Used for creation of output features : 
	private static char nameSeparator = '_';
	
	private static String internalNameSeparator = "##";
	
	// Used for storing reversing steps
	private Vector myReversingSteps = null;
	private Step theStepThisStepReverses = null;
	
	// Used for adapting a replaced input concept:
	private Concept oldInputConcept = null;
	private Concept replacingConcept = null;
	
	// Used for storing existing output attribs during update of output:
	Map oldNamesOfOutputAttribs = null;
	
	// Used for adapting input parameters of following steps
	// that are affected by changes to the output of this step:
	private Map mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames = null;
	
	// Used for breadth-first searches through the steps:
	private int bfsLevel = 0;
	
	/** @see edu.udo.cs.miningmart.m4.core.M4Data#Constructor */
    public Step(DB m4Db) {
		super(m4Db);
    }
	
	/**
	 * @see edu.udo.cs.miningmart.m4.core.M4Object#print
	 */
    public void print(){
    	Operator op = null;
		op = (Operator) this.getTheOperator();
    	String opName = (op == null ? "<null>" : op.getName());
		this.doPrint(Print.M4_OBJECT, "Step (Id = " + myId + ";" +
		      "      Operator = "   + opName + ";" +
		      "      Loop Count = " + this.getLoopCount() +";" +
		      "      Multi Step = " + this.getMultiStepCondition() + ")");
		      
		super.print();  // prints graphical info
    }
	
	/**
	 * @see edu.udo.cs.miningmart.m4.core.M4Data#getObjectsInNamespace(Class)
	 */
	protected Collection getObjectsInNamespace(Class typeOfObjects) throws M4Exception {
		if (typeOfObjects.isAssignableFrom(Parameter.class)) {
			return this.getParameterTuples();
		}
		else throw new M4Exception("Step.getObjectsInNamespace: unknown type of objects given: " + typeOfObjects.getName());
	}

	/**
	 * Active getter method.
	 * @return the M4 operator object related to this <code>Step</code>
	 */
    public edu.udo.cs.miningmart.m4.Operator getTheOperator() {
   		return this.myOperator; 
    }

	/**
	 * Setter method.
	 * There is no back-reference from <code>Operator</code>, so we do
	 * not need another primitive setter.
	 * 
	 * @param theOp The new operator
	 */
    public void setTheOperator(edu.udo.cs.miningmart.m4.Operator theOp)
    {   
	    this.setDirty();
    	this.myOperator = (Operator) theOp;
    }

	/** Getter method.
	 * 
	 * @return the <code>Chain</code> of this <code>Step</code>
	 */
	public edu.udo.cs.miningmart.m4.Chain getTheChain() {
		return this.myChain;	
	}

	/**
	 * Setter method.
	 * 
	 * @param chain The new chain
	 */
	public void setTheChain(edu.udo.cs.miningmart.m4.Chain chain) throws M4Exception {
		Chain.chain2step.checkNameExists(this, chain);
		Chain.chain2step.updateReferenceTo(this, chain);
	}

	/** Primitive setter method. Do not use it!
	 * 
	 * @param the <code>Chain</code> to be set.
	 */
	public void primitiveSetChain(edu.udo.cs.miningmart.m4.Chain chain) {
		this.setDirty();
		this.myChain = (Chain) chain;
	}

	/**
	 * Getter method.
	 * 
	 * @return the id
	 */
    public long getCaseId()
    {   return this.myCase.getId();  }

	/**
	 * Getter method.
	 * 
	 * @return the number
	 */
    public long getNumber()
    {   return this.myNr;  }

	/**
	 * Setter method.
	 * 
	 * @param myCase The new case
	 */
    public void setTheCase(edu.udo.cs.miningmart.m4.Case myCase) throws M4Exception {
    	Case.case2step.checkNameExists(this, myCase);
   		Case.case2step.updateReferenceTo(this, myCase);
    }

	/**
	 * Primitive setter method. Do not uise it!
	 * 
	 * @param myCase The <code>Case</code> to be set.
	 */
    public void primitiveSetCase(edu.udo.cs.miningmart.m4.Case myCase) {
    	this.setDirty();
   		this.myCase = (Case) myCase;  
    }

	/**
	 * Setter method.
	 * 
	 * @param nr the new step number
	 */
    public void setNumber(long nr)
    {   
    	this.setDirty();
    	this.myNr = nr;
    }

	/**
	 * Getter method.
	 * 
	 * @return the loop count
	 */
    public int getLoopCount()
    {   return this.loopCount;   }

	/**
	 * Getter method.
	 * 
	 * @return the multistep condition
	 */
    public String getMultiStepCondition()
    {   return this.multiStepCondition;   }

	/**
	 * Adds a parameter to this step's parameter list on tuple level.
	 *
	 * @param par the <code>Parameter</code> object representing the parameter tuple
	 */
	public void addParameterTuple(edu.udo.cs.miningmart.m4.Parameter par) throws M4Exception {
		step2par.checkNameExists((Parameter) par, this);
		step2par.add(this, (Parameter) par);
	}

	/**
	 * Removes a parameter from this step's parameter list on tuple level.
	 *
	 * @param par the <code>Parameter</code> object representing the parameter tuple
	 * @return <code>true</code> iff the object was part of the parameter list and
	 *         could be removed
	 */
	public boolean removeParameterTuple(edu.udo.cs.miningmart.m4.Parameter par) throws M4Exception {
		return step2par.remove(this, (Parameter) par);
	}

	public void removeAllParameterTuples() throws M4Exception {
		step2par.setCollectionTo(this, null);
	}
	
	public void removeParameter(String parName) throws M4Exception {
		
		Collection theParamTuples = this.getParameterTuples();
		Vector theCopy = new Vector(theParamTuples);
		Iterator tupleIt = theCopy.iterator();
		while (tupleIt.hasNext()) {
			Parameter par = (Parameter) tupleIt.next();
			if (par != null && par.getName().startsWith(parName)) {
				this.removeParameterTuple(par);
				par.deleteSoon();
			}
		}
	}
	
	/**
	 * Setter method.
	 * 
	 * @param lc the new loop count
	 */
    public void setLoopCount(int lc) throws M4Exception {   
    	if (this.loopCount > lc) {
    		// delete the parameters that are superfluous now:
    		Iterator it = this.getTheOperator().getOpParamsIterator();
    		while (it.hasNext()) {
				OpParam myOpPar = (OpParam) it.next();
				if (myOpPar.isLoopable()) {
					for (int loop = lc + 1; loop <= this.loopCount; loop++) {
						this.setParameter(myOpPar, new Vector(), loop);
					}
				}
    		}
    	}
    	this.setDirty();
    	this.loopCount = lc;   
    }

	/**
	 * Setter method.
	 * 
	 * @param msc The new multistep condition
	 */
    public void setMultiStepCondition(String msc)
    {   
    	this.setDirty();
    	this.multiStepCondition = msc;
    }

	/**
	 * Getter method.
	 * 
	 * @return this step's case
	 */
	public edu.udo.cs.miningmart.m4.Case getTheCase() {
		return this.myCase;
	}

	/**
	 * Returns the step whose feature construction is reversed
	 * by this step, or null if no such step exists.
	 * 
	 * @return
	 */
	public edu.udo.cs.miningmart.m4.Step getReversedStep() throws M4Exception {
		if (this.theStepThisStepReverses == null) {
			Case myCase = (Case) this.getTheCase();
			if (myCase != null) {
				Iterator stepIt = myCase.getStepIterator();
				while (stepIt.hasNext()) {
					Step aStep = (Step) stepIt.next();
					aStep.getDependentReversingSteps();
				}
			}
		}
		return this.theStepThisStepReverses;
	}

	/**
	 * Active getter for the parameters of this <code>Step</code>
	 * <b>in tuple-wise form</b> (not aggregated by <code>OpParam</code>
	 * arrays!).
	 * 
	 * @return Returns a Collection of <code>Parameter</code> objects
	 */
	public Collection getParameterTuples() throws M4Exception {
		if ( ! this.allParameterTuplesLoaded && ( ! this.isNew())) {
			this.allParameterTuplesLoaded = true;	
			this.readParameterTuplesFromDB();
		}
		return this.myParameterTuples;
	}

	/*
	 * Returns the parameter tuples for the given OperatorParameter.
	 */
	private Collection getParameterTuples(edu.udo.cs.miningmart.m4.OpParam theOpParam) throws M4Exception {
		Collection all = this.getParameterTuples();
		Vector ret = new Vector();
		Iterator it = all.iterator();
		while (it.hasNext()) {
			Parameter myPar = (Parameter) it.next();
			if (myPar.getName().startsWith(theOpParam.getName())) {
				ret.add(myPar);
			}
		}
		return ret;
	}

	/*
	 * Returns the parameter tuples for the given OperatorParameter and loop number.
	 */
	private Collection getParameterTuples(edu.udo.cs.miningmart.m4.OpParam theOpParam, int loopNr) throws M4Exception {
		Collection all = this.getParameterTuples();
		Vector ret = new Vector();
		Iterator it = all.iterator();
		while (it.hasNext()) {
			Parameter myPar = (Parameter) it.next();
			if (myPar.getName().startsWith(theOpParam.getName()) && myPar.getLoopNr() == loopNr) {
				ret.add(myPar);
			}
		}
		return ret;
	}
	
	/**
	 * Active getter for this <code>Step</code>'s successors.
	 * 
	 * @return <code>Collection</code> of dependent <code>Step</code> objects
	 */
	public Collection getSuccessors() throws M4Exception {
		if (this.allDependentStepsLoaded == false && this.isNew() == false)	{
			this.allDependentStepsLoaded = true;
			this.readSuccessorsFromDb();
		}
		return this.dependentSteps;
	}

	/**
	 * This method should only be called from the method
	 * <code>Case.addStepDependency(Step, Step)</code>!
	 * Please use that method instead of this one, because
	 * it also efficiently reorders the steps!
	 * 
	 * Adds a <code>Step</code> to this <code>Step</code>'s <code>Collection</code>
	 * of dependent <code>Step</code>s.
	 * 
	 * @param step the <code>Step</code> to add
	 */	
	public void addSuccessor(edu.udo.cs.miningmart.m4.Step step) throws M4Exception {
		if (step != null) {
			this.dependentStepsDirty = true;
			this.setDirty();
			this.primitiveAddSuccessor(step);
		}	
	}

	/**
	 * Same as <code>addSuccessor(Step) but without setting any of
	 * the dirty flags. This method should only be used while loading
	 * or if you know what you are doing!
	 */	
	public void primitiveAddSuccessor(edu.udo.cs.miningmart.m4.Step step) throws M4Exception {
		if (step != null) {
			this.getSuccessors().add(step);
		}			
	}

	/**
	 * Removes a <code>Step</code> from this <code>Step</code>'s <code>Collection</code>
	 * of dependent <code>Step</code>s.
	 * 
	 * @param step the <code>Step</code> to remove
	 * @return <code>true</code> iff the <code>Step</code> was found in the
	 *         <code>Collection</code> and could be successfully removed
	 */	
	public boolean removeSuccessor(edu.udo.cs.miningmart.m4.Step step) throws M4Exception {
		if (step != null) {
			boolean success = this.getSuccessors().remove(step);
			if (success) {
				this.dependentStepsDirty = true;
				this.setDirty();
			}
			return success;
		}
		else return false;
	}

	/**
	 * Removes a <code>Step</code> from this <code>Step</code>'s <code>Collection</code>
	 * of dependent <code>Step</code>s.
	 * 
	 * @param name the name of the step to remove
	 * @return <code>true</code> iff the <code>Step</code> was found in the
	 *         <code>Collection</code> and could be successfully removed
	 */	
	public boolean removeSuccessor(String name) throws M4Exception {
		if (name != null) {
			Step toRemove = null;
			Iterator it = this.getSuccessors().iterator();
			while (it.hasNext()) {
				Step st = (Step) it.next();
				if (st.getName().equals(name)) {
					toRemove = st;
				}
			}
			boolean success = this.removeSuccessor(toRemove);			
			return success;
		}
		else return false;
	}

	/**
	 * @see edu.udo.cs.miningmart.m4.Step#dependencyExists(Chain)
	 */
	public boolean dependencyExists(edu.udo.cs.miningmart.m4.Chain toChain) throws M4Exception {

		Collection allStepsOfToChain = toChain.getAllSteps();
		if (allStepsOfToChain == null) {
			throw new M4Exception("ToChain '" + toChain.getName() + "': got NULL when asking for all steps!");
		}
		Iterator it = allStepsOfToChain.iterator();
		while (it.hasNext()) {
			Step toStep = (Step) it.next();
			if (this.getTheCase().containsDependency(this, toStep)) {
				return true;
			}			
		}
		return false;
	}
	
	public boolean isRelationallyValid() {
		// TODO: implement this if needed
		return true;
	}
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#isSuccessorOf(Step)
	 */
	public boolean isSuccessorOf(edu.udo.cs.miningmart.m4.Step previousStep) throws M4Exception {
		if (previousStep == null)
			return false;
		Collection successorsOfPreviousStep =
			previousStep.getSuccessors();
		return successorsOfPreviousStep.contains(this);
	}
	
	/**
	 * @see Step#addPredecessor(Step)
	 */
	public void addPredecessor(edu.udo.cs.miningmart.m4.Step step) throws M4Exception {
		this.getTheCase().addStepDependency(step, this);		
	}	
	
	public void addReversingStep(edu.udo.cs.miningmart.m4.Step st) throws M4Exception {
		this.getDependentReversingSteps();
		if ( ! this.myReversingSteps.contains(st)) {
			if (this.linkExistsToHere((Step) st)) {
				throw new M4Exception("Cannot add step '" + st.getName() + "' as reversing step of '" +
						this.getName() + "' because the former precedes the latter in the transition graph!");
			}
			this.myReversingSteps.add(st);
			((Step) st).setReversedStep(this);
		}
	}
	
	// returns true iff this step follows the given step
	// in the data flow graph (of step dependencies)
	private boolean linkExistsToHere(Step aStep) throws M4Exception {
		Iterator stepIt = this.getTheCase().getDependentStepsFor(aStep).iterator();
		while (stepIt.hasNext()) {
			Step oneStep = (Step) stepIt.next();
			if (oneStep.equals(this)) {
				return true;
			}
		}
		return false;
	}
	
	public void setReversedStep(Step reversedStep) {
		this.theStepThisStepReverses = reversedStep;
	}
	
	public boolean materialisesOutput() throws M4Exception {
		try {
			ExecutableOperator myExecOp = CompilerAccessLogic.getExecutableOperator(this.getTheOperator());
			if (myExecOp instanceof ConceptOperator) {
				String outputCsType = ((ConceptOperator) myExecOp).getTypeOfNewColumnSet(0);
				if (outputCsType.equals(Columnset.TYPE_TABLE)) {
					return true;
				}
			}
			return false;
		}
		catch (M4CompilerError mce) {
			throw new M4Exception("Step '"  + this.getName() +
					"': could not determine executable operator: " + mce.getMessage());
		}
	}

	/**
	 * Returns true iff all input concepts of this step are
	 * of type DB.
	 * 
	 * @return
	 * @throws M4Exception
	 */
	public boolean hasOnlyInputConceptsOfTypeDB() throws M4Exception {
		Iterator it = this.getAllInputConcepts().iterator();
		while (it.hasNext()) {
			Concept inputConcept = (Concept) it.next();
			if ( ! inputConcept.getType().equals(Concept.TYPE_DB)) {
				return false;
			}
		}
		return true;
	}
	
	/**
	 * Returns the greatest number of steps between this step and the last
	 * of its predecessors that materialises its output. If all direct
	 * predecessors materialise their output, or if this step uses only DB
	 * Concepts as inputs, 0 is returned. Otherwise the maximum of these 
	 * results for the predecessors is increased by 1 and returned.
	 */
	public int getMaximumNumberOfStepsSinceLastMaterialisation() throws M4Exception {
		
		Collection directPredecessors = this.getAllPredecessors();
		if (directPredecessors.isEmpty()) {
			return 0;
		}
		Iterator predStepIt = directPredecessors.iterator();
		boolean allPredsMaterialise = true;
		while (predStepIt.hasNext()) {
			Step predecessor = (Step) predStepIt.next();
			if ( ! predecessor.materialisesOutput()) {
				allPredsMaterialise = false;
			}
		}
		if (allPredsMaterialise) {
			return 0;
		}
		predStepIt = directPredecessors.iterator();
		int maximum = 0;
		while (predStepIt.hasNext()) {
			Step predecessor = (Step) predStepIt.next();
			int predValue = predecessor.getMaximumNumberOfStepsSinceLastMaterialisation();
			if (predValue > maximum) {
				maximum = predValue;
			}
		}
		return maximum + 1;
	}

	/**
	 * Some steps, like those using evaluation operators, do not 
	 * produce any output concept or attribute. These steps return
	 * FALSE, all others return TRUE.
	 */
	public boolean createsConceptualOutput() throws M4Exception {
		return ( ! this.getTheOperator().getAllOutputOperatorParameters().isEmpty());
	}

	/**
	 * Returns TRUE iff this step has no successors or if it has
	 * only successors that do not produce conceptual output.
	 */
	public boolean isLastStepProducingConceptualOutput() throws M4Exception {
		Collection succ = this.getSuccessors();
		if (succ.isEmpty() && this.createsConceptualOutput()) {
			return true;
		}
		Iterator succStepIt = succ.iterator();
		while (succStepIt.hasNext()) {
			Step successorStep = (Step) succStepIt.next();
			if (	(successorStep.createsConceptualOutput())
					||
					( ! successorStep.getSuccessors().isEmpty())
				) {
				return false;
			}
		}
		if (this.createsConceptualOutput())
			return true;
		return false;
	}
	
	/**
	 * This method only works for steps that do not create an output concept.
	 * It returns an iterator of all successors of this step
	 * that use one of its input concepts as input concept, up to and 
	 * including the first step that creates an output concept (on every
	 * path that starts from this step).
	 * 
	 * @return an iterator of steps
	 * @throws M4Exception
	 */
	public Collection getSuccessorsUntilOutputConceptIsCreated()
	throws M4Exception {
		if (this.getOutputConcept() != null) {
			return new Vector();
		}
		Collection myInputs = this.getAllInputConcepts();
		Collection ret = new Vector();
		this.getSuccessorsUntilOutputConceptIsCreated(myInputs, ret);
		return ret;
	}
	
	private void getSuccessorsUntilOutputConceptIsCreated(Collection theInputs, Collection theSuccessors)
	throws M4Exception {
		
		theSuccessors.add(this);
		
		if (this.getOutputConcept() != null) {
			return;
		}
		Iterator succIt = (new Vector(this.getSuccessors())).iterator();
		while (succIt.hasNext()) {
			Step successor = (Step) succIt.next();
			successor.getSuccessorsUntilOutputConceptIsCreated(theInputs, theSuccessors);
		}
	}
	
	/**
	 * Returns true iff this step aggregates rows of
	 * its input concept
	 * 
	 * @return
	 * @throws M4Exception
	 */
	public boolean usesAggregation() throws M4Exception {
		// simply check for known aggregation parameters:
		Collection c = new Vector();
		OpParam groupByOpParam = (OpParam) this.getTheOperator().getOpParam(Pivotize.PARAMETER_GROUPBY_ATTR);
		if (groupByOpParam != null) {
			c = this.getParameterTuples(groupByOpParam);
			groupByOpParam = (OpParam) this.getTheOperator().getOpParam(SpecifiedStatistics.PARAMETER_ATTR_GROUPBY);
			if (groupByOpParam != null) {
				c.addAll(this.getParameterTuples(groupByOpParam));
			}
		}
		
		return ( ! c.isEmpty());
	}
	
	/**
	 * Overwrites the superclass method because the output that this step
	 * creates must be deleted, too.
	 * 
	 * @throws M4Exception
	 */
	public void deleteSoon() throws M4Exception {
		this.deleteOutputs(); // delete concepts etc. created as output by this step
		super.deleteSoon();
	}

	/**
	 * Overwrites the superclass method to ensure that
	 * the cross table entries are deleted first
	 */
	protected void deleteLocal() throws M4Exception {
		
		super.deleteLocal();
		
		this.deleteDbTrash();
		this.writeDbTrashToDb();
		this.deleteM4Trash();
		this.writeM4TrashToDb();
		
		String sql = "DELETE FROM " + M4_TABLE_NAME_STEPSEQ
		           + " WHERE " + ATTRIB_STEPSEQ_FROM_STEP
		           + " = " + this.getId();			
		this.executeM4SqlWrite(sql);

		sql = "DELETE FROM " + M4_TABLE_NAME_STEPSEQ
		    + " WHERE " + ATTRIB_STEPSEQ_TO_STEP
		    + " = " + this.getId();			
		this.executeM4SqlWrite(sql);
		
		if (this.reversingStepsTableExistsInM4()) {
			sql = "DELETE FROM " + REVSTEP_TABLENAME
				+ " WHERE " + ATTRIB_REV_REVSTEP
				+ " = " + this.getId();		
			this.executeM4SqlWrite(sql);
			
			sql = "DELETE FROM " + REVSTEP_TABLENAME
				+ " WHERE " + ATTRIB_REV_ORIGSTEP
				+ " = " + this.getId();	
			this.executeM4SqlWrite(sql);
		}
	}
	
	// ***** Helper methods for loading: *****
	
    /**
     * Active getter of parameters:
	 * Reads all parameters from the database having this step's ID as
	 * their step reference. This method does not use any information
	 * stored in the <code>OP_PARAM_T</code> table!
     */
	private void readParameterTuplesFromDB() throws M4Exception
    {	
    	Iterator it = this.getObjectsReferencingMe(Parameter.class).iterator();
    	while (it.hasNext()) {
    		this.addParameterTuple((Parameter) it.next());	
    	}
	}
    
    /**
	 * Reads all of this step's parameters specified in table <i>OP_PARAM_T</i> for
	 * the current <code>Step</code>  and stores them in a parameter dictionary.
	 * This is a service method for the compiler, aggregating all parameters to arrays
	 * as specified by the corresponding <code>OpParam</code> objects!
	 * 
	 * @param paramsMustExist If TRUE, it means that an exception ought to be 
	 * thrown if not all parameters are set. This parameter serves to enable the
	 * new GUI to set the parameters initially.
     */
	public edu.udo.cs.miningmart.m4.ParamDict readParametersFromDB(boolean paramsMustExist) 
	throws M4Exception {
		final ParamDict pd = new edu.udo.cs.miningmart.m4.core.ParamDict();
		Iterator it = this.getTheOperator().getOpParamsIterator();
		while (it.hasNext()) {
			final edu.udo.cs.miningmart.m4.OpParam op  = (OpParam) it.next();
			
			if (op.getMinArg() == 0) {
				// Optional parameters in combination with loops need special attention,
				// because the 'loopable' property of this kind of parameter is sometimes
				// not detectable from the meta-data!
				try {
					ParameterArray withoutLoop = (ParameterArray) this.loadParam(op, 0);
					if (withoutLoop.size() > 0 || this.getLoopCount() <= 0) {
						pd.put(op.getName(), withoutLoop);
					}
					else {
						final int loops = this.getLoopCount() + 1;
						for (int i = 1; i < loops; i++) {
							ParameterArray obj = (ParameterArray) this.loadParam(op, i);
							pd.put(op.getName(), i, obj);
						}
					}
				}
				catch (ParameterNotFoundException e) {
					throw new M4Exception("Internal compiler error." +
					"'ParameterNotFound' exception was thrown for an optional parameter:\n" +
					e.getMessage());
				}
			}
			else {
				// If the parameter is not optional, one can try to load it as non-looped,
				// catch the exception and try again, assuming it is looped.
				try {
					// load parameter: arrays and optional parameters are handled, as well!
					pd.put(op.getName(), this.loadParam(op, 0));
				}
				catch (ParameterNotFoundException e) {
					// The parameter could not be loaded as usual.
					// ==> Is it a looped parameter?
					if (this.getLoopCount() > 0) {
						final int loops = this.getLoopCount() + 1;
						try {
							for (int i = 1; i < loops; i++) {
								ParameterArray obj = (ParameterArray) this.loadParam(op, i);
								pd.put(op.getName(), i, obj);
							}
						}
						catch (ParameterNotFoundException pnfe) {
							if (paramsMustExist) {
								throw pnfe;
							}
						}
					}
					else
						if (paramsMustExist) throw e; // no: propagate exception!
				}
			}
		}
		return pd;
	}
	
	/**
	 * This method handles the minimum and maximum information for the parameter size.
	 * Depending of the values it calls methods for loading a single parameter or for
	 * loading a parameter array. Afterwards it is checked, whether the parameters read
	 * from the database match the constraints.
	 * @param op an <code>OpParam</code> object specifying which parameter(array) to load.
	 * @return either a single <code>M4Object</code> or an array of M4Objects.
	 * It is asserted that the minimum/maximum number of parameter constraints are met.
	 * Furthermore <code>null</code> is never returned, but a <code>ParameterArray</code>
	 * containing an array of size 0.
	 */
	private edu.udo.cs.miningmart.m4.ParameterArray loadParam(edu.udo.cs.miningmart.m4.OpParam op, int loopNr)
		throws M4Exception
	{
		ParameterArray ret = null;
		try {
			if (op.getMaxArg() == 1) {
				final Parameter obj = (Parameter) this.loadSingleParam(op, loopNr);
				if (obj != null) {
					ret = new ParameterArray(obj.getParObjectType());
					ret.addParameter(obj);
				}
			}
			else {
				ret = (ParameterArray) this.loadParamArray(op, loopNr);
			}
		}
		catch (ParameterError e) {} // This is handled below.
		 
		if (ret == null) {
			if (op.getMinArg() > 0) {
				this.throwParameterLoadException(loopNr, op.getName() + " not found.");
			}
			// create an empty array of the correct type:
			else ret = new ParameterArray(op.getType());
		}
		else {
			if (ret.size() < op.getMinArg())
				this.throwParameterLoadException(loopNr, op.getName() + " should be an Array of at least " +
												op.getMinArg() + " entries, but has only " + ret.size() + " elements!");
			if (op.getMaxArg() >= 0 && ret.size() > op.getMaxArg())
				this.throwParameterLoadException(loopNr, op.getName() + " should be an Array of at most " +
												op.getMinArg() + " entries, but has " + ret.size() + " elements!");
		}
		
		return ret;
	}

	private void throwParameterLoadException(int loopNr, String restOfMessage)
		throws M4Exception
	{
		String message = "Could not load parameters for operator " +
						 this.getTheOperator().getName() + " in step " + this.getId();
		if (loopNr > 0)
		{ message += ", loop nr." + loopNr; }
		message += " !\nParameter " + restOfMessage;
		throw new ParameterNotFoundException(message);		
	}

	// returns null if parameter is optional and not specified
	private edu.udo.cs.miningmart.m4.Parameter loadSingleParam(edu.udo.cs.miningmart.m4.OpParam op, int loopNr)
		throws M4Exception
	{
		// to make the parameter loading mechanism more versatile, we check 
		// first the Java level before reading the database:
		Parameter theSingleParam = null;
		if (this.myParameterTuples != null) {
			Iterator tupleIt = this.myParameterTuples.iterator();
			while (tupleIt.hasNext()) {
				Parameter myParam = (Parameter) tupleIt.next();
				if (myParam.getName().startsWith(op.getName())) {
					if (theSingleParam == null) {
						theSingleParam = myParam;
					}
					else {
						throw new M4Exception("Step.loadSingleParam(): found more than one Parameter '" +
								op.getName() + "' in Step '" + this.getName());
					}
				}
			}
		}
		if (theSingleParam != null) {
			return theSingleParam;
		}
		
		// now try reading the database:
		String query = this.createLoadParamQuery(op, loopNr);
	    Long parId = this.executeM4SingleValueSqlReadL(query);
	    if (parId == null) {
	    	return null;	
	    }
	    else {
	    	DB db = this.getM4Db();
	    	return (Parameter) db.getM4Object(parId.longValue(), Parameter.class);
	    }  
	}

	private edu.udo.cs.miningmart.m4.ParameterArray loadParamArray(edu.udo.cs.miningmart.m4.OpParam op, int loopNr)
		throws M4Exception {
		
		// to make the parameter loading mechanism more versatile, we check 
		// first the Java level before reading the database:
		ParameterArray ret = null;
		if (this.myParameterTuples != null) {
			ret = new ParameterArray(op.getType());
			Iterator tupleIt = this.myParameterTuples.iterator();
			while (tupleIt.hasNext()) {
				Parameter myParam = (Parameter) tupleIt.next();
				if (myParam.getName().startsWith(op.getName())) {
					ret.addParameter(myParam);
				}
			}
		}
		if (ret != null && (ret.size() != 0)) {
			return ret;
		}
		
		// now try the DB:
		ResultSet rs = null;
		try {
			String query = this.createLoadParamQuery(op, loopNr);
		    rs = this.executeM4SqlRead(query);

		    DB db = this.getM4Db();
			ret = new ParameterArray(op.getType());
		    while (rs.next()) {
		    	Long parId  = new Long(rs.getLong(1)); 
				Parameter p = (Parameter) db.getM4Object(parId.longValue(), Parameter.class);
				ret.addParameter(p);
	    	}				    
			return ret;
		}
	    catch (SQLException sqle) {
	    	throw new M4Exception(
	    		"SQL error when loading ParameterArray for Step "
	    		+ this.getId() + ": " + sqle.getMessage());
	    }
	    finally {
			DB.closeResultSet(rs);
		}
	}

	private String createLoadParamQuery(edu.udo.cs.miningmart.m4.OpParam op, int loopNr) {
		String parName = op.getName();
	    String LoopCondition;
	    if (loopNr == 0) {
	    	LoopCondition = "(" + Parameter.ATTRIB_PAR_LOOP_NR + " IS NULL OR "
	    	              + Parameter.ATTRIB_PAR_LOOP_NR + " = 0)";
	    }
	    else {
	    	LoopCondition = Parameter.ATTRIB_PAR_LOOP_NR + " = " + loopNr;
	   	}
	    String query =
	           "SELECT " + Parameter.ATTRIB_PARAMETER_ID
	           + " FROM " + Parameter.M4_TABLE_NAME
               + " WHERE " + Parameter.ATTRIB_PAR_STEP_ID
               + " = " + this.getId()
	           + " AND " + LoopCondition
	           + " AND " + DB.attribPrefix(Parameter.ATTRIB_PARAMETER_NAME, parName)
	           + " = " + DB.quote(parName)
			   + " ORDER BY " + Parameter.ATTRIB_PARAMETER_NR; // for arrays

	    return query;
	}
	
	/**	 Helper method: database reader for the dependent steps */
	private void readSuccessorsFromDb() throws M4Exception {
	    String query = "SELECT " + ATTRIB_STEPSEQ_TO_STEP
	    			 + " FROM "  + M4_TABLE_NAME_STEPSEQ
	    			 + " WHERE " + ATTRIB_STEPSEQ_FROM_STEP
	    			 + " = " + this.getId();

		ResultSet rs = null;
	    try {
		    rs = this.executeM4SqlRead(query);
			final DB db = this.getM4Db();
		    while (rs.next()) {
		    	long successorId = rs.getLong(1);
		    	if (!rs.wasNull()) {
					Step step = (Step) db.getM4Object(successorId, Step.class);
					if ( ! this.dependentSteps.contains(step)) { // debug
					    this.primitiveAddSuccessor(step); // does not set the dirty flag
					}
					else {
						throw new M4Exception("Step.readSuccessorsFromDb(): Found double entry for Step " +
								this.getId() + " and Successor Step " + step.getId() + "!");
					}
		    	}
		    }
	    }
	    catch (SQLException sqle) {
	    	throw new M4Exception(
	    		"SQL error when reading step sequence for case with Id "
	    		+ this.getId() + ": " + sqle.getMessage());
	    }
	    finally {
	    	DB.closeResultSet(rs);
	    }
	}

	/**
	 * @return a <code>Collection</code> of <code>Step</code> to <code>Step</code>
	 * dependencies for sequentializing all <code>Step</code>s before writing updates
	 * to the database.
	 */
	public Collection getCrossReferences() throws M4Exception {
		Collection ret = this.getSuccessors();
		return ret;
	}


 	/** <code>Step</code>s have coordinates. */
	protected boolean hasCoordinates() {
		return true;	
	}

	/**
	 * This method stores the <code>Collection</code> of dependent 
	 * <code>Step</code>s to the database (table <i>STEPSEQUENCE_T</i>)
	 * if there were any updates to that <code>Collection</code>.
	 */
	protected void storeLocal() throws M4Exception {
		super.storeLocal(); // stores the coordinates

		if (this.dependentStepsDirty) {
			
			{ // Delete all successors first
				String sql = "DELETE FROM " + M4_TABLE_NAME_STEPSEQ
				           + " WHERE " + ATTRIB_STEPSEQ_FROM_STEP
				           + " = " + this.getId();
			
				this.executeM4SqlWrite(sql);
			}
			
			{ // Then write all current successors
				Iterator it = this.getSuccessors().iterator();
				String prefix = "INSERT INTO " + M4_TABLE_NAME_STEPSEQ
							  + " ( " + ATTRIB_STEPSEQ_DEPENDENCY_ID
							  + ", " + ATTRIB_STEPSEQ_FROM_STEP
							  + ", " + ATTRIB_STEPSEQ_TO_STEP
							  + " ) VALUES ( ";
			
				while (it.hasNext()) {
					Step successor = (Step) it.next();
			
					String sql = prefix + this.getNextM4SequenceValue()
							   + ", " + this.getId()
							   + ", " + successor.getId() + " )";
					
					this.executeM4SqlWrite(sql);
				}
			}
		}
		
		this.writeM4TrashToDb();
		this.writeDbTrashToDb();
		
		this.writeThisStepAsReversedStepToDb();
	}
	
	/**
	 * This method is mainly to be used by the XML deserialization mechanism,
	 * but it robust enough not to crash if the Case is not yet set, and it
	 * maintains the Steps' order within the Case by using the Case's
	 * <code>addStepDependency(Step, Step)</code> method.
	 * 
	 * @param newSuccessors a <code>Collection</code> of successors <code>Step</code>s
	 */
	protected void setSuccessors(Collection newSuccessors) throws M4Exception {
		Iterator it = new Vector(this.getSuccessors()).iterator();
		while (it.hasNext()) {
			this.removeSuccessor((Step) it.next());
		}
		
		if (newSuccessors == null || newSuccessors.isEmpty())
			return;
			
		final edu.udo.cs.miningmart.m4.Case theCase = this.getTheCase();
		if (theCase != null) {
			it = newSuccessors.iterator();
			while (it.hasNext()) {
				theCase.addStepDependency(this, (Step) it.next());
			}
		}
		else this.getSuccessors().addAll(newSuccessors);
	}

   	/**
   	 * This method is to be written in a way that it can still be
   	 * called during updateDatabase()! You may only set objects dirty,
   	 * which are updated after and deleted before <code>Step</code>s.
   	 * 
   	 * @see M4Data#removeAllM4References()
   	 */
	protected void removeAllM4References() throws M4Exception {
		// This is necessary to avoid violations of foreign key
		// constraints in the M4 database:
		this.deleteM4Trash();
		this.deleteDbTrash();
		
		// Successors do not have to be set dirty.
		this.removeAllSuccessorsDuringDelete();
		
		// Parameters have to be set dirty and to be prepared for
		// being deleted.
		this.removeAllParametersDuringDelete();

		// changes to this step's DB tuple only:
		this.setTheCase(null);
		this.setTheChain(null);
		this.setTheOperator(null);
		
		// will be changed as the last table:
		this.removeDocObject();
	}

	/** @see M4Data#getDependentObjects */
	public Collection getDependentObjects() throws M4Exception {
		Collection ret = super.getDependentObjects();
		ret.addAll(this.getParameterTuples());
		return ret;
	}

	/**
	 * Helper method of removeAllM4References().
	 * Does not set the object's top-level dirty flag, because this would
	 * cause a CoModification Exception when done during updateDatabase().
	 * The dependentSteps dirty flag needs to be set, however, to make any
	 * updates to the successors collection written back to the database!
	 */
	private void removeAllSuccessorsDuringDelete() throws M4Exception {
		Collection suc = this.getSuccessors();
		if (suc != null && suc.size() > 0) {		
			suc.clear();
			this.dependentStepsDirty = true;
		}
	}

	/** Helper method of removeAllParameters() */
	private void removeAllParametersDuringDelete() throws M4Exception {
		Collection col = this.getParameterTuples();
		if (col != null && !col.isEmpty()) {
			Iterator it = (new Vector(col)).iterator(); // avoids a CoModification exception (?)
			while (it.hasNext())
				this.removeParameterTuple((Parameter) it.next());
		}
	}



	// ****** Handling of M4 Trash ******

	/** 
	 * The name of the M4 table holding the compiler's garbage collection
	 * information of M4 objects.
	 */
	public static final String M4_TRASH_TABLE = "m4trash_t";

	/** The M4 trash table's primary key attribute name. */
	public static final String M4_TRASH_ID = "m4id";

	/**
	 * The M4 trash table's attribute holding the referenced object's M4
	 * table name.
	 */
	public static final String M4_TRASH_OBJ_TABLE = "m4table";

	/** 
	 * The M4 trash table's attribute holding the object's creating
	 * Step ID.
	 */
	public static final String M4_TRASH_STEP_ID = "stepid";

	/** The compilation flag is also stored in the M4Trash table. */
	private boolean isCompiled = false;
	
	/**
	 * Active getter for the compiler's M4 Trash information. 
	 * @return a <code>Collection</code> of <code>M4Data</code>
	 * objects created by the M4 compiler that have to be removed
	 * before this <code>Step</code> can be compiled anew.
	 */
	public Collection getM4Trash() throws M4Exception {
		if (!this.allM4TrashLoaded && (!this.isNew())) {
			this.allM4TrashLoaded = true;
			try {
				this.readM4TrashFromDb();
			}
			catch (SQLException e) {
				throw new M4Exception(
					"SQLException caught when trying to read M4Trash information from DB:\n"
					+ e.getMessage());
			}
		}
		return this.myM4Trash;
	}

	/**
	 * This method should only be called by the compiler after a new M4 object
	 * has been created by calling a constructor.
	 * @param m4data the <code>M4Data</code> object to be added to the table
	 *        <code>M4TRASH_T</code> for being garbage collected before compiling
	 *        this <code>Step</code> or any of its predecessors.
	 */
	public void addToTrash(edu.udo.cs.miningmart.m4.M4Data m4data) throws M4Exception {
		if (m4data != null) {
			this.getM4Trash().add(m4data);
		}
		this.setM4TrashDirty();
	}

	/**
	 * This method realizes the M4Compiler's garbage collection on the objects
	 * created during compilation of this <code>Step</code>.
	 */
	public void deleteM4Trash() throws M4Exception {
		Iterator it = this.getM4Trash().iterator();
		try {
			while (it.hasNext()) {
				((M4Data) it.next()).deleteSoon();
				it.remove();
			}
		}
		catch (ClassCastException cce) {
			System.out.println("CCException in Step " + this.getName() + ": " + cce.getMessage());
			throw cce;
		}
		this.isCompiled = false;
		this.setM4TrashDirty();
	}

	/** Reads the information from the M4 trash table. */
	private void readM4TrashFromDb() throws M4Exception, SQLException {
		String sql = "SELECT " + M4_TRASH_ID + ", " + M4_TRASH_OBJ_TABLE
				   + " FROM "  + M4_TRASH_TABLE
				   + " WHERE " + M4_TRASH_STEP_ID + " = " + this.getId();
		
		final DB db = this.getM4Db();
		ResultSet rs = null;
		try {
			rs = this.executeM4SqlRead(sql);
		 L: while (rs.next()) {
				long   objId = rs.getLong(M4_TRASH_ID);
				String table = rs.getString(M4_TRASH_OBJ_TABLE);
				Class objClass;
				if (table.equalsIgnoreCase(Column.M4_TABLE_NAME)) {
					objClass = Column.class;
				}
				else if (table.equalsIgnoreCase(Columnset.M4_TABLE_NAME)) {
					objClass = Columnset.class;
				}
				else if (table.equalsIgnoreCase(Key.M4_TABLE_NAME)) {
					objClass = ForeignKey.class;
				}
				else if (table.equals(" ")) {
					this.isCompiled = true;
					continue L;			
				}
				else {
					throw new M4Exception(
						"Step.readM4TrashFromDb(): Found unsupported M4 table "
						+ table	+ " in attribute " + M4_TRASH_OBJ_TABLE
						+ " of table " + M4_TRASH_TABLE + "!");
				}

				try {
					M4Data obj = (M4Data) db.getM4Object(objId, objClass);
					if (obj != null) {
						this.getM4Trash().add(obj); // Avoids setting the dirty-flag.
					}
				}
				catch (ParameterNotFoundException e) {
					super.doPrint(Print.COMPILER_STEP_CONTROL,
						"Garbage Collection for Step " + this.getId()
						+ "did not find object with ID " + objId
						+ " from table " + table + "!");
				}
			}
		}
		finally {
			DB.closeResultSet(rs);
		}
	}


	// ***** Handling of DB Trash *****

	private void setDbTrashDirty() {
		if (this.dbTrashDirty == false) {
			this.dbTrashDirty = true;
			this.setDirty(); // Currently the complete Step needs to be set dirty!
		}
	}
	
	/** Helper method to store the information about a database object in the trash index */
	public void addDatabaseObjectToTrash(String objectName, String schemaName, String objectType)
		throws M4Exception
	{		
		String[] trashEntry = this.createDbTrashArray(objectName, schemaName, objectType);
        this.getDbTrash().add(trashEntry);
        this.setDbTrashDirty();
	}

	/**
	 * This method takes care of deleting trash objects from the
	 * business data schema.
	 */
	public void deleteDbTrash() throws M4Exception
	{
		Iterator it = this.getDbTrash().iterator();
		{ // Sort trash objects: 
			// 1. foreign keys, 2. primary keys,
			// 3. anything that is not under 4.,
			// 4. views ending in "_V", because they may have been created by the
			// method ConceptOperator.createViewAsSqlDef, and can only be deleted
			// after the view that depends on them has been deleted
			Vector fkList = new Vector();
			Vector pkList = new Vector();
			Vector other = new Vector();
			Vector intermediateViews = new Vector();
			while (it.hasNext()) {
				String[] trashArray = (String[]) it.next();
				String objectType = trashArray[2];
				if (objectType.equalsIgnoreCase(DBT_TYPE_FOREIGN_KEY)) {
					fkList.add(trashArray);
				}
				else if (objectType.equalsIgnoreCase(DBT_TYPE_PRIMARY_KEY)) {
					pkList.add(trashArray);
				}
				else if (objectType.equalsIgnoreCase(DBT_TYPE_VIEW)) {
					if (trashArray[0].endsWith(ConceptOperator.suffixForIntermediateViews)) {
						intermediateViews.add(trashArray);
					}
					else other.add(trashArray);
				}
				else other.add(trashArray);
			}
			fkList.addAll(pkList);
			fkList.addAll(other);
			fkList.addAll(intermediateViews);
			it = fkList.iterator();
		}
        
        while (it.hasNext()) {
        	final String[] trashArray = (String[]) it.next();
			String objectName = trashArray[0];
			String schemaName = trashArray[1];
			String objectType = trashArray[2];
			 
			String query = null;  
			boolean tableOrView = false;
            if (objectType.equalsIgnoreCase(DBT_TYPE_TABLE)) {
               	query = "DROP TABLE ";
               	tableOrView = true;
            }
            else if (objectType.equalsIgnoreCase(DBT_TYPE_VIEW)) {
              	query = "DROP VIEW ";
              	tableOrView = true;
            }
            else if (objectType.equalsIgnoreCase(DBT_TYPE_FUNCTION)) {
              	query = "DROP FUNCTION ";
            }
            else if (objectType.equalsIgnoreCase(DBT_TYPE_INDEX)) {
              	query = "DROP INDEX ";
            }
            else if (objectType.equalsIgnoreCase(DBT_TYPE_PRIMARY_KEY)
            	  || objectType.equalsIgnoreCase(DBT_TYPE_FOREIGN_KEY))
            {
            	int dotIndex = objectName.indexOf('.');
            	if (dotIndex < 1) {
            		throw new M4Exception("Step.deleteDbTrash: Found invalid constraint primary key or foreign key constraint:\n"
            				+ objectType + "'!");
            	}
            	String table = objectName.substring(0, dotIndex);
            	objectName   = objectName.substring(dotIndex+1, objectName.length());
              	query = "ALTER TABLE " + (schemaName == null ? "" : schemaName + ".") + table + " DROP CONSTRAINT ";
              	schemaName = null;
              	
              	// Method that helps to verify that the primary key objects exist:
              	// this.getM4Db().getBusinessDbCore().getPrimaryKeyColumnNames(table);
            }
    		else throw new M4Exception("Step.deleteDbTrash: Unknown object type '" + objectType + "'!");
    		            
            if (schemaName != null) {
               	query += schemaName + ".";
            }
            query += objectName;

			try {
				if (tableOrView && 
						( this.getM4Db().getBusinessDbms() == DB.POSTGRES ||
						  this.getM4Db().getBusinessDbms() == DB.MYSQL)) {
					query += " CASCADE";
				}            
               	this.executeBusinessSqlWrite(query);
            }
            catch (M4Exception sqle) {
               	this.doPrint(Print.MAX,
               		"WARNING: Error when attempting to remove a database object according to DBTrash_T: "
                       + sqle.getMessage());
                // here we need a commit for postgres, because the current transaction block
                // is messed up...
                try {
	                this.getM4Db().commitBusinessTransactions();
                }
                catch (DbConnectionClosed dbc) { 
                	throw new M4Exception("Closed Db connection during trash deletion:\n" + dbc.getMessage());
                }
                catch (SQLException se) { 
                	throw new M4Exception("Error trying to commit transactions during trash deletion:\n" + se.getMessage());
                }
            }
            catch (DbConnectionClosed dbc) { 
            	throw new M4Exception("Closed Db connection during trash deletion:\n" + dbc.getMessage());
            }
            this.getDbTrash().remove(trashArray);
	        this.setDbTrashDirty();
        }					     
	}

	/**
	 * Active getter for this <code>Step</code>'s DB trash.
	 * @return a <code>Collection</code> of <code>String[3]</code>
	 * objects with <ul>
	 * <li><code>String[0]</code>: object name</li>
	 * <li><code>String[1]</code>: schema of the object</li>
	 * <li><code>String[2]</code>: object type</li>
	 * </ul>
	 * */
	private Collection getDbTrash() throws M4Exception {
		if ( ! this.allDbTrashLoaded && ! this.isNew() ) {
			this.allDbTrashLoaded = true;
			this.readDbTrashFromDb();
		}
		return this.myDbTrash;
	}

	/** Reads the information from the DB trash table. */
	private void readDbTrashFromDb() throws M4Exception {
		String sql = "SELECT " + ATTRIB_DBT_OBJNAME
		                + ", " + ATTRIB_DBT_SCHEMA
		                + ", " + ATTRIB_DBT_OBJTYPE
				    + " FROM " + DB_TRASH_TABLENAME
				   + " WHERE " + ATTRIB_DBT_STEPID
				       + " = " + this.getId();
		
		ResultSet rs = null;
 		try {
			rs = this.executeM4SqlRead(sql);	
		 	while (rs.next()) {
				String[] trashEntry =
					this.createDbTrashArray(
						rs.getString(ATTRIB_DBT_OBJNAME),
						rs.getString(ATTRIB_DBT_SCHEMA),
						rs.getString(ATTRIB_DBT_OBJTYPE));
				this.myDbTrash.add(trashEntry);
	 		}
 		}
		catch (SQLException e) {
	 		this.doPrint(Print.MAX,
           		"WARNING: Error when attempting to read a database object from "
           		+ DB_TRASH_TABLENAME + " for step " + this.getId() + ":\n"
           		+ e.getMessage());
	 	}
		finally {
			DB.closeResultSet(rs);
		}
	}

	/** Local storage method for the database trash table. */
	private void writeDbTrashToDb() throws M4Exception {
		if (this.dbTrashDirty) {
	        String query = "DELETE FROM " + DB_TRASH_TABLENAME
	        			 + " WHERE " + ATTRIB_DBT_STEPID
	        			 + " = " + this.getId();

			this.executeM4SqlWrite(query);

	        final String attributes = ATTRIB_DBT_OBJTYPE + ", " +
    	    	  	                  ATTRIB_DBT_OBJNAME + ", " +
        		    	              ATTRIB_DBT_SCHEMA + ", " +
        		        	          ATTRIB_DBT_STEPID;
			
			Iterator it = this.getDbTrash().iterator();
			while (it.hasNext()) {
	        	String[] trashArray = (String[]) it.next();
				String objectName = trashArray[0];
				String schemaName = trashArray[1];
				String objectType = trashArray[2];

		        if ((schemaName == null) || (schemaName.equals("")))
    		    {   schemaName = "NULL";   }
        	    else
            	{   schemaName = DB.quote(schemaName);  }

		        query = "INSERT INTO " + DB_TRASH_TABLENAME
		                + " (" + attributes + ") VALUES ("
	        	        + DB.quote(objectType) + ", "
	        	        + DB.quote(objectName) + ", "
	        	        + schemaName + ", "
	        	        + this.getId() + ")";
	                        
				this.executeM4SqlWrite(query);
			}	
		}
		this.dbTrashDirty = false;	
	}

	/**
	 * Helper method to store the Strings in the right order and take
	 * care that the object type is known.
	 */
	private String[] createDbTrashArray(String objectName, String schemaName, String objectType)
		throws M4Exception
	{
       	// check if the object type is known:
       	if ( ! ( objectType.equals(DBT_TYPE_FUNCTION) || 
       	         objectType.equals(DBT_TYPE_INDEX) ||
       	         objectType.equals(DBT_TYPE_TABLE) ||
       	         objectType.equals(DBT_TYPE_VIEW) ||
       	      	 objectType.equals(DBT_TYPE_PRIMARY_KEY) ||
       	   		 objectType.equals(DBT_TYPE_FOREIGN_KEY)
       	        )
       	    )
       	{
       		throw new M4Exception(
       			"Error adding a database object to trash: Unknown type: "
       			+ objectType + "!");
       	}

		if (schemaName != null && schemaName.length() == 0)
			schemaName = null;

		String[] trashEntry = new String[3];
		trashEntry[0] = objectName;
		trashEntry[1] = schemaName;
		trashEntry[2] = objectType;
		return trashEntry;
	}

	// **********
	

	/** 
	 * After a <code>Step</code> has been successfully compiled a flag in the
	 * M4 database is set, which is removed by the next garbage collection.
	 * This method checks, if this flag is set.
	 * 
	 * @return <code>true</code> iff this <code>Step</code> has been
	 *         successfully compiled and no garbage collection involving this
	 *         <code>Step</code> has been run afterwards.
	 */	
	public boolean isCompiled() throws M4Exception {
		this.getM4Trash(); // makes sure the flag is up-to-date
		return this.isCompiled;
	}
	
	/**
	 * This method may only be called by the control structure!
	 * It marks this <code>Step</code> as successfully compiled.
	 * This information is used when it is checked for succeeding
	 * <code>Step</code>s, if all of the dependent resources are
	 * already available.
	 */
	public void setCompiled() throws M4Exception {
		// makes sure the avtive getter will not overwrite the flag later on:
		this.getM4Trash();
		this.isCompiled = true;
		this.setM4TrashDirty();
	}

	// Indicates whether we have to update the M4Trash table for this Step:
	private boolean m4trashDirty = false;

	/** Sets the M4Trash-dirty flag. */
	private void setM4TrashDirty() {
		this.m4trashDirty = true;		
		this.setDirty(); // currently the whole object has to be set dirty!
	}

	/** Updates the trash information in table M4Trash. */
	private void writeM4TrashToDb() throws M4Exception {
		if (this.m4trashDirty == true) {
			this.m4trashDirty = false;

			String delSql = "DELETE FROM " + M4_TRASH_TABLE
						  + " WHERE " + M4_TRASH_STEP_ID
						  + " = " + this.getId();
			this.executeM4SqlWrite(delSql);

			final String sqlPre =
				"INSERT INTO " + M4_TRASH_TABLE + " ( "
				+ M4_TRASH_ID + ", " + M4_TRASH_OBJ_TABLE + ", " + M4_TRASH_STEP_ID
				+ " ) VALUES ( ";
			final String sqlPost = ", " + this.getId() + " )";
						
			Iterator it = this.getM4Trash().iterator();
			while (it.hasNext()) {
				M4Data m4data = (M4Data) it.next();
				if (!m4data.isWaitingForDelete()) {
					long objId   = m4data.getId();
					String table = m4data.getM4TableName();
					String sql = sqlPre + objId + ", " + DB.quote(table) + sqlPost;
					this.executeM4SqlWrite(sql);
				}		
			}
			
			if (this.isCompiled()) {
				String sql = sqlPre + "0, ' '" + sqlPost;
				this.executeM4SqlWrite(sql);
			}
		}
	}

	/**
	 * Returns the direct predecessor.
	 * 
	 * @see edu.udo.cs.miningmart.m4.Step#getPredecessor()
	 */
	public edu.udo.cs.miningmart.m4.Step getPredecessor() throws M4Exception {
		// trying to find the predecessor by iterating backwards
		// through the step dependencies

		Iterator it = this.getTheCase().getReverseIterator();
		boolean currentStepFound = false;
			
		while (it.hasNext()) {
			Step step = (Step) it.next();
			if (this.equals(step)) {   
				currentStepFound = true;  
				continue; 
			}
			if (currentStepFound && this.getTheCase().containsDependency(step, this))				{
				return step;
			}
		}	
		return null;
	}
	
	/**
	 * Returns only the DIRECT predecessors!
	 * 
	 * @see Step#getAllPredecessors()
	 */
	public Collection getAllPredecessors() throws M4Exception {
		// trying to find all DIRECT predecessors by iterating backwards
		// through the step dependencies

		if (this.getTheCase() == null) {
			return null;
		}
		Iterator it = this.getTheCase().getReverseIterator();
		boolean currentStepFound = false;
		Vector thePredecessors = new Vector();
		
		while (it.hasNext()) {
			Step step = (Step) it.next();
			if (this.equals(step))
			{   currentStepFound = true;  
				continue; 
			}
			if (currentStepFound && this.getTheCase().containsDependency(step, this)) {
					thePredecessors.add(step);
			}
		}	
				
		return thePredecessors;		
	}	

	/**
	 * This method returns all steps that have been created automatically
	 * to reverse the attribute creation done by this step's operator.
	 * Such reversing steps are needed for deployment if the target attribute
	 * for mining has been changed.
	 * 
	 * @return a collection of steps
	 * @throws M4Exception
	 */
	public Collection getDependentReversingSteps() throws M4Exception {
		if (this.myReversingSteps == null) {
			this.readReversingStepsFromDb();
		}
		return this.myReversingSteps;
	}
	
	private void readReversingStepsFromDb() throws M4Exception {
		this.myReversingSteps = new Vector();
		try {
			if ( ! this.reversingStepsTableExistsInM4()) {
				return;
			}
			String query = "SELECT " + ATTRIB_REV_ID + ", " +
							ATTRIB_REV_REVSTEP + 
							" FROM " + REVSTEP_TABLENAME + 
							" WHERE " + ATTRIB_REV_ORIGSTEP + " = " + this.getId();
			ResultSet rs = this.executeM4SqlRead(query);
			while (rs.next()) {
				long revStepId = rs.getLong(ATTRIB_REV_REVSTEP);
				Step revStep = (Step) this.getM4Db().getM4Object(revStepId, Step.class);
				this.myReversingSteps.add(revStep);
				revStep.theStepThisStepReverses = this;
			}
			rs.close();
		}
		catch (SQLException se) {
			throw new M4Exception("Step '" + this.getName() + "': SQL error reading reversing steps from DB: " + se.getMessage());
		}
		catch (M4Exception m4e) {
			throw new M4Exception("Step '" + this.getName() + "': M4 error reading reversing steps from DB: " + m4e.getMessage());
		}
	}
	
	private void writeThisStepAsReversedStepToDb() throws M4Exception {
		if (this.getDependentReversingSteps().isEmpty()) {
			return;
		}
		try {
			if ( ! this.reversingStepsTableExistsInM4()) {
				return;
			}			
			long reversedStepId = this.getId();
			Iterator it = this.myReversingSteps.iterator();
			while (it.hasNext()) {
				Step reversingStep = (Step) it.next();
				long reversingStepId = reversingStep.getId();
			
				// the link might already exist in the M4 schema:
				String query = "SELECT " + ATTRIB_REV_ORIGSTEP + 
								" FROM " + REVSTEP_TABLENAME +
								" WHERE " + ATTRIB_REV_REVSTEP + 
								" = " + reversingStepId +
								" AND " + ATTRIB_REV_ORIGSTEP +
								" = " + reversedStepId;
				ResultSet rs = this.executeM4SqlRead(query);
				if (rs.next()) {
					query = "UPDATE " + REVSTEP_TABLENAME + 
								" SET " + ATTRIB_REV_ORIGSTEP + " = " + reversedStepId + 
								" WHERE " + ATTRIB_REV_REVSTEP + " = " + reversingStepId;
				}
				else {
					long newId = this.getM4Db().getNextM4SequenceValue();
					query = "INSERT INTO " + REVSTEP_TABLENAME + 
							" (" + ATTRIB_REV_ID + ", " + ATTRIB_REV_ORIGSTEP + ", " + ATTRIB_REV_REVSTEP + 
							") VALUES (" + newId + ", " + reversedStepId + ", " + reversingStepId + ")";
				}
				this.executeM4SqlWrite(query);
			}
		}
		catch (DbConnectionClosed d) {
			throw new M4Exception("Step '" + this.getName() + "': DB connection closed when writing reversing step to DB: " + d.getMessage());
		}
		catch (SQLException se) {
			throw new M4Exception("Step '" + this.getName() + "': SQL error writing reversing step to DB: " + se.getMessage());
		}
		catch (M4Exception m4e) {
			throw new M4Exception("Step '" + this.getName() + "': M4 error writing reversing step to DB: " + m4e.getMessage());
		}
	}
	
	private boolean reversingStepsTableExistsInM4() throws M4Exception {
		return this.getM4Db().tableExistsInM4(REVSTEP_TABLENAME);
	}
	
	/**
	 * Steps must additionally export the links to the steps that reverse
	 * them, if any reversing steps exist.
	 */
	public Collection exportLocal(Writer out, Collection dependent) 
	throws M4Exception, IOException {
		// export the dependent Steps first!
		Collection theStepsThatReverseThisStep = this.getDependentReversingSteps();
		if (theStepsThatReverseThisStep == null || theStepsThatReverseThisStep.isEmpty()) {
			return null;
		}
		Collection ret = new Vector();
		Iterator it = theStepsThatReverseThisStep.iterator();
		while (it.hasNext()) {
			Step revStep = (Step) it.next();
			long xmlIdOfReversedStep = M4Xml.export(revStep, out, dependent);
			String xmlIdentification = M4Xml.putInXmlTags(xmlIdOfReversedStep + "", XmlInfo.TAG_XML_ID);
			String xmlDescription = M4Xml.putInXmlTags(xmlIdentification, EXPORT_TAG_REVERSING_STEPS);
			ret.add(xmlDescription);
		}
		return ret;
	}
	/**
	 * Steps may have to import the steps they reverse.
	 * @param tag the tag indicating the local field to be imported
	 * @param embedded the <code>String</code> between the opening and closing tag
	 */
	public void importLocal(String tag, String embedded)
	throws XmlException, M4Exception {
		if (tag.equals(EXPORT_TAG_REVERSING_STEPS)) {
			// don't import it if the current M4 schema doesn't support it!
			if ( ! this.reversingStepsTableExistsInM4()) {
				return;
			}
			String[] tags = M4Xml.stripOuterTag(embedded);
			if (tags == null) {
				return;
			}
			String xmlTag = tags[0];
			if ( ! xmlTag.equals(XmlInfo.TAG_XML_ID)) {
				throw new M4Exception("Error importing steps that reverse step '" + 
						this.getName() + "': found wrong tag in Step.importLocal()!");
			}
			String xmlId = tags[1];
			Object o = M4Xml.createObjectFromXml(xmlTag, xmlId);
			if ( ! (o instanceof Step)) {
				throw new M4Exception("Error importing steps that reverse step '" + 
						this.getName() + "': found wrong object type in Step.importLocal()!");
			}
			Step reversingStep = (Step) o;
			this.addReversingStep(reversingStep);
		}
	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#belongsToChainOrSubChain(Chain)
	 */
	public boolean belongsToChainOrSubChain(edu.udo.cs.miningmart.m4.Chain theChain) throws M4Exception {
		if (theChain == null) { // everything belongs to the Case (the parent of the toplevel chains)!
			return true;
		}
		if (this.getTheChain().equals(theChain)) {
			return true;
		}
		Iterator chIt = theChain.getDirectSubChains().iterator();
		while (chIt.hasNext()) {
			theChain = (Chain) chIt.next();
			if (this.belongsToChainOrSubChain(theChain)) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#hasPredecessorOutsideChain
	 */
	public boolean hasPredecessorOutsideChain() throws M4Exception {
		Iterator predIt = this.getAllPredecessors().iterator();
		Chain myChain = (Chain) this.getTheChain();
		while (predIt.hasNext()) {
			Step aStep = (Step) predIt.next();
			if ( ! aStep.belongsToChainOrSubChain(myChain)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * @see edu.udo.cs.miningmart.m4.Step#hasSuccessorOutsideChain
	 */
	public boolean hasSuccessorOutsideChain() throws M4Exception {
		Iterator succIt = this.getSuccessors().iterator();
		Chain myChain = (Chain) this.getTheChain();
		while (succIt.hasNext()) {
			Step aStep = (Step) succIt.next();
			if ( ! aStep.belongsToChainOrSubChain(myChain)) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#removePredecessor(String)
	 */
	public void removePredecessor() throws M4Exception {
		Step fromStep = (Step) this.getPredecessor();	
		this.getTheCase().removeStepDependency(fromStep, this);
	}
	
	public edu.udo.cs.miningmart.m4.Parameter getParameterTuple(String name, int loopNr) throws M4Exception {
		Collection col;
		Iterator it;
		col = this.getParameterTuples();
		if (name != null && col != null && (it = col.iterator()) != null) {
			while (it.hasNext()) {
				Parameter par = (Parameter) it.next();
				if (par != null && par.getName().startsWith(name) && par.getLoopNr() == loopNr) {
					return par;	
				}
			}
		}
		return null;
	}

	public edu.udo.cs.miningmart.m4.ParameterObject getSingleParameterObject(String parameterName) {
		ParameterObject[] pos = this.getParameter(parameterName, 0);
		if (pos == null || pos.length == 0) {
			return null;
		}
		return pos[0];
	}
	
	/*
	 * Gives the ParameterArray that aggregates the parameter tuples corresponding
	 * to the given OpParam and the given loop number.
	 * Helper method of getParameterDictionary.
	 * 
	 * @param theOperatorParameter the given OpParam
	 * @param loopNumber the given loop number
	 * @return a ParameterArray
	 */
	private edu.udo.cs.miningmart.m4.ParameterArray getParameterArray(edu.udo.cs.miningmart.m4.OpParam theOperatorParameter, int loopNumber) 
	throws M4Exception {
		Collection allParamTuples = this.getParameterTuples();
		Iterator tupleIt = allParamTuples.iterator();
		ParameterArray thePA = new edu.udo.cs.miningmart.m4.core.ParameterArray(theOperatorParameter.getType());
		while (tupleIt.hasNext()) {
			Parameter myParam = (Parameter) tupleIt.next();
			if (this.parameterNamesMatch(theOperatorParameter.getName(), myParam) &&
				myParam.getLoopNr() == loopNumber) {
				thePA.addParameter(myParam);
			}
		}
		if (thePA.size() == 0) {
			return null;
		}
		else {
			return thePA;
		}
	}
	
	// this method returns true iff the given name matches the name of the
	// given parameter (the latter may have a number suffix needed for the loop mechanism).
	private boolean parameterNamesMatch(String name, Parameter param) {
		String parName = param.getName();
		if (parName.startsWith(name)) { // if the names match at the beginning...
			String suffix = parName.substring(name.length());
			if (suffix.length() == 0) {
				return true;
			}
			try {
				Integer.parseInt(suffix); // ... then the rest must be a number!
				return true;
			}
			catch (NumberFormatException n) {
				return false;
			}			
		}
		return false;
	}
	
	/**
	 * @see Step#createParameter(String, M4Object, Operator, int, String)
	 */
	public edu.udo.cs.miningmart.m4.Parameter createParameterTuple(
		String name,
		edu.udo.cs.miningmart.m4.ParameterObject object,
		long number,
		int loopNumber,
		String ioType)
		throws M4Exception
	{
		// check that a Parameter with the given name is allowed in this step's operator:
		if (this.getTheOperator().getOpParam(this.getParameterNameWithoutSuffix(name)) == null) {
			throw new M4Exception("Step.createParameter(...): name '" + name +
					"' is not an allowed parameter of operator '" + this.getTheOperator().getName() +
					"'!");
		}
		Parameter par = (Parameter) this.getM4Db().createNewInstance(Parameter.class);

		par.setParameterName(name);
		par.setParamNr(number);
		par.setInputParam(ioType);
		par.setTheStep(this); // takes care of back-reference
		par.setTheParameterObject(object);
		par.setTheOperator(this.getTheOperator());
		par.setLoopNr(loopNumber);
		
		return par;
	}
	
	/**
	 * @see Step#createValueParameter(String, String, String, Operator, int, String)
	 */
	public edu.udo.cs.miningmart.m4.Parameter createValueParameterTuple(
		String value,
		String datatype,
		String name,
		long number,
		int loopNumber,
		String type)
		throws M4Exception 
	{
		Value newVal = (Value) this.getM4Db().createNewInstance(Value.class);
		newVal.setName(name);
		newVal.setType(datatype);
		newVal.setValue(value);

		return this.createParameterTuple(name, newVal, number, loopNumber, type);
	}
	
	/**
	 * Make and return a copy of this Step that is attached to the
	 * given Chain in the given Case.
	 * 
	 * @see Step#copy(Case, Chain)
	 */
	public edu.udo.cs.miningmart.m4.Step copy(edu.udo.cs.miningmart.m4.Case newCase, 
			                               edu.udo.cs.miningmart.m4.Chain newChain)
		throws M4Exception {
		
		Step theCopy = new Step(this.getM4Db());
		
		try {
			theCopy.setTheCase(newCase);
			theCopy.setTheChain(newChain);
			
			// theCopy.setFunctionallyValid(this.isFunctionallyValid());
			theCopy.setLoopCount(this.getLoopCount());
			theCopy.setMultiStepCondition(this.getMultiStepCondition());
			String nameOfCopy = newChain.getValidName(this.getName(), Step.class);
			theCopy.setName(nameOfCopy);
			theCopy.setNumber(this.getNumber());
			theCopy.setTheOperator(this.getTheOperator());
			// theCopy.setStepLoopNr(this.getStepLoopNr());
			theCopy.setNumber(this.getNumber());
		}
		catch (M4Exception m4e)
		{   throw new M4Exception("Could not create copy of Step " + this.getId() + ": " + m4e.getMessage());  }
		
		return theCopy;
	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#getParameter(OpParam)
	 */
	public Collection getParameter(edu.udo.cs.miningmart.m4.OpParam theOpParam, int loopNr) throws M4Exception {
		ParameterObject[] parObjArray = this.getParameter(theOpParam.getName(), loopNr);
		
		if (parObjArray == null) {
			return null; // parameter does not to exist or other error
		}
		else {
			Vector vec = new Vector();
			for (int i=0; i<parObjArray.length; i++)
				vec.add(parObjArray[i]);
			return vec;
		}
	}
	
	/**
	 * An empty <code>Collection</code> for <code>theParameterObjects</code>
	 * makes this method remove the parameter (as otherwise the type would be
	 * unclear, anyway). For output feature parameters the old names of the 
	 * parameter objects are returned in a map from the parameters to strings. 
	 * 
	 * 
	 * @see edu.udo.cs.miningmart.m4.Step#setParameter(OpParam, Collection)
	 */
	public Map setParameter(edu.udo.cs.miningmart.m4.OpParam theOpParam, Collection theParameterObjects, int loopNr) throws M4Exception {
				
		if (theParameterObjects == null) {
			throw new M4Exception("Step.setParameter() was called with >null< Collection 'theParameterObjects'!");
		}		
		
		// sometimes value objects are reused by the GUI where they
		// shouldn't be reused because they are already waiting for deletion:
		if (theOpParam.isValueParameter()) {
			Vector newValueObjects = new Vector();
			Iterator it = theParameterObjects.iterator();
			while (it.hasNext()) {
				ParameterObject parObj = (ParameterObject) it.next();
				if ( ! (parObj instanceof Value)) {
					throw new M4Exception("Step.setParameter() was called with a Value OpParam but non-Value parameter objects!");
				}
				Value v = (Value) this.getM4Db().createNewInstance(Value.class);				
				v.setDocumentation(parObj.getDocumentation());
				v.setType(((Value) parObj).getTypeName());
				v.setValue(((Value) parObj).getValue());
				v.setName(parObj.getName());
				newValueObjects.add(v);
			}
			theParameterObjects = newValueObjects;
		}
		
		/* Second implementation... first one see below! */
		
		final String parameterName = theOpParam.getName();
		edu.udo.cs.miningmart.m4.ParameterArray paOld = null;

		Map mapParametersToOldNames = new HashMap();
		
		paOld = this.getParameterArray(theOpParam, loopNr);
		
		boolean needsSuffix = theOpParam.isArray() || theOpParam.isLoopable() || (loopNr > 0) || (paOld != null && paOld.size() > 1) || (theParameterObjects.size() > 1);
		final String inputOutputString = theOpParam.getInput();
		// if we set the input concept, we may have some work to do, so we save
		// some information in the following two variables:
		ParameterObject oldParameterObject = null, newParameterObject = null;

		// see if this parameter existed before:
		if (paOld == null) {
			// Parameter must be created.
			// Create a new (set of) parameter(s). We only need to 
			// create the tuples.
			if ( ! theParameterObjects.isEmpty()) {				
				Iterator it = theParameterObjects.iterator();
				while (it.hasNext()) {
					ParameterObject parObj = (ParameterObject) it.next();
					newParameterObject = parObj;
					long freeParNum = this.getFreeParameterNumber();
					int freeSuffixNum = this.getFreeParameterSuffix(theOpParam);
					this.createParameterTuple((needsSuffix ? parameterName + freeSuffixNum : parameterName),
								              parObj, freeParNum, loopNr, inputOutputString);
				}
			}
		}
		else {
			// parameter existed before:
			Object[] copyOfOldParams = paOld.getParameters().toArray();
			Object[] copyOfParObjs = theParameterObjects.toArray();
			
			// overwrite the old parameters:
			int index = 0;
			while (index < copyOfOldParams.length && index < copyOfParObjs.length) {
				Parameter oldPar = (Parameter) copyOfOldParams[index];
				oldParameterObject = (ParameterObject) oldPar.getTheParameterObject(); // for later we need only one
				ParameterObject newParObj = (ParameterObject) copyOfParObjs[index];
				if ( ! theOpParam.isInput()) { 
					// the old object may be equal to a new one:
					if ( ! this.checkOccurrence(oldParameterObject, copyOfParObjs)) {
						this.handleOldOutput(oldPar, mapParametersToOldNames, newParObj);
						// before changing the parameter object to the new object,
						// check that no output with the same name as the new object 
						// is ever used in a following step:
						this.giveValidOutputObjectName(newParObj, newParObj.getName());
					}
				}
				else {
					// before changing an old input parameter, see if output depends on it 
					// and delete it:
					this.handleOutputCreatedByOldInput(theOpParam, oldPar);
				}
				newParameterObject = newParObj;
				oldPar.setTheParameterObject(newParObj);
				index++;
			}
			// if there were more old parameters, delete them:
			while (index < copyOfOldParams.length) {
				Parameter oldPar = (Parameter) copyOfOldParams[index];
				oldParameterObject = (ParameterObject) oldPar.getTheParameterObject(); // for later we need only one
				if ( ! theOpParam.isInput()) { 
					this.handleOldOutput(oldPar, mapParametersToOldNames, null);
				}
				this.removeParameterTuple(oldPar);
				oldPar.deleteSoon(); // should delete M4 references
				index++;
			}
			// if there are extra new parameters, create them:
			while (index < copyOfParObjs.length) {
				ParameterObject newParObj = (ParameterObject) copyOfParObjs[index];
				long freeParameterNumber = this.getFreeParameterNumber();
				int freeSuffixNum = this.getFreeParameterSuffix(theOpParam);
				this.createParameterTuple((needsSuffix ? parameterName + freeSuffixNum : parameterName),
			             				  newParObj, freeParameterNumber, loopNr, inputOutputString);
				index++;
			}
		}		
		
		// if an input concept has changed, there is some work to do:
		if (oldParameterObject != null && theOpParam.isInput() && theOpParam.isConceptParameter() &&
			( ! oldParameterObject.equals(newParameterObject))) {
			
			if ( ( ! (oldParameterObject instanceof Concept)) ||
			     (newParameterObject != null && ( ! (newParameterObject instanceof Concept)))) {
				throw new M4Exception("Step.setParameter(): found something that is not a concept as (old) parameter object of a concept parameter!");
			}
			
			// keep the changed concepts for later (see method handleOldInput()):
			this.oldInputConcept = (Concept) oldParameterObject;
			this.replacingConcept = (Concept) newParameterObject;
		}

		return mapParametersToOldNames;
		
		/* ---
		 * First implementation from here on;
		 * it deletes and creates anew, but this led to problems with the store mechanism...!
		 * 
		// Step 1: Delete the old parameter tuples
		final String parameterName = theOpParam.getName();
		edu.udo.cs.miningmart.m4.ParameterArray paOld = pd.get(parameterName);
		pd.removeParamArray(paOld);
		Iterator itOld = (new Vector(this.getParameters())).iterator();
		long maxParNum = 0; // needed in 2nd step
		while (itOld.hasNext()) {
			Parameter par = (Parameter) itOld.next();
			if (par.getParameterName().startsWith(parameterName)) {
				maxParNum = Math.max(maxParNum, par.getParamNr());
				par.setParamArray(null);
				par.setTheParameterObject(null);
				par.setTheStep(null);
				par.deleteSoon();
			}
		}
		
		// Step 2: Create a new (set of) parameter(s) and update for both access methods.
		// For the access via ParamDict a ParameterArray containing all parameters needs to be set up.
		// The M4 generic getter access needs the parameters of an array to have 
		if ( ! theParameterObjects.isEmpty()) {
			ParameterObject parObj = (ParameterObject) theParameterObjects.iterator().next();
			final String typeString = Parameter.getTypeStringForParObject(parObj);
			ParameterArray pArr = new ParameterArray(typeString);			
			boolean needsSuffix = (theParameterObjects.size() > 1);
			
			Iterator it = theParameterObjects.iterator();
			while (it.hasNext()) {
				parObj = (ParameterObject) it.next();
				// Similar to database sequences: increment highest parameter number so far as the next number:
				maxParNum++;   
				edu.udo.cs.miningmart.m4.Parameter newPar =
					this.createParameter((needsSuffix ? parameterName + maxParNum : parameterName),
							             parObj, this.getTheOperator(), maxParNum, typeString);			
			
				pArr.addParameter(newPar);
			}
			
			pd.put(theOpParam.getName(), loopNr, pArr);
		}
		*/
	}
	
	// used by setParameter()
	private void handleOldOutput(
			edu.udo.cs.miningmart.m4.Parameter oldOutput,
			Map mapParametersToOldNames,
			edu.udo.cs.miningmart.m4.ParameterObject newObject) 
	throws M4Exception {
		if (oldOutput == null || mapParametersToOldNames == null) 
			return;
		if (mapParametersToOldNames.get(oldOutput) != null) 
			return;
		ParameterObject oldParObj = (ParameterObject) oldOutput.getTheParameterObject();
		if (oldParObj == null) 
			return;
		if (oldParObj instanceof Feature) {
			Feature outF = (Feature) oldParObj;
			// get the parameters where it is used as input:
			Collection inputParamsUsingOutF = this.getInputParametersUsingFeature(outF);
			String newName = (newObject != null ? newObject.getName() : null);
			this.createMapOfInputParamsToNames(
					inputParamsUsingOutF, 
					mapParametersToOldNames, 
					oldParObj.getName(),
					newName);
			oldParObj.deleteSoon();
		}
	}
	
	/**
	 * The given input operator parameter is about to be changed; the new parameter
	 * object is attached to oldPar. This method removes output parameter
	 * objects that are dependent on the old input parameter object.
	 * 
	 * @param inputOpParam Input operator parameter 
	 * @param oldPar parameter with old parameter object
	 */
	private void handleOutputCreatedByOldInput(
			edu.udo.cs.miningmart.m4.OpParam inputOpParam,
			Parameter oldPar) throws M4Exception {
		Collection constraints = inputOpParam.getApplicableConstraints();
		Iterator it = constraints.iterator();
		while (it.hasNext()) {
			Constraint inputConstraint = (Constraint) it.next();
			// find output features that have been created by this constraint
			// and delete them:
			if (oldPar.getTheParameterObject() instanceof BaseAttribute) {
				BaseAttribute inputBA = (BaseAttribute) oldPar.getTheParameterObject();
				Map namesMap = this.getMapOfNamesToNamesWithSuffixes(inputConstraint, new String[] { inputBA.getName() });
				if (namesMap == null)
					continue;
				Collection names = (Collection) namesMap.get(inputBA.getName());
				if (names == null)
					continue;
				this.deleteFeaturesFromConcept(names, (Concept) this.getOutputConcept());
			}
		}
	}
	
	private void deleteFeaturesFromConcept(Collection namesOfFeatures, Concept theConcept) 
	throws M4Exception {
		if (namesOfFeatures == null || namesOfFeatures.isEmpty() || theConcept == null)
			return;
		Collection bas = theConcept.getAllBaseAttributes();
		Iterator it = bas.iterator();
		while (it.hasNext()) {
			BaseAttribute ba = (BaseAttribute) it.next();
			if (namesOfFeatures.contains(ba.getName())) {
				theConcept.removeBaseAttribute(ba.getName());
			}
		}
	}
	
	// ensures that the given object gets a name that is never used by other
	// output objects of following steps
	private void giveValidOutputObjectName(ParameterObject newOutputObject, String newName)
	throws M4Exception {
		// concepts must have a globally unique name:
		if (newOutputObject instanceof Concept) {
			String validName = ((Case) this.getTheCase()).getValidName(newName, Concept.class, newOutputObject);
			newOutputObject.setName(validName);
			return;
		}
		// BaseAttributes must have a name that is unique in their concept,
		// and since the output attribute of this step occurs in the input of
		// following steps, it must have a name that is valid there, too!
		if (newOutputObject instanceof Feature) {
			// go through all output attributes created by any
			// following steps and get a valid name:
			Collection allOutputNames = this.getAllNamesOfOutputAttribsOfDependentSteps();
			String validName = this.getUniqueName(newName, allOutputNames);
			newOutputObject.setName(validName);
		}
	}
	
	// returns all names of output attributes of followers of this step
	private Collection getAllNamesOfOutputAttribsOfDependentSteps() 
	throws M4Exception {
		Collection ret = new Vector();
		Iterator succIt = this.getSuccessors().iterator();
		while (succIt.hasNext()) {
			Step succStep = (Step) succIt.next();
			succStep.getNamesOfOutputAttribs(ret);
		}
		return ret;
	}
	
	private void getNamesOfOutputAttribs(Collection namesSoFar) throws M4Exception {
		if (this.getTheOperator() == null) {
			return;
		}
		Collection outOpParams = this.getTheOperator().getAllOutputOperatorParameters();
		Iterator it = outOpParams.iterator();
		while (it.hasNext()) {
			OpParam anOpPar = (OpParam) it.next();
			if (anOpPar.isBaseAttribParameter() || anOpPar.isFeatureParameter()) {
				Collection attribs = this.getParameterObjects(anOpPar.getName());
				Iterator attrIt = attribs.iterator();
				while (attrIt.hasNext()) {
					Feature f = (Feature) attrIt.next();
					if ( ! namesSoFar.contains(f.getName()))
						namesSoFar.add(f.getName());
				}
			}
		}
		Iterator succIt = this.getSuccessors().iterator();
		while (succIt.hasNext()) {
			Step succStep = (Step) succIt.next();
			succStep.getNamesOfOutputAttribs(namesSoFar);
		}
	}
	
	// returns a collection of Parameters that use the given feature, or a copy
	// of it that is made in any (indirect) successor step of this step, as input parameter.
	private Collection getInputParametersUsingFeature(Feature theFeature) throws M4Exception {
		if (theFeature == null) {
			return new Vector();
		}
		Collection paramsUsingFeatureDirectly = this.getInputParametersFor(theFeature);
		
		// also find copies of the feature that are used as input params
		// in successors of this step:
		Iterator succIt = this.getSuccessors().iterator();
		while (succIt.hasNext()) {
			Step successor = (Step) succIt.next();
			Collection copiesOfTheFeature = successor.getCorrespondingOutputFeatures(theFeature.getConcept(), theFeature);
			if (copiesOfTheFeature != null) {
				Iterator copyIt = copiesOfTheFeature.iterator();
				while (copyIt.hasNext()) {
					Feature copyOfTheFeature = (Feature) copyIt.next();
					Collection additionalParams = successor.getInputParametersUsingFeature(copyOfTheFeature);
					Iterator addIt = additionalParams.iterator();
					while (addIt.hasNext()) {
						Parameter aPar = (Parameter) addIt.next();
						if ( ! paramsUsingFeatureDirectly.contains(aPar))
							paramsUsingFeatureDirectly.add(aPar);
					}
				}
			}
		}
		
		return paramsUsingFeatureDirectly;
	}
	
	private Collection getInputParametersFor(Feature f) throws M4Exception {
		Collection paramsUsingFeatureDirectly = f.getParameterReferences();
		Collection ret = new Vector();
		Iterator directIt = paramsUsingFeatureDirectly.iterator();
		while (directIt.hasNext()) {
			Parameter onePar = (Parameter) directIt.next();
			if (onePar.isInputParam()) {
				ret.add(onePar);
			}
		}
		return ret;
	}
	
	// add entries to the given map that map the given input parameters to
	// a two-element String[]; the first element is the old name, the second element
	// the given new name (may be null)
	private void createMapOfInputParamsToNames(
			Collection inputParams, 
			Map mapParametersToNames, 
			String oldName,
			String newName) 
	throws M4Exception {
		if (oldName == null) 
			return;
		Iterator parametersUsingOutF_It = inputParams.iterator();
		while (parametersUsingOutF_It.hasNext()) {
			Parameter aPar = (Parameter) parametersUsingOutF_It.next();
			if ((aPar != null) && (aPar.getTheStep() != null) && ( ! aPar.getTheStep().equals(this))) {
				if (aPar.isInputParam()) {
					// keep the info that this parameter object was deleted:
					if (aPar.getTheParameterObject() != null) {
						String oldNameWithLoop = this.addLoopNumberToName(oldName, aPar.getLoopNr());
						String newNameWithLoop = (newName != null ? this.addLoopNumberToName(newName, aPar.getLoopNr()) : null);
						String[] pairOfOldAndNewName = new String[] { oldNameWithLoop, newNameWithLoop };
						mapParametersToNames.put(aPar, pairOfOldAndNewName);
					}
				}
			}
		}
	}
	
	// Decides whether the given object occurs in the given array
	private boolean checkOccurrence(Object anObject, Object[] anArray) {
		for (int i = 0; i < anArray.length; i++) {
			if (anArray[i].equals(anObject)) {
				return true;
			}
		}
		return false;
	}

	// Decides whether the given object occurs in the given collection
	private boolean checkOccurrence(Object anObject, Collection aColl) {
		if (aColl != null) {
			Iterator it = aColl.iterator();
			while (it.hasNext()) {
				Object myObj = it.next();
				if (anObject.equals(myObj)) {
					return true;
				}
			}
		}
		return false;
	}
	
	// returns the highest loop number that any parameter tuple
	// in this step has, for the given OpParam
	private int getHighestLoopNr(edu.udo.cs.miningmart.m4.OpParam theOpParam) throws M4Exception {
		Iterator parIt = this.getParameterTuples(theOpParam).iterator();
		int highestLoopNr = 0;
		while (parIt.hasNext()) {
			Parameter myPar = (Parameter) parIt.next();
			if (myPar.getLoopNr() > highestLoopNr) {
				highestLoopNr = myPar.getLoopNr();
			}
		}
		return highestLoopNr;
	}
	
	private long getFreeParameterNumber() throws M4Exception {
		Collection theTuples = this.getParameterTuples();
		boolean found = false;
		long candidateNumber = 0;
		while ( ! found) {
			Iterator it = theTuples.iterator();
			boolean changed = false;
			while (it.hasNext()) {
				Parameter myPar = (Parameter) it.next();
				long number = myPar.getParamNr();
				if (number == candidateNumber) {
					candidateNumber++;
					changed = true;
				}
			}
			if ( ! changed) {
				found = true;
			}
		}
		return candidateNumber;
	}
	
	private int getFreeParameterSuffix(edu.udo.cs.miningmart.m4.OpParam anOpPar) throws M4Exception {
		Collection theTuples = this.getParameterTuples(anOpPar);
		boolean found = false;
		int candidateSuffix = 0;
		while ( ! found) {
			Iterator it = theTuples.iterator();
			boolean changed = false;
			while (it.hasNext()) {
				Parameter myPar = (Parameter) it.next();
				int suffix = this.getParameterNameSuffix(myPar);
				if (suffix == candidateSuffix) {
					candidateSuffix++;
					changed = true;
				}
			}
			if ( ! changed) {
				found = true;
			}
		}
		return candidateSuffix;
	}
	
	private int getParameterNameSuffix(Parameter aPar) {
		String parName = aPar.getName();
		int pos = parName.length() - 1;
		while (Character.isDigit(parName.charAt(pos))) {
			pos--;
		}
		if (pos == parName.length() - 1) {
			return -1;
		}
		return Integer.parseInt(parName.substring(pos + 1));
	}

	private String getParameterNameWithoutSuffix(String parName) {
		int pos = parName.length() - 1;
		while (Character.isDigit(parName.charAt(pos))) {
			pos--;
		}
		if (pos == -1) {
			return null;
		}
		return parName.substring(0, pos + 1);
	}
	
	/*
	 * This method deletes the conceptual output objects 
	 * created by this step. See Step.deleteSoon().
	 */
	private void deleteOutputs() throws M4Exception {
		Operator op = (Operator) this.getTheOperator();
		if (op != null) { // op == null can happen due to load mechanism
			Iterator outputOpParIt = op.getAllOutputOperatorParameters().iterator();
			int loops = this.getLoopCount();
			if (loops == 0) {
				loops = 1;
			}		
			boolean thisStepHadAnOutputConcept = false;
			while (outputOpParIt.hasNext()) {
				OpParam outOpPar = (OpParam) outputOpParIt.next();
				int startLoop = (outOpPar.isLoopable() && loops > 1 ? 1 : 0);
				int endLoop = loops;
				for (int loopNr = startLoop; loopNr <= endLoop; loopNr++) {
					Collection parObjs = this.getParameter(outOpPar, loopNr);
					if (parObjs != null) {					
						Iterator parObjsIt = parObjs.iterator();
						while (parObjsIt.hasNext()) {
							ParameterObject myParObj = (ParameterObject) parObjsIt.next();
							myParObj.deleteSoon();					
						}
					}
				}
				if (outOpPar.isConceptParameter()) {
					thisStepHadAnOutputConcept = true;
				}
			}
			// if only features are deleted, update dependent steps:
			if ( ! thisStepHadAnOutputConcept) {
				// ? this.adaptOutputToChangedInput(); // to adapt the input concept(s)
				this.propagateOutputChanges();
			}
		}
	}
	
	/**
	 * To be called after changing the input of this step; ensures
	 * that old output attributes are removed from a replaced input concept.
	 * 
	 * @throws M4Exception
	 */
	public void handleOldInput() throws M4Exception {
		if (this.oldInputConcept == null || this.replacingConcept == null) {
			return;
		}
		this.makeInputConceptChanges(this.oldInputConcept, this.replacingConcept);
		this.oldInputConcept = null;
		this.replacingConcept = null;
	}
	
	/*
	 * This method scans the chain of steps, starting in this step, for output attributes 
	 * of the given concept. They are removed from the concept.
	 */
	private void makeInputConceptChanges(
			edu.udo.cs.miningmart.m4.Concept oldInputConcept,
			edu.udo.cs.miningmart.m4.Concept newInputConcept) throws M4Exception {
		
		if (oldInputConcept != null && newInputConcept != null && this.getTheOperator() != null) {
			// see if any successors use the old input concept:
			Collection successorsWithSameOldInputConcept = this.getSuccessorsUntilOutputConceptIsCreated();
			if (successorsWithSameOldInputConcept != null && ( ! successorsWithSameOldInputConcept.isEmpty())) {
				// ask user...
				String message = "Some successor steps use the previous input concept '" + 
									oldInputConcept.getName() + 
									"':\n";
				Iterator succIt = successorsWithSameOldInputConcept.iterator();
				while (succIt.hasNext()) {
					Step mySuccessor = (Step) succIt.next();
					if ( ! mySuccessor.equals(this))
						message += " - " + mySuccessor.getName() + "\n";
				}
				message += "Should it be replaced by the new concept '" + 
							newInputConcept.getName() + "' in those steps, too?";
				int answer = JOptionPane.showConfirmDialog(null, message, "Replace concepts?", JOptionPane.YES_NO_OPTION);
				
				// if confirmed:
				if (answer == JOptionPane.YES_OPTION) {
					// go through the successor steps concerned:
					succIt = successorsWithSameOldInputConcept.iterator();
					while (succIt.hasNext()) {
						Step mySuccessor = (Step) succIt.next();
						// 1. change the input of the successor:
						mySuccessor.changeInputConcept(oldInputConcept, newInputConcept);
						// 2. check all output attribs of the successor step:
						Iterator outputOpParIt = mySuccessor.getTheOperator().getAllOutputOperatorParameters().iterator();						
						while (outputOpParIt.hasNext()) {
							OpParam outOpPar = (OpParam) outputOpParIt.next();
							if (outOpPar.isBaseAttribParameter() || outOpPar.isFeatureParameter()) {
								Collection coll = this.getParameterTuples(outOpPar);
								Iterator it = coll.iterator();
								while (it.hasNext()) {
									Parameter myPar = (Parameter) it.next();
									ParameterObject myParObj = (ParameterObject) myPar.getTheParameterObject();
									// remove output attrib (because it was created by a step that no
									// longer applies to that input concept):
									if (oldInputConcept.hasFeature((Feature) myParObj)) {
										oldInputConcept.removeFeature((Feature) myParObj);
									}
								}
							}			
						}
						// do not adapt output of successor to changed input,
						// as this is done by a global call to 'propagateOutputChanges'.
					} // end of loop through concerned successors
				}
			}
		}		
	}
	
	/*
	 * This method collects all output attributes that this step and its successors create on the
	 * same input concept. That is, the successors are only searched until the first step that 
	 * creates its own output concept. 
	 *
	private void getOutputAttribsOfThisAndFollowingSteps(Collection attribs) throws M4Exception {
		
		// if this step has an output concept, no new attribs are collected:
		if (this.getOutputConcept() != null) {
			return;
		}
		
		// otherwise add all output attribs of this step to the given collection:
		Iterator outputOpParIt = this.getTheOperator().getAllOutputOperatorParameters().iterator();
		
		while (outputOpParIt.hasNext()) {
			OpParam outOpPar = (OpParam) outputOpParIt.next();
			if (outOpPar.isBaseAttribParameter() || outOpPar.isFeatureParameter()) {
				Collection coll = this.getParameterTuples(outOpPar);
				Iterator it = coll.iterator();
				while (it.hasNext()) {
					Parameter myPar = (Parameter) it.next();
					attribs.add(myPar.getTheParameterObject());
				}
			}			
		}
		
		// collect from following steps:
		Iterator succStepsIt = this.getSuccessors().iterator();
		while (succStepsIt.hasNext()) {
			Step mySucc = (Step) succStepsIt.next();
			mySucc.getOutputAttribsOfThisAndFollowingSteps(attribs);
		}
	} */
	
	/*
	 * This method assumes that the input concept of this step has changed.
	 * It checks the Features that are used as input parameters.
	 * It tries to find Features in the new input concept that correspond to the 
	 * Features from the old input concepts that were used as input parameters so far, 
	 * and sets them as new parameters.
	 *
	private void checkInputAttributes(edu.udo.cs.miningmart.m4.Concept oldConcept) throws M4Exception {
		Iterator inputOpParIt = this.getTheOperator().getAllInputOperatorParameters().iterator();
		int loops = this.getLoopCount();
		if (loops == 0) {
			loops = 1;
		}
		while (inputOpParIt.hasNext()) {
			OpParam inputOpPar = (OpParam) inputOpParIt.next();
			if (inputOpPar.isFeatureParameter() || inputOpPar.isBaseAttribParameter()) {
				for (int loopNr = 0; loopNr < loops; loopNr++) {
					Iterator oldParsIt = this.getParameter(inputOpPar, loopNr).iterator();
					while (oldParsIt.hasNext()) {
						Vector correspondingFeatures = new Vector();
						ParameterObject myParObj = (ParameterObject) oldParsIt.next();
						if ( ! (myParObj instanceof Feature)) {
							throw new M4Exception ("Step '" + this.getName() + "': found something that is not a feature as parameter object of '" + inputOpPar.getName() + "'!");
						}
						Feature oldF = (Feature) myParObj;
						if ( ! oldConcept.hasFeature(oldF)) {
							// nothing to do, feature seems to be changed already, or belong
							// to a different input concept
						}
						else {
							// here we need to find a corresponding feature in the current
							// input concepts:
							Iterator inConsIt = this.getInputConceptsThatAreParameters().iterator();
							while (inConsIt.hasNext()) {
								Concept anInputConcept = (Concept) inConsIt.next();
								Feature inF = (Feature) anInputConcept.getFeature(oldF.getName());							
								if (inF != null) {
									correspondingFeatures.add(inF);
								}
							}
						}
						if ( ! correspondingFeatures.isEmpty()) {
							this.setParameter(inputOpPar, correspondingFeatures, loopNr);
						}
						else {
							throw new UserError("Step '" + this.getName() + 
									"': an input concept has changed. Please make sure that the parameter '" +
									inputOpPar.getName() + "' is adjusted (no automatic adjustment was possible).");
						}
					}
				}
			}			
		}
	}
	*/
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#checkInputParameterEntries()
	 */
	public boolean checkInputParameterEntries() throws M4Exception {
		
		// Ungetestet:
		// * parameter-spezifikation pruefen (array/single, optional, etc.)
		// * vorhandensein aller werte in allen loops ueberpruefen

		// TODO:
		// constraints abfragen und ueberpruefen		
		// Bei Loops noch zu pr??fen: Ist die loop-Nummer bei "nicht looped" gleich 0,
		// f??ngt sie bei loops bei 1 an und geht bis (also kleiner/gleich) this.getLoopCount() ??

 		String errorMsg = null; // A value different from null indicates that an error has been found.
	
		final int numLoops = this.getLoopCount();
		
		Collection opParams = this.getTheOperator().getOpParams();
		Iterator it = opParams.iterator();
		while (it.hasNext()) {
			OpParam opPar = (OpParam) it.next();

			// check only input params:
			if (opPar.isInput()) {
				// 	identify parameter by name
				String parName = opPar.getName();
			
				if ((numLoops > 0) && opPar.isLoopable()) {
					for (int i=1; i<=numLoops; i++) {
						ParameterObject[] pArray = this.getParameter(parName, i);
						if (pArray == null) {
							throw new M4Exception("Step '" + this.getName() + "', Parameter '" + parName +
									"', Loop " + i + ": no parameter object(s) found!");
						}
						errorMsg = this.checkSingleLoopOpParam(opPar, pArray);
						if (errorMsg != null)
							errorMsg += " (loop: " + i + ") ";
					}
				}
				else {					
					/* coordinated arrays are stored in ParameterDictionary under loopnr 0!
					if (opPar.isCoordinated()) {
						// this means that the loop mechanism is exploited for coordination of parameter values
						int loopNr = 1; 
						ParameterObject[] pArray = this.getParameter(parName, loopNr);
						while (pArray != null) { // number of parameter values is not known
							errorMsg = this.checkSingleLoopOpParam(opPar, pArray);
							loopNr++;
						}
					}*/
					errorMsg = this.checkSingleLoopOpParam(opPar, this.getParameter(parName, 0));
				}
			}
		}

		// Proceed only, if no error has been found up to here.
		if (errorMsg == null) {
			// Check constraints from table OP_CONSTR_T
			Collection allConstraints = this.getTheOperator().getConstraints();
			Map datatypeConstraints = new HashMap();			
			if ((allConstraints != null) && ((it = allConstraints.iterator()) != null)) {
				while (it.hasNext() && (errorMsg == null)) {
					Constraint constraint = (Constraint) it.next();
					// Special treatment for TYPE-Constraints: we collect
					// them in a Map to see if more than one TYPE-Constraint holds
					// for any of the input parameters. If yes it means this parameter
					// must have one of the types (disjunctive interpretation of constraints).
					if (constraint.getType().equals(Constraint.TYPE_DATATYPE)) {
						// the map maps obj1 to a list of its type constraints:
						Vector listOfConstraints = (Vector) datatypeConstraints.get(constraint.getObj1());
						if (listOfConstraints == null) {
							listOfConstraints = new Vector();
							datatypeConstraints.put(constraint.getObj1(), listOfConstraints);
						}
						listOfConstraints.add(constraint);
					}
					else {
						errorMsg = this.checkInputTypeConstraint(constraint);
					}
				}
			}
			if (errorMsg == null) {
				// now go through the TYPE constraints and check them:
				Iterator typeConstrIt = datatypeConstraints.entrySet().iterator();
				while (typeConstrIt.hasNext()) {
					Map.Entry myEntry = (Map.Entry) typeConstrIt.next();
					Vector listOfConstraints = (Vector) myEntry.getValue();
					if (listOfConstraints.size() == 1) {
						errorMsg = this.checkInputTypeConstraint((Constraint) listOfConstraints.get(0));
						if (errorMsg != null) { 
							break;
						}
					}
					else {
					// 	here listOfConstraints contains more than one constraint
					// 	(all for the same parameter obj1)!
						String obj1 = (String) myEntry.getKey();
						OpParam obj1OpPar = (OpParam) this.getTheOperator().getOpParam(obj1);
						int loopNum = (obj1OpPar.isLoopable() && this.getLoopCount() > 0) ? 1 : 0;
						ParameterObject[] pArray = this.getParameter(obj1, loopNum);
						if (pArray == null || pArray.length == 0 || pArray[0] == null) {
							if (obj1OpPar.isOptional()) {
								break; // parameter is missing but optional, so it cannot
										// break this constraint
							}
							else {
								throw new UserError("Constraint " + Constraint.TYPE_DATATYPE + " violated. No object(s) for mandatory parameter '" + obj1 + "' found!");					
							}
						}
						// check that one constraint is fulfilled for each parameter object:
						boolean oneConstraintFulfilledByEachObject = true;
						for (int i = 0; i < pArray.length; i++) {
							Iterator constrIt = listOfConstraints.iterator();
							boolean oneConstraintFulfilledByCurrentObject = false;
							while (constrIt.hasNext()) {
								Constraint myConstr = (Constraint) constrIt.next();
								String localErrorMsg = this.checkDatatypeConstraint(pArray[i], myConstr.getObj2());
								if (localErrorMsg == null) {
									oneConstraintFulfilledByCurrentObject = true;
								}
								else {
									errorMsg = localErrorMsg;
								}
							}
							if ( ! oneConstraintFulfilledByCurrentObject) {
								oneConstraintFulfilledByEachObject = false;
							}
						}
						if (oneConstraintFulfilledByEachObject) {
							errorMsg = null;
						}
					}
				}
			}
		}

		// special Hack for MRFC:
		if (errorMsg != null && 
			this.getTheOperator().getName().equals("MultiRelationalFeatureConstruction")) {
			// Check the IN constraints again.
			// First collect the parameter objects:
			Collection allConstraints = this.getTheOperator().getConstraints();
			Vector inputConcepts = new Vector();
			Vector attribsInInputConcept = new Vector();
			if ((allConstraints != null) && ((it = allConstraints.iterator()) != null)) {
				while (it.hasNext()) {
					Constraint constraint = (Constraint) it.next();
					if (constraint.getType().equals(Constraint.TYPE_CONTAINED_IN)) {
						ParameterObject[] concepts = getParameter(constraint.getObj2(), 0);
						ParameterObject[] attribs = this.getParameter(constraint.getObj1(), 0);
						if (concepts != null) { // output concept may not exist yet
							for (int i = 0; i < attribs.length; i++) {
								attribsInInputConcept.add(attribs[i]);
							}
							for (int i = 0; i < concepts.length; i++) {
								inputConcepts.add(concepts[i]);
							}
						}
					}
				}
			}
			// Second check that all input attributes belong to one of the input concepts:
			Iterator attrIt = attribsInInputConcept.iterator();
			errorMsg = null;
			while (attrIt.hasNext()) {
				BaseAttribute myBA = (BaseAttribute) attrIt.next();
				Iterator concIt = inputConcepts.iterator();
				boolean found = false;
				while (concIt.hasNext()) {
					Concept myCon = (Concept) concIt.next();
					if (myCon.getBaseAttribute(myBA.getName()) != null) {
						found = true;
					}
				}
				if ( ! found) {
					errorMsg = "Step '" + this.getName() + 
						"': BaseAttribute '" + myBA.getName() + "' does not belong to any of the input concepts!";
				}
			}
		}
		
		if (errorMsg == null) {
			return true;
		}
		else throw new UserError(errorMsg + " in step '" + this.getName() + "' (Id " + this.getId() + ")");
	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#checkOutputParameterEntries()
	 */
	public boolean checkOutputParameterEntries() throws M4Exception, UserError {
		Collection out = this.getTheOperator().getAllOutputOperatorParameters();
		String pref = "Step '" + this.getName() + "'";
		String errorPrefix = "";
		if (out == null || out.isEmpty())
			return true;
		Iterator it = out.iterator();
		while (it.hasNext()) {
			OpParam outputOpParam = (OpParam) it.next();
			String errPref = pref + ", output parameter '" + outputOpParam.getName() + "'";
			if (outputOpParam.isOptional()) {
				continue;
			}
			int startloop = 0; int endloop = 0;
			if (outputOpParam.isLoopable() && this.getLoopCount() > 0) {
				startloop = 1; endloop = this.getLoopCount();
			}
			for (int loop = startloop; loop <= endloop; loop++) {
				errorPrefix = (startloop > 0 ? errPref + ", loop " + loop + ": " : errPref + ": ");
				Collection itsTuples = this.getParameterTuples(outputOpParam, loop);
				if (itsTuples == null || itsTuples.isEmpty()) {
					throw new UserError(errorPrefix + "no entries found, but parameter is not optional!");
				}
				if (outputOpParam.getMinArg() > itsTuples.size()) {
					throw new UserError(errorPrefix + "at least " + outputOpParam.getMinArg() + " entries are needed!");
				}
				if (outputOpParam.getMaxArg() > -1 && itsTuples.size() > outputOpParam.getMaxArg()) {
					throw new UserError(errorPrefix + "at most " + outputOpParam.getMinArg() + " entries are possible!");				
				}
				Iterator tupleIt = itsTuples.iterator();
				while (tupleIt.hasNext()) {
					Parameter myPar = (Parameter) tupleIt.next();
					if (myPar == null) {
						throw new UserError(errorPrefix + "found NULL parameter!");
					}
					ParameterObject po = (ParameterObject) myPar.getTheParameterObject();
					if (po == null) {
						throw new UserError(errorPrefix + "found no parameter object!");
					}
					if (outputOpParam.isBaseAttribParameter() && ( ! (po instanceof BaseAttribute))) {
						throw new UserError(errorPrefix + "expected a BaseAttribute as parameter object!");
					}
					if (outputOpParam.isConceptParameter() && ( ! (po instanceof Concept))) {
						throw new UserError(errorPrefix + "expected a Concept as parameter object!");
					}
					if (outputOpParam.isFeatureParameter() && ( ! (po instanceof Feature))) {
						throw new UserError(errorPrefix + "expected a Feature as parameter object!");
					}
				}
			}
		}
		return true;
	}
	
	/** 
	 * This method checks the number and type constraints for a given parameter (single or array)
	 * at a single loop.
	 * @param opParam the <code>OpParam</code> object referring to the parameter under consideration
	 * @param pArray the parameter under consideration. Since it may be a parameter array and the
	 *        type of parameter objects is unknown it is given as an array of <code>ParameterObject</code>.
	 * @return the error message without the step identification suffix or <code>null</code>
	 *         if no error occurred.
	 */
	private String checkSingleLoopOpParam(OpParam opParam, ParameterObject[] pArray) {
		if (opParam.getMinArg() > 0 && pArray == null) {
			return "Step '" + this.getName() + "': non-optional parameter '" + opParam.getName() + "' is missing!";
		}
		else if (pArray == null) { // ie if op.getMinArg()== 0, so parameter is optional:
			return null;
		}
		if (pArray.length < opParam.getMinArg()) {
			return ("Non-optional parameter '" + opParam.getName() + "': Found " + pArray.length
					+ " entries, but expect at least " + opParam.getMinArg());
		}
		
		if ((opParam.getMaxArg() > 0) && (pArray.length > opParam.getMaxArg())) {
			return ("Parameter '" + opParam.getName() + "': Found " + pArray.length
					+ " entries, but expect at most " + opParam.getMaxArg());
		}

		final short parType   = opParam.getType();
		
		// At least one parameter present:
		for (int i=0; i<pArray.length; i++) {
			ParameterObject pObj = pArray[i];
			if (pObj == null)
				return ("Parameter '" + opParam.getName() + "': Found null entry at index " + i);
			
			String expected = null;
			if ((parType == Parameter.TYPE_CONCEPT) && !(pObj instanceof Concept)) {
				expected = "Concept";
			}
			else if ((parType == Parameter.TYPE_BASEATTRIBUTE) && !(pObj instanceof BaseAttribute)) {
				expected = "BaseAttribute";
			}
			else if ((parType == Parameter.TYPE_MULTICOLUMNFEATURE) && !(pObj instanceof MultiColumnFeature)) {
				expected = "MultiColumnFeature";
			}
			else if ((parType == Parameter.TYPE_FEATURE) && !(pObj instanceof Feature)) {
				expected = "Feature";
			}
			else if ((parType == Parameter.TYPE_VALUE) && !(pObj instanceof Value)) {
				expected = "Value";
			}
			else if ((parType == Parameter.TYPE_RELATION) && !(pObj instanceof Relation)) {
				expected = "Relation";
			}
			
			if (expected != null)
				return ("Parameter '" + opParam.getName() + "' at index " + i 
						+ ": Expected was a " + expected + ", but found an object of incompatible class "
						+ pObj.getClass().getName());
		}
		
		return null; // no errors
	}

	/** 
	 * @return the error message without the step identification suffix or <code>null</code>
	 *          if no error occurred.
	 */
	private String checkInputTypeConstraint(Constraint constraint) throws M4Exception {
		final String type = constraint.getType();
		String obj1 = constraint.getObj1(); // cannot be null
		String obj2 = constraint.getObj2();		

		// We cannot check any constraint that concerns output parameters, because
		// the latter may not exist yet!
		OpParam oneOpPar = (OpParam) this.getTheOperator().getOpParam(obj1);
		OpParam secondOpPar = null;
		if ( ! oneOpPar.isInput()) {		 
			return null;
		}
		if (obj2 != null) {
			secondOpPar = (OpParam) this.getTheOperator().getOpParam(obj2);
			if (secondOpPar != null && ( ! secondOpPar.isInput())) {
				return null;
			}
		}		
		
		// Now obj1 is always the name of an input parameter or a name prefix for an array

		// Use the right loop number, based on the type of the parameter obj1:
		int loopNum = 0;
		int lastLoop = this.getLoopCount() + 1;
		if (oneOpPar.isLoopable() && this.getLoopCount() > 0) {
			loopNum = 1;
		}
		if ( ! oneOpPar.isLoopable()) {
			lastLoop = 0;
		}
		String errMessg = null;
		do {
			ParameterObject[] pArray = this.getParameter(obj1, loopNum);
			if (pArray == null || pArray.length == 0 || pArray[0] == null) {
				if (oneOpPar.isOptional()) {
					return null;
				}
				else {
					throw new UserError("Constraint " + type + " violated. No object(s) for mandatory parameter '" + obj1 + "' found!");					
				}
			}
			// else if ((pArray.length == 0) || (pArray[0] == null))
			//	return null; // just a simplification before branching into the separate methods 
		
			if (type.equals(Constraint.TYPE_ONE_OF))
				errMessg = this.checkOneOfConstraint(pArray, obj2);
			else if (type.equals(Constraint.TYPE_CONTAINED_IN)) {
				if (secondOpPar == null) {
					throw new M4Exception("Step '" + this.getName() + "': IN Constraint found whose object 2 is not a parameter!");
				}
				int loopNumberForSecondParameter = (secondOpPar.isLoopable() ? loopNum : 0);
				ParameterObject[] obj2ParArray = this.getParameter(obj2, loopNumberForSecondParameter);
				errMessg =  this.checkContainedInConstraint(pArray, obj2ParArray, loopNum);				
			}
			else if (type.equals(Constraint.TYPE_DATATYPE)) {
				String theType = obj2;
				// but it may also be a Value parameter whose value gives the type:
				if (secondOpPar != null) {
					int loopNumberForSecondParameter = (secondOpPar.isLoopable() ? loopNum : 0);
					ParameterObject[] obj2ParArray = this.getParameter(obj2, loopNumberForSecondParameter);
					if (obj2ParArray != null && obj2ParArray.length > 0) {
						 theType = ((Value) obj2ParArray[0]).getValue();
					}
				}
				errMessg =  this.checkDatatypeConstraint(pArray, theType);
			}
			else if (type.equals(Constraint.TYPE_GREATER_OR_EQUAL))
				errMessg =  this.checkComparisonConstraint(pArray, obj2, loopNum, true, true);  // orEqual = true, greater = true
			else if (type.equals(Constraint.TYPE_GREATER_THAN))
				errMessg =  this.checkComparisonConstraint(pArray, obj2, loopNum, false, true); // orEqual = false, greater = true
			else if (type.equals(Constraint.TYPE_LESSER_OR_EQUAL))
				errMessg =  this.checkComparisonConstraint(pArray, obj2, loopNum, true, false);  // orEqual = true, greater = false
			else if (type.equals(Constraint.TYPE_LESSER_THAN))
				errMessg =  this.checkComparisonConstraint(pArray, obj2, loopNum, false, false); // orEqual = false, greater = false
			else if (type.equals(Constraint.TYPE_SAME_DATATYPE))
				errMessg =  this.checkSameDataTypeConstraint(pArray, obj2, loopNum);
			else if (type.equals(Constraint.TYPE_SAME_FEATURES))
				errMessg =  this.checkSameFeatureConstraint(pArray, obj2, loopNum);
			else if (type.equals(Constraint.TYPE_SUM_MUST_BE_ONE))
				errMessg =  this.checkSumsToOne(pArray);
			else if (type.equals(Constraint.TYPE_COORDINATED_ARRAYS))
				errMessg =  this.checkArrayCoordination(pArray, obj1, obj2, loopNum);
			else if (type.equals(Constraint.TYPE_NO_COMMON_FEATURES))
				errMessg =  this.checkNoCommonFeaturesExceptRenamedFeatures(pArray);
			loopNum++;
		}
		while (loopNum < lastLoop && (errMessg == null));
		
		// Ignore all unrecognized constraint types!
		
		/*
		  Not checked:
		  TYPE_IS_LOOPED        = "IS_LOOPED"; // supported otherwise, no need to be checked
		  TYPE_COMPATIBLE       = "COMP"; // Rather complex constraint: The only application is with the MRFC-operator,
		                                  // which works beyond the specification. Implementation delayed ...
		*/	

		
		return errMessg;
	}
	
	private String checkNoCommonFeaturesExceptRenamedFeatures(ParameterObject[] theConcepts) 
	throws M4Exception {
		if (theConcepts == null || theConcepts.length < 2) {
			return null;
		}
		// the given concepts must have no features in common
		// except those that are matched or renamed by a pair of parameters;
		// this matchning or renaming is possible if such constraints exist:
		Constraint theRenameConstraint = this.getRenamingConstraint();
		Constraint theMatchConstraint = this.getMatchingConstraint();
		String inputParamName = null, outputParamName = null;
		Collection listOfRenamedOrMatchedFeatures = null;
		if (theRenameConstraint != null) {
			// check that no doublettes occur in renamed names:
			String error = this.checkNoDoublesInOutputNames(theRenameConstraint);
			if (error != null)
				return error;
			
			// store the map from old to new names:
			inputParamName = theRenameConstraint.getObj1();
			outputParamName = theRenameConstraint.getObj2();
			listOfRenamedOrMatchedFeatures = this.getListOfRenamedAndMatchedFeatures(theRenameConstraint, theMatchConstraint);
		}
		
		// go through the features of all concepts and make sure they
		// have no common names except among the mapped ones:
		int level = 0;
		String errorMessage = this.checkNoCommonFeaturesPairwise(level, theConcepts, listOfRenamedOrMatchedFeatures);
		if (errorMessage != null) {
			errorMessage += "\nPlease use the Parameters '" + inputParamName + "' and '" + 
							outputParamName + "' to map the names to different output names!";
			return errorMessage;
		}
		return null;
	}	
	
	private String checkNoCommonFeaturesPairwise(
			int currentLevel, 
			ParameterObject[] theConcepts, 
			Collection allowedFeatures) throws M4Exception {
		if (currentLevel < 0 || currentLevel >= theConcepts.length) {
			return null;
		}
		if ( ! (theConcepts[currentLevel] instanceof Concept)) {
			throw new M4Exception("Step '" + this.getName() + "': found non-Concept as object of a concept parameter!");
		}
		Concept currentConcept = (Concept) theConcepts[currentLevel];
		for (int nextLevel = currentLevel + 1; nextLevel < theConcepts.length; nextLevel++) {

			if ( ! (theConcepts[nextLevel] instanceof Concept)) {
				throw new M4Exception("Step '" + this.getName() + "': found non-Concept as object of a concept parameter!");
			}
			Concept nextConcept = (Concept) theConcepts[nextLevel];
			Iterator currentBaIt = currentConcept.getAllBaseAttributes().iterator();
			while (currentBaIt.hasNext()) {
				BaseAttribute currBa = (BaseAttribute) currentBaIt.next();
				BaseAttribute nextBa = (BaseAttribute) nextConcept.getBaseAttribute(currBa.getName());
				if (nextBa != null) {
					// Found two features with the same name in two input concepts!
					// See if they are matched or mapped to different names:
					if ((allowedFeatures == null)
							||
					    ( ! (this.checkOccurrence(currBa, allowedFeatures) &&
						 	 this.checkOccurrence(nextBa, allowedFeatures)))) {
						return "A feature named '" + currBa.getName() + 
							"' occurs in the input concepts '" + currentConcept.getName() +
							"' and '" + nextConcept.getName() + "'!";
					}
				}
			}
			String message = this.checkNoCommonFeaturesPairwise(currentLevel + 1, theConcepts, allowedFeatures);
			if (message != null) {
				return message;
			}
		}
		return null;
	}
	
	// returns a message hinting at a name that occurs more than once
	// in the parameter given by obj2 of the given constraint
	private String checkNoDoublesInOutputNames(Constraint theRenameConstraint)
	throws M4Exception {
		if (theRenameConstraint == null)
			return null;
		
		Vector outputMappedFeatures = new Vector();
		
		OpParam outOpPar = (OpParam) this.getTheOperator().getOpParam(theRenameConstraint.getObj2());
		// the parameters are coordinated so their loops start with 1:
		int startLoop = 1;
		int endLoop = this.getHighestLoopNr(outOpPar);
		if (endLoop > 0) {
			for (int loop = startLoop; loop <= endLoop; loop++) {
				Parameter outParam = (Parameter) this.getParameterTuple(outOpPar.getName(), loop);
				if (outParam != null) {
					ParameterObject outObj = (ParameterObject) outParam.getTheParameterObject();
					if (outObj != null && (outObj instanceof Feature)) 
						outputMappedFeatures.add(outObj.getName());
				}
			}
		}
		
		// no doubles must exist in 'outputMappedFeatures':
		int noOfElems = outputMappedFeatures.size();
		for (int i = 0; i < noOfElems; i++) {
			String oneName = (String) outputMappedFeatures.remove(0);
			Iterator it = outputMappedFeatures.iterator();
			while (it.hasNext()) {
				String s = (String) it.next();
				if (s.equalsIgnoreCase(oneName)) {
					return "The name '" + oneName + 
						"' occurs more than once in Parameter '" + 
						outOpPar.getName() + "' (ignoring case)!"; 
				}
			}
		}
		return null;
	}
	
	// returns all input features that are mapped or matched
	private Collection getListOfRenamedAndMatchedFeatures(Constraint theRenameConstraint, Constraint theMatchConstraint) 
	throws M4Exception {
		
		if (theRenameConstraint == null && theMatchConstraint == null)
			return null;
		
		Collection listOfFeatures = new Vector(); // to be returned
		Collection outputMappedFeatures = new Vector();
		
		// first handle the renaming constraint:
		OpParam inOpPar = (OpParam) this.getTheOperator().getOpParam(theRenameConstraint.getObj1());
		OpParam outOpPar = (OpParam) this.getTheOperator().getOpParam(theRenameConstraint.getObj2());
		// the parameters are coordinated so their loops start with 1:
		int startLoop = 1;
		int endLoop = this.getHighestLoopNr(inOpPar);
		if (endLoop > 0) {
			for (int loop = startLoop; loop <= endLoop; loop++) {
				Parameter inParam = (Parameter) this.getParameterTuple(inOpPar.getName(), loop);
				Parameter outParam = (Parameter) this.getParameterTuple(outOpPar.getName(), loop);
				if (inParam != null) {
					ParameterObject inObj = (ParameterObject) inParam.getTheParameterObject();
					if (inObj != null && (inObj instanceof Feature)) 
						listOfFeatures.add(inObj);
				}
				if (outParam != null) {
					ParameterObject outObj = (ParameterObject) outParam.getTheParameterObject();
					if (outObj != null && (outObj instanceof Feature)) 
						outputMappedFeatures.add(outObj);
				}
			}
		}
		
		
		// second handle the matching constraint:
		OpParam matchOpParam = (OpParam) this.getTheOperator().getOpParam(theMatchConstraint.getObj1());
		if (matchOpParam != null) {
			ParameterObject[] pos = this.getParameter(matchOpParam.getName(), 0);
			if (pos != null)
				for (int i = 0; i < pos.length; i++) {
					if (pos[i] instanceof Feature) {
						listOfFeatures.add(pos[i]);
					}
				}
		}
		
		return listOfFeatures;
	}		

	private Constraint getRenamingConstraint() throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		Collection theConstraints = this.getTheOperator().getConstraints();
		if (theConstraints != null) {
			Iterator it = theConstraints.iterator();
			while (it.hasNext()) {
				Constraint constr = (Constraint) it.next();
				if (constr.getType().equals(Constraint.TYPE_RENAME_ATTRIB)) {
					return constr;
				}
			}
		}
		return null;
	}

	private Constraint getMatchingConstraint() throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		Collection theConstraints = this.getTheOperator().getConstraints();
		if (theConstraints != null) {
			Iterator it = theConstraints.iterator();
			while (it.hasNext()) {
				Constraint constr = (Constraint) it.next();
				if (constr.getType().equals(Constraint.TYPE_MATCH_ATTRIBS_BY_CONCEPTS)) {
					return constr;
				}
			}
		}
		return null;
	}
	
	/** 
	 * Checks whether all values in pArray are of type Value and in the list of alternatives
	 * specified by obj2.
	 * @param pArray a single or an array of Value parameters
	 * @param obj2 a comma separated list of alternative values
	 * @return an error message if the constraint is violated, <code>null</code> otherwise
	 */
	private String checkOneOfConstraint(ParameterObject[] pArray, String obj2) {
		if (obj2 == null)
			return null; // Nothing to check. Should not occur.

		// turn comma-separated list of possible values into a Vector
		StringTokenizer st = new StringTokenizer(obj2);
		Vector alternatives = new Vector();
		while (st.hasMoreElements()) {
			alternatives.add(st.nextElement());
		}
		
		// Make sure that each entry is of type Value and is in the Vector of altertnatives:
		for (int i=0; i<pArray.length; i++) {
			if (pArray[i] instanceof Value) {
				String value = ((Value) pArray[i]).getValue();
				if ( ! alternatives.contains(value)) {
						return "'ONE OF'-Constraint violated: value " + value
						       + " not found in list of alternatives ("
							   + alternatives.toString() + ").";
			
				}
			}
			else return "'ONE OF'-Constraint found for parameter of different type than Value.";
		}
		
		return null; // constraint not violated
	}

	/**
	 * Checks whether all features in an array or in a concept (obj1) are also part
	 * of the concept specified by obj2.
	 * @param pArray a single input concept, or a single or array of features
	 * @param obj2 a single concept
	 * @param loopNum the current loop number or 0
	 * @return an error message if the constraint is violated, <code>null</code> otherwise
	 * @throws M4Exception
	 */
	private String checkContainedInConstraint(ParameterObject[] pArray, ParameterObject[] obj2, int loopNum)
		throws M4Exception
	{
		// Corresponds to the "IN" constraint. Currently only the inputs of a step
		// are validated, so if the object referred to by obj2 is not yet present this
		// method does not throw an exception, because this probably means that it is
		// an output concept which need to be created by a subsequently applied method.
		
		// The union of obj2 if the same constraint occurs several times for the same
		// obj1 object is ignored yet. MultiColumnFeatures are not supported yet.
		
		// obj2 should be one or more concepts
		
		for (int i = 0; i < obj2.length; i++) {
			if ( ! (obj2[i] instanceof Concept)) {
				return "Constraint 'IN' expects a Concept as its second parameter. Found an object of class "
					   + obj2[i].getClass().getName() + " instead!";
			}
		}
		
		// Case 1: Obj1 corresponds to one or more features
		if (pArray[0] instanceof Feature) {
			for (int i=0; i<pArray.length; i++) {
				if (pArray[i] instanceof Feature) {		
					if ( ! this.findFeatureInConcepts((Feature) pArray[i], obj2))
						return "'IN'-constraint violated for feature '" + ((Feature) pArray[i]).getName() + "', it does not belong to the input concept(s) in loop number " + loopNum + "!";
				}
				else return "Found unsupported object of class " + pArray[i].getClass().getName()
				            + " in first parameter of 'IN' constraint!";
			}
		}
		else if (pArray[0] instanceof Concept) { // case 2: concept
			Iterator it = ((Concept) pArray[0]).getFeatures().iterator();
			while (it.hasNext()) {
				Feature f = (Feature) it.next();
				if ( ! this.findFeatureInConcepts(f, obj2))
					return "'IN'-constraint violated for feature '" + f.getName() + "' of concept '" + pArray[0].getName() + "'!";
			}			
		}
		else return "Found unsupported object of class " + pArray[0].getClass().getName()
					+ " in first parameter of 'IN' constraint!";

		return null;
	}
	
	// checks if given Feature exists in the given array of concepts
	private boolean findFeatureInConcepts(Feature f, ParameterObject[] concepts) throws M4Exception {
		for (int i = 0; i < concepts.length; i++) {
			Concept myCon = (Concept) concepts[i];
			Iterator theFeaturesIt = myCon.getFeatures().iterator();
			while (theFeaturesIt.hasNext()) {
				Feature myF = (Feature) theFeaturesIt.next();
				if (myF.equals(f)) {
					return true;
				}
			}
		}
		return false;
	}
	
	/**
	 * Checks whether all features in one concept (obj1) are also part of the other (obj2).
	 * As specified in deliverable 18 only the names are compared. A crucial point is that
	 * both concepts may be extended by feature construction operators in other steps.
	 * This means that only the <i>visible</i> features are compared to each other, but
	 * these need to match exactly.
	 * MultiColumnFeatures are not yet fully supported!
	 * 
	 * @param pArray a single input concept, or a single or array of features
	 * @param obj2 the name of a concept
	 * @param loopNum the current loop number or 0
	 * @return an error message if the constraint is violated, <code>null</code> otherwise
	 * @throws M4Exception
	 */
	private String checkSameFeatureConstraint(ParameterObject[] pArray, String obj2, int loopNum)
		throws M4Exception
	{
		// Corresponds to the "SAME_FEAT"
		Concept inConcept;
		Vector otherConcepts = new Vector();

		{
			// obj1 should be an array of concepts
			for (int i=0; i<pArray.length; i++) {
				if ( !(pArray[i] instanceof Concept) )
					return "Constraint 'SAME_FEAT' expects a single or an array of Concept(s) as its first parameter. Found an object of class "
					+ pArray[i].getClass().getName() + " instead!";
				else otherConcepts.add(pArray[i]);
			}
		
			// obj2 should be a single concept
			{
				ParameterObject[] pArray2 = this.getParameter(obj2, loopNum);
				if ((pArray2 == null) || (pArray2.length == 0))
					return null;
			
				if ( ! (pArray2[0] instanceof Concept)) {
					return "Constraint 'SAME_FEAT' expects a Concept as its second parameter. Found an object of class "
						   + pArray[0].getClass().getName() + " instead!";
				}
				else inConcept = (Concept) pArray2[0]; 
			}
		}

		Vector inputFeatureNames = new Vector();
		{
			Iterator feaIt = inConcept.getFeatures().iterator();
			while (feaIt.hasNext()) {
				Feature fea = (Feature) feaIt.next();
				if (this.isVisible(fea)) {
					inputFeatureNames.add(fea.getName());
				}
			}
		}
		
		
		Iterator it = otherConcepts.iterator();
		while (it.hasNext()) {
			Concept outConcept = (Concept) it.next();
			Iterator feaIt = outConcept.getFeatures().iterator();
			Vector ifn = (Vector) (inputFeatureNames.clone());
			while (feaIt.hasNext()) {
				Feature fea = (Feature) feaIt.next();
				if (this.isVisible(fea) && !(ifn.contains(fea.getName()))) {
					return "Constraint 'SAME_FEAT' violated. Concept '" + outConcept.getName()
					       + "' contains additional feature '" + fea.getName() + "'!"; 
				}
				else ifn.remove(fea.getName());
			}
			if ( ! ifn.isEmpty()) {
				String additional = (String) ifn.firstElement();
				return "Constraint 'SAME_FEAT' violated. Concept '" + inConcept.getName()
			       + "' contains feature '" + additional + "', which is not part of concept '"
				   + outConcept.getName() + "'!";
				
			}
		}
		
		return null; // constraint not violated
	}
	
	private String checkDatatypeConstraint(ParameterObject[] pArray, String obj2)
		throws M4Exception {
		for (int i=0; i<pArray.length; i++) {
			ParameterObject current = pArray[i];
			String error = this.checkDatatypeConstraint(current, obj2);
			if (error != null) {
				return error;
			}
		}
		return null;
	}
	
	private String checkDatatypeConstraint(ParameterObject theObject, String obj2) throws M4Exception {
		String datatype = null;
		if (theObject instanceof Value) {
			datatype = ((Value) theObject).getTypeName();
		}
		else {
			if (theObject instanceof BaseAttribute) {
				datatype = ((BaseAttribute) theObject).getConceptualDataTypeName();
			}
		    else {
		     	return "Found unsupported object of class " + 
				       theObject.getClass().getName() + 
					   " in first parameter of 'TYPE' constraint!";
		    }
		}		     
		
		// MultiColumnFeatures are not supported yet!
		// Multiple occurences of TYPE constraints for the same obj1 are ignored.
		// Due to the specification of deliverable 18 they should be interpreted
		// as a disjunction of possible types!
		
		if ( ! ConceptualDatatypes.checkDatatypeCompatibility(obj2, datatype)) {
			return "'TYPE' constraint violated. BaseAttribute or Value '" + 
			       theObject.getName() + "' is of type " + datatype +
				   ", but the type must be " + obj2 + "!";				
		}
		else {
			return null;
		}
	}
	
	/*
	 * If param greater is true, it is a greater-than or greater-or-equal constraint,
	 * depending on param orEqual. If param greater is false, it is a lesser-than or
	 * lesser-or-equal constraint, again depending on orEqual.
	 */
	private String checkComparisonConstraint(ParameterObject[] pArray, String obj2, int loopNr, boolean orEqual, boolean greater) throws M4Exception {
		String constrType = null;
		if (greater) {
			constrType = (orEqual ? Constraint.TYPE_GREATER_OR_EQUAL : Constraint.TYPE_GREATER_THAN);
		}
		else {
			constrType = (orEqual ? Constraint.TYPE_LESSER_OR_EQUAL : Constraint.TYPE_LESSER_THAN);
		}
		
		// obj2 may be a parameter!
		OpParam secondObjOpPar = (OpParam) this.getTheOperator().getOpParam(obj2);
		Double obj2D = null;
		
		if (secondObjOpPar == null) { // obj2 is no parameter
			if (obj2 != null) {
				try {
					obj2D = new Double(obj2);
				}
				catch (NumberFormatException e) {}
			}
		}
		else {
			Iterator it = this.getParameter(secondObjOpPar, loopNr).iterator();
			// for now we assume only one parameter, no array!
			ParameterObject po = (ParameterObject) it.next();
			if ( ! (po instanceof Value)) {
				throw new M4Exception("Step '" + this.getName() + "', Constraint " + constrType +
						": Second constraint object is neither a number nor a Value parameter!");
			}
			obj2D = ((Value) po).getDouble();
		}
		if (obj2D == null)
			return "Second argument of " + constrType + "constraint invalid: " + obj2;
		double obj2d = obj2D.doubleValue();
		
		for (int i=0; i<pArray.length; i++) {
			ParameterObject current = pArray[i];
			if ( ! (current instanceof Value)) {
				return "Found unsupported object of class " + current.getClass().getName()
	                   + " in first parameter of " + constrType + " constraint!";
			}
			String valueS = ((Value) current).getValue();
			Double value = null;
			if (valueS != null) {
				try {
					value = new Double(valueS);
				}
				catch (NumberFormatException e) {
					return "Found unsupported Value object for 'GE' or 'GT' constraint:"
					       + "The constraint implies a numeric value, but the value found is '"
						   + valueS + "'!";
				}
			}
			if (greater) {
				if ((value.doubleValue() < obj2d) || ( !orEqual && (value.doubleValue() <= obj2d)))
					return "Constraint " + constrType + " violated: Found parameter value " + value + "!";
			}
			else {
				if ((value.doubleValue() > obj2d) || ( !orEqual && (value.doubleValue() >= obj2d)))
					return "Constraint " + constrType + " violated: Found parameter value " + value + "!";				
			}
		}
		return null; //  Constraint not violated.
	}
	
	private String checkSumsToOne(ParameterObject[] pArray) {
		double sum = 0;
		String constraint = "'" + Constraint.TYPE_SUM_MUST_BE_ONE + "'";
		for (int i=0; i<pArray.length; i++) {
			ParameterObject current = pArray[i];
			if ( ! (current instanceof Value)) {
				return "Found unsupported object of class " + current.getClass().getName()
	                   + " in first parameter of " + constraint + " constraint!";
			}
			String valueS = ((Value) current).getValue();
			Double value = null;
			if (valueS != null) {
				try {
					value = new Double(valueS);
				}
				catch (NumberFormatException e) {
					return "Found unsupported Value object for " + constraint + " constraint:"
					       + "The constraint implies a numeric value, but the value found is '"
						   + valueS + "'!";
				}
			}
			sum += value.doubleValue();			
		}

		if (sum != 1) {
			return "Constraint " + constraint + " violated. Sum is " + sum + " rather than 1.0 !";
		}
		else return null; //  Constraint not violated.		
	}
	
	private String checkArrayCoordination(ParameterObject[] theArray,
			                              String nameOf1stArray,
			                              String nameOf2ndArray, 
										  int loopNum) {
		ParameterObject[] pArray2 = this.getParameter(nameOf2ndArray, loopNum);
		
		// - both parameters (theArray and pArray2) might be optional!
		// - their contents are related at each position, but this cannot be checked
		// -> all that can be checked is that both arrays have the same length:
		if (theArray != null && pArray2 != null && theArray.length != pArray2.length) {
			return "Constraint '" + Constraint.TYPE_COORDINATED_ARRAYS + 
			       "' (Array coordination) is violated for parameters '" +
				   nameOf1stArray + "' and '" + nameOf2ndArray + "' (number of entries not equal), ";
		}
		
		return null; // Constraint not violated
	}
	
	private String checkSameDataTypeConstraint(ParameterObject[] pArray, String obj2, int loopNum) {
		ParameterObject[] pArray2 = this.getParameter(obj2, loopNum);
		if ((pArray2 == null) || (pArray2.length == 0) || (pArray2[0] == null)) {
			return "Second argument of 'SAME_TYPE' constraint missing!"; 
		}
		
		if ( ! (pArray[0] instanceof BaseAttribute)) {
			return "First argument of 'SAME_TYPE' constraint should be a BaseAttribute. Found an object of class "
			       + pArray[0].getClass().getName() + " instead!";
		}
		
		if ( ! (pArray2[0] instanceof BaseAttribute)) {
			return "Second argument of 'SAME_TYPE' constraint should be a BaseAttribute. Found an object of class "
			       + pArray2[0].getClass().getName() + " instead!";
		}
		
		BaseAttribute firstBa  = (BaseAttribute) pArray[0];
		BaseAttribute secondBa = (BaseAttribute) pArray2[0];
		
		if (firstBa.getConceptualDataType() != secondBa.getConceptualDataType()) {
			return "Constraint 'SAME_TYPE' violated for BaseAttributes '"
			       + firstBa.getName() + "' and '" + secondBa.getName() + "'!";			
		}
		
		// During input checking the equivalence of corresponding relational
		// datatypes is not yet possible, because these types are created by
		// the compiler.
		
		return null; //  Constraint not violated.
	}
	
	/**
	 * Used by the constraint checking mechanism and GUI convenience methods. 
	 * 
	 * @param name the name of a parameter to be looked up 
	 * @param loopNr the loop for which the parameter should be returned
	 * @return an array of <code>ParameterObject</code>s.
	 * A return value of <code>null</code> indicates, that the parameter was not found.
	 * */
	private ParameterObject[] getParameter(String name, int loopNr) {
		try {
			// ParamDict pd = this.getParameterDictionary();
			edu.udo.cs.miningmart.m4.ParameterArray p = this.getParameterDictionary(false).get(name, loopNr);
			if (p == null) {
				return null;
			}
			else { 
				return (ParameterObject[]) p.getParameterObjectArray();
			}
		}
		catch (M4Exception e) {
			this.doPrint(Print.MAX,
				"Warning: Exception caught when trying to fetch parameters for Step "
				+ this.getId());
			this.doPrint(e);
			return null;
		}
	}
	
	/*
	 * Returns all ParameterObjects of parameters of this step with the given name.
	 */
	private Collection getParameterObjects(String name) throws M4Exception {
		Collection c = this.getParameterTuples();
		if (c != null) {
			Iterator it = c.iterator();
			Vector ret = new Vector();
			while (it.hasNext()) {
				Parameter aPar = (Parameter) it.next();
				if (this.parameterNamesMatch(name, aPar)) {
					ret.add(aPar.getTheParameterObject());
				}
			}
			return ret;
		}
		return null;
	}

	public Collection getPossibleConceptsForParam(edu.udo.cs.miningmart.m4.OpParam opParam) throws M4Exception {
		// find OpParams as specified by the constraints:
		Collection c = this.getOpParamsToSelectFeatureParamObjectsFrom(opParam);
		if (c == null) {
			return null;
		}
		// find concepts for the OpParams:
		Collection theConcepts = new Vector(); // to be returned
		Iterator it = c.iterator();
		while (it.hasNext()) {
			OpParam anOpPar = (OpParam) it.next();
			if (anOpPar.isConceptParameter()) {
				theConcepts.addAll(this.getParameterObjects(anOpPar.getName()));
			}
			if (anOpPar.isRelationParameter()) {
				// get the relation:
				Relation rel = (Relation) this.getSingleParameterObject(anOpPar.getName());
				// return either the FromConcept or the ToConcept!
				// to decide, check the constraints:
				boolean takeFromConcept = false;
				boolean takeToConcept = false;
				Collection constrs = anOpPar.getApplicableConstraints();
				if (constrs != null && ( ! constrs.isEmpty())) {
					Iterator it2 = constrs.iterator();
					while (it2.hasNext()) {
						Constraint myRelConstraint = (Constraint) it2.next();
						if (myRelConstraint.getType().equals(Constraint.TYPE_ATTRIB_IN_REL_FROM)) {
							takeFromConcept = true;
						}
						else if (myRelConstraint.getType().equals(Constraint.TYPE_ATTRIB_IN_REL_TO)) {
							takeToConcept = true;
						}
					}
					if (takeFromConcept) {
						theConcepts.add(rel.getTheFromConcept());
					}
					if (takeToConcept) {
						theConcepts.add(rel.getTheToConcept());
					}
				}
			}
		}
		return theConcepts;
	}

	/**
	 * Returns TRUE iff there is a constraint indicating that the feature parameter
	 * indicated by theAttribOpParam is to be taken from the FromConcept of the
	 * relation given by theRelationOpParam.
	 * @param theAttribOpParam
	 * @param theRelationOpParam
	 * @return
	 * @throws M4Exception
	 */
	public boolean isAttribToTakeFromFromConcept(edu.udo.cs.miningmart.m4.OpParam theAttribOpParam, edu.udo.cs.miningmart.m4.OpParam theRelationOpParam)
	throws M4Exception {
		Collection constrs = theAttribOpParam.getApplicableConstraints();
		if (constrs == null || constrs.isEmpty()) {
			return false;
		}
		Iterator it = constrs.iterator();
		while (it.hasNext()) {
			Constraint aConstr = (Constraint) it.next();
			if (aConstr.getType().equals(Constraint.TYPE_ATTRIB_IN_REL_FROM)
					&&
				( aConstr.getObj2().equalsIgnoreCase(theRelationOpParam.getName())
						||
				  aConstr.getObj1().equalsIgnoreCase(theRelationOpParam.getName()))) {
				return true;
			}			
		}
		return false;
	}

	/**
	 * Returns TRUE iff there is a constraint indicating that the feature parameter
	 * indicated by theAttribOpParam is to be taken from the ToConcept of the
	 * relation given by theRelationOpParam.
	 * @param theAttribOpParam
	 * @param theRelationOpParam
	 * @return
	 * @throws M4Exception
	 */
	public boolean isAttribToTakeFromToConcept(edu.udo.cs.miningmart.m4.OpParam theAttribOpParam, edu.udo.cs.miningmart.m4.OpParam theRelationOpParam)
	throws M4Exception {
		Collection constrs = theAttribOpParam.getApplicableConstraints();
		if (constrs == null || constrs.isEmpty()) {
			return false;
		}
		Iterator it = constrs.iterator();
		while (it.hasNext()) {
			Constraint aConstr = (Constraint) it.next();
			if (aConstr.getType().equals(Constraint.TYPE_ATTRIB_IN_REL_TO)
					&&
				( aConstr.getObj2().equalsIgnoreCase(theRelationOpParam.getName())
						||
				  aConstr.getObj1().equalsIgnoreCase(theRelationOpParam.getName()))) {
				return true;
			}			
		}
		return false;
	}
	
	public Collection getOpParamsToSelectFeatureParamObjectsFrom(edu.udo.cs.miningmart.m4.OpParam opParam)
		throws M4Exception
	{
		if (opParam == null)
			return null;
		
		// This method is used by the GUI to decide about whether attribute parameters
		// are selected from the input concept or given new names. Therefore we distinguish
		// the following cases:
		// Case 1 - there is only a constraint "attribute is IN TheOutputConcept"
		//                In this case the attribute is not selected from the input concept(s)
		// Case 2 - there is both a constraint "attribute is IN TheOutputConcept" and
		//                and a constraint "attribute is IN TheInputConcept"
		//                In this case the attribute is selected from the input concept(s)
		// Case 3 - there is only a constraint "attribute is IN TheInputConcept" and
		//                no output concept exists
		//                In this case the attribute is newly created (and added to the input concept later),
		//                if it is an output attribute; if it is an input attribute, it is selected from the input concept
		// Case 4 - there is only a constraint "attribute is IN TheInputConcept" and
		//                an output concept exists
		//                In this case the attribute is selected from the input concept(s)
		// Case 5 - there is only a constraint "attribute is IN_REL_FROM/IN_REL_TO TheRelation"
		//                In this case return the relation OpParam
		
		Class typeOfParameter = Parameter.getClassForParameterType(opParam.getType());
		if ( ! (Feature.class.isAssignableFrom(typeOfParameter))) {
			throw new M4Exception("Step.getPossibleConceptsForParam: "
					+ "Expected a Feature parameter, but found an object of class "
					+ typeOfParameter.getName()
			);
		}
		
		String opParamName = opParam.getName();
		Iterator it = this.getTheOperator().getConstraints().iterator();
		Collection inputConOpParams = new Vector(); 
		OpParam outputOpParam = null;
		OpParam relationOpParam = null;
		
		// loop through the constraints and store the information found:
		while (it.hasNext()) {
			Constraint constraint = (Constraint) it.next();
			if (constraint.getType().equals(Constraint.TYPE_CONTAINED_IN)
				&& constraint.getObj1().equalsIgnoreCase(opParamName))
			{
				String obj2 = constraint.getObj2();
				if ((obj2 != null) && obj2.length() > 0) {
					// do not return output OpParams:
					OpParam obj2OpPar = (OpParam) this.getTheOperator().getOpParam(obj2);
					if (obj2OpPar.isInput() && obj2OpPar.isConceptParameter()) {
						inputConOpParams.add(obj2OpPar);
					}
					if (( ! obj2OpPar.isInput()) && obj2OpPar.isConceptParameter()) {
						outputOpParam = obj2OpPar;
					}
				}
			}
			if ( (constraint.getType().equals(Constraint.TYPE_ATTRIB_IN_REL_FROM) ||
				  constraint.getType().equals(Constraint.TYPE_ATTRIB_IN_REL_TO)) 
				&& constraint.getObj1().equalsIgnoreCase(opParamName)) {

				String obj2 = constraint.getObj2();
				if ((obj2 != null) && obj2.length() > 0) {
					// do not return output OpParams:
					OpParam obj2OpPar = (OpParam) this.getTheOperator().getOpParam(obj2);
					if (obj2OpPar.isInput() && obj2OpPar.isRelationParameter()) {
						relationOpParam = obj2OpPar;
					}
				}
			}				
		}
		
		// Case 5:
		if (relationOpParam != null) {
			Vector v = new Vector();
			v.add(relationOpParam);
			return v;
		}		
		// Case 1:
		if (outputOpParam != null && inputConOpParams.isEmpty()) {
			return null; // no concepts to select the attribs from
		}
		// Case 2:
		else if (outputOpParam != null && ( ! inputConOpParams.isEmpty())) {
			return inputConOpParams;
		}
		// Case 3:
		else if (( ! inputConOpParams.isEmpty()) && (this.getOutputConcept() == null)) {
			if (opParam.isInput()) {
				return inputConOpParams;
			}
			else	return null; // output attrib is created and added to input concept later, but not chosen from it
		}
		// Case 4:
		else if (( ! inputConOpParams.isEmpty()) && (this.getOutputConcept() != null)) {
			return inputConOpParams;
		}
		else return null; // if there are no constraints, take it to be an output
	}
	
	public Collection getPossibleInputConcepts() throws M4Exception {
		// return all Concepts of this Case that are available for this step
		
		Vector ret = new Vector(); // to be returned
		
		// first, find the DB concepts:
		Iterator it = this.getTheCase().getConcepts().iterator();
		while (it.hasNext()) {
			Concept myCon = (Concept) it.next();
			if (myCon.getType().equals(Concept.TYPE_DB)) {
				ret.add(myCon);
			}
		}
		
		// second, add all output concepts of previous steps:
		this.addOutputConceptsOfPredecessors(ret);
		
		ret.trimToSize();
		return ret;
	}
	
	private void addOutputConceptsOfPredecessors(Collection outCons) throws M4Exception {
		Iterator it = this.getAllPredecessors().iterator();
		while (it.hasNext()) {
			Step myPredStep = (Step) it.next();
			Concept anOutCon = (Concept) myPredStep.getOutputConcept();
			if (anOutCon != null && ( ! outCons.contains(anOutCon))) {
				outCons.add(anOutCon);
			}
			myPredStep.addOutputConceptsOfPredecessors(outCons);
		}
	}
	
	/**
	 * This method must only be called for an output feature. It is checked 
	 * whether the output feature is constrained to be IN the input concept.
	 * 
	 * @param outputFeature the OpParam representing the output feature
	 * @return TRUE iff the outputFeature's name should be selected from the
	 *         names of the features of the input concept; FALSE if any new
	 *         name can be given to the outputFeature.
	 */
	public boolean isContainedInInputConcept(edu.udo.cs.miningmart.m4.OpParam outputFeature) throws M4Exception {
		
	    if (outputFeature.isInput()) {
	    	throw new M4Exception("Step.isContainedInInputConcept: can only be called with an output parameter!");
	    }
	    
	    boolean isContained = false; // to be returned
		Collection applicableConstraints = outputFeature.getApplicableConstraints();
		Iterator theConstraintsIt = applicableConstraints.iterator();
		
		// go through the constraints and look for an IN constraint:
		while (theConstraintsIt.hasNext()) {
			Constraint aConstr = (Constraint) theConstraintsIt.next();
			// if an IN constraint is found:
			if (aConstr.getType().equalsIgnoreCase(Constraint.TYPE_CONTAINED_IN)) {
				String containedIn = aConstr.getObj2(); 
				// 'containedIn' can only be TheInputConcept or TheOutputConcept!
				Parameter conceptPar = (Parameter) this.getParameterTuple(containedIn, 0);
				// check that it is a concept:
				if ( ! conceptPar.getParObjectType().equalsIgnoreCase(Parameter.typeStringForTypeConst(Parameter.TYPE_CONCEPT))) {
					throw new M4Exception("Step '" + this.getName() + 
							              "': found IN constraint for output BA without a Concept in object 2!");
				}
				// check if it is the input concept:
				if (conceptPar.isInputParam()) {
					isContained = true;
				}
			}
		}
		return isContained;
	}
	
	/**
	 * Active getter for the parameters of this <code>Step</code>
	 * <b>in aggregated form</b>.
	 * 
	 * @return Returns a ParamDict
	 */
	public edu.udo.cs.miningmart.m4.ParamDict getParameterDictionary(boolean expectingAllParamsToExist) 
	throws M4Exception {
		// Read the current status of parameter tuples and aggregate them.
		// Thus we have an on-demand-aggregation.	

		final ParamDict pd = new edu.udo.cs.miningmart.m4.core.ParamDict();
		Iterator it = this.getTheOperator().getOpParamsIterator();
		while (it.hasNext()) {
			final OpParam op  = (OpParam) it.next();
			final String paramMissingMsg = "Parameter '" + op.getName() + "' missing in Step '" + this.getName() + "'!";
			
			if (op.getMinArg() == 0) {
				// Optional parameters in combination with loops need special attention,
				// because the 'loopable' property of this kind of parameter is sometimes
				// not detectable from the meta-data!
				try {
					ParameterArray withoutLoop = (ParameterArray) this.getParameterArray(op, 0);
					if ((withoutLoop != null) && 
						(withoutLoop.size() > 0 || this.getLoopCount() <= 0)) {
						pd.put(op.getName(), withoutLoop);
					}
					else {
						if (op.isCoordinated()) {
							// special: loop mechanism starting from 1 for coordination
							int loopNr = 1;
							ParameterArray myPar = (ParameterArray) this.getParameterArray(op, loopNr);
							if (myPar != null) {
								// we add the tuples from higher loops to the parameter array:
								loopNr++;
								ParameterArray additionalParArr = (ParameterArray) this.getParameterArray(op, loopNr);
								while (additionalParArr != null) {
									// add tuples to myPar:
									Iterator it2 = additionalParArr.getParameters().iterator();
									while (it2.hasNext()) {
										Parameter aParam = (Parameter) it2.next();
										myPar.addParameter(aParam);
									}
									loopNr++;
									additionalParArr = (ParameterArray) this.getParameterArray(op, loopNr);
								}
								// by convention, we store coordinated arrays in the PD under loopnr 0!!
								pd.put(op.getName(), 0, myPar);
							}
						}
						else {
							final int loops = this.getLoopCount() + 1;
							for (int i = 1; i < loops; i++) {
								ParameterArray obj = (ParameterArray) this.getParameterArray(op, i);
								if (obj != null) {
									pd.put(op.getName(), i, obj);
								}
							}
						}
					}
				}
				catch (ParameterNotFoundException e) {
					throw new M4Exception("Internal error." +
					"'ParameterNotFound' exception was thrown for an optional parameter:\n" +
					e.getMessage());
				}
			}
			else {
				// If the parameter is not optional, one can try to load it as non-looped,
				// catch the exception and try again, assuming it is looped.
				try {
					// load parameter: arrays and optional parameters are handled, as well!
					ParameterArray obj = (ParameterArray) this.getParameterArray(op, 0);
					if (obj == null) {
						throw new ParameterNotFoundException(paramMissingMsg);
					}
					pd.put(op.getName(), obj);
				}
				catch (ParameterNotFoundException e) {
					// The parameter could not be loaded as usual.
					// ==> Is it a looped or coordinated parameter?
					if (op.isCoordinated()) {
						// special: loop mechanism starting from 1 for coordination
						int loopNr = 1;
						ParameterArray myPar = (ParameterArray) this.getParameterArray(op, loopNr);
						if (myPar == null) {
							if (expectingAllParamsToExist) {
								throw new ParameterNotFoundException(paramMissingMsg);
							}
						}
						else {
							// we add the tuples from higher loops to the parameter array:
							loopNr++;
							ParameterArray additionalParArr = (ParameterArray) this.getParameterArray(op, loopNr);
							while (additionalParArr != null) {
							    // add tuples to myPar:
								Iterator it2 = additionalParArr.getParameters().iterator();
								while (it2.hasNext()) {
									Parameter aParam = (Parameter) it2.next();
									myPar.addParameter(aParam);
								}
								loopNr++;
								additionalParArr = (ParameterArray) this.getParameterArray(op, loopNr);
							}
						    // by convention, we store coordinated arrays in the PD under loopnr 0!!
							pd.put(op.getName(), 0, myPar);
						}
					}
					else {
						if (this.getLoopCount() > 0) {
							final int loops = this.getLoopCount() + 1;
							for (int i = 1; i < loops; i++) {
								try {
									ParameterArray obj = (ParameterArray) this.getParameterArray(op, i);
									if (obj == null) {
										throw new ParameterNotFoundException(paramMissingMsg);
									}
									pd.put(op.getName(), i, obj);
								}
								catch (ParameterNotFoundException pnfe) {
									if (expectingAllParamsToExist) {
										throw pnfe;
									}
								}
							}
						}
						else
							if (expectingAllParamsToExist) throw e; // no: propagate exception!
					}
				}
			}
		}
		return pd;
	}	

	/**
	 * @see edu.udo.cs.miningmart.m4.Step#isVisible(Feature)
	 */
	public boolean isVisible(edu.udo.cs.miningmart.m4.Feature theFeature) throws M4Exception {
		// First draft of an implementation
		
		if (this.getTheCase() == null) {
			return false;
		}
		
		// first: check if the Feature is of type DB
		//    - if yes, it's visible
		if (theFeature instanceof BaseAttribute) {
			if (((BaseAttribute) theFeature).isDBAttrib()) {
				return true;
			}
		}
		else {
			boolean allBAsAreDB = true;
			MultiColumnFeature mcf = (MultiColumnFeature) theFeature; 
			Iterator it = mcf.getBaseAttributes().iterator();
			if ( ! it.hasNext()) {
				throw new M4Exception("Step.isVisible(Feature): found MCF '" + mcf.getName() + "'without BAs!");
			}
			while (it.hasNext()) {
				BaseAttribute ba = (BaseAttribute) it.next();
				if ( ! ba.isDBAttrib()) {
					allBAsAreDB = false;
				}				
			}
			if (allBAsAreDB) {
				return true;
			}
		}
		
		// second: check if the feature is an output parameter somewhere
		edu.udo.cs.miningmart.m4.Parameter featureOutputParam = theFeature.getParameterWhereThisIsOutputFeature();
		// if it is never output, it is visible:
		if (featureOutputParam == null) {
			return true;
		}
		// perhaps it is an output of THIS step:
		if (this.getParameterTuples().contains(featureOutputParam))
			return true;
		
		// otherwise it is only visible if it is output somewhere preceding this step:
		Collection predSteps = this.getTheCase().getStepsToCompileBefore(this, true);
		if (predSteps == null) {
			throw new M4Exception("Step.isVisible(Feature): got <null> Collection when asking for predecessors!");
		}
		Iterator stepIterator = predSteps.iterator();
		while (stepIterator.hasNext()) {
			Step myPrecedingStep = (Step) stepIterator.next();
			if (myPrecedingStep.getParameterTuples().contains(featureOutputParam)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * This method finds all features in the output concept of this step
	 * that correspond to the given feature in the given input concept
	 * of this step. If the step has no output concept, the given feature is 
	 * returned in a collection.
	 * 
	 * @param theInputConcept
	 * @param theFeature
	 * @return all corresponding features
	 * @throws M4Exception
	 */
	public Collection getCorrespondingOutputFeatures(
			edu.udo.cs.miningmart.m4.Concept theInputConcept, 
			edu.udo.cs.miningmart.m4.Feature theFeature) 
	throws M4Exception {
		Concept outputConcept = (Concept) this.getOutputConcept();
		if (outputConcept == null) {
			Vector ret = new Vector();
			ret.add(theFeature);
			return ret;
		}
		Feature directCorrespondence = (Feature) outputConcept.getFeature(theFeature.getName());
		if (directCorrespondence != null) {
			Vector ret = new Vector();
			ret.add(directCorrespondence);
			return ret;
		}
		// look for features whose correspondence to the input cannot be
		// seen directly from the name:
		Iterator constrIt = this.getTheOperator().getConstraints().iterator();
		while (constrIt.hasNext()) {
			Constraint theConstraint = (Constraint) constrIt.next();
			if (theConstraint.getType().equals(Constraint.TYPE_CREATE_ATTRIB_BY) ||
				theConstraint.getType().equals(Constraint.TYPE_CREATE_ATTRIB_SUFF)) {
				// obj1 gives the input attrib parameter:
				OpParam inputOpParam = (OpParam) this.getTheOperator().getOpParam(theConstraint.getObj1());
				Collection inputAttribsOfThisParam = this.getParameterObjects(inputOpParam.getName());
				Iterator inputAttrIt = inputAttribsOfThisParam.iterator();
				String[] inputBaNames = new String[inputAttribsOfThisParam.size()];
				int i = 0;
				boolean constraintApplies = false;
				while (inputAttrIt.hasNext()) {
					BaseAttribute in = (BaseAttribute) inputAttrIt.next();
					inputBaNames[i] = in.getName();
					if (in.correspondsTo(theFeature)) {
						constraintApplies = true;
					}
					i++;
				}
				// if the constraint applies to this input feature, find out
				// which output feature was created from it:
				if (constraintApplies) {
					Map mapInputFeaturesToOutputFeatureNames = 
						this.getMapOfNamesToNamesWithSuffixes(theConstraint, inputBaNames);
					Vector correspondingOutputFeatureNames = (Vector) mapInputFeaturesToOutputFeatureNames.get(theFeature.getName());
					Vector outputFeatures = new Vector();
					Iterator namesIt = correspondingOutputFeatureNames.iterator();
					while (namesIt.hasNext()) {
						String outName = (String) namesIt.next();
						Feature f = (Feature) outputConcept.getFeature(outName);
						if (f != null)
							outputFeatures.add(f);					
					}
					return outputFeatures;
				}
			}
			if (theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_RENAME_ATTRIB)) {
				// the input feature may have been renamed; if so, the renaming is
				// given by two corresponding parameter arrays:
				OpParam inputOpParam = (OpParam) this.getTheOperator().getOpParam(theConstraint.getObj1());
				OpParam outputOpParam = (OpParam) this.getTheOperator().getOpParam(theConstraint.getObj2());
				ParameterArray paIn = (ParameterArray) this.getParameterArray(inputOpParam, 0);
				ParameterArray paOut = (ParameterArray) this.getParameterArray(outputOpParam, 0);
				ParameterObject[] inputFeatures = (ParameterObject[]) paIn.getParameterObjectArray();
				ParameterObject[] outputFeatures = (ParameterObject[]) paOut.getParameterObjectArray();
				Vector ret = new Vector();
				for (int i = 0; i < inputFeatures.length; i++) {
					if (((Feature) inputFeatures[i]).correspondsTo(theFeature)) {
						ret.add(outputFeatures[i]);
					}
				}
				return ret;
			}
		}
		return new Vector();
	}
	
	/**
	 * This method checks if any problems with the validity of this 
	 * step would occur if the given feature were removed from
	 * the given input concept of this step. A String describing the
	 * problem is returned, or NULL if no problem would occur.
	 * 
	 * @param theInputConcept input concept in which the feature would be deleted
	 * @param nameOfFeature
	 * @return a hypothetical error message, or NULL if no error would occur
	 * @throws M4Exception
	 */
	public String checkFeatureRemoval(
			edu.udo.cs.miningmart.m4.Concept theInputConcept,
			edu.udo.cs.miningmart.m4.Feature theFeature) throws M4Exception {
		
		// some debugging:
		if ( ! this.checkOccurrence(theInputConcept, this.getAllInputConcepts())) {
			throw new M4Exception("Step '" + this.getName() + 
					"', method checkFeatureRemoval: got unknown input concept!");
		}
		if ( ! theInputConcept.hasFeature(theFeature)) {
			throw new M4Exception("Step '" + this.getName() + 
			"', method checkFeatureRemoval: got unknown input Feature!");
		}
		
		// first see if any parameters exist that involve the
		// given feature:
		Collection myParameters = this.getParameterTuples();
		Iterator parIt = myParameters.iterator();
		while (parIt.hasNext()) {
			Parameter myPar = (Parameter) parIt.next();
			ParameterObject myParObj = (ParameterObject) myPar.getTheParameterObject();
			if (myParObj instanceof Feature && 
				theInputConcept.hasFeature((Feature) myParObj) &&
				myParObj.getName().equalsIgnoreCase(theFeature.getName())) {
				// Now we know that the given feature from the given input concept
				// is used as a parameter in this step. Decide whether it is 
				// a single parameter and whether it is obligatory:
				OpParam theParamInfo = this.findOpParamForParam(myPar);
				if (theParamInfo == null) {
					throw new M4Exception("Step '" + this.getName() +
							"', method checkFeatureRemoval: found no OpParam for Parameter '" +
							myPar.getName() + "'!");
				}
				if (theParamInfo.isOptional()) {
					return null; // an optional parameter may be deleted
				}
				if (theParamInfo.isArray()) {
					// if there are more than one entry, the given feature
					// may be deleted since other entries remain. But the
					// given feature must not be the last one:
					ParameterArray pa = (ParameterArray) this.getParameterArray(theParamInfo, myPar.getLoopNr());
					if (pa.getParameterObjectArray().length > 1) {
						return null;
					}
					else {
						return "The list of entries in the obligatory parameter '" +
							theParamInfo.getName() + "' in loop number " +
							myPar.getLoopNr() + " of Step '" + this.getName() +
							"' would be empty!";
					}
				}
				else {
					return "The obligatory parameter '" +
							theParamInfo.getName() + "' in loop number " +
							myPar.getLoopNr() + " of Step '" + this.getName() +
							"' would be deleted!";
				}
			}
		}
		
		return null;
	}
	
	// attempts to find the OpParam of this step's operator that 
	// holds the info for the given parameter
	private OpParam findOpParamForParam(Parameter myParameter) throws M4Exception {
		Collection allOpParams = this.getTheOperator().getOpParams();
		if (allOpParams != null) {
			Iterator it = allOpParams.iterator();
			while (it.hasNext()) {
				OpParam myOpPar = (OpParam) it.next();
				if (this.parameterNamesMatch(myOpPar.getName(), myParameter)) {
					return myOpPar;
				}
			}
		}
		return null;
	}	
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#usesLoopsForCoordination()
	 */
	public boolean usesLoopsForCoordination() throws M4Exception {
		
		// check if any parameters exist that are coordinated:
		Iterator it = this.getTheOperator().getOpParamsIterator();
		while (it.hasNext()) {
			OpParam anOpPar = (OpParam) it.next();
			if (anOpPar.isCoordinated()) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * This method must only be called if no output objects existed
	 * for this Step before!
	 * This method creates the output concept or output feature(s) 
	 * for this step. What is to be created is specified by the given 
	 * OpParam object. The name(s) for the new object(s) are given 
	 * in the collection theNames. If there are both output concept AND
	 * output feature(s), please call this method FIRST with the concept!
	 * 
	 * @param theOpParam the OpParam object specifying which output parameter is meant
	 * @param theNames a Collection of Strings giving the name(s) for the new object(s)
	 * @throws M4Exception
	 */
	public void createOutput(edu.udo.cs.miningmart.m4.OpParam theOpParam, Collection theNames) throws M4Exception {
		boolean create = true;
		if (theNames == null || theNames.isEmpty()) {
			return;
		}
		this.workOnOutput(theOpParam, theNames, create);
	}

	/**
	 * @see edu.udo.cs.miningmart.m4.Step#propagateOutputChanges
	 */
	public void propagateOutputChanges() throws M4Exception {
		
		// Secondly search through the successors of this step and
		// adapt their outputs to changed inputs, until no output
		// is changed on a complete search level.
		// A breadth-first search is used.
		
		if (this.getTheCase() == null)
			return;
		
		// initialise all steps to have search level 0:
		Iterator stepsIt = this.getTheCase().getStepIterator();
		while (stepsIt.hasNext()) {
			Step oneStep = (Step) stepsIt.next();
			oneStep.bfsLevel = 0;
		}
		// use a queue to organise the BFS:
		Vector myQueue = new Vector();
		
		// initialise values:
		this.bfsLevel = 1;
		Step nextStep = this;
		int currentBfsLevel = 1;
		boolean noStepOfThisLevelHasChangedItsOutput = true;
		Collection stepsToBeUpdated = this.getTheCase().getDependentStepsFor(this);
		
		// loop until the queue is empty:
		while (nextStep != null) {
			
			// if any predecessors of the current step have not been visited yet,
			// ignore this step here (and wait till it is visited for the last time):
			if ((this != nextStep) && 
					( ! this.havePredecessorsBeenVisited(nextStep, myQueue, stepsToBeUpdated))) {
				nextStep.bfsLevel = 0;
				nextStep = (myQueue.isEmpty() ? null : (Step) myQueue.remove(0));
				continue;
			}
			int nextBfsLevel = nextStep.bfsLevel;
			// check if a new level is reached:
			if (nextBfsLevel > currentBfsLevel) {
				currentBfsLevel = nextBfsLevel;
				// if there have been no changes to the output and 
				// there are no more input parameters of following steps to re-create,
				// stop propagation:
				if (noStepOfThisLevelHasChangedItsOutput && 
						(this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames == null ||
						 this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames.isEmpty())) {
					return; 
				}
				noStepOfThisLevelHasChangedItsOutput = true;
			}
			
			// also recreate the input parameters of the successors:
			if (this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames != null)
				nextStep.recreateInputParametersForThisStep(this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames);
			
			// see if any changes occur on this level:
			if (nextStep.equals(this) || nextStep.adaptOutputToChangedInput()) {
				noStepOfThisLevelHasChangedItsOutput = false;
			}
			// enqueue (the visible part of) the next level:
			Collection successors = new Vector(nextStep.getSuccessors()); 
			// also add all successors that use the same input without
			// creating an output concept:
			successors.addAll(nextStep.getSuccessorsUntilOutputConceptIsCreated());
			if (successors != null && ( ! successors.isEmpty())) {
				Iterator it = successors.iterator();
				while (it.hasNext()) {
					Step succ = (Step) it.next();
					if (succ.bfsLevel == 0 && ( ! myQueue.contains(succ))) {
						succ.bfsLevel = nextBfsLevel + 1;
						myQueue.add(succ);
					}
				}
			}
			nextStep = (myQueue.isEmpty() ? null : (Step) myQueue.remove(0));
		}
		// no longer valid:
		this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames = null;
	}
	
	// checks whether ALL predecessors of the given step that are
	// among the given relevantSteps have already been visited during 
	// the current propagation (see method propagateOutputChanges)
	private boolean havePredecessorsBeenVisited(
			Step currentStep, 
			Vector theSearchQueue,
			Collection relevantSteps) 
	throws M4Exception {
		Collection preds = currentStep.getAllPredecessors();
		if (preds != null && ( ! preds.isEmpty())) {
			Iterator predIt = preds.iterator();
			while (predIt.hasNext()) {
				Step predecessor = (Step) predIt.next();
				if ( ! relevantSteps.contains(predecessor))
					continue;
				if (predecessor.bfsLevel == 0)
					return false;
				if (theSearchQueue.contains(predecessor))
					return false;
			}
		}
		return true;
	}
	
	/*
	 * This method handles changes and updates to the output concept and output 
	 * attributes of this step. The first parameter is the OpParam of the output
	 * parameter that has been changed or must be created. When several output
	 * parameters of this step exist, it is necessary to call this method FIRST
	 * with the concept OpParams, THEN with the BaseAttribute OpParams.
	 * If the OpParam is NULL, this means that all outputs of the step have to
	 * be adjusted to possibly changed inputs.
	 * The second parameter of this method gives the new or changed names of 
	 * the objects that realise the output parameter. It should be NULL whenever
	 * the first parameter of this method is NULL.
	 * The third parameter of this method is a boolean flag that indicates whether
	 * the output of this step is to be created from scratch, or is to be updated
	 * from existing output.
	 */
	private void workOnOutput(
			edu.udo.cs.miningmart.m4.OpParam theOpParam, 
			Collection theNames, 
			boolean create) 
	throws M4Exception {
		
		// ensure it IS an output param:
		if (theOpParam != null && theOpParam.isInput()) {
			throw new M4Exception("Step.create/updateOutput: got an input operator parameter!");
		}
		
		// if old parameter values exist for some reason, it's easiest to delete them and
		// create new ones (but only if we are in creation mode, ie create==TRUE):
		if (theOpParam != null && theNames != null && create) {
			this.deleteParameters(theOpParam);
		}
		
		// this collection will contain the constraints that apply to the 
		// parameter that is worked on:
		Collection applicableConstraints = this.getApplicableOutputConstraints(theOpParam);
		
		// If theNames == null, it means this method was called during propagation
		// of changed information. So we have to save the old names of the output
		// objects before updating them, because an update may include deletion,
		// at least for attributes! OutputConcepts are never deleted, only their features 
		// may change completely.
		Collection outputFeatureOpParams = null;
		if (( ! create) && (this.oldNamesOfOutputAttribs == null)) {
			outputFeatureOpParams = this.getOutputFeatureOpParams(); 
			this.oldNamesOfOutputAttribs = this.storeNamesOfOpParams(outputFeatureOpParams);
		}

		// we store the names of old parameter objects of input parameters of
		// following steps in these two maps:
		Map mapChangedInputsToOldNames = null;
		Map secondMapChangedInputsToOldNames = null;
		
		// If theOpParam == null, it means this method was called during
		// propagation of changed information; thus the updates have to be
		// done for all OpParams: relations, concepts and attributes, in this order. 
		// Otherwise only for the type given by theOpParam. 
		
		// check if we work on a Relation:
		if (theOpParam == null || theOpParam.isRelationParameter()) {
			this.checkWorkOnOutputRelation(theOpParam, theNames, create);
		}
		
		// check if we work on a Concept:
		if (theOpParam == null || theOpParam.isConceptParameter()) {
			
			String theName = (create ? this.extractSingleName(theNames) : null);			
			mapChangedInputsToOldNames = this.workOnOutputConcept(theName, create);			
		}		
		
		// check if we work on a Feature:
		if (theOpParam == null || theOpParam.isBaseAttribParameter() || theOpParam.isFeatureParameter()) {
			
			// determine the right loop numbering:
			int startLoopNr = this.getStartOrEndLoopNumber(theOpParam, theNames, true);
			int endLoopNr = this.getStartOrEndLoopNumber(theOpParam, theNames, false);
			
			// work on one output BA (array) for each loop:
			for (int loopNr = startLoopNr; loopNr <= endLoopNr; loopNr++) {
				if (create) {
					// Here we create the output BAs, using the given names:
					Collection namesForLoop = this.getNamesForThisLoop(theOpParam, theNames, startLoopNr, loopNr);
					secondMapChangedInputsToOldNames = this.workOnOutputBAs(applicableConstraints, theOpParam, namesForLoop, loopNr, create);
				}
				else {
					// Here we only update the output BAs.
					
					// Either there is one given OpParam to be updated, or the given OpParam
					// is NULL meaning that all output feature parameters must be updated.
					// The Collection 'outputFeatureOpParams' already contains all 
					// output feature OpParams:
					if (theOpParam != null) {
						outputFeatureOpParams = new Vector();
						outputFeatureOpParams.add(theOpParam);
					}
					
					// Now updating all 'outputFeatureOpParams' is correct in any case.
					secondMapChangedInputsToOldNames = 
						this.updateOutputFeatures(outputFeatureOpParams, this.oldNamesOfOutputAttribs, startLoopNr, loopNr);					
				}
			}
			// now that the features have been dealt with, forget the old map:
			this.oldNamesOfOutputAttribs = null;
		}
		
		// merge the two maps (mapping input parameters of following steps
		// to their old parameter object names) and store the merged map:
		if (mapChangedInputsToOldNames == null) 
			this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames = secondMapChangedInputsToOldNames;
		else if (secondMapChangedInputsToOldNames == null)
			this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames = mapChangedInputsToOldNames;
		else {
			// add entries from second map to first map for merging:
			Iterator entryIt = secondMapChangedInputsToOldNames.entrySet().iterator();
			while (entryIt.hasNext()) {
				Map.Entry myEntry = (Map.Entry) entryIt.next();
				mapChangedInputsToOldNames.put(myEntry.getKey(), myEntry.getValue());
			}
			this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames = mapChangedInputsToOldNames;
		}
	}

	// This method adjusts the given output feature OpParams, whose old names
	// are given in the map, according to the constraints.
	private Map updateOutputFeatures(
			Collection featureOpParams, 
			Map mapFromOpParamsToOldNames,
			int startLoopNr,
			int currentLoopNumber)
	throws M4Exception {
		if (mapFromOpParamsToOldNames == null || featureOpParams == null) {
			return null;
		}
		Iterator it = featureOpParams.iterator();
		Map mapChangedInputParamsOfFollowingStepsToOldNames = null;
		while (it.hasNext()) {
			OpParam myOutputFeatureOpPar = (OpParam) it.next();
			Collection applicableConstraints = this.getApplicableOutputConstraints(myOutputFeatureOpPar);
			Collection names = (Collection) mapFromOpParamsToOldNames.get(myOutputFeatureOpPar);
			if ( ! myOutputFeatureOpPar.isCoordinated()) {
				Collection namesForThisLoop = this.removeLoopsFromNames(names, currentLoopNumber);
				mapChangedInputParamsOfFollowingStepsToOldNames = this.workOnOutputBAs( 
						applicableConstraints, 
			            myOutputFeatureOpPar, 
						namesForThisLoop, 
						currentLoopNumber, 
						false);
			}
			else {
				// for coordinated parameters use loops 
				// (the outer for-loop is executed only once in this case)
				int newStartLoopNr = 1;
				int newEndLoopNr = (names != null ? names.size() : this.getHighestLoopNr(myOutputFeatureOpPar));					
				for (int loop = newStartLoopNr; loop <= newEndLoopNr; loop++) {
					Collection namesForThisLoop = this.removeLoopsFromNames(names, loop);
					mapChangedInputParamsOfFollowingStepsToOldNames = this.workOnOutputBAs( 
							applicableConstraints, 
							myOutputFeatureOpPar, 
							namesForThisLoop,
							loop, 
							false);
				}
			}
		}
		return mapChangedInputParamsOfFollowingStepsToOldNames;
	}
	
	// returns a collection of the names in the given collection which
	// have their loop-identifying suffix removed, and have the given 
	// loop number
	private Collection removeLoopsFromNames(Collection theNames, int loopNr) {
		if (theNames == null) {
			return null;
		}
		Vector ret = new Vector();
		Iterator it = theNames.iterator();
		while (it.hasNext()) {
			String aName = (String) it.next();
			String suffix = internalNameSeparator + loopNr;
			if (aName.endsWith(suffix)) {
				int endIndex = aName.length() - suffix.length();
				String originalName = aName.substring(0, endIndex);
				ret.add(originalName);
			}
		}
		return ret;
	}
	
	// adds the loop number to the name, separated by an internal marker
	private String addLoopNumberToName(String name, int loopNr) {
		return name + internalNameSeparator + loopNr;
	}
	
	// separates the name from its loop number
	private String getNameWithoutLoopNumber(String nameWithLoop) {
		if (nameWithLoop == null)
			return null;
		int index = nameWithLoop.indexOf(internalNameSeparator);
		if (index > -1) {
			String prefix = nameWithLoop.substring(0, index);
			return prefix;
		}
		return nameWithLoop;
	}
	
	// separates the loop number
	private int getLoopNumberWithoutName(String nameWithLoop) {
		if (nameWithLoop == null)
			return -1;
		int index = nameWithLoop.indexOf(internalNameSeparator);
		if (index > -1) {
			String suffix = nameWithLoop.substring(index + internalNameSeparator.length());
			int loop = -1;
			try {
				loop = Integer.parseInt(suffix);
			}
			catch (NumberFormatException n) {	}
			return loop;
		}		
		return -1;
	}
	
	/** 
	 * This method ensures that the output parameters of this Step
	 * reflect the specification of the operator and the current instances
	 * of the input parameters.
	 * TRUE is returned iff anything in the output has changed
	 */
	public boolean adaptOutputToChangedInput() throws M4Exception {
		// store the previous output:
		Map outputOpParamsToInfos = this.storeCurrentOutput();
		// do any updates:
		this.workOnOutput(null, null, false);
		// see if anything has changed:
		return this.compareOutputs(outputOpParamsToInfos);
	}
	
	// compares the conceptual infos of output parameter objects of this step to those
	// given in the map; returns TRUE if there are any differences
	private boolean compareOutputs(Map outputOpParamsToParamObjects) throws M4Exception {
		if (outputOpParamsToParamObjects == null) {
			return false;
		}
		Map currentMap = this.storeCurrentOutput();
		// now two maps must be compared:
		Iterator allOpParamsIt = currentMap.keySet().iterator();
		while (allOpParamsIt.hasNext())		 {
			OpParam outputOpParam = (OpParam) allOpParamsIt.next();
			Collection current = (Collection) currentMap.get(outputOpParam);
			Collection previous = (Collection) outputOpParamsToParamObjects.get(outputOpParam);
			if (current.size() != previous.size()) {
				return true;
			}
			Iterator currentIt = current.iterator();
			boolean differenceFound = false;
			while (currentIt.hasNext() && ( ! differenceFound)) {
				Object currentInfoObj = currentIt.next();
				if (currentInfoObj instanceof ConceptualAttributeInfo) {
					if ( ! ((ConceptualAttributeInfo) currentInfoObj).hasMatchIn(previous)) {
						differenceFound = true;
					}
				}
				if (currentInfoObj instanceof ConceptualConceptInfo) {
					if ( ! ((ConceptualConceptInfo) currentInfoObj).hasMatchIn(previous)) {
						differenceFound = true;
					}
				}
			}
			if (differenceFound)
				return true;
		}
		return false;
	}
	
	// returns a map that maps the output OpParams of this step
	// to their current parameter objects
	private Map storeCurrentOutput() throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		Iterator outOpParamsIt = this.getTheOperator().getAllOutputOperatorParameters().iterator();
		Map theMap = new HashMap();
		while (outOpParamsIt.hasNext()) {
			OpParam outputOpPar = (OpParam) outOpParamsIt.next();
			Iterator paramsIt = this.getParameterTuples(outputOpPar).iterator();
			Vector infoObjects = new Vector();
			while (paramsIt.hasNext()) {
				Parameter myParam = (Parameter) paramsIt.next();
				ParameterObject theObj = (ParameterObject) myParam.getTheParameterObject();
				if (theObj instanceof Concept) {
					infoObjects.add(new ConceptualConceptInfo((Concept) theObj));
				}
				if (theObj instanceof BaseAttribute) {
					infoObjects.add(new ConceptualAttributeInfo((BaseAttribute) theObj));
				}
			}
			theMap.put(outputOpPar, infoObjects);
		}
		return theMap;
	}
	
	// This method returns the correct INTERNAL loop numbering
	// to start with or to end with, for the given OpParam; the boolean
	// flag decides between start and end (TRUE -> start).
	// The given collection (theNames) can decide about the end loop number
	// (by its size) if it is not NULL.
	private int getStartOrEndLoopNumber(
			edu.udo.cs.miningmart.m4.OpParam theOpParam, 
			Collection theNames,
			boolean start) 
	throws M4Exception {
		int startLoopNr = 0, endLoopNr = 0;
		if (this.getLoopCount() > 0) {
			startLoopNr = 1;
			endLoopNr = this.getLoopCount();
		}
		if (theOpParam != null && theOpParam.isCoordinated()) { // if coordinated it can't be loopable
			startLoopNr = 1;
			if (theNames != null) {
				endLoopNr = theNames.size();					
			}
			else {
				endLoopNr = this.getHighestLoopNr(theOpParam);
			}					
		}
		return (start ? startLoopNr : endLoopNr);
	}
	
	// this method ensures that any output relation is updated:
	private void checkWorkOnOutputRelation(edu.udo.cs.miningmart.m4.OpParam theOpParam, Collection theNames, boolean create) 
	throws M4Exception {
		if (this.getTheOperator() == null)
			return;
		// theOpParam may be null because this method might have been called 
		// during propagation of updates. If so, check if this operator produces 
		// a relation at all:
		OpParam relationOpParam = null;
		if (theOpParam == null) {
			// iterate through all output OpParams to find relations:
			Iterator it = this.getTheOperator().getAllOutputOperatorParameters().iterator();
			while (it.hasNext()) {
				OpParam anOpPar = (OpParam) it.next();
				if (anOpPar.isRelationParameter()) {
					relationOpParam = anOpPar;
				}
			}			
		}
		else {
			relationOpParam = (OpParam) theOpParam;
		}
		
		// If a relation was found, go ahead:
		if (relationOpParam != null) {
			// extract the name:
			String theName = null;
			if (create) {
				theName = this.extractSingleName(theNames);			
			}

			// Finally, work with the relation (create or adapt it):
			this.workOnOutputRelation(relationOpParam, theName, create);				
		}
	}
	
	// this method expects and returns a single String, either directly
	// in the given collection, or in a single collection inside that collection.
	private String extractSingleName(Collection theNames) throws M4Exception {
		if (theNames == null || theNames.size() != 1) {
			throw new M4Exception("Step '" + this.getName() + "': found 0 or several names for output parameter, but need exactly one!");
		}
		// extract the name:
		String theName = null;
		try {
			theName = (String) theNames.iterator().next();
		}
		catch (ClassCastException cce) {
			// it may have been a collection of collections, since some OpParams
			// have parameter arrays for different loops:
			Collection c = (Collection) theNames.iterator().next();
			theName = (String) c.iterator().next();
		}
		return theName;
	}
	
	// This method removes all parameters and parameter objects from this Step
	// that belong to the given OpParam
	private void deleteParameters(edu.udo.cs.miningmart.m4.OpParam theOpParam) 
	throws M4Exception {
		int endLoop = this.getHighestLoopNr(theOpParam);
		int startLoop = (theOpParam.isCoordinated() || (theOpParam.isLoopable() && this.getLoopCount() > 0)) ? 1 : 0;
		for (int l = startLoop; l <= endLoop; l++) {
			Collection c = this.getParameter(theOpParam, l);
			if (c != null) {
				Iterator parIt = c.iterator();
				while (parIt.hasNext()) {
					ParameterObject myParObj = (ParameterObject) parIt.next();
					myParObj.deleteSoon();
				}
				// still necessary??:
				this.setParameter(theOpParam, new Vector(), l);
			}
		}
	}
	
	// returns all constraints that involve the given OpParam; if
	// the given OpParam is null, this method returns all constraints 
	// that apply to any output OpParam of this Step
	private Collection getApplicableOutputConstraints(edu.udo.cs.miningmart.m4.OpParam theOpParam) 
	throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		Collection applicableConstraints = new Vector(); 
		
		if (theOpParam != null) {
			applicableConstraints = theOpParam.getApplicableConstraints();
			if (applicableConstraints.isEmpty() && 
					( ! this.getTheOperator().getName().equals("SpecifiedStatistics")) &&
					( ! this.getTheOperator().getName().equals("SegmentationStratified")) &&
					( ! this.getTheOperator().getName().equals("Unsegment")) &&
					( ! this.getTheOperator().getName().equals("CreateOneToManyRelation")) &&
					( ! this.getTheOperator().getName().equals("CreateManyToManyRelation")) &&
					( ! this.getTheOperator().getName().equals("ReversePivotize"))) {
				throw new M4Exception("Step '" + this.getName() + "': the operator does not specify the output parameters (missing constraints on output parameters)!");
			}
		}
		else {
			Iterator it = this.getTheOperator().getAllOutputOperatorParameters().iterator();
			while (it.hasNext()) {
				OpParam anOutputOpPar = (OpParam) it.next();
				if (anOutputOpPar.isConceptParameter()) {
					applicableConstraints.addAll(anOutputOpPar.getApplicableConstraints());
				}
			}
		}
		
		return applicableConstraints;		
	}
	
	// create or adapt a relation that is created by this step
	private void workOnOutputRelation(edu.udo.cs.miningmart.m4.OpParam theOpParam, String theName, boolean create) 
	throws M4Exception {
		// first ensure that there is a relation:
		Relation rel = null;
		boolean isManyToMany = false;
		if (create) {
			 isManyToMany = this.createOutputRelationWithoutKeys(theOpParam, theName);
		}
		
		// fetch the relation:
		rel = (Relation) this.getSingleParameterObject(theOpParam.getName());
		
		// second, create or set its keys:
		if (rel != null) {
			this.adjustKeysOfRelation(theOpParam, rel, isManyToMany);
		}
	}
	
	// only creates the relation with FromConcept and ToConcept, and with
	// cross columnset if any exists. Returns TRUE iff the relation created is
	// a many-to-many relation, and FALSE iff it is a one-to-many relation.
	private boolean createOutputRelationWithoutKeys(edu.udo.cs.miningmart.m4.OpParam theOpParam, String theName) 
	throws M4Exception {
		Relation rel = (Relation) this.getM4Db().createNewInstance(Relation.class);
		rel.setName(theName);
		Vector v = new Vector();
		v.add(rel);
		this.setParameter(theOpParam, v, 0);
		
		Concept fromConcept = null;
		Concept toConcept = null;
		Columnset crossLinkCs = null;
		boolean isManyToMany = false;
		
		// use the constraints to get the parameter names that
		// give the from and to and cross concept:
		String fromConceptName = null;
		String toConceptName = null;
		String crossConceptName = null;
		Collection constrColl = theOpParam.getApplicableConstraints();
		if (constrColl != null) {
			Iterator constrIt = constrColl.iterator();
			while (constrIt.hasNext()) {
				Constraint myConstraint = (Constraint) constrIt.next();
				if (myConstraint.getType().equals(Constraint.TYPE_TO_CON_FOR_REL)) {
					toConceptName = myConstraint.getObj2();
				}
				if (myConstraint.getType().equals(Constraint.TYPE_FROM_CON_FOR_REL)) {
					fromConceptName = myConstraint.getObj2();
				}
				if (myConstraint.getType().equals(Constraint.TYPE_CROSS_CON_FOR_REL)) {
					crossConceptName = myConstraint.getObj2();
				}
			}
		}
		if (fromConceptName == null || toConceptName == null) {
			throw new M4Exception("Step '" + this.getName() + "': could not determine the FROM or TO concept parameter name for creating the relation!");
		}
		// now get the concepts:
		fromConcept = (Concept) this.getSingleParameterObject(fromConceptName);			
		toConcept = (Concept) this.getSingleParameterObject(toConceptName);
		Concept crossConcept = (Concept) this.getSingleParameterObject(crossConceptName);
		if (crossConcept != null) {
			isManyToMany = true;
			crossLinkCs = (Columnset) crossConcept.getCurrentColumnSet();
		}
		
		if (fromConcept == null || toConcept == null) {
			throw new M4Exception("Step '" + this.getName() + "': could not determine FromConcept and/or ToConcept for output relation!");
		}
		rel.setTheFromConcept(fromConcept);
		rel.setTheToConcept(toConcept);
		if (crossLinkCs != null) {
			rel.setCrossLinkColumnSet(crossLinkCs);
		}
		return isManyToMany;
	}
	
	// sets or changes the keys of the given relation, according to the
	// parameter settings of this step
	private void adjustKeysOfRelation(
			edu.udo.cs.miningmart.m4.OpParam theOpParam,
			Relation rel, 
			boolean isManyToMany) throws M4Exception {
		
		// old keys might exist, so we delete them:
		ForeignKey oldFrom = (ForeignKey) rel.getFromKey();
		if (oldFrom != null) oldFrom.deleteSoon();
		rel.removeFromKey();
		ForeignKey oldTo = (ForeignKey) rel.getToKey();
		if (oldTo != null) oldTo.deleteSoon();
		rel.removeToKey();
		
		// these collections will take the key BaseAttributes:
		Collection fromConceptKeyAttribs = null;
		Collection toConceptKeyAttribs = null;
		Collection crossConceptToFromAttribs = null;
		Collection crossConceptToToAttribs = null;

		// use the constraints to get the parameter names that
		// give the from and to and cross keys:
		String fromConceptKeyName = null;
		String toConceptKeyName = null;
		String crossConceptFromKeyName = null;
		String crossConceptToKeyName = null;
		Collection constrColl = theOpParam.getApplicableConstraints();
		if (constrColl != null) {
			Iterator constrIt = constrColl.iterator();
			while (constrIt.hasNext()) {
				Constraint myConstraint = (Constraint) constrIt.next();
				if (myConstraint.getType().equals(Constraint.TYPE_FROM_KEY_FOR_REL)) {
					fromConceptKeyName = myConstraint.getObj2();
				}
				if (myConstraint.getType().equals(Constraint.TYPE_TO_KEY_FOR_REL)) {
					toConceptKeyName = myConstraint.getObj2();
				}
				if (myConstraint.getType().equals(Constraint.TYPE_CROSS_FROM_KEY_FOR_REL)) {
					crossConceptFromKeyName = myConstraint.getObj2();
				}
				if (myConstraint.getType().equals(Constraint.TYPE_CROSS_TO_KEY_FOR_REL)) {
					crossConceptToKeyName = myConstraint.getObj2();
				}
			}
		}
		if (fromConceptKeyName == null || toConceptKeyName == null) {
			throw new M4Exception("Step '" + this.getName() + "': could not determine the FROM or TO KEY parameter name for creating the relation!");
		}
			
		// get attribs from parameter settings:
		fromConceptKeyAttribs = this.getParameterObjects(fromConceptKeyName);
		toConceptKeyAttribs = this.getParameterObjects(toConceptKeyName);
			
		if (isManyToMany) {
			crossConceptToFromAttribs = this.getParameterObjects(crossConceptFromKeyName);
			crossConceptToToAttribs = this.getParameterObjects(crossConceptToKeyName);
		}
				
		// now the attribs for the keys are known, set them:
		if ( ! isManyToMany) {
			// one foreign key is needed for a one-to-many relation:
			ForeignKey theForeignKey = (ForeignKey) this.getM4Db().createNewInstance(ForeignKey.class);
			theForeignKey.setForeignKeyColumnset(rel.getTheFromConcept().getCurrentColumnSet());
			theForeignKey.setPrimaryKeyColumnset(rel.getTheToConcept().getCurrentColumnSet());		
			theForeignKey.createColumnLinksFromAttribs(fromConceptKeyAttribs, toConceptKeyAttribs);		
			theForeignKey.setName(rel.getName() + "_FK");
			
			rel.setFromKey(theForeignKey);
			rel.setToKey(null); // just to make it clear
			rel.setCrossLinkColumnSet(null); // dito				
		}
		else {
			// create two foreign keys for the many-to-many relation:
			ForeignKey linkCrossToFromConcept = (ForeignKey) this.getM4Db().createNewInstance(ForeignKey.class);
			ForeignKey linkCrossToToConcept = (ForeignKey) this.getM4Db().createNewInstance(ForeignKey.class);
			String crossTableName = rel.getCrossLinkColumnSet().getName();
			linkCrossToFromConcept.setName(crossTableName + "_" + rel.getTheFromConcept().getName() + "_FK");
			linkCrossToToConcept.setName(crossTableName + "_" + rel.getTheToConcept().getName() + "_FK");
			
			// set the columnsets into the two foreign keys:
			linkCrossToFromConcept.setPrimaryKeyColumnset(rel.getTheFromConcept().getCurrentColumnSet());
			linkCrossToFromConcept.setForeignKeyColumnset(rel.getCrossLinkColumnSet());
			linkCrossToToConcept.setPrimaryKeyColumnset(rel.getTheToConcept().getCurrentColumnSet());
			linkCrossToToConcept.setForeignKeyColumnset(rel.getCrossLinkColumnSet());
			
			// create the column links:
			linkCrossToFromConcept.createColumnLinksFromAttribs(crossConceptToFromAttribs, fromConceptKeyAttribs);
			linkCrossToToConcept.createColumnLinksFromAttribs(crossConceptToToAttribs, toConceptKeyAttribs);
			
			// set the foreign keys into the relation:
			rel.setFromKey(linkCrossToFromConcept);
			rel.setToKey(linkCrossToToConcept);
		}
	}
	
	/*
	 * Collects all OpParams that represent output feature parameters.
	 */
	private Collection getOutputFeatureOpParams() throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		Iterator it = this.getTheOperator().getAllOutputOperatorParameters().iterator();
		Vector ret = new Vector();
		while (it.hasNext()) {
			OpParam outputOpPar = (OpParam) it.next();
			if (outputOpPar.isBaseAttribParameter() || outputOpPar.isFeatureParameter()) {
				ret.add(outputOpPar);
			}
		}
		return ret;
	}
	
	private Collection getNamesForThisLoop(edu.udo.cs.miningmart.m4.OpParam theOpParam, Collection theNames, int startLoopNr, int loopNr) throws M4Exception {

		if (theOpParam != null && (theOpParam.isLoopable() || theOpParam.isCoordinated())) {
			// here we have a collection of collections:
			Iterator it = theNames.iterator();
			int counter = startLoopNr;
			while (it.hasNext()) {
				Collection namesForOneLoop = (Collection) it.next();
				if (counter == loopNr) {
					return namesForOneLoop;
				}
				counter++;
			}		
			throw new M4Exception("Step.create/updateOutput: method getNamesForThisLoop could not find the names for loop number " + loopNr + "!");				
		}
		return theNames;
	}
	
	/*
	 * Creates a HashMap that maps each given operator parameter to a Collection 
	 * with the names occurring in this parameter (there may be more
	 * than one name per parameter, if it is an array or looped parameter).
	 */
	private HashMap storeNamesOfOpParams(Collection opParams) throws M4Exception {
		if (opParams == null) {
			return null;
		}
		Iterator it = opParams.iterator();
		HashMap theMap = new HashMap();
		while (it.hasNext()) {
			OpParam outputOpPar = (OpParam) it.next();
			Collection theNames = new Vector();
			Collection params = this.getParameterTuples(outputOpPar);			
			Iterator tupleIt = params.iterator();
			while (tupleIt.hasNext()) {
				Parameter myPar = (Parameter) tupleIt.next();
				ParameterObject myParObj = (ParameterObject) myPar.getTheParameterObject();
				if (myParObj != null) {
					String nameToStore = this.addLoopNumberToName(myParObj.getName(), myPar.getLoopNr());
					theNames.add(nameToStore);
				}
			}
			theMap.put(outputOpPar, theNames);			
		}
		return theMap;
	}
	
	/*
	 * This method creates or adjusts the output concept of this step
	 * according to the constraints and the current input parameters.
	 */
	private Map workOnOutputConcept( 
			String theName,
			boolean create) throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		OpParam theOutputConceptOpParam = this.getOutputConceptOpParamForThisStep();
		
		// some steps never have an output concept:
		if (theOutputConceptOpParam == null) {
			return null;
		}
		
		// first create an empty output concept; in case of 
		// update it exists already:					
		if (create) {
			this.createEmptyOutputConceptForThisStep(theName, theOutputConceptOpParam);
		}
			// get the freshly created (or the old) output concept:
		Concept outputConcept = this.checkConceptExists(theOutputConceptOpParam);
	
		// second find out which features are to be copied from the
		// input concept into the output concept, without change:
		Collection featuresForOutputConcept = this.getInputFeaturesForOutputConceptByConstraints();
	
		// check that all input features exist: some may have been deleted by the user;
		// if so, they are removed from the collection:
		featuresForOutputConcept = this.checkExistenceOfInputFeatures(featuresForOutputConcept);

		// third copy those features to the output concept:
		Map oldInputParameterNames = null;
		if (featuresForOutputConcept != null && ( ! featuresForOutputConcept.isEmpty())) {
			oldInputParameterNames = this.copyFeaturesIntoConcept(featuresForOutputConcept, outputConcept);
		}
		
		// fourth check for other constraints that do not involve output feature
		// parameters, but apply to input feature parameters:
		this.checkForMatchingOutputAttribs(outputConcept);
		this.checkForOutputAttribsToCreateFromInput(outputConcept);

		// fifth check if any input parameters of following steps that
		// have been deleted can be re-assigned:
		// -- is now done in global call to propagateOutputChanges!!
		// if (oldInputParameterNames != null) {
		// 	this.recreateInputParameters(oldInputParameterNames);
		// }
		
		// finally, check for assertions and add projections, subconcept links
		// and relationships:
		if (create) {
			this.createSemanticLinksAccordingToAssertions();
		}
		return oldInputParameterNames;
	}
	
	// The given map maps parameters of other steps to the old names
	// of their parameter objects. If such objects still exist in the
	// output of this step, the assignment is done again.
	/*
	private void recreateInputParameters(Map parametersToOldNames) throws M4Exception {
		Collection allInputParametersThatWereDeleted = parametersToOldNames.keySet();
		Iterator paramIt = allInputParametersThatWereDeleted.iterator();
		while (paramIt.hasNext()) {
			Parameter inputParam = (Parameter) paramIt.next();
			// only do this if the parameter is indeed deleted:
			if (inputParam != null && inputParam.getTheParameterObject() == null) {
				// get the step whose input parameter it is (that has been deleted):
				Step itsStep = (Step) inputParam.getTheStep();
				if (itsStep != null) {
					OpParam inputOpParam = (OpParam) itsStep.getTheOperator().getOpParam(this.getParameterNameWithoutSuffix(inputParam.getName()));
					if (inputOpParam != null) {
						// this is how the parameter object was called before deletion:
						String[] pairOfNames = (String[]) parametersToOldNames.get(inputParam);
						String oldParamName = this.getNameWithoutLoopNumber(pairOfNames[0]);
						String newParamName = (pairOfNames[1] != null ? this.getNameWithoutLoopNumber(pairOfNames[1]) : null);
						int oldLoopNumber = this.getLoopNumberWithoutName(pairOfNames[0]);
						if (oldLoopNumber == -1) {
							throw new M4Exception("Unrecognised loop number in Step.recreateInputParameters!");
						}
						if (inputOpParam.isConceptParameter()) {
							// the deleted parameter object can only be
							// the output of this step:
							Concept myOutCon = (Concept) this.getOutputConcept();
							if (myOutCon != null && 
									(  myOutCon.getName().equals(oldParamName)
									   ||
									   myOutCon.getName().equals(newParamName))) {
								inputParam.setTheParameterObject(myOutCon);
							}
						}
						if (inputOpParam.isBaseAttribParameter() || inputOpParam.isFeatureParameter()) {
							// look for BAs with the old (deleted) name in the output of this step,
							// because the output of this step has been modified and is used
							// as input of 'itsStep':
							Collection allMyOutputs = this.getTheOperator().getAllOutputOperatorParameters();
							Iterator outOpParIt = allMyOutputs.iterator();
							while (outOpParIt.hasNext()) {
								OpParam myOutputOpPar = (OpParam) outOpParIt.next();
								Iterator outParamsIt = this.getParameterTuples(myOutputOpPar).iterator();
								while (outParamsIt.hasNext()) {
									Parameter outPar = (Parameter) outParamsIt.next();
									if (outPar.getLoopNr() == oldLoopNumber) {
										ParameterObject outObject = (ParameterObject) outPar.getTheParameterObject();
										if (outObject instanceof Concept) {
											Iterator attribsIt = ((Concept) outObject).getAllBaseAttributes().iterator();
											while (attribsIt.hasNext()) {
												BaseAttribute myOutBA = (BaseAttribute) attribsIt.next();
												if (myOutBA.getName().equals(oldParamName) || myOutBA.getName().equals(newParamName)) {
													if (itsStep.checkFulfillsInInputConceptConstraint(inputOpParam, myOutBA))
														inputParam.setTheParameterObject(myOutBA);
												}
											}
										}
										if (outObject instanceof BaseAttribute) {
											BaseAttribute myOutBA = (BaseAttribute) outObject;
											if (myOutBA.getName().equals(oldParamName)) {
												if (itsStep.checkFulfillsInInputConceptConstraint(inputOpParam, myOutBA))
													inputParam.setTheParameterObject(myOutBA);
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	*/

	private void recreateInputParametersForThisStep(Map parametersToOldNames) throws M4Exception {
		Collection allInputParametersThatWereDeleted = new Vector(parametersToOldNames.keySet());
		Iterator paramIt = allInputParametersThatWereDeleted.iterator();
		while (paramIt.hasNext()) {
			Parameter inputParam = (Parameter) paramIt.next();
			// only do it for this step:
			if (inputParam.getTheStep() != null && inputParam.getTheStep().equals(this)) {
				String[] pairOfNames = (String[]) parametersToOldNames.get(inputParam);
				// in any case that parameter has now been handled:
				parametersToOldNames.remove(inputParam);
				// only do this if the parameter is indeed deleted:
				if (inputParam != null && inputParam.getTheParameterObject() == null) {
					OpParam inputOpParam = (OpParam) this.getTheOperator().getOpParam(this.getParameterNameWithoutSuffix(inputParam.getName()));
					if (inputOpParam != null) {
						// this is how the parameter object was called before deletion:
						String oldParamName = this.getNameWithoutLoopNumber(pairOfNames[0]);
						String newParamName = (pairOfNames[1] != null ? this.getNameWithoutLoopNumber(pairOfNames[1]) : null);
						int oldLoopNumber = this.getLoopNumberWithoutName(pairOfNames[0]);
						if (oldLoopNumber == -1) {
							throw new M4Exception("Unrecognised loop number in Step.recreateInputParameters!");
						}
						if (inputOpParam.isConceptParameter()) {
							// the deleted parameter object can only be
							// the output of a previous step:
							Collection possibleInputs = this.getPossibleConceptsForParam(inputOpParam);
							Iterator conceptsIt = possibleInputs.iterator();
							while (conceptsIt.hasNext()) {
								Concept myCandidate = (Concept) conceptsIt.next();
								if (myCandidate.getName().equals(oldParamName)
										||
									myCandidate.getName().equals(newParamName)) {
									inputParam.setTheParameterObject(myCandidate);
								}
							}
						}
						if (inputOpParam.isBaseAttribParameter() || inputOpParam.isFeatureParameter()) {
							// look for BAs with the old (deleted) name in the input of this step:							
							Collection allMyInputs = this.getAllInputConcepts();
							Iterator inConcIt = allMyInputs.iterator();
							while (inConcIt.hasNext()) {
								Concept inputConcept = (Concept) inConcIt.next();
								Iterator inAttribsIt = inputConcept.getAllBaseAttributes().iterator();
								while (inAttribsIt.hasNext()) {
									BaseAttribute myInBA = (BaseAttribute) inAttribsIt.next();
									if (myInBA.getName().equals(oldParamName) || myInBA.getName().equals(newParamName)) {
										if (this.checkFulfillsInInputConceptConstraint(inputOpParam, myInBA)) {
											inputParam.setTheParameterObject(myInBA);
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	
	// checks if the given BA fulfills an IN constraint of this step
	// saying that it must occur IN some input concept
	private boolean checkFulfillsInInputConceptConstraint(
			OpParam inputOpParamOfThisStep,
			BaseAttribute theParamObject)
	throws M4Exception {
		// go through the constraints:
		Iterator constrIt = inputOpParamOfThisStep.getApplicableConstraints().iterator();
		while (constrIt.hasNext()) {
			Constraint myConstr = (Constraint) constrIt.next();
			if (myConstr.getType().equals(Constraint.TYPE_CONTAINED_IN)) {
				String otherOpParName = myConstr.getObj1();
				if (otherOpParName.equalsIgnoreCase(inputOpParamOfThisStep.getName()))
					otherOpParName = myConstr.getObj2();
				OpParam otherOpPar = (OpParam) this.getTheOperator().getOpParam(otherOpParName);
				// the other parameter mentionend in the constrained ought to be a concept parameter:
				if (otherOpPar.isInput() && otherOpPar.isConceptParameter()) {
					// so go through the input concepts and see if the given BA 
					// occurs there:
					Iterator theConceptsIt = this.getParameterObjects(otherOpPar.getName()).iterator();
					while (theConceptsIt.hasNext()) {
						Concept oneInputConcept = (Concept) theConceptsIt.next();
						if (oneInputConcept.hasFeature(theParamObject)) {
							return true;
						}
					}
				}
			}
		}
		return false;
	}
	
	// return a Collection of BaseAttributes, all of which must be copied
	// from the input concept to the output concept 
	private Collection getInputFeaturesForOutputConceptByConstraints() 
	throws M4Exception {
		Iterator theConstraintsIt = this.getTheOperator().getConstraints().iterator();
		// Iterate through the constraints. For each constraint decide
		// what to do:
		Collection ret = new Vector();
		boolean exceptionsDone = false; 
		while (theConstraintsIt.hasNext()) {
		
			Constraint aConstr = (Constraint) theConstraintsIt.next();
			
			// simple case: output concept is copy of input concept
			if (this.outputConceptIsCopyOfInput(aConstr))  {
				
				// get the input concept:
				String nameOfSecondConstraintObject = aConstr.getObj2();
				OpParam inputConcOpPar = (OpParam) this.getTheOperator().getOpParam(nameOfSecondConstraintObject);				
				Concept inputConcept = this.checkConceptExists(inputConcOpPar);				
				if (inputConcept == null) {
					throw new M4Exception("Step '" + this.getName() +
							"': could not find input concept for copying its features to the output concept!");
				}
				ret.addAll(inputConcept.getFeatures());
			}
			
			// similarly simple: output concept features are subset of input
			// concept features; in this case it is expected that the output
			// concept features are given by an output BA parameter, so they
			// will be created in this.workOnOutputBAs()
			if (this.outputConceptHasSubsetOfInputConceptFeatures(aConstr)) {
				return new Vector();
			}
			
			// somewhat more complex: the output concept features are taken from 
			// an input parameter:
			OpParam inputOpPar = null;
			if ((inputOpPar = (OpParam) this.addFeaturesFromInputParameter(aConstr)) != null) {
							
				// if input features must go into output concept according to the current constraint, do it:
				if (inputOpPar != null) {
					// no such input parameter is looped, so use loopNr 0:
					Collection theInputFeaturesForOutputConcept = this.getParameter(inputOpPar, 0);
					if (theInputFeaturesForOutputConcept != null) 
						ret.addAll(theInputFeaturesForOutputConcept);
				}
			}			
			
			// another case is that the constraint specifies an exception
			// but otherwise all features are copied:
			if (this.copyFeaturesToOutputWithException(aConstr) &&
				 ( ! exceptionsDone)) {
				// unfortunately there can be more than one exception!
				// So collect ALL features first, then delete the exceptions:
				Iterator inputConceptsIt = this.getAllInputConcepts().iterator();
				Collection allFeatures = new Vector();
				while (inputConceptsIt.hasNext()) {
					Concept anInConc = (Concept) inputConceptsIt.next();
					allFeatures.addAll(anInConc.getAllBaseAttributes());
				}
				ret.addAll(this.removeExceptions(allFeatures));
				exceptionsDone = true;
			}
			
			// yet another case is that the features are copied from a
			// concept that is attached to an input relation:
			Concept relatedConcept = null;
			if ((relatedConcept = this.getRelatedConceptWithFeaturesToCopy(aConstr)) != null) {
				ret.addAll(relatedConcept.getAllBaseAttributes());
			}
			
			// again simple: other constraints on the output concept concern
			// additional features for them; they are dealt with in another call
			// to update/createOutput (with the OpParam for those features)
			// -> nothing else to do here
				
		} // end of loop through the constraints
		
		return ret;
	}	
	
	// returns a concept that is related to the input of this step by
	// a relation and whose features are to be copied to the output concept
	// of this step
	private Concept getRelatedConceptWithFeaturesToCopy(Constraint theConstraint) 
	throws M4Exception {
		boolean takeFeaturesFromFromConcept = false;
		if ((takeFeaturesFromFromConcept = theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_FEAT_FROM_RELFR)) 
				||
			theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_FEAT_FROM_RELTO)) {
			// find out which relation it is:
			String relParamName = theConstraint.getObj1();
			OpParam relOpPar = (OpParam) this.getTheOperator().getOpParam(relParamName);
			Collection rels = this.getParameter(relOpPar, 0);
			Iterator relIt = rels.iterator();
			Relation inputRel = null;
			if (relIt.hasNext()) {
				inputRel = (Relation) relIt.next();
			}
			if (inputRel == null) {
				throw new M4Exception("Step '" + this.getName() +
						"': could not find input relation for a constraint that specifies that output features come from a related concept!");
			}
			// now take the concept:
			Concept theConcept = (Concept) (takeFeaturesFromFromConcept ? inputRel.getTheFromConcept() : inputRel.getTheToConcept());
			if (theConcept == null) {
				throw new M4Exception("Step '" + this.getName() +
						"': could not find the related concept, for a constraint that specifies that output features come from a related concept!");
			}
			return theConcept;
		}
		return null;
	}
	
	// This method checks if there are any constraints for this step 
	// that say certain attributes in the output concept have to be 
	// matched and merged together (like the keys in joins). If such
	// a constraint exists, the matching and merging is done and the
	// attributes are added to the given concept.
	private void checkForMatchingOutputAttribs(Concept theOutputConcept) 
	throws M4Exception {
		// check if there is such a constraint:
		Iterator constrIt = this.getTheOperator().getConstraints().iterator();
		while (constrIt.hasNext()) {
			Constraint matchingConstraint = (Constraint) constrIt.next();
			if (matchingConstraint.getType().equals(Constraint.TYPE_MATCH_ATTRIBS_BY_CONCEPTS)) {
				// find out which parameter it applies to:
				String parName = matchingConstraint.getObj1();
				BaseAttribute[] matchingAttribs = (BaseAttribute[]) this.getParameter(parName, 0);
				if (matchingAttribs != null) {
					// now find all attribs of one (arbitrary) concept:
					Concept oneConcept = null;
					Vector allMatchAttribsOfOneConcept = new Vector();			
					for (int i = 0; i < matchingAttribs.length; i++) {
						if (oneConcept == null) {
							oneConcept = (Concept) matchingAttribs[i].getConcept();
							allMatchAttribsOfOneConcept.add(matchingAttribs[i]);
						}
						else {
							if (oneConcept.equals(matchingAttribs[i].getConcept())) {
								allMatchAttribsOfOneConcept.add(matchingAttribs[i]);
							}
						}
					}
					if (oneConcept == null) {
						throw new M4Exception("Step '" + this.getName() + 
								"': could not find concept(s) for attribs from parameter '" +
								parName + "', which are to be matched!");
					}
					// add only the attribs from one of the concepts:
					Iterator it = allMatchAttribsOfOneConcept.iterator();
					while (it.hasNext()) {
						BaseAttribute newBa = (BaseAttribute) it.next();
						this.updateOrCreateBaInConcept(theOutputConcept, newBa);				
					}
				}
			}
		}
	}
	
	/*
	 * This method creates or updates the output attributes that are created
	 * by using some input attributes as templates, but extending the names
	 * with some suffix.
	 */
	private void checkForOutputAttribsToCreateFromInput(Concept theOutputConcept)
	throws M4Exception {
		// check for constraints for the creation of attributes by values of
		// some parameter:
		Iterator constrIt = this.getTheOperator().getConstraints().iterator();
		while (constrIt.hasNext()) {
			Constraint createConstraint = (Constraint) constrIt.next();
			if (createConstraint.getType().equals(Constraint.TYPE_CREATE_ATTRIB_BY) ||
				createConstraint.getType().equals(Constraint.TYPE_CREATE_ATTRIB_SUFF)) {
				// find out which BaseAttribute parameter the constraint applies to:
				String parName = createConstraint.getObj1();
				BaseAttribute[] templateAttribs = (BaseAttribute[]) this.getParameter(parName, 0);
				// the parameters may be optional!:
				if (templateAttribs == null || templateAttribs.length == 0) {
					continue;
				}
				
				// if the template attribs have been renamed, get the output
				// attributes that are based on their old names and delete them:
				for (int i = 0; i < templateAttribs.length; i++) {
					String oldName = this.getTheCase().getOldNameOfChangedFeature(templateAttribs[i].getName());
					if (oldName != null) {
						Map mapToCorrespondingNames = this.getMapOfNamesToNamesWithSuffixes(createConstraint, new String[] { oldName });
						if (mapToCorrespondingNames == null)
							continue;
						Collection correspondingNames = (Collection) mapToCorrespondingNames.get(oldName);
						if (correspondingNames == null)
							return;
						this.deleteFeaturesFromConcept(correspondingNames, theOutputConcept);
					}
				}
				
				// find the names of the output attribs:
				String[] namesOfNewAttribs = this.getNamesBySuffixes(createConstraint, templateAttribs);
				// find their types:
				String typeOfNewAttribs = this.getTypeOfCreatedOutputAttribs(parName);
				// if the type could not be determined, try the type of the
				// template BA:
				if (typeOfNewAttribs == null) {
					typeOfNewAttribs = templateAttribs[0].getConceptualDataTypeName();
				}				
				if (typeOfNewAttribs == null) {
					throw new M4Exception("Step '" + this.getName() +
							"': could not determine type of some created output attributes!");
				}
				String[] types = new String[namesOfNewAttribs.length];
				for (int i = 0; i < namesOfNewAttribs.length; i++) {
					types[i] = typeOfNewAttribs;
				}
				// use the role of the template BA:
				String roleOfNewAttribs = templateAttribs[0].getRoleName();
				String[] roles = new String[namesOfNewAttribs.length];
				for (int i = 0; i < namesOfNewAttribs.length; i++) {
					roles[i] = roleOfNewAttribs;
				}
				this.createBAsInConcept(namesOfNewAttribs, types, roles, theOutputConcept);
			}
		}
	}
	
	// try to find a constraint that says what type the output attribute created
	// by using the given parameter as template should have 
	private String getTypeOfCreatedOutputAttribs(String nameOfConstrainedParameter) 
	throws M4Exception {
		String type = null;
		Iterator constrIt = this.getTheOperator().getConstraints().iterator();
		while (constrIt.hasNext()) {
			Constraint aConstraint = (Constraint) constrIt.next();
			if (aConstraint.getObj1().equalsIgnoreCase(nameOfConstrainedParameter)
					&&
				aConstraint.getType().equals(Constraint.TYPE_OUT_ATTRIB_TYPE)) {
				// check whether obj2 of this constraint gives the output type
				// directly, or via the type of another Feature parameter:
				String parName = aConstraint.getObj2();
				BaseAttribute[] templates = (BaseAttribute[]) this.getParameter(parName, 0);
				if (templates != null && templates.length > 0) {
					// obj2 seems to be a parameter name:
					type = templates[0].getConceptualDataTypeName();
				}
				else {
					// obj2 seems to be a datatype:
					type = parName;
				}
			}
		}
		return type;
	}
	
	/*
	 * Returns a map (from Strings to Collections) that associates each name of the given
	 * array of input feature names with a collection of output feature names,
	 * if this step has constraints that let the step create output features 
	 * from the given input features. In other words, this method can be used to
	 * associate names of input and output features although the names are not identical,
	 * but only if constraints exist that specify how to create these output features from
	 * these input features. 
	 */
	private Map getMapOfNamesToNamesWithSuffixes(Constraint theCreateConstraint, String[] templateBaNames)
	throws M4Exception {
		// either there is a fixed suffix:
		Map theMap = new HashMap();
		if (theCreateConstraint.getType().equals(Constraint.TYPE_CREATE_ATTRIB_SUFF)) {
			String suffix = theCreateConstraint.getObj2();
			for (int i = 0; i < templateBaNames.length; i++) {
				Vector theNames = new Vector();
				theNames.add(templateBaNames[i] + nameSeparator + suffix);
				theNames.trimToSize();
				theMap.put(templateBaNames[i], theNames);
			}
			return theMap;
		}
		// or the suffix is to be taken from a value parameter:
		if (theCreateConstraint.getType().equals(Constraint.TYPE_CREATE_ATTRIB_BY)) {
			
			// get the value parameter with the suffixes:
			String secondParamName = theCreateConstraint.getObj2();
			Value[] theValues = (Value[]) this.getParameter(secondParamName, 0);
			
			// get the BA parameter also to decide how to add suffixes:
			String firstParamName = theCreateConstraint.getObj1();
			OpParam baOpParam = (OpParam) this.getTheOperator().getOpParam(firstParamName);
			
			// if there is only one possible BA parameter but several suffixes,
			// we have to build the combinations of all suffixes! Otherwise
			// each value gives a suffix list for one BA!
			if (baOpParam.isArray()) {
				// here each value contains a list of suffixes to be
				// attached to one BA. So there must be as many values as BAs:
				if (theValues.length != templateBaNames.length) {
					throw new M4Exception("Step '" + this.getName() +
						"': there must be as many values for parameter '" + secondParamName +
						"' as for the corresponding grouped parameter!");
				}
				for (int i = 0; i < templateBaNames.length; i++) {
					String valueList = theValues[i].getValue();
					Vector theNames = new Vector();
					StringTokenizer st = new StringTokenizer(valueList, ",");
					while (st.hasMoreTokens()) {
						String value = st.nextToken().trim();
						value.replace(' ', nameSeparator);
						theNames.add(templateBaNames[i] + nameSeparator + value);
					}
					theNames.trimToSize();
					theMap.put(templateBaNames[i], theNames);
				}
			}
			else {
				// here there is only one BA, and each combination of suffixes
				// gives one name of an output BA:
				
				// except if it's a numeric value parameter, then we use one suffix
				// for each number 1 to n if n is the value:
				OpParam valueOpParam = (OpParam) this.getTheOperator().getOpParam(secondParamName);
				
				if (theValues.length == 1 && 
					valueOpParam.isValueParameter()) {
					int n = -1;
					try {
						n = Integer.parseInt(theValues[0].getValue());
					}
					catch (NumberFormatException nfe) {
						n = -1;
					}
					if (n > 0) {
						String namePrefix = templateBaNames[0];
						Vector theNames = new Vector();
						for (int i = 1; i <= n; i++) {
							theNames.add(namePrefix + nameSeparator + i);
						}
						theNames.trimToSize();
						theMap.put(templateBaNames[0], theNames);
						return theMap;
					}
				}
				String namePrefix = templateBaNames[0];
				int recursionLevel = 0;
				Vector theNames = new Vector();
				this.recurseThroughValueLists(theNames, namePrefix, theValues, recursionLevel);
				theNames.trimToSize();
				theMap.put(templateBaNames[0], theNames);
			}
		}
		// throw new M4Exception("Step '" + this.getName() + "': unknown constraint in Step.getNamesBySuffixes!");
		return theMap;
	}
	
	private String[] getNamesBySuffixes(Constraint theCreateConstraint, BaseAttribute[] templateBAs) 
	throws M4Exception {
		String[] templateNames = new String[templateBAs.length];
		for (int i = 0; i < templateNames.length; i++) {
			templateNames[i] = templateBAs[i].getName();
		}
		Map theMap = this.getMapOfNamesToNamesWithSuffixes(theCreateConstraint, templateNames);
		if (theMap == null) {
			return null;
		}
		
		Vector allOutputNames = new Vector();
		Iterator it = theMap.values().iterator();
		while (it.hasNext()) {
			Vector theOutNames = (Vector) it.next();
			allOutputNames.addAll(theOutNames);
		}
		
		// return the names with suffixes as an array:
		String[] ret = new String[allOutputNames.size()];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = (String) allOutputNames.get(i);
		}
		return ret;
	}
	
	private void recurseThroughValueLists(Vector namesToCreate, String namePrefix, Value[] valueLists, int level) {
		if (level < valueLists.length) {
			String oneValueList = valueLists[level].getValue();
			StringTokenizer st = new StringTokenizer(oneValueList, ", ");
			while (st.hasMoreTokens()) {
				String value = st.nextToken();
				String newPrefix = namePrefix + nameSeparator + value;
				this.recurseThroughValueLists(namesToCreate, newPrefix, valueLists, level+1);
			}
		}
		else {
			namesToCreate.add(namePrefix);
		}
	}
	
	// removes, from the given collection of features, all features that
	// should not be in the output concept according to a constraint
	// that says all features should be copied except those
	private Collection removeExceptions(Collection features) throws M4Exception {
		Iterator it = this.getTheOperator().getConstraints().iterator();
		Collection ret = new Vector(features);
		while (it.hasNext()) {
			Constraint constr = (Constraint) it.next();
			if (this.copyFeaturesToOutputWithException(constr)) {
				// find the parameter objects that correspond to this constraint:
				String paramName = constr.getObj1();
				OpParam opPar = (OpParam) this.getTheOperator().getOpParam(paramName);
				Iterator paramsIt = this.getParameterTuples(opPar).iterator();
				while (paramsIt.hasNext()) {
					Parameter myParam = (Parameter) paramsIt.next();
					// ... and remove them:
					ret.remove(myParam.getTheParameterObject());
				}
			}
		}
		return ret;
	}
	
	// returns true iff the given constraint says the output concept
	// should have the same features as the input concept
	private boolean outputConceptIsCopyOfInput(Constraint theConstraint) {
		if (theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_SAME_FEATURES)
				&&
			theConstraint.getObj1().equalsIgnoreCase("TheOutputConcept")
				&&
			theConstraint.getObj2().equalsIgnoreCase("TheInputConcept")) {
			return true;
		}
		return false;
	}
	
	// returns true iff the given constraint says that the output concept
	// should have a subset of the features of the input concept's features
	private boolean outputConceptHasSubsetOfInputConceptFeatures(Constraint theConstraint) {
		if (theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_CONTAINED_IN)
				&&
			theConstraint.getObj1().equalsIgnoreCase("TheOutputConcept")
				&&
			theConstraint.getObj2().equalsIgnoreCase("TheInputConcept")) {
			return true;
		}
		return false;
	}
	
	// returns the input parameter iff the given constraint says that the output concept
	// has some features which are given by an input parameter
	private edu.udo.cs.miningmart.m4.OpParam addFeaturesFromInputParameter(Constraint theConstraint) 
	throws M4Exception {		
		if (theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_CONTAINED_IN)
				&&
	        theConstraint.getObj2().equalsIgnoreCase("TheOutputConcept")) {
			String parName = theConstraint.getObj1();
			OpParam opPar = (OpParam) this.getTheOperator().getOpParam(parName);
			if (opPar != null) {
				if (opPar.isInput() && 
						(opPar.isFeatureParameter() || opPar.isBaseAttribParameter())) {
					return opPar;
				}
			}
		}
		return null;
	}
	
	private boolean copyFeaturesToOutputWithException(Constraint theConstraint) {
		if (theConstraint.getType().equalsIgnoreCase(Constraint.TYPE_SAME_FEAT_EXCEPT)
				&&
			theConstraint.getObj2().equalsIgnoreCase("TheOutputConcept")) {
			return true;
		}
		return false;
	}
	
	// This method iterates through the assertions that the operator attached to
	// this step makes. It creates the projections, subconcept links and relationships.
	private void createSemanticLinksAccordingToAssertions() throws M4Exception {

		Iterator assIt = this.getTheOperator().getAssertions().iterator();
		while (assIt.hasNext()) {
			Assertion myAss = (Assertion) assIt.next();
			// Projections (specialisations):
			if (myAss.getAssertionType().equals(Assertion.TYPE_PROJECTION)) {
				String fromConceptParName = myAss.getObj2(); // fromConcept has more features than toConcept 
				String toConceptParName = myAss.getObj1();
				ParameterObject[] fromConcepts = this.getParameter(fromConceptParName, 0);
				ParameterObject[] toConcepts = this.getParameter(toConceptParName, 0);
				// only one of the arrays can have more than one element!
				// therefore we can safely use two loops without overlap:
				for (int i = 0; i < fromConcepts.length; i++) {
					for (int j = 0; j < toConcepts.length; j++) {
						String projName = null;
						((Concept) fromConcepts[i]).createProjectionToConcept((Concept) toConcepts[j], projName);							
					}
				}
			}
			boolean isFrom;
			// Projections to concepts attached to a relationship:
			if ((isFrom = myAss.getAssertionType().equals(Assertion.TYPE_PROJ_REL_FROM)) ||
				myAss.getAssertionType().equals(Assertion.TYPE_PROJ_REL_TO)) {
				// find the from concept of the projection as the From/ToConcept of
				// a given relation:
				Relation rel = this.getRelationFromAssertion(myAss);
				if (rel == null) {
					throw new M4Exception("Step '" + this.getName() + 
							"': could not find relation for assertion of type '" +
							(isFrom ? Assertion.TYPE_PROJ_REL_FROM : Assertion.TYPE_PROJ_REL_TO) +
							"'!");
				}
				Concept from = this.getConceptFromAssertion(myAss);
				if (from == null) {
					throw new M4Exception("Step '" + this.getName() + 
							"': could not find concept for assertion of type '" +
							(isFrom ? Assertion.TYPE_PROJ_REL_FROM : Assertion.TYPE_PROJ_REL_TO) +
							"'!");
				}
				Concept to = (Concept) (isFrom ? rel.getTheFromConcept() : rel.getTheToConcept());
				String projName = null;
				from.createProjectionToConcept(to, projName);					
			}
			// Subconcept links (separations):
			if (myAss.getAssertionType().equals(Assertion.TYPE_SUBCONCEPT)) {
				String subConceptParName = myAss.getObj1(); // subconcept has fewer rows
				String superConceptParName = myAss.getObj2();
				ParameterObject[] subConcepts = this.getParameter(subConceptParName, 0);
				ParameterObject[] superConcepts = this.getParameter(superConceptParName, 0);
				// only one of the arrays can have more than one element!
				// therefore we can safely use two loops without overlap:
				for (int i = 0; i < subConcepts.length; i++) {
					for (int j = 0; j < superConcepts.length; j++) {
						((Concept) subConcepts[i]).addSuperConcept((Concept) superConcepts[j]);							
					}
				}
			}
		} // end of loop through assertions
		
		// Check for Relationships:
		this.createOneToManyRelationByAssertion();
	}	
	
	// Checks the assertions of this Step's operator and creates
	// a one-to-many relationship if they prescribe it
	private void createOneToManyRelationByAssertion() throws M4Exception {
		
		BaseAttribute[][] allKeys = (BaseAttribute[][]) this.getKeysOfRelationshipAssertion();
		if (allKeys != null) {
			BaseAttribute[] manyKeys = allKeys[0];
			BaseAttribute[] oneKeys = allKeys[1];
			if (oneKeys != null && manyKeys != null && oneKeys.length > 0
					&& manyKeys.length > 0) {
				Concept fromConcept = (Concept) manyKeys[0].getConcept();
				Concept toConcept = (Concept) oneKeys[0].getConcept();
				String nameForRel = fromConcept.getName() + "_" + toConcept.getName() + "_REL";
				// check if the relation has already been created:
				Relation rel = (Relation) fromConcept.getRelationshipToConcept(toConcept);
				if (rel == null) {
					// change the arrays into collections:
					Collection mk = new Vector(), ok = new Vector();
					for (int i = 0; i < oneKeys.length; i++) {
						mk.add(manyKeys[i]);
						ok.add(oneKeys[i]);
					}
					this.getTheCase().createOneToManyRelation(nameForRel, mk, ok);
				}
			}
		}	
	}
	
	// This method assumes that the given assertion has one relation
	// and one non-relation parameter in its obj1/obj2. It returns the
	// relation (the parameter object).
	private Relation getRelationFromAssertion(Assertion ass) throws M4Exception {
		String obj = ass.getObj1();
		OpParam opPar = (OpParam) this.getTheOperator().getOpParam(obj);
		if ( ! opPar.isRelationParameter()) {
			obj = ass.getObj2();
			opPar = (OpParam) this.getTheOperator().getOpParam(obj);
			if ( ! opPar.isRelationParameter()) {
				return null;
			}
		}
		return (Relation) this.getSingleParameterObject(opPar.getName());
	}

	// This method assumes that the given assertion has one concept
	// and one non-concept parameter in its obj1/obj2. It returns the
	// concept (the parameter object).
	private Concept getConceptFromAssertion(Assertion ass) throws M4Exception {
		String obj = ass.getObj1();
		OpParam opPar = (OpParam) this.getTheOperator().getOpParam(obj);
		if ( ! opPar.isConceptParameter()) {
			obj = ass.getObj2();
			opPar = (OpParam) this.getTheOperator().getOpParam(obj);
			if ( ! opPar.isConceptParameter()) {
				return null;
			}
		}
		return (Concept) this.getSingleParameterObject(opPar.getName());
	}
	
	// Returns the OpParam that corresponds to the output concept of this Step.
	// An exception is thrown if there are more than one!
	private OpParam getOutputConceptOpParamForThisStep() throws M4Exception {
		try {
			Collection allOutOpParams = this.getTheOperator().getAllOutputOperatorParameters();
			Collection outConceptOpParams = new Vector();
			if (allOutOpParams != null) {
				Iterator it = allOutOpParams.iterator();
				while (it.hasNext()) {
					OpParam outOpParam = (OpParam) it.next();
					if (outOpParam.isConceptParameter()) {
						outConceptOpParams.add(outOpParam);
					}
				}
			}
			if (outConceptOpParams.isEmpty()) {
				return null;
			}
			if (outConceptOpParams.size() > 1) {
				throw new M4Exception("Found more than on output concept OpParam!");
			}
			return (OpParam) outConceptOpParams.iterator().next();
		}
		catch (M4Exception m4e) {
			throw new M4Exception("Step '" + this.getName() + "': M4 error trying to access output concept OpParam: " + m4e.getMessage());
		}
	}
	
	/**
	 * This method checks if this Step creates a relationship from or to its
	 * output concept. If no, null is returned. If yes, it checks whether the 
	 * output concept and its features have been created yet. If yes, it 
	 * returns the BaseAttributes that form the keys of the relationship
	 * in a two-dimensional array.
	 * 
	 * @return a two-dimensional array of the BaseAttributes that are the
	 * keys of the relationship created by this Step, or null
	 * @throws M4Exception
	 */
	public edu.udo.cs.miningmart.m4.BaseAttribute[][] getKeysOfRelationshipAssertion() throws M4Exception {

		Iterator assIt = this.getTheOperator().getAssertions().iterator();
		Concept manyConcept = null, oneConcept = null; // needed for relationship-assertions
		BaseAttribute[] manyKeys = null, oneKeys = null; // dito
		while (assIt.hasNext()) {
			Assertion myAss = (Assertion) assIt.next();
			boolean manyToOne = false;
			if ((manyToOne = myAss.getAssertionType().equals(Assertion.TYPE_REL_MANY_TO_ONE)) ||
				myAss.getAssertionType().equals(Assertion.TYPE_REL_ONE_TO_MANY)) {
				String manyConceptParName = (manyToOne ? myAss.getObj1() : myAss.getObj2());
				String oneConceptParName = (manyToOne ? myAss.getObj2() : myAss.getObj1());
				ParameterObject[] manyConcepts = this.getParameter(manyConceptParName, 0);
				ParameterObject[] oneConcepts = this.getParameter(oneConceptParName, 0);
				if (manyConcepts != null && manyConcepts.length > 0)
					manyConcept = (Concept) manyConcepts[0];
				if (oneConcepts != null && oneConcepts.length > 0)
					oneConcept = (Concept) oneConcepts[0];					
			}
			if (myAss.getAssertionType().equals(Assertion.TYPE_REL_MANY_KEY)) {
				String manyKeysParName = myAss.getObj1();
				ParameterObject[] manys = this.getParameter(manyKeysParName, 0);
				if (manys != null && manys.length > 0) {
					manyKeys = new BaseAttribute[manys.length];
					for (int i = 0; i < manys.length; i++) {
						manyKeys[i] = (BaseAttribute) manys[i];
					}
				}
			}
			if (myAss.getAssertionType().equals(Assertion.TYPE_REL_ONE_KEY)) {
				String oneKeysParName = myAss.getObj1();
				ParameterObject[] ones = this.getParameter(oneKeysParName, 0);
				if (ones != null && ones.length > 0) {
					oneKeys = new BaseAttribute[ones.length];
					for (int i = 0; i < ones.length; i++) {
						oneKeys[i] = (BaseAttribute) ones[i];
					}
				}
			}
		}
		// for relationship-assertions we have all the info now:
		if (manyConcept != null && oneConcept != null && manyKeys != null && oneKeys != null) {
			// check that the attributes belong indeed to the concepts; all keys might
			// belong to the same concept.
			if (this.attribsBelongToConcept(manyKeys, oneConcept)) {
				manyKeys = this.getCorrespondingAttribs(manyKeys, manyConcept);
			}
			if (this.attribsBelongToConcept(oneKeys, manyConcept)) {
				oneKeys = this.getCorrespondingAttribs(oneKeys, oneConcept);
			}
			if (manyKeys == null || oneKeys == null || manyKeys.length != oneKeys.length) {
				throw new M4Exception("Step '"+this.getName()+"': found nonmatching number of keys for relationship between input and output!");
			}
			BaseAttribute[][] ret = new BaseAttribute[][] { manyKeys, oneKeys };
			return ret;
		}
		return null;
	}
	
	// this method returns the attributes of the given concept that correspond by name
	// to the given attributes; the method returns null if any of the given attributes does not have
	// a corresponding attribute in the given concept
	private BaseAttribute[] getCorrespondingAttribs(BaseAttribute[] theAttribs, Concept theConcept) 
	throws M4Exception {
		if (this.attribsBelongToConcept(theAttribs, theConcept)) {
			return theAttribs;
		}
		if (theAttribs != null) {
			BaseAttribute[] correspondents = new BaseAttribute[theAttribs.length];
			for (int i = 0; i < theAttribs.length; i++) {
				correspondents[i] = (BaseAttribute) theConcept.getBaseAttribute(theAttribs[i].getName());
				if (correspondents[i] == null) {
					return null;					
				}
			}	
			return correspondents;
		}
		return null;
	}

	// this method returns the attributes of the given concept that correspond by name
	// to the given attributes; if no correspondence could be found, the entry in the array
	// is NULL
	private BaseAttribute[] getAllCorrespondingAttribs(BaseAttribute[] theAttribs, Concept theConcept) 
	throws M4Exception {
		if (this.attribsBelongToConcept(theAttribs, theConcept)) {
			return theAttribs;
		}
		if (theAttribs != null) {
			BaseAttribute[] correspondents = new BaseAttribute[theAttribs.length];
			for (int i = 0; i < theAttribs.length; i++) {
				correspondents[i] = (BaseAttribute) theConcept.getBaseAttribute(theAttribs[i].getName());
			}	
			return correspondents;
		}
		return null;
	}
	
	// This method returns true if all given attributes belong to the given concept.
	private boolean attribsBelongToConcept(BaseAttribute[] theAttribs, Concept theConcept)
	throws M4Exception {
		if (theAttribs != null && theConcept != null) {
			for (int i = 0; i < theAttribs.length; i++) {
				if ( ! theAttribs[i].getConcept().equals(theConcept)) {
					return false;
				}
			}
			return true;
		} 
		return false;
	}
	
	/*
	// This method returns true if the output concept creation cannot be done by the generic mechanism;
	// in this case, the output concept creation is done in this method. If false is returned, this method has
	// done nothing.
	private boolean workOnOutputConceptForSpecialOperators(String theName, edu.udo.cs.miningmart.m4.OpParam outOpPar, boolean create)  throws M4Exception {
		String name = this.getTheOperator().getName();
		
		// check if we are dealing with Join/UnionByKey:
		if (name.equalsIgnoreCase("JoinByKey") || name.equalsIgnoreCase("UnionByKey")) {
			
			if (create) {
				this.createEmptyOutputConceptForThisStep(theName, outOpPar);
			}			

		   	Concept outputConcept = this.getConceptFromOpPar(outOpPar);

			// If we only update the output, we simply remove all features from output concept 
			// and create them anew below:			
			if ( ! create) {
				outputConcept.removeAllFeatures();
			}
			
			// Collect the Features of all input Concepts and their types, but check
			// for the keys! Check also for mappings from input features to new names,
			// but the names for the mappings are not known yet. Therefore we use special 
			// suffixes to mark the mapped features in the output concept, and we
			// deal with the mappings again in their own call to "create/updateOutput"!
			ParameterObject[] myInConcepts = this.getParameter("TheConcepts", 0);
			ParameterObject[] myKeys = this.getParameter("TheKeys", 0);
			ParameterObject[] myMappedInputs = this.getParameter("MapInput", 0);
			
			if (myInConcepts == null || myKeys == null) {
				throw new M4Exception("Step '" + this.getName() + "': could not read input parameters for output creation!");
			}
			
			boolean keyDealtWith = false;
			
			for (int i = 0; i < myInConcepts.length; i++) {
				Concept inCon = (Concept) myInConcepts[i];
				Iterator fIt = inCon.getFeatures().iterator();
				while (fIt.hasNext()) {
					Feature myF = (Feature) fIt.next();
					if (this.isVisible(myF)) {
						// check if myF is a key:
						for (int j = 0; j < myKeys.length; j++) {
							if (myF.equals(myKeys[j])) {
								if ( ! keyDealtWith) {
									keyDealtWith = true;
								}
								else {
									myF = null; // found second or more key, do not add it to output
									break;  // do not compare to myF (which is now null) any longer
								}
							}
						}
						if (myF != null) {
							String nameOfFeatureInOutput = myF.getName();
							// check if it is a mapped feature:
							if (myMappedInputs != null) {
								for (int j = 0; j < myMappedInputs.length; j++) {
									Feature mapped = (Feature) myMappedInputs[j];
									if (myF.equals(mapped)) {
										// 	assuming j+1 gives the loop number because the parameters are read 
										// in the right order:
										nameOfFeatureInOutput += specialNameMarker + (j+1);
										break;
									}
								}
							}
							// copy this feature to output:
							// default data type:
							String conDt = edu.udo.cs.miningmart.m4.ConceptualDatatypes.CDT_NOMINAL;
							if (myF instanceof BaseAttribute) {
								conDt = ((BaseAttribute) myF).getConceptualDataTypeName();
							} // ignore MCFs				
							String[] types = new String[1];
							types[0] = conDt;
							this.createBAsInConcept(new String[] { nameOfFeatureInOutput }, types, outputConcept);
						}
					}
				}
			}

			// output concept worked on
			return true;
		}
		
		// check if we are dealing with SpecifiedStatistics:
		if (name.equalsIgnoreCase("SpecifiedStatistics")) {
			
			Concept outputConcept = this.prepareOutputConcept(outOpPar, theName, create);

			// deal with all features of the output concept here, 
			// because no output OpParams exist for them :-( :
			
			Concept inputConcept = (Concept) this.getInputConceptsThatAreParameters().iterator().next();
			
			// loop through the output attribute parameters:
			String[] outputAttrParameterNames = 
				new String[] { SpecifiedStatistics.PARAMETER_ATTR_AVG,
							   SpecifiedStatistics.PARAMETER_ATTR_COUNT,
							   SpecifiedStatistics.PARAMETER_ATTR_GROUPBY,
							   SpecifiedStatistics.PARAMETER_ATTR_MAX,
							   SpecifiedStatistics.PARAMETER_ATTR_MIN,
							   SpecifiedStatistics.PARAMETER_ATTR_SUM,
							   SpecifiedStatistics.PARAMETER_ATTR_UNIQUE
							  };			
		 	
			for (int n = 0; n < outputAttrParameterNames.length; n++) {
				
				ParameterObject[] attrs = this.getParameter(outputAttrParameterNames[n], 0);
				if (attrs != null) {
					String[] types = new String[attrs.length];
					String[] names = new String[attrs.length];
					boolean createSomething = false;
					for (int i = 0; i < attrs.length; i++) {
						String inputFeatName = attrs[i].getName();
						BaseAttribute inputBa = (BaseAttribute) inputConcept.getBaseAttribute(inputFeatName);
						String outputFeatName = this.getSpecStatOutAttrSuffix(outputAttrParameterNames[n], inputFeatName);
						names[i] = outputFeatName;
						createSomething = true;
						if (outputAttrParameterNames[n].equals(SpecifiedStatistics.PARAMETER_ATTR_GROUPBY)) {
							types[i] = inputBa.getConceptualDataTypeName();
						}
						else {
							types[i] = edu.udo.cs.miningmart.m4.ConceptualDatatypes.CDT_NUMERIC;
						}
					}
					if (createSomething) {
						this.createBAsInConcept(names, types, outputConcept);
					}
				}
			}
			
			// special things to do with parameter "AttributesComputeDistrib":
			ParameterObject[] distrAttrs = this.getParameter(SpecifiedStatistics.PARAMETER_ATTR_DISTRIB, 0);
			if (distrAttrs != null && distrAttrs.length > 0) {
				ParameterObject[] distrVals = this.getParameter(SpecifiedStatistics.PARAMETER_DISTRIB_VAL, 0);
				
				// this should have been checked already, but let's be safe:
				if (distrVals == null || distrVals.length != distrAttrs.length) {
					throw new M4Exception("Step '" + this.getName() + "': number of values for parameters '" +
							SpecifiedStatistics.PARAMETER_ATTR_DISTRIB + "' and '" +
							SpecifiedStatistics.PARAMETER_DISTRIB_VAL + "' is not equal!");
				}
				Vector names = new Vector();
				for (int i = 0; i < distrAttrs.length; i++) {
					String inputAttrName = distrAttrs[i].getName();
					String valuesOfInputAttr = ((Value) distrVals[i]).getValue();
					StringTokenizer st = new StringTokenizer(valuesOfInputAttr, ",");
					while (st.hasMoreTokens()) {
						String oneValueOfAttr = st.nextToken().trim();
						String outputFeatName = inputAttrName + "_" + oneValueOfAttr;
						names.add(outputFeatName);
					}
				}
				String[] types = new String[names.size()];
				String[] namesArray = new String[names.size()];
				for (int i = 0; i < types.length; i++) {
					namesArray[i] = (String) names.get(i);
					types[i] = edu.udo.cs.miningmart.m4.ConceptualDatatypes.CDT_NUMERIC;
				}
				this.createBAsInConcept(namesArray, types, outputConcept);
			}
			
			// output concept worked on
			return true;
		}
		
		// check if we are dealing with operator "Pivotize":
		if (name.equalsIgnoreCase("Pivotize")) {

			Concept outputConcept = this.prepareOutputConcept(outOpPar, theName, create);

			// deal with all features of the output concept here, 
			// because no output OpParams exist for them :-( :
			
			// get all combinations of the values of index attributes;
			// create an output BaseAttribute for each combination:
			ParameterObject[] indexAttrs = this.getParameter(Pivotize.PARAMETER_INDEX_ATTR, 0);
			ParameterObject[] indexVals = this.getParameter(Pivotize.PARAMETER_VALUES, 0);
			ParameterObject[] pivot = this.getParameter(Pivotize.PARAMETER_PIVOT_ATTR, 0);
			ParameterObject[] aggOp = this.getParameter(Pivotize.PARAMETER_AGGREGATION, 0);
			
			String[] indexValues;
			BaseAttribute[] indexAttributes;
			BaseAttribute pivotAttribute;
			String aggregationOperator;
			try {
				Value[] someVals = (Value[]) indexVals;
				indexValues = new String[someVals.length];
				for (int i = 0; i < someVals.length; i++) {
					indexValues[i] = someVals[i].getValue();
				}
				indexAttributes = (BaseAttribute[]) indexAttrs;
				pivotAttribute = (BaseAttribute) pivot[0];
				aggregationOperator = ((Value[]) aggOp)[0].getValue();
			}
			catch (ClassCastException cce) {
				throw new M4Exception("Step '" + this.getName() + "': Could not cast parameter(s) to expected type!");
			}
			
			Collection theCombis = Pivotize.getCombinations(indexValues, indexAttributes, pivotAttribute);
			if (theCombis == null || theCombis.isEmpty()) {
				throw new M4Exception("Step '" + this.getName() + "': Could not compute combinations of index values!");
			}
			
			// also get the group by attributes:
			ParameterObject[] groupBy = this.getParameter(Pivotize.PARAMETER_GROUPBY_ATTR, 0);
			BaseAttribute[] groupByBAs;
			if (groupBy != null) {
				groupByBAs = new BaseAttribute[groupBy.length];
				for (int i = 0; i < groupBy.length; i++) {
					groupByBAs[i] = (BaseAttribute) groupBy[i];
				}
			}
			else {
				groupByBAs = new BaseAttribute[0];
			}
			
			// loop through the combinations:
			Iterator it = theCombis.iterator();
			String[] types = new String[theCombis.size() + groupByBAs.length];
			String[] names = new String[types.length];
			int i = 0;
			boolean createSomething = false;
			while (it.hasNext()) {
				Pivotize.AttributeValueCombination avc = (Pivotize.AttributeValueCombination) it.next();
				names[i] = avc.getOutputBaName();
				createSomething = true;
				// if any aggregation takes place, then the type is numeric 
				// because the aggregation operators are sum, count, avg etc.:
				if (aggregationOperator.equalsIgnoreCase(Pivotize.NO_AGGREGATION_VALUE)) {
					types[i] = pivotAttribute.getConceptualDataTypeName();
				}
				else {
					types[i] = ConceptualDatatypes.CDT_NUMERIC; 
				}
				i++;
			}
			
			// add the group by attributes:
			for (int j = i; j < i + groupByBAs.length; j++) {
				names[j] = groupByBAs[j-i].getName();
				types[j] = groupByBAs[j-i].getConceptualDataTypeName();
				createSomething = true;
			}
			
			// create the attributes in the output concept:
			if (createSomething) {
				this.createBAsInConcept(names, types, outputConcept);
			}
			
			// output concept worked on
			return true;
		}

		// check if we are dealing with operator "ReversePivotize":
		if (name.equalsIgnoreCase("ReversePivotize")) {

			Concept outputConcept = this.prepareOutputConcept(outOpPar, theName, create);

			// deal with all features of the output concept here, 
			// because no output OpParams exist for them :-( :
			
			// all input BaseAttributes are copied to the output except those 
			// which are given as parameter 'ThePivotizedAttributes':
			BaseAttribute[] pivotAttrs = (BaseAttribute[]) this.getParameter(ReversePivotize.PARAMETER_PIVOT_ATTRS, 0);
			
			Concept inCon = (Concept) this.getAllInputConcepts().iterator().next();
			Iterator inFeatIt = inCon.getAllBaseAttributes().iterator();
			Vector names = new Vector();
			Vector baTypes = new Vector();
			while (inFeatIt.hasNext()) {
				BaseAttribute inBa = (BaseAttribute) inFeatIt.next();
				if ( ! nameOccursIn(inBa, pivotAttrs)) {
					names.add(inBa.getName());
					baTypes.add(inBa.getConceptualDataTypeName());
				}
			}
			
			// now add the BaseAttributes for the Pivot and Index Attributes:
			String nameOfPivot = ((Value[]) this.getParameter(ReversePivotize.PARAMETER_NAME_PIVOT, 0))[0].getValue();
			Value[] indexAttrs = (Value[]) this.getParameter(ReversePivotize.PARAMETER_NAMES_INDEX_ATTRS, 0);
			Value[] indexVals = (Value[]) this.getParameter(ReversePivotize.PARAMETER_VALUES, 0);
			String[] indexValues = new String[indexVals.length];
			for (int i = 0; i < indexVals.length; i++) {
				indexValues[i] = indexVals[i].getValue();
			}
			String[] namesOfIndexAttrs = new String[indexAttrs.length];
			for (int i = 0; i < indexAttrs.length; i++) {
				namesOfIndexAttrs[i] = indexAttrs[i].getValue();
			}
			names.add(nameOfPivot);
			baTypes.add(ReversePivotize.getConDataTypeOfPivotAttr(pivotAttrs));
			for (int i = 0; i < namesOfIndexAttrs.length; i++) {
				names.add(namesOfIndexAttrs[i]);
				baTypes.add(ReversePivotize.getConDataTypeOfIndexAttr(namesOfIndexAttrs[i], indexValues, namesOfIndexAttrs));
			}
			String[] types = new String[baTypes.size()];
			String[] namesArray = new String[types.length];
			for (int i = 0; i < types.length; i++) {
				types[i] = (String) baTypes.get(i);
				namesArray[i] = (String) names.get(i);
			}
			// create the attributes in the output concept:
			if (names.size() > 0) {
				this.createBAsInConcept(namesArray, types, outputConcept);
			}
			// output concept worked on
			return true;
		}
		
		if (name.startsWith("Segmentation") || name.equals("Unsegment")) {			

			Concept outputConcept = this.prepareOutputConcept(outOpPar, theName, create);

			BaseAttribute targetAttr = null;
			if (name.equals("SegmentationStratified")) {
				ParameterObject[] po = this.getParameter("TheAttribute", 0);
				if (po == null || po.length == 0) {
					throw new M4Exception("Step '" + this.getName() + "': could not find parameter 'TheAttribute'!");
				}
				targetAttr = (BaseAttribute) po[0];
			}
			
			// now create all the features of the input concept in the output concept;
			// the only exception is TheAttribute for SegmentationStratified:
			Concept inputConcept = (Concept) this.getInputConceptsThatAreParameters().iterator().next();
			Iterator inputAttribsIt = inputConcept.getAllBaseAttributes().iterator();
			Vector outputBaNames = new Vector();
			Vector outputBaTypes = new Vector();
			while (inputAttribsIt.hasNext()) {
				BaseAttribute inBa = (BaseAttribute) inputAttribsIt.next();
				if (this.isVisible(inBa) &&
						(targetAttr == null || name.equals("Unsegment") || 
								(targetAttr != null && ( ! inBa.getName().equals(targetAttr.getName()))))) {
					outputBaNames.add(inBa.getName());
					outputBaTypes.add(inBa.getConceptualDataTypeName());
				}
			}
			// for Unsegment the additional attribute is created in a separate call
			// to createOutput with the corresponding OpParam; see method
			// Step.workOnOutputBAsForSpecialOperators()
			
			String[] outBaTypes = new String[outputBaTypes.size()];
			String[] outNames = new String[outBaTypes.length];
			for (int i = 0; i < outBaTypes.length; i++) {
				outBaTypes[i] = (String) outputBaTypes.get(i);
				outNames[i] = (String) outputBaNames.get(i);
			}
			// create the attributes in the output concept:
			if (outputBaNames.size() > 0) {
				this.createBAsInConcept(outNames, outBaTypes, outputConcept);
			}
			else {
				throw new M4Exception("Step '" + this.getName() + "': no attributes were copied to output concept!");
			}
			return true;
		}

		if (name.equals("FeatureConstructionByRelation")) {			

			Concept outputConcept = this.prepareOutputConcept(outOpPar, theName, create);
			
			// now create all the features of the ToConcept in the output concept:
			Concept toConcept = (Concept) ((Relation) this.getSingleParameterObject("TheRelation")).getTheToConcept();
			Iterator inputAttribsIt = toConcept.getAllBaseAttributes().iterator();
			Vector outputBaNames = new Vector();
			Vector outputBaTypes = new Vector();
			while (inputAttribsIt.hasNext()) {
				BaseAttribute inBa = (BaseAttribute) inputAttribsIt.next();
				if (this.isVisible(inBa)) {
					outputBaNames.add(inBa.getName());
					outputBaTypes.add(inBa.getConceptualDataTypeName());
				}
			}
			
			String[] outBaTypes = new String[outputBaTypes.size()];
			String[] outNames = new String[outBaTypes.length];
			for (int i = 0; i < outBaTypes.length; i++) {
				outBaTypes[i] = (String) outputBaTypes.get(i);
				outNames[i] = (String) outputBaNames.get(i);
			}
			// create the attributes in the output concept:
			if (outputBaNames.size() > 0) {
				this.createBAsInConcept(outNames, outBaTypes, outputConcept);
			}
			else {
				throw new M4Exception("Step '" + this.getName() + "': no attributes were copied to output concept!");
			}
			return true;
		}
		// add further checks for special operators here:
		// if name.equalsIgnoreCase(mySpecialOpName) {  ... return true; }
		
		// no special operator:
		return false;
	}
	
	
	// This method either creates an empty output concept, or deletes
	// all features that can safely be deleted from an existing output concept
	private Concept prepareOutputConcept(
			edu.udo.cs.miningmart.m4.OpParam outOpPar, 
			String theName,
			boolean create)
	throws M4Exception {
		
		if (create) {
			this.createEmptyOutputConceptForThisStep(theName, outOpPar);
		}			

	   	Concept outputConcept = this.getConceptFromOpPar(outOpPar);

		// If we only update the output, we simply remove all features 
	   	// from output concept and create them anew below:			
		if ( ! create) {
			// but we cannot remove those features that are used as 
			// parameters by later steps!
			Iterator featureIt = outputConcept.getFeatures().iterator();
			Vector featuresToDelete = new Vector();
			while (featureIt.hasNext()) {
				Feature myF = (Feature) featureIt.next();
				if (myF.getParameterReferences().isEmpty()) {
					featuresToDelete.add(myF);
				}
			}
			featureIt = featuresToDelete.iterator();
			while (featureIt.hasNext()) {
				Feature myF = (Feature) featureIt.next();
				outputConcept.removeFeature(myF);
			}
		}			
		return outputConcept;
	}
	
	
	private Concept getConceptFromOpPar(edu.udo.cs.miningmart.m4.OpParam conceptOpPar) throws M4Exception {
		if (conceptOpPar == null) {
			throw new M4Exception("Step '" + this.getName() + "': request for concept with NULL OpParam!");			
		}
		if ( ! conceptOpPar.isConceptParameter()) {
			throw new M4Exception("Step '" + this.getName() + "': request for concept with non-concept OpParam '" + conceptOpPar.getName() + "'!");
		}
	   	ParameterObject[] po = this.getParameter(conceptOpPar.getName(), 0);
	   	if (po == null) {
	   		throw new M4Exception("Step '" + this.getName() + "': could not find output concept!");
	   	}
	   	Concept outputConcept = (Concept) po[0];
	   	return outputConcept;
	}
	
	private boolean nameOccursIn(BaseAttribute ba, BaseAttribute[] list) {
		for (int i = 0; i < list.length; i++) {
			if (ba.getName().equalsIgnoreCase(list[i].getName())) {
				return true;
			}
		}
		return false;
	}
	
	private String getSpecStatOutAttrSuffix(String parName, String parObjectName) throws M4Exception {
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_SUM)) {
			return parObjectName + SpecifiedStatistics.SUM_SUFFIX;
		}
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_COUNT)) {
			return parObjectName + SpecifiedStatistics.COUNT_SUFFIX;
		}
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_AVG)) {
			return parObjectName + SpecifiedStatistics.AVG_SUFFIX;
		}
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_MIN)) {
			return parObjectName + SpecifiedStatistics.MIN_SUFFIX;
		}
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_MAX)) {
			return parObjectName + SpecifiedStatistics.MAX_SUFFIX;
		}
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_UNIQUE)) {
			return parObjectName + SpecifiedStatistics.UNIQUE_SUFFIX;
		}
		if (parName.equals(SpecifiedStatistics.PARAMETER_ATTR_GROUPBY)) {
			return parObjectName;
		}
		throw new M4Exception("Step '" + this.getName() + 
				"', creating output: got unknown parameter name '" +
				parName + "'!");
	}
	*/
	
	private Concept checkConceptExists(OpParam theConceptOpParam) throws M4Exception {
		Concept con = null;
		try {
			Parameter theConcPar = (Parameter) this.getParameterTuple(theConceptOpParam.getName(), 0);
			if (theConcPar == null) {
				throw new M4Exception("Step '" + this.getName() + "': no Parameter for '" + theConceptOpParam.getName() + "' has been created!");
			}
		    con = (Concept) theConcPar.getTheParameterObject();
		}
		catch (ClassCastException cce) {
			throw new M4Exception("Step '" + this.getName() + 
					"': found something different from a concept as parameter object of '" + theConceptOpParam.getName() + "'!");
		}
		if (con == null) {
			throw new M4Exception("Step '" + this.getName() +
					"': failed to reach concept '" + theConceptOpParam.getName() + "'!");
		}	
		return con;
	}
	
	private Collection checkExistenceOfInputFeatures(Collection theInputFeatures) throws M4Exception {
		// simply check if every feature belongs to a concept that is input of this step:
		Vector ret = new Vector();
		if (theInputFeatures != null) {
			Iterator it = theInputFeatures.iterator();
			Collection incons = this.getAllInputConcepts();
			while (it.hasNext()) {
				Feature myF = (Feature) it.next();
				Concept theConcept = (Concept) myF.getConcept();
				if (theConcept != null) {
					if (incons.contains(theConcept)) {
						ret.add(myF);
					}
				}
			}
		}
		return ret;
	}
	
	private Map copyFeaturesIntoConcept(Collection theFeatures, edu.udo.cs.miningmart.m4.Concept theConcept) throws M4Exception {
		// if the output concept is updated (rather than created),
		// it has features already; check if they are in the feature set, 
		// and delete them if not:
		Collection theFeats = theConcept.getFeatures();
		// use an array to avoid a ConcurrentModificationException:
		Feature[] toDelete = new Feature[theFeats.size()];
		Iterator outputFeatureIt = theFeats.iterator();
		int i = 0;
		Map mapFromParametersToTheirOldNames = new HashMap();
		while (outputFeatureIt.hasNext()) {
			Feature outF = (Feature) outputFeatureIt.next();
			Iterator inputFeatureIt = theFeatures.iterator();
			// a feature in theConcept is deleted if it is not used as a parameter
			// in a different step and if it has no corresponding feature in the
			// given feature list (in the latter case it is updated as a copy of the
			// corresponding feature).
			boolean hasCorrespondentInInputConcept = false;
			boolean isUsedAsOutputParameterByDifferentStep = false;
			boolean isRenamed = false;
			while (inputFeatureIt.hasNext()) {
				Feature inF = (Feature) inputFeatureIt.next();
				if (this.isVisible(inF) && inF.correspondsTo(outF)) {
					hasCorrespondentInInputConcept = true;
				}
			}
			
			// See if the output feature can be assigned to an input
			// feature by a globally stored map:
			isRenamed = (this.getTheCase().getNewNameOfChangedFeature(outF.getName()) != null);
			
			Iterator parametersUsingOutF_It = outF.getParameterReferences().iterator();
			while (parametersUsingOutF_It.hasNext()) {
				Parameter aPar = (Parameter) parametersUsingOutF_It.next();
				if ((aPar != null) && (aPar.getTheStep() != null) && ( ! aPar.getTheStep().equals(this))) {
					if ( ! aPar.isInputParam()) {
						isUsedAsOutputParameterByDifferentStep = true;
					}
					else {
						// keep the info that this parameter object was deleted:
						if ((! isRenamed) && aPar.getTheParameterObject() != null) {
							String nameWithLoop = this.addLoopNumberToName(aPar.getTheParameterObject().getName(), aPar.getLoopNr());
							String[] pairOfOldAndNewNames = new String[] { nameWithLoop, null };
							mapFromParametersToTheirOldNames.put(aPar, pairOfOldAndNewNames);
						}
					}
				}
			}
			toDelete[i] = (hasCorrespondentInInputConcept || isRenamed || isUsedAsOutputParameterByDifferentStep ? null : outF);
			i++;
		}					
		for (i = 0; i < toDelete.length; i++) {
			if (toDelete[i] != null) toDelete[i].deleteSoon();
		}
		
		// now go through the visible feature list and ensure
		// they exist in output concept:
		Iterator inputFeatureIt = theFeatures.iterator();
		while (inputFeatureIt.hasNext()) {
			Feature myF = (Feature) inputFeatureIt.next();
			if (this.isVisible(myF)) {
				if (myF instanceof MultiColumnFeature) {
					Iterator it = ((MultiColumnFeature) myF).getBaseAttributes().iterator();
					while (it.hasNext()) {
						BaseAttribute ba = (BaseAttribute) it.next();
						this.updateOrCreateBaInConcept(theConcept, ba);							
					}
				}
				else {
					BaseAttribute ba = (BaseAttribute) myF;
					this.updateOrCreateBaInConcept(theConcept, ba);	
				}
			}
		}
		return mapFromParametersToTheirOldNames;
	}
	
	private void updateOrCreateBaInConcept(
			edu.udo.cs.miningmart.m4.Concept theConcept, 
			edu.udo.cs.miningmart.m4.BaseAttribute templateBa)
	throws M4Exception {
		// check if a BA with the right name exists already.
		// (if the name has changed, the BA was either deleted
		// or its old and new name are stored in a global map)
		Iterator baIt = theConcept.getAllBaseAttributes().iterator();
		boolean wasUpdated = false;
		while (baIt.hasNext()) {
			BaseAttribute myBA = (BaseAttribute) baIt.next();
			if (myBA.correspondsTo(templateBa)) {
				// update:
				myBA.setConceptualDataTypeName(templateBa.getConceptualDataTypeName());
				myBA.setRole(templateBa.getRole());
				wasUpdated = true;
			}
			// maybe the output BA (myBA) has an old name for which
			// a new name is globally stored:
			String newName = this.getTheCase().getNewNameOfChangedFeature(myBA.getName());
			if (newName != null) {
				myBA.setName(newName);
				// data type and role are kept
				wasUpdated = true;
			}
		}
		if ( ! wasUpdated) {			
			// check that the name doesn't exist yet:
			if (theConcept.getBaseAttribute(templateBa.getName()) != null) {
				throw new M4Exception("Step '" + this.getName() +
						"': attempt to add a double BaseAttribute to the output concept!");
			}
			theConcept.createBaseAttribute( templateBa.getName(), 
					                        templateBa.getConceptualDataTypeName(), 
											BaseAttribute.TYPE_MINING,
											templateBa.getRoleName());
		}		
	}
	
	private void createEmptyOutputConceptForThisStep( String theName, 
			edu.udo.cs.miningmart.m4.OpParam theOutputConceptOpParam) 
	    throws M4Exception {		
		
		Concept existingConcept = null;
		edu.udo.cs.miningmart.m4.Parameter existingParameter = null;
		try {
			existingParameter = (Parameter) this.getParameterTuple(theOutputConceptOpParam.getName(), 0);
			if (existingParameter != null) {
				existingConcept = (Concept) existingParameter.getTheParameterObject();
			}
		}
		catch (ClassCastException cce) {
			throw new M4Exception("Step '" + this.getName() + "': Parameter object for output concept was not a concept!");
		}
		if (existingConcept != null) {
			throw new M4Exception("Step '" + this.getName() + "': createOutput was called with output concept, but it exists already!");
		}		
		
		// for debugging:
		if (theName.equals("")) {
			throw new UserError("Step '" + this.getName() + "': empty output object name!");
		}

		// now we know that the output concept was not a parameter of this
		// step before. However there might already exist a concept somewhere in the case with 
		// the name that the user gave:
		Concept conceptWithGivenName = (Concept) this.getTheCase().getConcept(theName);
		Concept newOutputConcept = null;
		if (conceptWithGivenName != null) {
			// we will use this concept but delete all its features! so that 
			// they can be adapted to the step's parameter settings:
			conceptWithGivenName.removeAllFeatures();
			newOutputConcept = conceptWithGivenName;
		}
		else {
			newOutputConcept = (Concept) this.getTheCase().createConcept(theName, Concept.TYPE_MINING);			
		}
		if (existingParameter == null) {
			existingParameter = this.createParameterTuple(theOutputConceptOpParam.getName(), 
					             	newOutputConcept, 
									this.getFreeParameterNumber(),
									0, // loopNr
									Parameter.TYPE_OUTPUT);
			
			// now the Parameter tuple is created; set it into parameter dictionary:
			// final String typeString = Parameter.getTypeStringForParObject(newOutputConcept);
			// ParameterArray pArr = new ParameterArray(typeString);	
			// pArr.addParameter(existingParameter);			
			// this.getParameterDictionary(false).put(theOutputConceptOpParam.getName(), 0, pArr);
		}
		else {
			existingParameter.setTheParameterObject(newOutputConcept);
		}
	}
	
	// This method creates or updates feature output parameters. How to do that is 
	// specified in the constraints of this step's operator. The given
	// theOutputOpParam is never NULL (see calling method)!
	private Map workOnOutputBAs( Collection theConstraints, 
								  edu.udo.cs.miningmart.m4.OpParam theOutputOpParam,
								  Collection theNames,
								  int loopNr,
								  boolean create) 
	        throws M4Exception {

		if (theConstraints == null || this.getTheOperator() == null) {
			return null;
		}
		String type = null;
		String role = null;
		Constraint inOutputConceptConstraint = null, inInputConceptConstraint = null;
		Concept outputConcept = null, inputConcept = null;
		Iterator theConstraintsIt = theConstraints.iterator();
		
		// go through the constraints and use the information in them:
		// (the calling method ensures that the constraints apply to output features)
		while (theConstraintsIt.hasNext()) {
			Constraint aConstr = (Constraint) theConstraintsIt.next();
			
			// - the datatype for the new/updated BA may be given directly:
			if (aConstr.getType().equalsIgnoreCase(Constraint.TYPE_DATATYPE)) {
				type = aConstr.getObj2();
				// sometimes obj2 is a value parameter whose value gives the type:
				OpParam valueOpPar = (OpParam) this.getTheOperator().getOpParam(type);
				if (valueOpPar != null && valueOpPar.isValueParameter()) {
					Value v = (Value) this.getParameterArray(valueOpPar, loopNr).getParameterObjectArray()[0];
					if (v != null) {
						type = v.getValue();
					}
				}
			}
		
			// - the datatype for the new/updated BA may have to be found out from the
			// corresponding input BA:
			if (aConstr.getType().equalsIgnoreCase(Constraint.TYPE_SAME_DATATYPE)) {
				String[] result = this.getTypeAndRoleFromCorrespondingBA(aConstr, loopNr);
				type = result[0];
				role = result[1];
			}
			
			// there may be attributes from the input concepts, to be copied
			// but at the same time to be renamed:
			if (aConstr.getType().equalsIgnoreCase(Constraint.TYPE_RENAME_ATTRIB)) {
				this.addRenamedFeaturesToOutput(aConstr, theNames, loopNr);
			}
			
			// - or there can be IN constraints for the new BA, saying that the new
			// BA is IN the input concept or IN the output concept or BOTH!
			if (aConstr.getType().equalsIgnoreCase(Constraint.TYPE_CONTAINED_IN)) {
				
				// check that this is the right type of IN constraint,
				// ie it applies to output feature parameters:
				String whatIsContained = aConstr.getObj1();
				OpParam containedOpPar = (OpParam) this.getTheOperator().getOpParam(whatIsContained);
				if (containedOpPar == null) {
					throw new M4Exception("Step '" + this.getName() + 
							"': found IN constraint whose object 1 is not a parameter!");
				}
				if ( ( ! containedOpPar.isBaseAttribParameter()
						&&
					   ! containedOpPar.isFeatureParameter())
					||
					containedOpPar.isInput()) {
					continue; // wrong constraint
				}
				
				// now see what the constraint says about the output feature parameter:
				String containedIn = aConstr.getObj2(); 
				// 'containedIn' can only be TheInputConcept or TheOutputConcept!
				// A constraint that says the output feature is IN SEVERAL concepts 
				// is not expected and not handled here...
				OpParam containedInOpPar = (OpParam) this.getTheOperator().getOpParam(containedIn);
				if (containedInOpPar == null) {
					throw new M4Exception("Step '" + this.getName() + 
							"': found IN constraint whose object 2 is not a parameter!");
				}		
				// check that it is a concept:
				if ( ! containedInOpPar.isConceptParameter()) {
					throw new M4Exception("Step '" + this.getName() + 
					  "': found IN constraint for output BA without a Concept in object 2!");
				}
				Parameter conceptPar = (Parameter) this.getParameterTuple(containedInOpPar.getName(), 0);
				// maybe the concept has been deleted:
				if (conceptPar == null) {
					return null; // then we cannot create any output feature
					// throw new M4Exception("Step '" + this.getName() + 
					//  		"': found IN CONCEPT constraint for output BA, but the concept does not exist in this Step!");
				}
				Concept conceptFromInConstraint = (Concept) conceptPar.getTheParameterObject();
				// Now we have the concept IN which the output feature must be; is it
				// an input or output concept?:
				if ( ! conceptPar.isInputParam()) {
					// here we know that 'conceptFromInConstraint' is 
					// the output concept
					// so if it does not exist yet, it must be created first:
					if (conceptFromInConstraint == null) {
						throw new M4Exception("Step '" + this.getName() +
											  "': method workOnOutput was called with an output BA " +
											  "that belongs to an output concept; " +
											  "please create that concept first!");
					}
					else {
						// here the output concept that the new/updated BA(s) belong to
						// is already created; if the new/updated BA(s) are not yet in it
						// we add them. But first we need to find their types, if possible.
						// They can be concluded if there is also an IN constraint specifying
						// that the BAs occur also in the input concept. So we keep the
						// current constraint and work on it later, after the while loop that
						// goes through the constraints.
						inOutputConceptConstraint = aConstr;
						outputConcept = conceptFromInConstraint;
					}
				}
				else {
					// here we know that 'conceptFromInConstraint' is
					// the input concept. In most cases the new/updated BA is simply
					// added to the input concept. But there may also be an output
					// concept, which we find out by looking at all constraints.
					// So we wait until after the while loop before we work on 
					// this constraint.
					inInputConceptConstraint = aConstr;
					inputConcept = conceptFromInConstraint;
					// but maybe the input concept has been deleted (and then
					// the propagation to this step has been called):
					if (inputConcept == null) {
						// in this case we stop the propagation and thus
						// keep the old output feature(s) in place:
						return null;
					}
				}				
			} // end If constraint is of type IN
		} // end while loop through constraints
	
		// if the role could not be determined, use another source of information,
		// namely assertions (awaiting new constraints about roles...):
		if (role == null)
			role = this.determineRoleByAssertions(theOutputOpParam, loopNr);
		
		// handle the information found in the constraints:
		Map mapInputParamsToOldNames = this.handleBaConstraints(
				inInputConceptConstraint, 
				inOutputConceptConstraint, 
				theNames, 
				(OpParam) theOutputOpParam, 
				type,
				role,
				inputConcept, 
				outputConcept, 
				create, 
				loopNr);
	
		// now check for the attributes of relationships:
		this.createOneToManyRelationByAssertion();
		
		return mapInputParamsToOldNames;
	}
	
	// This method creates or updates an output feature parameter according
	// to the constraints that apply to it and the concept it must belong to.
	// At most one of the two constraints or the two concepts given as a parameter
	// to this method must be NULL. The collection theNames gives the names
	// of the output feature parameter objects. From its size the method knows
	// how many Features to create.
	// The method returns a map from input feature parameters of other steps to the names
	// of their old objects, or null.
	private Map handleBaConstraints(
			Constraint inInputConceptConstraint, 
			Constraint inOutputConceptConstraint,
			Collection theNames,
			OpParam theOutputOpParam,
			String type,
			String roleName,
			Concept inputConcept,
			Concept outputConcept,
			boolean create,
			int loopNr)
	throws M4Exception {
	
		if (inInputConceptConstraint == null && inOutputConceptConstraint == null) {
			return null;
		}
	
		String[] typesOfInputBAs = new String[theNames.size()];
		Collection baseAttribsOfThisOutputParam = null;
	
		/*
		 * Decide whether the new BA(s) have to be added to the input,
		 * or to the output, or whether they have to be copied to the 
		 * output:
		 */ 
		if (inOutputConceptConstraint == null) {
			// only add BA(s) to input concept, or update them there, 
			// but do not add them to output:
			String[] theTypes = new String[theNames.size()];
			String[] namesArray = new String[theNames.size()];
			String[] roles = new String[theNames.size()];
			// the conceptual data type is given, one hopes, but 
			// to be sure a default can be used:
			if (type == null) 
				type = ConceptualDatatypes.CDT_NOMINAL; 
			if (roleName == null) 
				roleName = Roles.ROLE_NOROLE; 
			Iterator namesIt = theNames.iterator();
			int j = 0;
			while (namesIt.hasNext()) {
				namesArray[j] = (String) namesIt.next();
				theTypes[j] = type;
				roles[j] = roleName;
				j++;
			}
			baseAttribsOfThisOutputParam = this.createBAsInConcept(namesArray, theTypes, roles, inputConcept); // only updates if BA exists
		}
		else {
			// Either create new BAs or copy them from input!
			// loop through the name(s) for the new BA (array) to see if it occur(s) 
			// in input concept:
			Iterator theNamesIt = theNames.iterator();
			Vector namesOfBAsForOutput = new Vector();
			Vector typesOfBAsForOutput = new Vector();
			Vector rolesOfBAsForOutput = new Vector();
			while (theNamesIt.hasNext()) {
				String oneNewName = (String) theNamesIt.next();
				if (oneNewName.equals("")) {
					throw new UserError("Step '" + this.getName() + "': empty output object name!");
				}
				// see if new BAs are to be copied from input:
				if (inInputConceptConstraint != null) {
					// yes - copy from input, so find the corresponding feature
					// in input:
					Feature inF = (Feature) inputConcept.getFeature(oneNewName);
					// if the feature is found, we now know its type, to be also
					// used for the output feature:
					if (inF != null && this.isVisible(inF) && inF instanceof BaseAttribute) { // ignoring MCFs
						typesOfBAsForOutput.add(((BaseAttribute) inF).getConceptualDataTypeName());
						rolesOfBAsForOutput.add(((BaseAttribute) inF).getRoleName());
						namesOfBAsForOutput.add(oneNewName);
					}
					// but if the feature is not found, we do NOT copy the feature to the 
					// output (even though the parameter specification
					// (given in 'theNames') says it should be copied), because we assume
					// it has been deleted from the input on purpose!
				}
				else {
					// no - do not copy the new BA from input but create it.
					
					// check for a really special case:
					boolean done = this.handleUnsegment(typesOfBAsForOutput, namesOfBAsForOutput, rolesOfBAsForOutput, oneNewName);
					if ( ! done) {
						typesOfBAsForOutput.add(type);
						rolesOfBAsForOutput.add(roleName);
						namesOfBAsForOutput.add(oneNewName);
					}
				}
			}
			String[] outNames = new String[namesOfBAsForOutput.size()];
			typesOfInputBAs = new String[typesOfBAsForOutput.size()];
			String[] roles = new String[rolesOfBAsForOutput.size()];
			Iterator namesIt = namesOfBAsForOutput.iterator();
			Iterator typesIt = typesOfBAsForOutput.iterator();
			Iterator rolesIt = rolesOfBAsForOutput.iterator();
			int i = 0;
			while (namesIt.hasNext()) {
				outNames[i] = (String) namesIt.next();
				typesOfInputBAs[i] = (String) typesIt.next();
				roles[i] = (String) rolesIt.next();
				i++;
			}
			baseAttribsOfThisOutputParam = this.createBAsInConcept(outNames, typesOfInputBAs, roles, outputConcept); 
		}
	
		if (baseAttribsOfThisOutputParam != null) {
			Map oldInputParamObjectNames = this.setParameter(theOutputOpParam, baseAttribsOfThisOutputParam, loopNr);
			return oldInputParamObjectNames;
		}
		return null;
	}
	
	private boolean handleUnsegment(Collection types, Collection names, Collection roles, String nameOfOutputAttrib) 
	throws M4Exception {
		if (this.getTheOperator().getName().equals("Unsegment")) {
			// if the operator reverses a random segmentation:
			if (nameOfOutputAttrib.equals("(Random)")) {
				// simply do not create anything in the output concept!
				return true;
			}
			else {
				// find the step that is unsegmented:
				Iterator stepIt = this.getTheCase().getStepsToCompileBefore(this, true).iterator();
				while (stepIt.hasNext()) {
					Step predecessor = (Step) stepIt.next();
					if (predecessor.getTheOperator().getName().equals("SegmentationStratified")) {
						Parameter originalAttribParam = (Parameter) predecessor.getParameterTuple("TheAttribute", 0);
						if (originalAttribParam != null) {
							BaseAttribute origAttrib = (BaseAttribute) originalAttribParam.getTheParameterObject();
							if (origAttrib != null) {
								String type = origAttrib.getConceptualDataTypeName();
								types.add(type);
								names.add(nameOfOutputAttrib);
								roles.add(origAttrib.getRoleName());
								return true;
							}
						}
					}
				}
			}
		}
		return false;
	}
	
	// Attempts to find an assertion of this step's operator that
	// associates the output attribute to some input attribute. If such
	// an assertion exists, this method returns the role of that input attribute.
	private String determineRoleByAssertions(
			edu.udo.cs.miningmart.m4.OpParam theOutputOpParam,
			int loopNr) 
	throws M4Exception {
		Collection assertions = this.getTheOperator().getAssertions();
		assertions = this.getRelevantAssertions(theOutputOpParam.getName(), assertions);
		Iterator it = assertions.iterator();
		while (it.hasNext()) {
			Assertion ass = (Assertion) it.next();
			if (ass.getAssertionType().equalsIgnoreCase(Assertion.ESTIMATE_MV_TAKE_FROM)) {
				String otherParamOfAssertion = ass.getObj1();
				if (ass.getObj1().equalsIgnoreCase(theOutputOpParam.getName()))
					otherParamOfAssertion = ass.getObj2();
				BaseAttribute correspondingAttrib = 
					this.getAssociatedAttribute(theOutputOpParam.getName(), otherParamOfAssertion, loopNr);

				if (correspondingAttrib != null)
					return correspondingAttrib.getRoleName();
			}
		}
		return Roles.ROLE_NOROLE;
	}
	
	/*
	 * The given constraint says to copy some features from the input 
	 * concept(s) and to rename them at the same time
	 */
	private void addRenamedFeaturesToOutput(
			Constraint theRenameConstraint,
			Collection newNames,
			int loopNr)
	throws M4Exception {
		if ( ! theRenameConstraint.getType().equalsIgnoreCase(Constraint.TYPE_RENAME_ATTRIB)) {
			return;
		}
		String inputParamName = theRenameConstraint.getObj1();
		String outputParamName = theRenameConstraint.getObj2();
		OpParam outOpPar = (OpParam) this.getTheOperator().getOpParam(outputParamName);
		OpParam inOpPar = (OpParam) this.getTheOperator().getOpParam(inputParamName);
		Collection sourceFeatureParams = this.getParameterTuples(inOpPar, loopNr);		
		if (sourceFeatureParams.size() != newNames.size()) {
			throw new M4Exception("Step '" + this.getName() +
					"': error matching parameters for a RENAME constraint!");
		}
		BaseAttribute[] sourceBAs = new BaseAttribute[sourceFeatureParams.size()];
		Iterator it = sourceFeatureParams.iterator();
		int i = 0;
		while (it.hasNext()) {
			Parameter aPar = (Parameter) it.next();
			sourceBAs[i] = (BaseAttribute) aPar.getTheParameterObject();
			i++;
		}
		String[] namesForOutBAs = new String[newNames.size()];
		it = newNames.iterator();
		String[] typesForOutBAs = new String[namesForOutBAs.length];
		String[] rolesForOutBAs = new String[namesForOutBAs.length];
		for (i = 0; i < namesForOutBAs.length; i++) {
			namesForOutBAs[i] = (String) it.next();
			typesForOutBAs[i] = sourceBAs[i].getConceptualDataTypeName();
			rolesForOutBAs[i] = sourceBAs[i].getRoleName();
		}

		Concept outputConcept = (Concept) this.getOutputConcept();
		// ensure that there are no double names in the attribs to be created!
		// otherwise only one of them is created
		for (int j = 0; j < namesForOutBAs.length; j++) {
			Iterator baIt = outputConcept.getAllBaseAttributes().iterator();
			while (baIt.hasNext()) {
				BaseAttribute ba = (BaseAttribute) baIt.next();
				if (ba.getName().equalsIgnoreCase(namesForOutBAs[j])) 
					throw new M4Exception("Step '" + this.getName() + 
							"': cannot create features of output concept because output parameter\n'" + outputParamName +
							"' contains the name '" + namesForOutBAs[j] + "', which either occurs more than once (ignoring case)\nin this parameter, or it already occurs as another input feature!");
			}
		}
		Collection outputBAs = this.createBAsInConcept(namesForOutBAs, typesForOutBAs, rolesForOutBAs, outputConcept);
		this.setParameter(outOpPar, outputBAs, loopNr);
	}
	
	/*
	 // This method returns true if the output BA(s) creation cannot be done by the 
	 // generic mechanism; in this case, the output BA(s) creation is done in this method. 
	 // If false is returned, this method has done nothing.
	 private boolean workOnOutputBAsForSpecialOperators(
	 		Collection theNames, 
	 		edu.udo.cs.miningmart.m4.OpParam outOpPar, 
			boolean create, 
			int loopNr)  
	 throws M4Exception {
		 String name = this.getTheOperator().getName();
		
		 // check if we are dealing with Join/UnionByKey:
		 if (name.equalsIgnoreCase("JoinByKey") || name.equalsIgnoreCase("UnionByKey")) {
		 	
		 	// all we need to do here is go through the output concept and replace the special names
		 	// with the names in the given collection		 	
			ParameterObject[] po = this.getParameter("TheOutputConcept", 0);
			if (po == null || po.length == 0) {
				throw new M4Exception("Step '" + this.getName() + "': could not find output concept!");
			}
			Concept outputConcept = (Concept) po[0];
		 	Iterator fIt = outputConcept.getFeatures().iterator();
		 	
		 	Object[] namesMapOutput = theNames.toArray(); 
		 	// there should be only one name for the current loop number,
		 	// as 'MapOutput' is a coordinated parameter:
		 	if (namesMapOutput.length != 1) {
		 		throw new M4Exception("Creating output for Join/UnionByKey: found more than one name for '" + outOpPar.getName() + "' for loop number " + loopNr + "!");
		 	}
			String mapOutputName = (String) namesMapOutput[0];
			
		 	// we also set all output mapped attributes as parameters into the step		 	
		 	while (fIt.hasNext()) {
				Feature myFeature = (Feature) fIt.next();				
				String myFname = myFeature.getName();
				// ugly but sort of necessary ;-) :
				int positionOfMarker = myFname.indexOf(specialNameMarker);
				if (positionOfMarker > -1) {
					// find the index for MapOutput:
					String indexString = myFname.substring(positionOfMarker + specialNameMarker.length());
					int index = -1;
					try {
						index = Integer.parseInt(indexString);
					}
					catch (NumberFormatException nfe) {
						throw new M4Exception("Step '" + this.getName() + "': could not find MapInput for MapOutput (" + indexString + ")!");
					}
					
					// get the indexed name from MapOutput:
					if (index == loopNr) {						
						// change the special name to the name it is mapped to:
						myFeature.setName(mapOutputName);
					
						// set this feature as output parameter into this step:
						if (create) {
							Vector parObjColl = new Vector();
							parObjColl.add(myFeature);					
							this.setParameter(outOpPar, parObjColl, loopNr);
						}
					}
				}
			}
		 	return true;
		}
		
		if (name.equals("Unsegment")) {
			// simply add the UnsegmentAttribute to the output concept,
			// if there is such an attribute:
			ParameterObject[] po = this.getParameter("TheOutputConcept", 0);
			if (po == null || po.length == 0) {
				throw new M4Exception("Step '" + this.getName() + "': could not find output concept!");
			}
			Concept outputConcept = (Concept) po[0];
			
			if (theNames == null || theNames.isEmpty()) {
				throw new M4Exception("Step '" + this.getName() + "': no name found to create 'UnsegmentAttribute'!");
			}
			String attribName = (String) theNames.iterator().next();
			BaseAttribute newBa = (BaseAttribute) outputConcept.createBaseAttribute(attribName, ConceptualDatatypes.CDT_NOMINAL, BaseAttribute.TYPE_MINING);

			// set this feature as output parameter into this step:
			if (create) {
				Vector parObjColl = new Vector();
				parObjColl.add(newBa);					
				this.setParameter(outOpPar, parObjColl, loopNr);
			}
			return true;
		}
		
		// add further checks for special operators here:
		// if name.equalsIgnoreCase(mySpecialOpName) {  ... return true; }
		
		// no special operator:
		return false;
	 }
	*/
	
	// creates BAs in the given Concept with the given names and the given type, 
	// if such names do not exist in that concept yet. 
	// Returns a Collection with all BAs that were created or updated
	private Collection createBAsInConcept(
			String[] theNames, 
			String[] types,
			String[] roleNames,
			edu.udo.cs.miningmart.m4.Concept theConcept) 
	    throws M4Exception {
		
		Vector ret = new Vector(); // to be returned
		if (types.length != theNames.length) {
			throw new M4Exception("Step '" + this.getName() + "': could not find suitable datatype(s) for new output BA(s)!");
		}
		
		for (int i = 0; i < theNames.length; i++) {			
			String oneNewName = theNames[i];
			if (oneNewName == null) {
				continue;
			}
			// using NOMINAL as default datatype if nothing is known about the type:
			String datatype = (types[i] == null ? edu.udo.cs.miningmart.m4.ConceptualDatatypes.CDT_NOMINAL : types[i]);
			
			// using NO_ROLE as default role if nothing is known about the role:
			String role = (roleNames[i] == null ? Roles.ROLE_NOROLE : roleNames[i]);
			// check if the BA exists:
			BaseAttribute oldBa = (BaseAttribute) theConcept.getBaseAttribute(oneNewName);
			if (oldBa != null) {
				// only update it:
				oldBa.setConceptualDataTypeName(datatype);
				oldBa.setRoleName(role);
				ret.add(oldBa);
			}
			else {
				// debug:
				if (oneNewName.equals("")) {
					throw new UserError("Step '" + this.getName() + "': empty output object name!");
				}
				BaseAttribute ba = (BaseAttribute) theConcept.createBaseAttribute(oneNewName, datatype, BaseAttribute.TYPE_MINING, role);
				ret.add(ba);
			}
		}		
		return ret;
	}
	
	// returns a two-cell String array with the type in the first cell and the role in the second
	private String[] getTypeAndRoleFromCorrespondingBA( edu.udo.cs.miningmart.m4.Constraint aConstr, 
											   int loopNr) 
		throws M4Exception {

		// we know that we must find the target attribute 
		// (from which to get the types) in the constraint:
		String nameOfParamWithSameType = aConstr.getObj2();
		String nameOfConstrainedParam = aConstr.getObj1();
		
		if (nameOfParamWithSameType == null) {
			throw new M4Exception("Step '" + this.getName() + 
					"': a SAME_TYPE constraint was found without second object!");
		}
		// sometimes the target attribute is in obj1 instead obj2, so
		// check that the target is an input param:
		OpParam targetOpPar = (OpParam) this.getTheOperator().getOpParam(nameOfParamWithSameType);
		if (targetOpPar == null) {
			throw new M4Exception("Step '" + this.getName() + 
					"': a same-type constraint was found that does not use a parameter of this step!");
		}
		if ( ! targetOpPar.isInput()) {
			// now try obj2:
			nameOfParamWithSameType = aConstr.getObj1();
			nameOfConstrainedParam = aConstr.getObj2();
			targetOpPar = (OpParam) this.getTheOperator().getOpParam(nameOfParamWithSameType);
			// for debugging make it safe:
			if (targetOpPar == null || ( ! targetOpPar.isInput())) {
				throw new M4Exception("Step '" + this.getName() +
						"': a same-type constraint was found that does not use an input parameter of this step!");
			}
		}
		
		BaseAttribute theTarget = this.getAssociatedAttribute(nameOfConstrainedParam, nameOfParamWithSameType, loopNr);
		
		if (theTarget == null) {
			// use a default:
			return new String[] { edu.udo.cs.miningmart.m4.ConceptualDatatypes.CDT_NOMINAL,
					              Roles.ROLE_NOROLE };
		}
		
		// 	now we know the type and role for the new BA:
		return new String[] { theTarget.getConceptualDataTypeName(),
				              theTarget.getRoleName() };
	}
	
	// This method assumes that the two baseattribute parameters whose 
	// parameter names are given are related by a constraint or assertion. It
	// returns the BaseAttribute that is the instance of the second parameter
	// name in this step.
	private BaseAttribute getAssociatedAttribute(
			String nameOfParameterOfGivenAttrib, 
			String nameOfParameterOfAssociatedAttrib,
			int loopNr) 
	throws M4Exception {

		// decide if both parameters use the same loop numbering:
		OpParam constrainedOpPar = (OpParam) this.getTheOperator().getOpParam(nameOfParameterOfGivenAttrib);
		OpParam targetOpPar = (OpParam) this.getTheOperator().getOpParam(nameOfParameterOfAssociatedAttrib);

		if (constrainedOpPar.isCoordinated() && ( ! targetOpPar.isCoordinated())) {
			loopNr = 0;
		}
		if (targetOpPar.isCoordinated() && ( ! constrainedOpPar.isCoordinated())) {
			loopNr = 1;
		}
		ParameterObject[] targets = this.getParameter(nameOfParameterOfAssociatedAttrib, loopNr);
		
		if (targets == null || targets.length == 0) {
			// this can happen when the target attribute was deleted by the user!
			// So let's be robust:
			this.doPrint(Print.MAX, "WARNING: Step '" + this.getName() + 
					              "': no parameter '" + nameOfParameterOfAssociatedAttrib +
								  "' found, though this name is used in a constraint!");
			return null;
		}
		
		// we assume that the target object from the constraint is a BA:
		BaseAttribute theTarget = null;
		try {
			theTarget = (BaseAttribute) targets[0];
		}
		catch (ClassCastException cce) {
			throw new M4Exception("Step '" + this.getName() +
				              	"': a same-type constraint or assertion was found that does not use the name of a BA parameter!");
		}
		if (theTarget == null) {
			throw new M4Exception("Step '" + this.getName() +
				              	"': could not find a parameter called '" + 
								nameOfParameterOfAssociatedAttrib + 
							  	"', though it was specified in a same-type constraint!");
		}
		if ( ! this.isVisible(theTarget)) {
			throw new M4Exception("Step '" + this.getName() + "': The target attribute '" + 
					theTarget.getName() + "' is not visible here!");
		}
		return theTarget;
	}
	
	/**
	 * This method changes the names of the output parameter objects; the latter
	 * are assumed to exist. If the input parameters of the step have also changed,
	 * call the method <code>updateOutput</code> before this one. 
	 * 
	 * @param theOpParam the OpParam object specifying which output parameter is meant
	 * @param theNames a Collection of Strings giving the new name(s)
	 * @throws M4Exception
	 */
	public void renameOutput(edu.udo.cs.miningmart.m4.OpParam theOpParam, Collection theNames) throws M4Exception {
	
		if (theOpParam.isInput()) {
			return;
		}
		// deal with the special case of coordination:
		Collection flatNames = theNames;
		if (theOpParam.isCoordinated() || theOpParam.isLoopable()) {
			flatNames = this.flattenCollections(theNames);
		}
		Iterator namesIt = flatNames.iterator();
		int startLoopNr, endLoopNr;
		if  ((theOpParam.isLoopable() && this.getLoopCount() > 0) || theOpParam.isCoordinated()) {
			startLoopNr = 1;
			endLoopNr = this.getHighestLoopNr(theOpParam);
		}
		else {
			startLoopNr = endLoopNr = 0;
		}
		if (namesIt.hasNext()) {
			for (int loopNr = startLoopNr; loopNr <= endLoopNr; loopNr++) {
				Parameter thePar = (Parameter) this.getParameterTuple(theOpParam.getName(), loopNr);
				if (thePar == null) {
					this.createOutput(theOpParam, theNames);
					return;
					/* oder aber:
					throw new M4Exception("Step '" + this.getName() + "': parameter '" +
							theOpParam.getName() + "' is missing!");
					*/
				}
				ParameterObject myParObj = (ParameterObject) thePar.getTheParameterObject();
				String oldName = myParObj.getName();
				String newName = (String) namesIt.next();
				if (myParObj instanceof BaseAttribute) {
					// get the parameters where the attrib is used as input:
					Collection inputParamsUsingAttr = this.getInputParametersUsingFeature((Feature) myParObj);
					this.giveValidOutputObjectName(myParObj, newName);
					this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames = new HashMap();
					this.createMapOfInputParamsToNames(
							inputParamsUsingAttr, 
							this.mapFromInputParametersOfFollowingStepsToTheirOldParameterObjectNames,
							oldName,
							newName);
				}
				else {
					this.giveValidOutputObjectName(myParObj, newName);
				}
			}
		}
	}
	
	// gets a collection of collections of objects, and returns
	// a collection of the objects
	private Collection flattenCollections(Collection coll) throws M4Exception {
		if (coll == null) {
			throw new M4Exception("Step '" + this.getName() + "' (method flattenCollections): got >null< instead of names for output objects!");
		}
		Iterator it1 = coll.iterator();
		Vector ret = new Vector();
		while (it1.hasNext()) {
			Collection innerColl = (Collection) it1.next();
			if (innerColl == null) {
				throw new M4Exception("Step '" + this.getName() + "' (method flattenCollections): got >null< instead of names for output objects (inner collection)!");
			}
			Iterator it2 = innerColl.iterator();
			while (it2.hasNext()) {
				ret.add(it2.next());
			}
		}
		return ret;
	}
	
	/*
	private void propagateNameChange(edu.udo.cs.miningmart.m4.BaseAttribute changedBA) throws M4Exception {		
		
		// propagate the new name through the later steps:
		Collection laterSteps = this.getSuccessors(); // returns only the direct successors
		Iterator stepIt = laterSteps.iterator();
		while (stepIt.hasNext()) {
			Step oneSuccessor = (Step) stepIt.next();
			
			// now find the changed BA in the output of the successor step:			
			Concept outputConcept = (Concept) oneSuccessor.getOutputConcept();
			if (outputConcept == null) {
				// this step has no output concept, but the propagation
				// must go on nevertheless:
				oneSuccessor.propagateNameChange(changedBA);
			}
			else {
				// find the corresponding BA in the output of the successor
				// step, rename it and propagate further:
				Collection outputBAs = outputConcept.getAllBaseAttributes();
				Iterator it = outputBAs.iterator();
				boolean correspondingBaFound = false;
				while (it.hasNext()) {
					BaseAttribute myBA = (BaseAttribute) it.next();
					if (myBA.equals(changedBA)) {
						myBA.setName(changedBA.getName());
						oneSuccessor.propagateNameChange(myBA);
						correspondingBaFound = true;
					}
				}
				// for debugging:
				if ( ! correspondingBaFound) {
					throw new M4Exception("Step '" + oneSuccessor.getName() + 
							"': could not find corresponding output BA for BA '" +
							changedBA.getName() + "'!");
				}
			}
		}
	}
	*/
	
	/**
	 * This method returns all concepts that are input concept to this step's
	 * operator. (For Join and other operators, there can be more than one input
	 * concept.) Note that concepts attached to an input relation are not returned!
	 */
	public Collection getInputConceptsThatAreParameters() throws M4Exception {
		Collection theInputOpParams = this.getTheOperator().getAllInputOperatorParameters();
		Iterator it = theInputOpParams.iterator();
		Vector theInputConcepts = new Vector();
		while (it.hasNext()) {
			OpParam myOpPar = (OpParam) it.next();
			if (myOpPar.isConceptParameter()) {
				Collection inputObjects = this.getParameter(myOpPar, 0); // an input concept is never looped!
				if (inputObjects != null) { // some input concept parameters are optional!
					theInputConcepts.addAll(inputObjects);
				}
			}
		}
		return theInputConcepts;
	}
	
	public Collection getAllInputConcepts() throws M4Exception {
		Collection directInputs = this.getInputConceptsThatAreParameters();

		// now add the concepts attached to input relations:
		Iterator it = this.getAllInputRelations().iterator();
		while (it.hasNext()) {
			Relation inRel = (Relation) it.next();
			Concept from = (Concept) inRel.getTheFromConcept();
			Concept to = (Concept) inRel.getTheToConcept();
			if ( ! directInputs.contains(from)) {
				directInputs.add(from);
			}
			if ((to != null) && ( ! directInputs.contains(to))) {
				directInputs.add(to);
			}			
		}	
		return directInputs;
	}
	
	/**
	 * Returns all relations that are a parameter object of an input
	 * parameter of this Step. 
	 * @return a collection of relations
	 * @throws M4Exception
	 */
	public Collection getAllInputRelations() throws M4Exception {
		Collection theInputOpParams = this.getTheOperator().getAllInputOperatorParameters();
		Iterator it = theInputOpParams.iterator();
		Vector theInputRels = new Vector();
		while (it.hasNext()) {
			OpParam myOpPar = (OpParam) it.next();
			if (myOpPar.isRelationParameter()) {
				Collection relPars = this.getParameterTuples(myOpPar);
				Iterator parIt = relPars.iterator();
				while (parIt.hasNext()) {
					Parameter myPar = (Parameter) parIt.next();
					theInputRels.add(myPar.getTheParameterObject());
				}
			}
		}
		return theInputRels;		
	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.Step#getOutputConcept()
	 */
	public edu.udo.cs.miningmart.m4.Concept getOutputConcept() throws M4Exception {
		if (this.getTheOperator() == null) {
			return null;
		}
		Collection theOutputOpParams = this.getTheOperator().getAllOutputOperatorParameters();
		Iterator it = theOutputOpParams.iterator();
		while (it.hasNext()) {
			OpParam myOpPar = (OpParam) it.next();
			if (myOpPar.isConceptParameter()) {
				// an output concept is never looped:
				int loopNr = 0;
				Collection outputConcepts = this.getParameter(myOpPar, loopNr);
				if (outputConcepts == null) {
					return null;
				}
				// There can be atmost one output concept:
				if (outputConcepts.size() > 1) {
					throw new M4Exception("Step '" + this.getName() + 
							"': found more than one output concept (Parameter '" + 
							myOpPar.getName() + "'!");
				}
				if ( ! outputConcepts.isEmpty()) {
					return ((Concept) outputConcepts.iterator().next());
				}
			}
		}
		return null;
	}
	
	/**
	 * One of the input concepts of this step is given, and is 
	 * set to be the other given concept. All parameters that depend
	 * on it are changed too, if possible. For example, if this step
	 * has a parameter 'TheTargetAttribute', and the new concept has an
	 * attribute whose name equals the old target attribute, the target
	 * attribute is changed to be the attribute of the new concept.
	 * 
	 * @param oldConcept existing input concept, to be replaced
	 * @param newConcept replacing input concept
	 * @throws M4Exception
	 */
	public void changeInputConcept(
			edu.udo.cs.miningmart.m4.Concept oldConcept,
			edu.udo.cs.miningmart.m4.Concept newConcept)
	throws M4Exception {
		if (oldConcept == null || newConcept == null) {
			return;
		}
		if (this.getTheOperator() == null) {
			return;
		}
		// find out the parameter of the old input concept:
		Collection inOpPars = this.getTheOperator().getAllInputOperatorParameters();
		if (inOpPars != null) {
			Iterator inOpParIt = inOpPars.iterator();
			Parameter paramToBeChanged = null;
			OpParam opParamToBeChanged = null;
			while (inOpParIt.hasNext()) {
				OpParam anInputOpPar = (OpParam) inOpParIt.next();
				if (anInputOpPar.isConceptParameter()) {
					Collection allInputConceptParams = this.getParameterTuples(anInputOpPar);
					if (allInputConceptParams != null) {
						Iterator paramsIt = allInputConceptParams.iterator();
						while (paramsIt.hasNext()) {
							Parameter myInConPar = (Parameter) paramsIt.next();
							if (myInConPar.getTheParameterObject().equals(oldConcept)) {
								paramToBeChanged = myInConPar;
								opParamToBeChanged = anInputOpPar;
							}								
						}
					}
				}
			}
			if (paramToBeChanged == null || opParamToBeChanged == null) {
				// the old concept is not an input concept of this step, so
				// we do not need to replace it:
				return;
			}
			
			// set the new input concept:
			paramToBeChanged.setTheParameterObject(newConcept);
			
			// find out if any other input parameters depend on this input concept:
			Collection constrs = opParamToBeChanged.getApplicableConstraints();
			if (constrs != null) {
				Iterator constrIt = constrs.iterator();
				while (constrIt.hasNext()) {
					Constraint myConstr = (Constraint) constrIt.next();
					// only IN constraints signal the dependence on the input concept:
					if (myConstr.getType().equals(Constraint.TYPE_CONTAINED_IN)) {
						// find out what is constrained to be IN the input concept:
						String otherOpParOfThisConstraint = null;
						if (myConstr.getObj1().equalsIgnoreCase(opParamToBeChanged.getName())) 
							otherOpParOfThisConstraint = myConstr.getObj2();
						else 
							otherOpParOfThisConstraint = myConstr.getObj1();
						OpParam otherOpPar = (OpParam) this.getTheOperator().getOpParam(otherOpParOfThisConstraint);
						// only features can depend on the input concept:
						if (otherOpPar != null && otherOpPar.isInput() && 
							(otherOpPar.isBaseAttribParameter() || otherOpPar.isFeatureParameter())) {
							Iterator otherParamsIt = this.getParameterTuples(otherOpPar).iterator();
							while (otherParamsIt.hasNext()) {
								Parameter inputFParam = (Parameter) otherParamsIt.next();
								ParameterObject po = (ParameterObject) inputFParam.getTheParameterObject();
								if (po instanceof Feature) {
									Feature oldFeature = (Feature) po;
									if (oldFeature.getConcept().equals(oldConcept)) {
										// if the new input concept has a corresponding feature, set it:
										Feature newFeature = (Feature) newConcept.getFeature(po.getName());
										if (newFeature != null) {
											inputFParam.setTheParameterObject(newFeature);
										}
									}
								}
							}
						}
					}					
				}
			}
		}
	}
	
	/** 
	 * Returns a collection of concepts; these concepts represent
	 * the data model that the current Case produced up to this Step.
	 * 
	 * @return a collection of concepts
	 * @throws M4Exception
	 */
	public Collection getResultingDataModel() throws M4Exception {
		// a data model is simply a collection of concepts.
		Collection inputDataModelOfCase = new Vector(this.getTheCase().getInputDataModel());
		
		// move backwards from this step towards the input of the first step
		// that this step depends on. If the input concept of this step is ever
		// an output concept of one of the preceding steps, it replaces the
		// case input.
		Collection conceptsToBeReplaced = new Vector();
		Collection visitedSteps = new HashSet();
		Concept replacingConcept = this.getReplacingConcept(this, conceptsToBeReplaced, visitedSteps);
		
		if (replacingConcept != null) {
			inputDataModelOfCase.removeAll(conceptsToBeReplaced);
			inputDataModelOfCase.add(replacingConcept);
		}
		return inputDataModelOfCase;
	}
	
	/*
	 * returns the last concept in the chain leading to this step
	 * that is an output concept, ie it replaces the input of its step
	 */
	private Concept getReplacingConcept(Step current, Collection conceptsToBeReplaced, Collection visitedSteps) throws M4Exception {
		
		if (current == null)
			return null;
		
		visitedSteps.add(current);
		
		Concept ret = null;
		Collection preds = current.getAllPredecessors();
		if (preds != null && ( ! preds.isEmpty())) {
			Iterator it = preds.iterator();
			// find out if the step has fewer predecessors than input concepts!:
			Vector outputConceptsOfIncomingSteps = new Vector();
			while (it.hasNext()) {
				Step next = (Step) it.next();
				if ( ! visitedSteps.contains(next)) {
					ret = this.getReplacingConcept(next, conceptsToBeReplaced, visitedSteps);
					outputConceptsOfIncomingSteps.add(ret);
				}
			}
			// if the current step has additional input concepts of type DB,
			// add the extra input concepts to the set of concepts to be replaced:
			Collection inCons = current.getAllInputConcepts();
			Iterator concIt = inCons.iterator();
			while (concIt.hasNext()) {
				Concept anInputConcept = (Concept) concIt.next();
				if ((anInputConcept.getType().equals(Concept.TYPE_DB)) 
						&& ( ! outputConceptsOfIncomingSteps.contains(anInputConcept))) {
					conceptsToBeReplaced.add(anInputConcept);
				}
			}
		}
		else {
			// current step has no predecessor, so add its input concepts to the
			// concepts to be replaced, if they are not in there yet:
			Iterator it = current.getAllInputConcepts().iterator();
			while (it.hasNext()) {
				Concept inCon = (Concept) it.next();
				if ( ! conceptsToBeReplaced.contains(inCon)) {
					conceptsToBeReplaced.add(inCon);
				}
			}			
		}
		if (current.getOutputConcept() != null) {
			// current step has output concept, so it has replaced its
			// input concepts:
			ret = (Concept) current.getOutputConcept();
		}
		return ret;
	}
	
	/** 
	 * This method must only be called if any input parameter of this Step
	 * was changed. It ensures consistency of the output parameter objects
	 * with the input parameter objects. If the name of the output parameter
	 * has also changed, call the method <code>renameOutput</code> afterwards.
	 * 
	 * @param theOpParam the OpParam object specifying which output parameter is meant
	 * @throws M4Exception
	 */
	public void updateOutput(edu.udo.cs.miningmart.m4.OpParam theOpParam) throws M4Exception {
		boolean create = false;
		this.workOnOutput(theOpParam, null, create);
	}
	
	/**
	 * Returns TRUE iff a change to the given input OpParam can have any effect
	 * on the output parameters of this Step. For example, if the NewRangeMin-parameter
	 * of a Scaling-Step is changed then the output parameters are not affected. 
	 * In contrast, if the number of features selected in a manual FeatureSelection
	 * step changes, then the output concept must be updated.
	 * 
	 * @param theOpParam an input OpParam of the operator of this step
	 * @return TRUE if the output may have to be updated if the input was updated
	 * @throws M4Exception
	 */
	public boolean inputChangeCanAffectOutput(edu.udo.cs.miningmart.m4.OpParam theOpParam)
	throws M4Exception {
		if ( ! theOpParam.isInput()) {
			return false;
		}
		// any change to an input concept can affect the output:
		if (theOpParam.isConceptParameter()) {
			return true;
		}
		Iterator it = theOpParam.getApplicableConstraints().iterator();
		while (it.hasNext()) {
			Constraint myConstr = (Constraint) it.next();
			if (myConstr.getType().equals(Constraint.TYPE_CREATE_ATTRIB_BY) ||
				myConstr.getType().equals(Constraint.TYPE_CREATE_ATTRIB_SUFF)) {
				return true;
			}
			String otherParameter = myConstr.getObj1();
			if (otherParameter.equalsIgnoreCase(theOpParam.getName())) {
				otherParameter = myConstr.getObj2();
			}
			// now otherParameter is the "constraining" parameter:
			OpParam otherOpPar = (OpParam) this.getTheOperator().getOpParam(otherParameter);
			if (otherOpPar != null && ( ! otherOpPar.isInput())) {
				return true;
			}			
		}
		/*
		// there are some special cases unfortunately:
		if ( ! constraintConnectingGivenOpParamToAnOutputOpParam_Exists) {
			if (this.getTheOperator().getName().equalsIgnoreCase("Pivotize")) {
				if (theOpParam.isBaseAttribParameter()) { 
					return true; // all attributes (index, pivot, groupby) affect the output
				}
				if (theOpParam.getName().equalsIgnoreCase(Pivotize.PARAMETER_VALUES)) {
					return true; // each index value becomes an output attribute
				}
			}
			if (this.getTheOperator().getName().equalsIgnoreCase("SpecifiedStatistics")) {
				if (theOpParam.isBaseAttribParameter() || theOpParam.getName().equals(SpecifiedStatistics.PARAMETER_DISTRIB_VAL)) { 
					return true; // 
				}				
			}
		}
		*/
		return false;
	}
	
	public edu.udo.cs.miningmart.m4.EstimatedStatistics estimateStatisticsForOutputConcept()
	throws M4Exception {
		// to be safe check the existence of an output concept:
		Concept outputConcept = (Concept) this.getOutputConcept();
		if (outputConcept == null) {
			return null;
		}
		if (outputConcept.getType() == Concept.TYPE_DB) {
			throw new M4Exception("Error in step '" + this.getName() + "': output concept of this step is of type DB!");
		}
		
		EstimatedStatistics outputEstimations = new EstimatedStatistics(outputConcept);
		
		
		// Do ESTIMATION. Get the input's estimations, there may be several:
		Collection inputConcepts = this.getAllInputConcepts();
		if (inputConcepts.size() == 1) {
			// step 1: assume the estimations can be copied to the output:
			Concept inputConcept = (Concept) inputConcepts.iterator().next();
			EstimatedStatistics inputEstimations = (EstimatedStatistics) inputConcept.getEstimatedStatistics(this);
			this.copyEstimationsForMatchingAttribs(inputEstimations, outputEstimations);
			
			// step 2: modify the estimations and add those for nonmatching attribs:
			this.applyEstimationsOneInput(inputEstimations, outputEstimations);
		}
		else {
			// merge the estimations from the input concepts:
			outputEstimations = this.mergeEstimations(inputConcepts, outputConcept);
		}
		
		// maybe some ACTUAL statistics are available; if yes,
		// let them overwrite the estimated ones:
		outputEstimations.readAvailableStatisticsFromDb();
		
		return outputEstimations;
	}
	
	// this method applies any assertions that apply to merging the estimations
	// from more than one input concept
	private EstimatedStatistics mergeEstimations(Collection inputConcepts, Concept outputConcept)
	throws M4Exception {
		// check that there are more than one input concepts
		if (inputConcepts == null) {
			return null;
		}
		if (inputConcepts.size() > 1) {
			EstimatedStatistics[] allESs = new EstimatedStatistics[inputConcepts.size()];
			Iterator it = inputConcepts.iterator();
			int i = 0;
			while (it.hasNext()) {
				Concept myInputConcept = (Concept) it.next();
				allESs[i] = (EstimatedStatistics) myInputConcept.getEstimatedStatistics(this);
				i++;
			}
			EstimatedStatistics outputES = new EstimatedStatistics(outputConcept);
			Collection assertions = this.getTheOperator().getAssertions();
			Assertion[] sortedAssertions = this.sortAssertions(assertions);
			for (i = 0; i < sortedAssertions.length; i++) {
				sortedAssertions[i].applyEstimationSeveralInputs(this, allESs, outputES);
			}
			return outputES;
		}		
		return null;
	}

	private void copyEstimationsForMatchingAttribs(EstimatedStatistics inputEstimations, EstimatedStatistics outputEstimations)
	throws M4Exception {
		// first copy global estimations:
		outputEstimations.setNumberOfRows(inputEstimations.getNumberOfRows());
		
		// second copy estimations for each attribute:
		Concept outputConcept = (Concept) outputEstimations.getConcept();
		Concept inputConcept = (Concept) inputEstimations.getConcept();
		Collection allOutAttribs = outputConcept.getAllBaseAttributes();
		BaseAttribute[] outAttribs = this.getBaseAttribArray(allOutAttribs);
		BaseAttribute[] inAttribs = this.getAllCorrespondingAttribs(outAttribs, inputConcept);
		for (int i = 0; i < inAttribs.length && i < outAttribs.length; i++) {
			if (inAttribs[i] != null) {
				outputEstimations.setBiggestValue(outAttribs[i].getName(), inputEstimations.getBiggestValue(inAttribs[i].getName()));
				outputEstimations.setLowestValue(outAttribs[i].getName(), inputEstimations.getLowestValue(inAttribs[i].getName()));
				outputEstimations.setNumberOfMissingValues(outAttribs[i].getName(), inputEstimations.getNumberOfMissingValues(inAttribs[i].getName()));
				Vector valueList;
				outputEstimations.setValueList(outAttribs[i].getName(), (valueList = inputEstimations.getValueList(inAttribs[i].getName())));
				Iterator valueIt = valueList.iterator();
				while (valueIt.hasNext()) {
					String value = (String) valueIt.next();
					outputEstimations.setNumberOfOccurrences(outAttribs[i].getName(), value, inputEstimations.getNumberOfOccurrences(inAttribs[i].getName(), value));
				}
			}
		}
	}
	
	private BaseAttribute[] getBaseAttribArray(Collection baseAttribCollection) {
		if (baseAttribCollection == null || baseAttribCollection.isEmpty()) {
			return null;
		}
		BaseAttribute[] ret = new BaseAttribute[baseAttribCollection.size()];
		Iterator it = baseAttribCollection.iterator();
		int i = 0;
		while (it.hasNext()) {
			ret[i] = (BaseAttribute) it.next();
			i++;
		}
		return ret;
	}
	
	private void applyEstimationsOneInput(EstimatedStatistics inputEstimations, EstimatedStatistics outputEstimations)
	throws M4Exception {
		// apply all assertions of this step's operator and modify the 
		// outputEstimations accordingly
		Collection assertions = this.getTheOperator().getAssertions();
		Assertion[] sortedAssertions = this.sortAssertions(assertions);
		for (int i = 0; i < sortedAssertions.length; i++) {
			sortedAssertions[i].applyEstimationOneInput(this, inputEstimations, outputEstimations);
		}
	}
	
	public void estimateStatisticsForOutputAttrib(
			edu.udo.cs.miningmart.m4.EstimatedStatistics currentOutputEstimations,
			edu.udo.cs.miningmart.m4.BaseAttribute outputAttribute) 
	throws M4Exception {
		// check that the output attribute is indeed an output attrib of this step
		Collection outOpParamsOfThisStep = this.getTheOperator().getAllOutputOperatorParameters();
		boolean found = false;
		String nameOfOutputAttribsParameter = null;
		Iterator it = outOpParamsOfThisStep.iterator();
		while (it.hasNext()) {
			OpParam outOpPar = (OpParam) it.next();
			Iterator outParIt = this.getParameterTuples(outOpPar).iterator();
			while (outParIt.hasNext()) {
				Parameter myPar = (Parameter) outParIt.next();
				if (outputAttribute.equals(myPar.getTheParameterObject())) {
					found = true;
					nameOfOutputAttribsParameter = this.getParameterNameWithoutSuffix(myPar.getName());
					break;
				}
			}
		}
		if (found) {
			Collection assertions = this.getTheOperator().getAssertions();
			assertions = this.getRelevantAssertions(nameOfOutputAttribsParameter, assertions);
			Assertion[] sortedAssertions = this.sortAssertions(assertions);
			for (int i = 0; i < sortedAssertions.length; i++) {
				sortedAssertions[i].applyEstimationOneInput(this, currentOutputEstimations, currentOutputEstimations);
			}			
		}
	}
	
	private Collection getRelevantAssertions(String paramName, Collection givenAssertions) {
		if (givenAssertions == null)
			return null;
		Iterator it = givenAssertions.iterator();
		Vector ret = new Vector();
		while (it.hasNext()) {
			Assertion anAssertion = (Assertion) it.next();
			if (anAssertion.getObj1().equalsIgnoreCase(paramName) ||
				(anAssertion.getObj2() != null && anAssertion.getObj2().equalsIgnoreCase(paramName))) {
				ret.add(anAssertion);
			}
		}
		return ret;
	}
	
	// unfortunately some assertions have to be handled before others!
	// This method does the sorting
	private Assertion[] sortAssertions(Collection unsortedAssertions) {
		if (unsortedAssertions == null)
			return null;
		Assertion[] ret = new Assertion[unsortedAssertions.size()];
		Iterator it = unsortedAssertions.iterator();
		Vector vlAssertions = new Vector();
		Vector vfAssertions = new Vector();
		Vector otherAssertions = new Vector();
		// ensure that value list assertions are sorted before value frequency 
		// assertions!
		while (it.hasNext()) {
			Assertion myAssertion = (Assertion) it.next();
			if (myAssertion.isValueListAssertion()) {
				if (myAssertion.getAssertionType().equals(Assertion.ESTIMATE_VALLIST_ADD_VALUE) ||
					myAssertion.getAssertionType().equals(Assertion.ESTIMATE_VALLIST_BY_SYMBOL)) {
					vlAssertions.add(vlAssertions.size(), myAssertion);
				}
				else {
					vlAssertions.add(0, myAssertion);
				}
			}
			else if (myAssertion.isValueFrequencyAssertion()) {
				vfAssertions.add(myAssertion);
			}
			else otherAssertions.add(myAssertion);
		}
		
		for (int i = 0; i < ret.length; i++) {
			if (i < vlAssertions.size()) {
				ret[i] = (Assertion) vlAssertions.get(i);
			}
			else if (i < vlAssertions.size() + vfAssertions.size()) {
				ret[i] = (Assertion) vfAssertions.get(i - vlAssertions.size());
			}
			else ret[i] = (Assertion) otherAssertions.get(i - vlAssertions.size() - vfAssertions.size());
		}
		return ret;
	}
	
	private class ConceptualAttributeInfo {
		String name;
		String conceptualDataType;
		String role;
		
		ConceptualAttributeInfo(BaseAttribute theAttribute) throws M4Exception {
			
			if (theAttribute == null) {
				throw new M4Exception("Step.ConceptualAttributeInfo, constructor: got NULL attribute!");
			}
			this.name = theAttribute.getName();
			this.conceptualDataType = theAttribute.getConceptualDataTypeName();
			this.role = theAttribute.getRoleName();
		}
		
		boolean hasMatchIn(Collection otherAttribInfos) {
			Iterator it = otherAttribInfos.iterator();
			while (it.hasNext()) {
				ConceptualAttributeInfo otherInfo = (ConceptualAttributeInfo) it.next();
				if (this.name.equals(otherInfo.name) &&
					this.conceptualDataType.equalsIgnoreCase(otherInfo.conceptualDataType) &&
					this.role.equalsIgnoreCase(otherInfo.role)) {
					return true;
				}
			}
			return false;
		}
	}
	
	private class ConceptualConceptInfo {
		String name;
		Vector attribInfos;
		
		ConceptualConceptInfo(Concept theConcept) throws M4Exception {
			if (theConcept == null) {
				throw new M4Exception("Step.ConceptualConceptInfo, constructor: got NULL concept!");
			}
			this.name = theConcept.getName();
			Iterator it = theConcept.getAllBaseAttributes().iterator();
			this.attribInfos = new Vector();
			while (it.hasNext()) {
				BaseAttribute ba = (BaseAttribute) it.next();
				this.attribInfos.add(new ConceptualAttributeInfo(ba));
			}
		}
		
		// returns TRUE if the given collection of conceptual concept info objects
		// has an object that represents an exactly matching concept
		boolean hasMatchIn(Collection otherInfos) {
			Iterator it = otherInfos.iterator();
			while (it.hasNext()) {
				ConceptualConceptInfo info = (ConceptualConceptInfo) it.next();
				if (this.name.equals(info.name)) {					
					Iterator attrInfoIt = this.attribInfos.iterator();
					boolean allAttribsMatch = true;
					while (attrInfoIt.hasNext()) {
						ConceptualAttributeInfo ai = (ConceptualAttributeInfo) attrInfoIt.next();
						if ( ! ai.hasMatchIn(info.attribInfos)) {
							allAttribsMatch = false;
						}
					}
					// reverse direction:
					if (allAttribsMatch) {
						attrInfoIt = info.attribInfos.iterator();
						while (attrInfoIt.hasNext()) {
							ConceptualAttributeInfo ai = (ConceptualAttributeInfo) attrInfoIt.next();
							if ( ! ai.hasMatchIn(this.attribInfos)) {
								allAttribsMatch = false;
							}
						}
					}
					if (allAttribsMatch)
						return true;
				}
			}
			return false;
		}
	}
}
/*
 * Historie
 * --------
 * 
 * $Log: Step.java,v $
 * Revision 1.71  2006/10/01 19:14:22  euler
 * Mysql works
 *
 * Revision 1.70  2006/09/27 14:59:59  euler
 * New version 1.1
 *
 * Revision 1.69  2006/09/21 13:39:28  euler
 * Bugs fixed
 *
 * Revision 1.68  2006/09/21 11:42:42  euler
 * *** empty log message ***
 *
 * Revision 1.67  2006/09/20 21:17:00  euler
 * Working finally
 *
 * Revision 1.66  2006/09/20 15:23:44  euler
 * Bugfixes and extensions
 *
 * Revision 1.65  2006/09/20 07:35:22  euler
 * *** empty log message ***
 *
 * Revision 1.64  2006/09/19 10:54:17  euler
 * Bugfixes
 *
 * Revision 1.63  2006/09/18 15:25:48  euler
 * Bugs fixed
 *
 * Revision 1.62  2006/09/14 13:03:07  euler
 * *** empty log message ***
 *
 * Revision 1.61  2006/09/14 11:51:07  euler
 * bugs fixed
 *
 * Revision 1.60  2006/09/13 16:20:24  euler
 * *** empty log message ***
 *
 * Revision 1.59  2006/09/13 12:19:43  euler
 * *** empty log message ***
 *
 * Revision 1.58  2006/09/13 08:34:20  euler
 * *** empty log message ***
 *
 * Revision 1.57  2006/09/12 15:29:29  euler
 * bugs fixed
 *
 * Revision 1.56  2006/09/12 11:42:55  euler
 * bugs fixed
 *
 * Revision 1.55  2006/09/12 09:56:31  euler
 * *** empty log message ***
 *
 * Revision 1.54  2006/09/11 14:46:38  euler
 * *** empty log message ***
 *
 * Revision 1.53  2006/09/10 10:49:59  euler
 * *** empty log message ***
 *
 * Revision 1.52  2006/09/06 16:05:52  euler
 * *** empty log message ***
 *
 * Revision 1.51  2006/09/04 17:21:40  euler
 * Bugfixes around statistics estimation
 *
 * Revision 1.50  2006/09/04 08:36:21  euler
 * *** empty log message ***
 *
 * Revision 1.49  2006/09/02 12:59:33  euler
 * *** empty log message ***
 *
 * Revision 1.48  2006/08/30 11:51:50  euler
 * *** empty log message ***
 *
 * Revision 1.47  2006/08/25 17:43:51  euler
 * *** empty log message ***
 *
 * Revision 1.46  2006/08/24 17:59:28  euler
 * *** empty log message ***
 *
 * Revision 1.45  2006/08/24 13:01:25  euler
 * Started implementation of statistics estimation
 *
 * Revision 1.44  2006/08/21 15:18:52  euler
 * Bugfix
 *
 * Revision 1.43  2006/08/21 13:59:07  euler
 * Bugfixes
 *
 * Revision 1.42  2006/08/20 16:55:07  euler
 * bugs fixed
 *
 * Revision 1.41  2006/08/19 18:43:26  euler
 * Some Bugfixes
 *
 * Revision 1.40  2006/08/18 15:04:22  euler
 * Some Bugfixes
 *
 * Revision 1.39  2006/08/17 16:26:04  euler
 * Some changes, still buggy
 *
 * Revision 1.38  2006/08/16 15:09:12  euler
 * *** empty log message ***
 *
 * Revision 1.37  2006/08/15 09:57:34  euler
 * New constraints for creating output concepts
 *
 * Revision 1.36  2006/08/11 15:33:23  euler
 * Bugfixes, updates
 *
 * Revision 1.35  2006/08/10 14:38:02  euler
 * New mechanism for reversing steps
 *
 * Revision 1.34  2006/08/07 16:42:18  euler
 * *** empty log message ***
 *
 * Revision 1.33  2006/08/05 14:14:10  euler
 * New mechanism for checking if deleting Features from concepts
 * violates step validities.
 *
 * Revision 1.32  2006/08/04 14:49:29  euler
 * *** empty log message ***
 *
 * Revision 1.31  2006/08/03 17:43:20  euler
 * *** empty log message ***
 *
 * Revision 1.30  2006/08/03 10:47:19  euler
 * Reorganised methods for output creation.
 *
 * Revision 1.29  2006/08/01 14:47:13  euler
 * Cleaned Code
 *
 * Revision 1.28  2006/06/20 07:51:42  euler
 * Bugfixes
 *
 * Revision 1.27  2006/06/19 15:37:13  euler
 * Bugfixes
 *
 * Revision 1.26  2006/06/18 15:13:05  euler
 * Bugfixes
 *
 * Revision 1.25  2006/06/16 17:33:02  scholz
 * *** empty log message ***
 *
 * Revision 1.24  2006/06/16 16:26:32  euler
 * New operator "FeatureConstructionByRelation"
 *
 * Revision 1.23  2006/06/14 12:36:09  euler
 * New relation-creating operators exist.
 *
 * Revision 1.22  2006/06/13 13:30:42  euler
 * Updates
 *
 * Revision 1.21  2006/05/23 08:08:34  euler
 * *** empty log message ***
 *
 * Revision 1.20  2006/05/19 16:24:05  euler
 * New operator 'ReversePivotize'.
 *
 * Revision 1.19  2006/04/11 14:10:14  euler
 * Updated license text.
 *
 * Revision 1.18  2006/04/06 16:31:13  euler
 * Prepended license remark.
 *
 * Revision 1.17  2006/03/24 13:14:58  euler
 * Bugfix
 *
 * Revision 1.16  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.15  2006/03/22 16:49:27  euler
 * *** empty log message ***
 *
 * Revision 1.14  2006/03/22 13:08:52  euler
 * *** empty log message ***
 *
 * Revision 1.13  2006/03/22 09:51:23  euler
 * *** empty log message ***
 *
 * Revision 1.12  2006/03/21 12:03:12  euler
 * Creating relations automatically works
 *
 * Revision 1.11  2006/03/21 09:50:34  euler
 * *** empty log message ***
 *
 * Revision 1.10  2006/03/20 16:44:21  euler
 * Bugfixes
 *
 * Revision 1.9  2006/03/20 09:15:46  euler
 * Updated Pivotize to allow pivotisation without aggregation.
 *
 * Revision 1.8  2006/03/16 14:53:38  euler
 * *** empty log message ***
 *
 * Revision 1.7  2006/03/02 16:49:59  euler
 * Many bugfixes
 *
 * Revision 1.6  2006/01/27 17:27:17  euler
 * Bugfix
 *
 * Revision 1.5  2006/01/26 14:57:45  euler
 * bugfix
 *
 * Revision 1.4  2006/01/18 16:58:58  euler
 * Added some basic estimations of statistics.
 * Will need improvements.
 *
 * Revision 1.3  2006/01/06 16:25:04  euler
 * Updates and bugfixes in the delete-Mechanism for M4Data objects.
 *
 * Revision 1.2  2006/01/05 14:11:22  euler
 * Bugfixes
 *
 * Revision 1.1  2006/01/03 09:54:17  hakenjos
 * Initial version!
 *
 */
