/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.core;

import java.io.Serializable;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Vector;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.DbConnectionClosed;
import edu.udo.cs.miningmart.exception.M4Exception;

/**
 * This class represents the relational datatypes stored in the M4 model
 * and takes care of mapping between the ID and name representations.
 * One instance of this class (e.g. <code>static</code> in class
 * <code>Column</code> is sufficient, adding/changing the predefined
 * set of relational datatypes at runtime is not supported.
 * 
 * @author Martin Scholz
 * @version $Id: RelationalDatatypes.java,v 1.4 2006/09/27 14:59:59 euler Exp $
 */
public class RelationalDatatypes implements Serializable, edu.udo.cs.miningmart.m4.RelationalDatatypes {

	/** Name of the M4 table mapping IDs to relational datatype names */
	public static final String M4_TABLE_NAME = "col_datatype_t";
	
	/** Database attribute name storing the datatype IDs */
	public static final String ATTRIB_DATATYPE_ID   = "coldt_id"; // NUMBER NOT NULL,

	/** Database attribute name storing the datatype names */
	public static final String ATTRIB_DATATYPE_NAME = "coldt_name"; // VARCHAR2 (100) NOT NULL,

	// ***** Private fields *****
	
	/** Data structure for mapping the IDs to names: */
	private HashMap idToName = new HashMap();
	
	/** Data structure for mapping the names to IDs: */
	private HashMap nameToId = new HashMap();
	
	/* Data structure for a collection of the names: */
	private Collection allNames = new Vector();

	
	public RelationalDatatypes(DB db) throws M4Exception, DbConnectionClosed {
		String query = "SELECT " + ATTRIB_DATATYPE_ID + ", " + ATTRIB_DATATYPE_NAME
					 + " FROM "  + M4_TABLE_NAME;	
					 
		ResultSet rs = null;
		try {	 	
			rs = db.executeM4SqlRead(query);
			L: while (rs.next()) {
				Integer id = new Integer(rs.getInt(ATTRIB_DATATYPE_ID));
				if (rs.wasNull()) {
					continue L;
				}
				String name = rs.getString(ATTRIB_DATATYPE_NAME);
				idToName.put(id, name);
				nameToId.put(name, id);
				allNames.add(name);
			}
		}
		catch (SQLException e) {
			throw new M4Exception(
				"SQLException when trying to read the relational datatypes:\n" +
				e.getMessage());
		}
		finally {
			DB.closeResultSet(rs);
		}
	}
	
	/**
	 * @param name the name of a relational datatype (case senitive)
	 * @return the id representing this datatype or <code>0</code> if not found
	 */
	public long getIdForName(String name) {
		Integer id = (Integer) this.nameToId.get(name);
		if (id != null) {
			return id.longValue();
		}
		else {
			return 0;	
		}
	}

	/**
	 * @param id the ID of a relational datatype
	 * @return the relational datatype's name
	 */
	public String getNameForId(long id) {
		Integer idI = new Integer((int) id);
		return (String) this.idToName.get(idI);
	}
	
	public Collection getAllRelationalDatatypeNames() {
		return this.allNames;
	}
}
/*
 * Historie
 * --------
 * 
 * $Log: RelationalDatatypes.java,v $
 * Revision 1.4  2006/09/27 14:59:59  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:14  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:13  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:18  hakenjos
 * Initial version!
 *
 */
