/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4.core;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4CompilerError;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.utils.M4Info;
import edu.udo.cs.miningmart.m4.utils.M4InfoEntry;
import edu.udo.cs.miningmart.m4.utils.Print;
import edu.udo.cs.miningmart.m4.utils.XmlInfo;
import edu.udo.cs.miningmart.operator.ExecutableOperator;
import edu.udo.cs.miningmart.operator.JoinByKey;
import edu.udo.cs.miningmart.operator.RowSelectionByQuery;

/**
 * Objects of this class represent M4 assertions.
 * These assertions are tuples stored in the table <code>OP_ASSERT_T</code>.
 * 
 * @author Martin Scholz
 * @version $Id: Assertion.java,v 1.15 2006/09/27 14:59:59 euler Exp $
 */
public class Assertion extends M4Data implements XmlInfo, edu.udo.cs.miningmart.m4.Assertion {

	/** The name of the corresponding M4 table. */
	public static final String M4_TABLE_NAME = "op_assert_t";

	/** db level: name of the assertion's id attribute */
	public static final String ATTRIB_ASSERT_ID = "assert_id";

	/** db level: name of the operator id attribute */
	public static final String ATTRIB_OPERATOR_ID = "assert_opid";

	/** db level: name of the attribute specifying the assertion type */
	public static final String ATTRIB_ASSERT_TYPE = "assert_type";

	/** db level: name of the attribute for first argument */
	public static final String ATTRIB_ASSERT_OBJ1 = "assert_obj1";

	/** db level: name of the attribute for second argument */
	public static final String ATTRIB_ASSERT_OBJ2 = "assert_obj2";

	// Constants that distinguish the assertion types:
	public static final String ASSERTION_NOT_NULL    = "NOT_NULL";
	public static final String ASSERTION_UNIQUE      = "UNIQUE";
	
	// ----------------------
	// Assertion types for Estimation of Statistics:
	// ----------------------
	//
	// Assertions related to size of output concept:
	public static final String ESTIMATE_SIZE_BY_RELATIONS = "SZ_BY_REL";
	public static final String ESTIMATE_SIZE_BY_VALUE     = "SZ_BY_VAL";
	public static final String ESTIMATE_SIZE_REDUCE_MV    = "SZ_MIN_MV";
	public static final String ESTIMATE_SIZE_BY_DIVISION  = "SZ_DIV_BY";
	public static final String ESTIMATE_SIZE_BY_VALLIST   = "SZ_BY_VL";
	public static final String ESTIMATE_SIZE_ADD_INPUTS   = "SZ_ADD";
	public static final String ESTIMATE_SIZE_MULTIPLY_BY  = "SZ_MULT_NO";
	// Assertions related to the list of values of an output attribute:
	public static final String ESTIMATE_VALLIST_TAKE_FROM = "VL_FROM";
	public static final String ESTIMATE_VALLIST_UNCHANGED = "VL_UNCH";
	public static final String ESTIMATE_VALLIST_ADD_VALUE = "VL_ADD";
	public static final String ESTIMATE_VALLIST_FROM_PARAM = "VL_BY_PAR";
	public static final String ESTIMATE_VALLIST_BY_SYMBOL = "VL_BY_SYM";
	public static final String ESTIMATE_VALLIST_COMBINE   = "VL_COMB";
	public static final String ESTIMATE_VALLIST_FROM_LIST = "VL_BY_LIST";
	// Assertions related to the mininum and maximum values of an output attribute:
	public static final String ESTIMATE_MINMAX_TAKE_FROM  = "MM_FROM";
	public static final String ESTIMATE_MINMAX_UNCHANGED  = "MM_UNCH";
	public static final String ESTIMATE_MINIMUM_FROM_PARAM = "MIN_FROM";
	public static final String ESTIMATE_MAXIMUM_FROM_PARAM = "MAX_FROM";
	public static final String ESTIMATE_MINMAX_COMBINE     = "MM_COMB";
	// Assertions related to the value frequencies of an output attribute:
	public static final String ESTIMATE_VALFREQ_REPL_MV    = "VF_REPL_MV";
	public static final String ESTIMATE_VALFREQ_BY_SELECTIVITY = "VF_BY_SEL";
	public static final String ESTIMATE_VALFREQ_AGGREGATE  = "VF_BY_AGG";
	public static final String ESTIMATE_VALFREQ_ADD_INPUTS = "VF_ADD";
	public static final String ESTIMATE_VALFREQ_TAKE_FROM  = "VF_FROM";
	public static final String ESTIMATE_VALFREQ_MULT_BY    = "VF_MULT_NO";
	// Assertions related to the number of missing values:
	public static final String ESTIMATE_MV_BY_SELECTIVITY  = "MV_BY_SEL";
	public static final String ESTIMATE_MV_TAKE_FROM       = "MV_FROM";
	public static final String ESTIMATE_MV_ADD_INPUTS      = "MV_ADD";	
	// General assertions:
	public static final String ESTIMATE_NO_CHANGE          = "NO_CHANGE";	
	// a particular assertion to trigger the complex estimations
	// of "RowSelectionByQuery":
	public static final String ESTIMATE_ROW_SELECTION      = "ES_SELECT";
	
	// a symbol used in output label construction, by grouping and 
	// discretisation operators:
	private static final String OUTVAL_SYMBOL = "L";
	
	/** Cache for getM4Info() */
	private static M4Info m4Info = null;

	/** @see M4Table.getM4TableName() */
	public String getM4TableName() {
		return M4_TABLE_NAME;	
	}

	/** @see M4Table.getIdAttributeName() */
	public String getIdAttributeName() {
		return ATTRIB_ASSERT_ID;
	}

	/** @see M4Table.getM4Info() */
	public M4Info getM4Info() {
		if (m4Info == null) {
			M4InfoEntry[] m4i = {
				new M4InfoEntry(ATTRIB_ASSERT_ID,   "getId",            "setId",                long.class,     NOT_NULL),
				new M4InfoEntry(ATTRIB_OPERATOR_ID, "getTheOperator",   "primitiveSetOperator", edu.udo.cs.miningmart.m4.Operator.class, NOT_NULL),
				new M4InfoEntry(ATTRIB_ASSERT_TYPE, "getAssertionType", "setAssertionType",     String.class,   NOT_NULL),
				new M4InfoEntry(ATTRIB_ASSERT_OBJ1, "getObj1",          "setObj1",              String.class,   NOT_NULL),
				new M4InfoEntry(ATTRIB_ASSERT_OBJ2, "getObj2",          "setObj2",              String.class)
			};
			m4Info = new M4Info(m4i);
		}
		return m4Info;
	}
	
	/** Cache for getXmlInfo() */
	private static M4Info xmlInfo = null;

	/** @see M4Table.getM4Info() */
	public M4Info getXmlInfo() {
		if (xmlInfo == null) {
			M4InfoEntry[] m4i = {
				new M4InfoEntry("Operator", "getTheOperator",   "setTheOperator",   edu.udo.cs.miningmart.m4.Operator.class),
				new M4InfoEntry("Type",     "getAssertionType", "setAssertionType", String.class),
				new M4InfoEntry("Obj1",     "getObj1",          "setObj1",          String.class),
				new M4InfoEntry("Obj2",     "getObj2",          "setObj2",          String.class),
				new M4InfoEntry("Docu",     "getDocumentation", "setDocumentation", String.class)
			};
			xmlInfo = new M4Info(m4i);
		}
		return xmlInfo;
	}

	// The private fields of assertion objects:
	private Operator myOperator;
	private String myType;
	private String myObj1;
	private String myObj2;

	/**
	 * @see edu.udo.cs.miningmart.m4.core.M4Data#Constructor
	 */
	public Assertion(DB db) {
		super(db);	
	}

	/**
	 * @see M4Object#print()
	 */
	public void print() {
		edu.udo.cs.miningmart.m4.Operator op = this.getTheOperator();
		String opName = (op == null) ? null : op.getName();

		this.doPrint(Print.M4_OBJECT, "Assertion (Id = " + this.getId() + ";"
				+ " Name = " + this.getName() + "; TYPE = " + this.getAssertionType()
				+ " OBJ1 = " + this.getObj1() + "; OBJ2 = " + this.getObj2()
				+ " Operator = " + opName + ")");

	}
	
	/**
	 * @see edu.udo.cs.miningmart.m4.core.M4Data#getObjectsInNamespace(Class)
	 */
	protected Collection getObjectsInNamespace(Class typeOfObjects) throws M4Exception {
		return null;
	}
	
	/**
	 * Gets the myObj1.
	 * @return Returns a String
	 */
	public String getObj1() {
		return myObj1;
	}

	/**
	 * Gets the myObj2.
	 * @return Returns a String
	 */
	public String getObj2() {
		return myObj2;
	}

	/**
	 * Gets the Operator
	 * @return Returns an <code>Operator</code>
	 */
	public edu.udo.cs.miningmart.m4.Operator getTheOperator() {
		return myOperator;
	}

	/**
	 * Gets the myType.
	 * @return Returns a String
	 */
	public String getAssertionType() {
		return myType;
	}

	/**
	 * Sets the myObj1.
	 * @param myObj1 The myObj1 to set
	 */
	public void setObj1(String myObj1) {
		this.setDirty();
		this.myObj1 = myObj1;
	}

	/**
	 * Sets the myObj2.
	 * @param myObj2 The myObj2 to set
	 */
	public void setObj2(String myObj2) {
		this.setDirty();
		this.myObj2 = myObj2;
	}

	/**
	 * Sets the Operator.
	 * @param operator The <code>Operator</code> to set
	 */
	public void setTheOperator(edu.udo.cs.miningmart.m4.Operator operator) throws M4Exception {
		Operator.op2assert.updateReferenceTo(this, operator);
	}

	/** Primitive setter method. Do not use it! */
	public void primitiveSetOperator(edu.udo.cs.miningmart.m4.Operator operator) {
		this.setDirty();
		this.myOperator = (Operator) operator;	
	}

	/**
	 * Sets the myType.
	 * @param myType The myType to set
	 */
	public void setAssertionType(String myType) {
		this.setDirty();
		this.myType = myType;
	}

	/** @see M4Data#removeAllM4References */
	protected void removeAllM4References() throws M4Exception {
		this.setTheOperator(null);
		this.removeDocObject();
	}
	
	
	public void writeAssertion(ExecutableOperator op)
	throws M4CompilerError {
		String assertionType = this.getAssertionType();
		for (int loop=0; loop < op.getNumberOfLoops(); loop++) {
			if (assertionType.equals(ASSERTION_NOT_NULL))
				assertNotNull(op, loop);
			else if (assertionType.equals(ASSERTION_UNIQUE))
				assertUniqueness(op, loop);
		}
	}
	
	public void applyEstimationOneInput(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation)
	throws M4Exception {
		
		// maybe this assertion triggers the complex RowSelectionByQuery estimations:
		if (this.doRowSelectionEstimation(whichStep, inputEstimation, outputEstimation)) {
			return;
		}
		
		// maybe this assertion concerns the output size:
		this.doSizeEstimationSingleInput(whichStep, inputEstimation, outputEstimation);
		
		// or maybe something else:
		this.doOtherEstimations(whichStep, inputEstimation, outputEstimation);
	}
	
	public void applyEstimationSeveralInputs(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics[] inputEstimations,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation)
	throws M4Exception {
		
		// maybe this assertion concerns the output size:
		this.doSizeEstimationSeveralInputs(whichStep, inputEstimations, outputEstimation);
		
		// or something else:
		for (int i = 0; i < inputEstimations.length; i++) {
			this.doOtherEstimations(whichStep, inputEstimations[i], outputEstimation);
		}
	}
	
	public boolean isValueListAssertion() {
		return this.getAssertionType().startsWith("VL");
	}
	
	public boolean isValueFrequencyAssertion() {
		return this.getAssertionType().startsWith("VF");
	}
	
	private void doOtherEstimations(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		// maybe this assertion concerns the value lists of an output attribute:
		this.doValueListEstimation(whichStep, inputEstimation, outputEstimation);
		
		// or maybe the minimum/maximum values of an output attribute:
		this.doBoundsEstimation(whichStep, inputEstimation, outputEstimation);
		
		// or maybe the number of missing values of an output attribute:
		this.doMissingValuesEstimation(whichStep, inputEstimation, outputEstimation);
		
		// and perhaps the value frequencies of an output attribute:
		this.doValueFrequenciesEstimation(whichStep, inputEstimation, outputEstimation);
	}
	
	private void doValueFrequenciesEstimation(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		if (this.getAssertionType().equals(ESTIMATE_NO_CHANGE)) {
			Concept outputConcept = (Concept) whichStep.getOutputConcept();
			Collection outBAs = outputConcept.getAllBaseAttributes();
			Iterator baIt = outBAs.iterator();
			while (baIt.hasNext()) {
				BaseAttribute outBA = (BaseAttribute) baIt.next();
				if (whichStep.isVisible(outBA)) {
					Vector inValList = inputEstimation.getValueList(outBA.getName());
					if (inValList != null) {
						Iterator valIt = inValList.iterator();
						while (valIt.hasNext()) {
							String value = (String) valIt.next();
							int freq = inputEstimation.getNumberOfOccurrences(outBA.getName(), value);
							outputEstimation.setNumberOfOccurrences(outBA.getName(), value, freq);
						}
					}
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALFREQ_REPL_MV)) {
			String[] outputBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			String value = this.getValueParameter(this.getObj2(), (Step) whichStep);
			// we have to find the inputs that correspond to the ouputBas:
			String[] inputBaNames = this.getInputAttribsForOutputAttribs(outputBaNames, this.getObj1(), (Step) whichStep);
			if (inputBaNames != null) {
				if (inputBaNames.length != outputBaNames.length) {
					throw new M4Exception("Assertion '" + this.getAssertionType() +
							"' in step '" + whichStep.getName() + "': unmatched numbers of input and output attributes!");
				}
				for (int i = 0; i < inputBaNames.length; i++) {
					if (inputBaNames[i] != null) {
						int freq = inputEstimation.getNumberOfOccurrences(inputBaNames[i], value);
						
						if (freq == EstimatedStatistics.VALUE_INT_UNKNOWN) {
							outputEstimation.setNumberOfOccurrences(outputBaNames[i], value, EstimatedStatistics.VALUE_INT_UNKNOWN);
						}
						else {
							int noOfMissVals = inputEstimation.getNumberOfMissingValues(inputBaNames[i]);
							if (noOfMissVals == EstimatedStatistics.VALUE_INT_UNKNOWN) {
								if (freq == 0) {
									outputEstimation.addValueInformation(outputBaNames[i], value, EstimatedStatistics.VALUE_INT_UNKNOWN);
								}
								else {									
									outputEstimation.setNumberOfOccurrences(outputBaNames[i], value, EstimatedStatistics.VALUE_INT_UNKNOWN);
								}
							}
							else {
								// perhaps the value whose frequency is estimated is new!
								if (freq == 0) {
									if (noOfMissVals > 0) {
										outputEstimation.addValueInformation(outputBaNames[i], value, noOfMissVals);
									}
								}
								else {
									outputEstimation.setNumberOfOccurrences(outputBaNames[i], value, freq + noOfMissVals);
								}
							}
						}
					}
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALFREQ_BY_SELECTIVITY)) {
			// estimate the frequency of each value according to the selectivity of the operator:
			if (inputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN &&
				outputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN) {

				// apply the selectivity to all attributes:
				Concept outputConcept = (Concept) whichStep.getOutputConcept();
				Concept inputConcept = (Concept) inputEstimation.getConcept();
				Iterator it = outputConcept.getAllBaseAttributes().iterator();
				while (it.hasNext()) {
					BaseAttribute outputBa = (BaseAttribute) it.next();
					String outBaName = outputBa.getName();
					BaseAttribute inputBa = (BaseAttribute) inputConcept.getBaseAttribute(outBaName);
					if (inputBa == null) {
						inputBa = this.getMappedInput(outputBa, inputConcept, (Step) whichStep);
					}
					if (inputBa != null) {
						String inBaName = inputBa.getName();
						this.applyVfSelectivity((EstimatedStatistics) inputEstimation, (EstimatedStatistics) outputEstimation, inBaName, outBaName);						
					}
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALFREQ_AGGREGATE)) {
			String[] namesOfAttribs = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			if (namesOfAttribs == null) {
				return; // no changes to the frequencies
			}
			// the group by combinations give the (upper bounds of the) frequencies:
			for (int i = 0; i < namesOfAttribs.length; i++) {
				int freq = this.getProductOfOtherFrequencies(namesOfAttribs, i, (EstimatedStatistics) inputEstimation);
				Vector vl = inputEstimation.getValueList(namesOfAttribs[i]);
				if (vl != null) {
					Iterator it = vl.iterator();
					while (it.hasNext()) {
						String value = (String) it.next();
						outputEstimation.setNumberOfOccurrences(namesOfAttribs[i], value, freq);
					}
				}
			}			
		}
		if (this.getAssertionType().equals(ESTIMATE_VALFREQ_ADD_INPUTS)) {
			// EITHER obj2 is an attrib parameter, then it is an output
			// attrib and obj1 is a list of input attribs whose estimations
			// are to be combined
			
			// OR obj1 and obj2 are concept parameters, this means to combine
			// all corresponding attributes in the input concepts (given by obj1)
			
			String outBaName = this.getNameOfAttribParameter(this.getObj2(), (Step) whichStep);
			StringTokenizer st = new StringTokenizer(this.getObj1(), ", ");
			String[] inNames = new String[st.countTokens()];
			int i = 0;
			while (st.hasMoreTokens()) {
				inNames[i] = st.nextToken();
				i++;
			}
			
			// now decide between the two cases above:
			if (outBaName != null) {
				// "EITHER-case":
				// add the frequencies of those values that occur in all input attribs:
				Map valuesToFrequencies = new HashMap();
				for (int j = 0; j < inNames.length; j++) {
					String[] moreAttribNames = this.getNamesOfAttribParameterForAllLoops(inNames[j], (Step) whichStep);
					if (moreAttribNames != null) {
						for (int k = 0; k < moreAttribNames.length; k++) {
							Vector vl = inputEstimation.getValueList(moreAttribNames[k]);
							if (vl != null) {
								Iterator it = vl.iterator();
								while (it.hasNext()) {
									String value = (String) it.next();
									int freq = inputEstimation.getNumberOfOccurrences(moreAttribNames[k], value);
									this.accumulateFrequencies(valuesToFrequencies, value, freq);
								}
							}
						}
					}
				}
				Collection allValues = valuesToFrequencies.keySet();
				Iterator it = allValues.iterator();
				while (it.hasNext()) {
					String oneValue = (String) it.next();
					Integer frequency = (Integer) valuesToFrequencies.get(oneValue);
					int fr = frequency.intValue();
					if (fr != 0) {
						outputEstimation.setNumberOfOccurrences(outBaName, oneValue, fr);
					}
				}
			}
			else {
				// "OR-case":
				Concept[] inputConcepts = this.getConceptsForParamNames(inNames, (Step) whichStep);
				
				// compute the estimated statistics for each input concept only once:
				EstimatedStatistics[] inputESs = new EstimatedStatistics[inputConcepts.length];
				for (int j = 0; j < inputConcepts.length; j++) {
					inputESs[j] = (EstimatedStatistics) inputConcepts[j].getEstimatedStatistics(whichStep);
				}
				// use the first concept as reference:
				Concept refConc = inputConcepts[0];
				Collection refBAs = refConc.getAllBaseAttributes();
				Iterator baIt = refBAs.iterator();
				while (baIt.hasNext()) {
					BaseAttribute refBA = (BaseAttribute) baIt.next();
					if (whichStep.isVisible(refBA)) {
						Map valuesToFrequencies = new HashMap();
						for (int j = 0; j < inputESs.length; j++) {
							BaseAttribute corrBA = (BaseAttribute) inputConcepts[j].getBaseAttribute(refBA.getName());
							if (corrBA != null) {
								Vector valueList = inputESs[j].getValueList(corrBA.getName());
								if (valueList != null) {
									Iterator valIt = valueList.iterator();
									while (valIt.hasNext()) {
										String value = (String) valIt.next();
										int freq = inputESs[j].getNumberOfOccurrences(corrBA.getName(), value);
										this.accumulateFrequencies(valuesToFrequencies, value, freq);
									}
								}
							}
						}
						Collection allValues = valuesToFrequencies.keySet();
						Iterator it = allValues.iterator();
						while (it.hasNext()) {
							String oneValue = (String) it.next();
							Integer frequency = (Integer) valuesToFrequencies.get(oneValue);
							int fr = frequency.intValue();
							if (fr != 0) {
								outputEstimation.setNumberOfOccurrences(refBA.getName(), oneValue, fr);
							}
						}
					}
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALFREQ_TAKE_FROM)) {
			// this assertion means that the value frequencies for an output
			// attribute, given in obj2, can be taken from an input
			// attribute, given in obj1

			// if obj2 is null, the parameter given in obj1 is available
			// in input and output, OR it is only in input but a CREATE_BY
			// constraint exists that maps it to the output
			
			// the whole thing only works if the step does not use aggregation!
			if ( ! whichStep.usesAggregation()) {
				String[] outBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj2(), (Step) whichStep);
				String[] inBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
				if (outBaNames == null) {
					outBaNames = this.getOutputAttribsForInputAttribs(
							inBaNames, 
							(Step) whichStep);
				}
				// a special case:
				boolean done = this.handleReversingSteps(inBaNames, outBaNames, (EstimatedStatistics) outputEstimation, (Step) whichStep);
				if (done) {
					return;
				}
				
				// transfer estimations; there may be more output than input attribs:
				if (outBaNames.length > inBaNames.length) {
					if (inBaNames.length != 1) {
						throw new M4Exception("Assertion '" + this.getAssertionType() +
								"' in Step '" + whichStep.getName() + "': don't know how to map a few input attribs to more output attribs!");
					}
				}
				for (int outputIndex = 0; outputIndex < outBaNames.length; outputIndex++) {
					// check that our input BA actually corresponds to the current 
					// input estimations:
					int inputIndex = (inBaNames.length > 1 ? outputIndex : 0);
					if (this.attribBelongsToConcept((Step) whichStep, inBaNames[inputIndex], (Concept) inputEstimation.getConcept())) {
						Vector vl = inputEstimation.getValueList(inBaNames[inputIndex]);
						if (vl != null) {
							Iterator it = vl.iterator();
							while (it.hasNext()) {
								String value = (String) it.next();
								int inputFrequency = inputEstimation.getNumberOfOccurrences(inBaNames[inputIndex], value);
								outputEstimation.setNumberOfOccurrences(outBaNames[outputIndex], value, inputFrequency);
							}
						}
					}
				}
			}			
		}
		if (this.getAssertionType().equals(ESTIMATE_VALFREQ_MULT_BY)) {
			String[] inputAttribs = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			String[] outputAttribs = this.getNamesOfAttribParameterForAllLoops(this.getObj2(), (Step) whichStep);
			int factor = 1;
			if (outputAttribs == null) {
				String theFactor = this.getValueParameter(this.getObj1(), (Step) whichStep);
				try {
					factor = Integer.parseInt(theFactor);
				}
				catch (NumberFormatException nfe) {
					throw new M4Exception("Assertion '" + this.getAssertionType() +
							"' in Step '" + whichStep.getName() + "': expected an attrib or numeric value parameter or a numeric constant in Obj1!");
				}
			}
			else {
				factor = outputAttribs.length;
			}
			for (int i = 0; i < inputAttribs.length; i++) {
				Vector vl = inputEstimation.getValueList(inputAttribs[i]);
				if (vl != null) {
					Iterator it = vl.iterator();
					while (it.hasNext()) {
						String value = (String) it.next();
						int freq = inputEstimation.getNumberOfOccurrences(inputAttribs[i], value);
						freq = (freq == EstimatedStatistics.VALUE_INT_UNKNOWN ? freq : factor * freq);
						// set the frequency to the output attrib with the same name as the input attrib:
						outputEstimation.setNumberOfOccurrences(inputAttribs[i], value, freq);
					}
				}
			}
		}
	}
	
	private void applyVfSelectivity(
			EstimatedStatistics inputEstimation, 
			EstimatedStatistics outputEstimation,
			String inBaName,
			String outBaName) {
		// apply the selectivity to all values:
		double selectivity = (double) outputEstimation.getNumberOfRows() / (double) inputEstimation.getNumberOfRows();
		Vector vl = inputEstimation.getValueList(inBaName);
		if (vl != null && ( ! vl.isEmpty())) {
			Iterator valIt = vl.iterator();
			while (valIt.hasNext()) {
				String value = (String) valIt.next();
				int inputValFreq = inputEstimation.getNumberOfOccurrences(inBaName, value);
				int outputValFreq = inputValFreq;
				if (inputValFreq != EstimatedStatistics.VALUE_INT_UNKNOWN) {
					outputValFreq = (int) (selectivity * inputValFreq);
				}
				// repair rounding to 0:
				if (outputValFreq == 0 && inputValFreq >= 1) {
					outputValFreq = 1;
				}
				outputEstimation.setNumberOfOccurrences(outBaName, value, outputValFreq);
			}
		}
	}
	
	private void accumulateFrequencies(Map valuesToFrequencies, String value, int frequency) {
		if (valuesToFrequencies == null || value == null) {
			return;
		}
		Integer frequencySoFar = (Integer) valuesToFrequencies.get(value);
		if (frequencySoFar == null) {
			frequencySoFar = new Integer(frequency);
			valuesToFrequencies.put(value, frequencySoFar);
		}
		else {
			int oldFreq = frequencySoFar.intValue();
			if (oldFreq != EstimatedStatistics.VALUE_INT_UNKNOWN) {
				int newFreq = oldFreq + frequency;
				valuesToFrequencies.put(value, new Integer(newFreq));
			}
		}
	}
	
	// returns the product of numbers of distinct values of the attribs
	// given in the array, leaving out the attrib at the given position
	private int getProductOfOtherFrequencies(
			String[] namesOfAttribs, 
			int positionToLeaveOut, 
			EstimatedStatistics inputEstimation) {
		int product = 1;
		for (int i = 0; i < namesOfAttribs.length; i++) {
			if (i != positionToLeaveOut) {
				Vector vl = inputEstimation.getValueList(namesOfAttribs[i]);
				if (vl != null) {
					product *= vl.size();
				}
			}
		}
		return product;
	}
	
	private void doBoundsEstimation(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		if (this.getAssertionType().equals(ESTIMATE_NO_CHANGE)) {
			Concept outputConcept = (Concept) whichStep.getOutputConcept();
			Collection outBAs = outputConcept.getAllBaseAttributes();
			Iterator baIt = outBAs.iterator();
			while (baIt.hasNext()) {
				BaseAttribute outBA = (BaseAttribute) baIt.next();
				if (whichStep.isVisible(outBA)) {
					outputEstimation.setBiggestValue(outBA.getName(), inputEstimation.getBiggestValue(outBA.getName()));
					outputEstimation.setLowestValue(outBA.getName(), inputEstimation.getLowestValue(outBA.getName()));
				}
			}
		}
		boolean max = false;
		if ((max = this.getAssertionType().equals(ESTIMATE_MAXIMUM_FROM_PARAM)) ||
			this.getAssertionType().equals(ESTIMATE_MINIMUM_FROM_PARAM)) {
			String[] attribNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			double value = this.getDoubleFromValueParameter(this.getObj2(), (Step) whichStep);
			if (attribNames != null) {
				for (int i = 0; i < attribNames.length; i++) {
					if (max)
						outputEstimation.setBiggestValue(attribNames[i], value);
					else
						outputEstimation.setLowestValue(attribNames[i], value);
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_MINMAX_UNCHANGED)) {
			Concept outputConcept = (Concept) whichStep.getOutputConcept();
			Concept inputConcept = (Concept) inputEstimation.getConcept();
			Iterator it = outputConcept.getAllBaseAttributes().iterator();
			while (it.hasNext()) {
				BaseAttribute outputBa = (BaseAttribute) it.next();
				String outBaName = outputBa.getName();
				BaseAttribute inputBa = (BaseAttribute) inputConcept.getBaseAttribute(outBaName);
				if (inputBa == null) {
					inputBa = this.getMappedInput(outputBa, inputConcept, (Step) whichStep);
				}
				if (inputBa != null) {
					String inBaName = inputBa.getName();
					outputEstimation.setBiggestValue(outBaName, inputEstimation.getBiggestValue(inBaName));
					outputEstimation.setLowestValue(outBaName, inputEstimation.getLowestValue(inBaName));
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_MINMAX_TAKE_FROM)) {
			// this assertion means that the min/max bound for an output
			// attribute, given in obj2, can be taken from an input
			// attribute, given in obj1

			// if obj2 is null, the parameter given in obj1 is available
			// in input and output, OR it is only in input but a CREATE_BY
			// constraint exists that maps it to the output
			
			// the whole thing only works if the step does not use aggregation!
			if ( ! whichStep.usesAggregation()) {
				String[] outBaNames = this.getNamesOfAttribParameter(this.getObj2(), (Step) whichStep);
				String[] inBaNames = this.getNamesOfAttribParameter(this.getObj1(), (Step) whichStep);
				if (outBaNames == null) {
					outBaNames = this.getOutputAttribsForInputAttribs(
							inBaNames, 
							(Step) whichStep);
				}
				boolean done = this.handleReversingSteps(inBaNames, outBaNames, (EstimatedStatistics) outputEstimation, (Step) whichStep);
				if (done) {
					return;
				}
				// transfer estimations; there may be more output than input attribs:
				if (outBaNames.length > inBaNames.length) {
					if (inBaNames.length != 1) {
						throw new M4Exception("Assertion '" + this.getAssertionType() +
								"' in Step '" + whichStep.getName() + "': don't know how to map a few input attribs to more output attribs!");
					}
				}
				for (int outputIndex = 0; outputIndex < outBaNames.length; outputIndex++) {
					int inputIndex = (inBaNames.length > 1 ? outputIndex : 0);
					// the inBa must be in the concept whose estimations are currently used: 
					if (this.attribBelongsToConcept((Step) whichStep, inBaNames[inputIndex], (Concept) inputEstimation.getConcept())) {
						outputEstimation.setBiggestValue(outBaNames[outputIndex], inputEstimation.getBiggestValue(inBaNames[inputIndex]));
						outputEstimation.setLowestValue(outBaNames[outputIndex], inputEstimation.getLowestValue(inBaNames[inputIndex]));
					}
				}
			}			
		}
		if (this.getAssertionType().equals(ESTIMATE_MINMAX_COMBINE)) {
			// EITHER obj2 is an attrib parameter, then it is an output
			// attrib and obj1 is a list of input attribs whose estimations
			// are to be combined
			
			// OR obj1 and obj2 are concept parameters, this means to combine
			// all corresponding attributes in the input concepts (given by obj1)
			
			String outBaName = this.getNameOfAttribParameter(this.getObj2(), (Step) whichStep);
			StringTokenizer st = new StringTokenizer(this.getObj1(), ", ");
			String[] inNames = new String[st.countTokens()];
			int i = 0;
			while (st.hasMoreTokens()) {
				inNames[i] = st.nextToken();
				i++;
			}
			
			// now decide between the two cases above:
			if (outBaName != null) {
				// "EITHER-case":
				double maxSoFar = Double.MIN_VALUE;
				double minSoFar = Double.MAX_VALUE;
				boolean maxUnknown = false, minUnknown = false;
				for (int j = 0; j < inNames.length; j++) {
					String[] moreInputNames = this.getNamesOfAttribParameterForAllLoops(inNames[j], (Step) whichStep);
					if (moreInputNames != null) {
						for (int k = 0; k < moreInputNames.length; k++) {
							double maxi = inputEstimation.getBiggestValue(moreInputNames[k]);
							if (this.doublesAreEqual(maxi, EstimatedStatistics.VALUE_DOUBLE_UNKNOWN)) {
								maxUnknown = true;
							}
							if (maxi > maxSoFar) maxSoFar = maxi;
							double mini = inputEstimation.getLowestValue(moreInputNames[k]);
							if (this.doublesAreEqual(mini, EstimatedStatistics.VALUE_DOUBLE_UNKNOWN)) {
								minUnknown = true;
							}
							if (mini > minSoFar) minSoFar = mini;
						}
					}
				}
				maxSoFar = (maxUnknown ? EstimatedStatistics.VALUE_DOUBLE_UNKNOWN : maxSoFar);
				minSoFar = (minUnknown ? EstimatedStatistics.VALUE_DOUBLE_UNKNOWN : minSoFar);
				outputEstimation.setLowestValue(outBaName, minSoFar);
				outputEstimation.setBiggestValue(outBaName, maxSoFar);
			}
			else {
				// "OR-case":
				Concept[] inputConcepts = this.getConceptsForParamNames(inNames, (Step) whichStep);
				
				// compute the estimated statistics for each input concept only once:
				EstimatedStatistics[] inputESs = new EstimatedStatistics[inputConcepts.length];
				for (int j = 0; j < inputConcepts.length; j++) {
					inputESs[j] = (EstimatedStatistics) inputConcepts[j].getEstimatedStatistics(whichStep);
				}
				// use the first concept as reference:
				Concept refConc = inputConcepts[0];
				Collection refBAs = refConc.getAllBaseAttributes();
				Iterator baIt = refBAs.iterator();
				while (baIt.hasNext()) {
					BaseAttribute refBA = (BaseAttribute) baIt.next();
					if (whichStep.isVisible(refBA)) {
						double[] inputMinimums = new double[inputConcepts.length];
						double[] inputMaximums = new double[inputConcepts.length];
						inputMinimums[0] = inputESs[0].getLowestValue(refBA.getName());
						inputMaximums[0] = inputESs[0].getBiggestValue(refBA.getName());
						for (int j = 1; j < inputConcepts.length; j++) {
							BaseAttribute corrBa = (BaseAttribute) inputConcepts[j].getBaseAttribute(refBA.getName());
							if (corrBa != null) {
								inputMinimums[j] = inputESs[j].getLowestValue(corrBa.getName());
								inputMaximums[j] = inputESs[j].getBiggestValue(corrBa.getName());
							}
						}
						outputEstimation.setLowestValue(refBA.getName(), this.combineBounds(inputMinimums, true));
						outputEstimation.setBiggestValue(refBA.getName(), this.combineBounds(inputMaximums, false));
					}
				}
			}
		}
	}			

	// For "RowSelectionByQuery" the estimations are not so easy to
	// specify by assertions. Therefore there is a particular assertion
	// that triggers the execution of this method. This method does the 
	// estimations and is tailored towards "RowSelectionByQuery".
	private boolean doRowSelectionEstimation(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		if (this.getAssertionType().equals(ESTIMATE_ROW_SELECTION)) {
			// debug check:
			if ( ! whichStep.getTheOperator().getName().equals("RowSelectionByQuery")) {
				throw new M4Exception("Assertion '" + this.getAssertionType() +
						"' in Step '" + whichStep.getName() + "': expected the operator to be 'RowSelectionByQuery'!");
			}			

			// get the parameters:
			String[] leftConditions = this.getNamesOfAttribParameterForAllLoops(RowSelectionByQuery.PARAMETER_LEFT_CONDITION, (Step) whichStep);
			String[] rightConditions = this.getValueParametersForAllLoops(RowSelectionByQuery.PARAMETER_RIGHT_CONDITION, (Step) whichStep);
			String[] conditionOps = this.getValueParametersForAllLoops(RowSelectionByQuery.PARAMETER_CONDITION_OP, (Step) whichStep);
			
    		// attempt to infer some values:
			int startLoop = (whichStep.getLoopCount() > 0 ? 1 : 0);
			int endLoop = whichStep.getLoopCount();
    		for (int i = startLoop; i <= endLoop; i++) {
    			// no missing values are created:
    			if (inputEstimation.getNumberOfMissingValues(leftConditions[i]) == 0) {
    				outputEstimation.setNumberOfMissingValues(leftConditions[i], 0);
    			}
    			Vector valueList = inputEstimation.getValueList(leftConditions[i]);
    			if (valueList != null) {
    				// if the comparison is by equality...
    				if (conditionOps[i].equals("=")) {
    					// ... then only one value remains
    					rightConditions[i] = this.dequote(rightConditions[i]);
    					if (valueList.contains(rightConditions[i])) {
    						int noOfOcc = inputEstimation.getNumberOfOccurrences(leftConditions[i], rightConditions[i]);
    						// the single value determines the number of rows in the output:
    						if (whichStep.getLoopCount() == 0) {
    							outputEstimation.setNumberOfRows(noOfOcc);
    						}
    						// if the value is numeric, it is also the max and min:
							double numericValue = this.isNumeric(rightConditions[i]);
							if ( ! this.doublesAreEqual(numericValue, Double.NaN)) {
								outputEstimation.setBiggestValue(leftConditions[i], numericValue);
								outputEstimation.setLowestValue(leftConditions[i], numericValue); 
							}    						
    						// the frequency of such a value also upperbounds its 
    						// frequency in the output:
    						int freq = inputEstimation.getNumberOfOccurrences(leftConditions[i], rightConditions[i]);
    						// only the single value remains:
    						outputEstimation.setValueList(leftConditions[i], new Vector());
    						outputEstimation.addValueInformation(leftConditions[i], rightConditions[i], freq);
    					}
    				}
    				else {  
    					// see which values fulfill the condition (ie remain in output):
    					Vector remainingValues = this.findRemainingValues(valueList, conditionOps[i], rightConditions[i]);
    					if (remainingValues != null) { // if null they could not be found
    						// remove the previous values, so that only the remaining 
    						// values are added below:
    						outputEstimation.setValueList(leftConditions[i], new Vector());
    						// several values fulfill the condition
    						Iterator allValIt = remainingValues.iterator();
    						Integer numberOfRemainingRows = new Integer(0);
    						double biggestRemainingValue = Double.MIN_VALUE;
    						double smallestRemainingValue = Double.MAX_VALUE;
    						while (allValIt.hasNext()) {
    							String aValue = (String) allValIt.next();
    							double numericValue = this.isNumeric(aValue);
    							if ( ! this.doublesAreEqual(numericValue, Double.NaN)) {
    								if (numericValue > biggestRemainingValue) {
    									biggestRemainingValue = numericValue;
    								}
    								if (numericValue < smallestRemainingValue) {
    									smallestRemainingValue = numericValue;
    								}
    							}
    							int noOfOcc = EstimatedStatistics.VALUE_INT_UNKNOWN;
    							// this value also occurs in the output. the number of occurrences is unchanged
    							noOfOcc = inputEstimation.getNumberOfOccurrences(leftConditions[i], aValue);
    							if (noOfOcc != EstimatedStatistics.VALUE_INT_UNKNOWN && 
    								noOfOcc > 0) {
    								if (numberOfRemainingRows != null) {
    									// the sum of occurrences is the upper bound for the output
    									numberOfRemainingRows = new Integer(numberOfRemainingRows.intValue() + noOfOcc);
    								}		        				
    							}
    							else {
    								// number of remaining rows cannot be known
    								numberOfRemainingRows = null;
    							}
    							if (noOfOcc > 0) {
    								outputEstimation.addValueInformation(leftConditions[i], aValue, noOfOcc);
    							}
    						}
    						// if the number is known, set it:
    						if (numberOfRemainingRows != null) {
    							outputEstimation.setNumberOfRows(numberOfRemainingRows.intValue());
    						}
    						if ( ! this.doublesAreEqual(biggestRemainingValue, Double.MIN_VALUE)) {
    							outputEstimation.setBiggestValue(leftConditions[i], biggestRemainingValue);
    						}
    						if ( ! this.doublesAreEqual(smallestRemainingValue, Double.MAX_VALUE)) {
    							outputEstimation.setLowestValue(leftConditions[i], smallestRemainingValue);
    						}
    					}
    				}
    			}
    		} // end for-loop through loops of step 			
			
    		// if any selection took place, no inferences are possible
    		// for the attributes that did not take part in the selection.
    		// But optimistic estimation says the old values are still there
    		// with reduced frequencies...
    		long inputSize = inputEstimation.getNumberOfRows();
    		long outputSize = outputEstimation.getNumberOfRows();
    		if (inputSize != EstimatedStatistics.VALUE_INT_UNKNOWN &&
    			outputSize != EstimatedStatistics.VALUE_INT_UNKNOWN &&
				outputSize < inputSize) {
    			Concept outputConcept = (Concept) outputEstimation.getConcept();
    			Iterator baIt = outputConcept.getAllBaseAttributes().iterator();
    			while (baIt.hasNext()) {
					BaseAttribute outBA = (BaseAttribute) baIt.next();
					if (whichStep.isVisible(outBA) &&
						( ! this.arrayContains(leftConditions, outBA.getName()))) {
						// decide here between conservative and optimistic estimation!
						if (true) {
							// optimistic!
							this.applyVfSelectivity((EstimatedStatistics) inputEstimation, (EstimatedStatistics) outputEstimation, outBA.getName(), outBA.getName());
							double selectivity = (double) outputSize / (double) inputSize;
							int inputMv = inputEstimation.getNumberOfMissingValues(outBA.getName());
							int outputMv = (int) (selectivity * inputMv);
							outputEstimation.setNumberOfMissingValues(outBA.getName(), outputMv);
						}
						else {
							// conservative!
							outputEstimation.setBiggestValue(outBA.getName(), EstimatedStatistics.VALUE_DOUBLE_UNKNOWN);
							outputEstimation.setLowestValue(outBA.getName(), EstimatedStatistics.VALUE_DOUBLE_UNKNOWN);
							if (inputEstimation.getNumberOfMissingValues(outBA.getName()) == 0) {
								outputEstimation.setNumberOfMissingValues(outBA.getName(), 0);
							}
							else {
								outputEstimation.setNumberOfMissingValues(outBA.getName(), EstimatedStatistics.VALUE_INT_UNKNOWN);
							}
							outputEstimation.setValueList(outBA.getName(), new Vector());
						}
					}
				}
    		}
    		
			return true;
		}
		return false;
	}
	
	private boolean arrayContains(String[] anArray, String anElement) {
		for (int i = 0; i < anArray.length; i++) {
			if (anArray[i].equals(anElement)) {
				return true;
			}
		}
		return false;
	}
	
	private String dequote(String quoted) {
		quoted = quoted.trim();
		while (quoted.startsWith("'")) {
			quoted = quoted.substring(1);
			if (quoted.endsWith("'")) {
				quoted = quoted.substring(0, quoted.length()-1);
			}
		}
		return quoted.trim();
	}

	private String debracket(String bracketed) {
		bracketed = bracketed.trim();
		while (bracketed.startsWith("(")) {
			bracketed = bracketed.substring(1);
			if (bracketed.endsWith(")")) {
				bracketed = bracketed.substring(0, bracketed.length()-1);
			}
		}
		return bracketed.trim();
	}
	
	private double isNumeric(String value) {
		double ret = Double.NaN;
		try {
			ret = Double.parseDouble(value); 
		}
		catch (NumberFormatException nfe) { }
		return ret;
	}
	
	// returns a vector with values from the given listOfValues, namely those
	// values that fulfill the condition given by the operator and the right hand side
    private Vector findRemainingValues(Vector listOfValues, String operator, String right) {
    	// maybe the right hand side is a list of possible values:
    	if (operator.equalsIgnoreCase("in")) {
    		// yes - so return all values that occur in the given list:
    		right = this.debracket(right).trim();
    		StringTokenizer st = new StringTokenizer(right, ",");
    		Vector theValues = new Vector();
    		while (st.hasMoreTokens()) {
    			String oneValue = st.nextToken();
    			oneValue = this.dequote(oneValue).trim();
    			if (listOfValues.contains(oneValue)) {
    				theValues.add(oneValue);
    			}
    		}
    		if (theValues.isEmpty()) {
    			return null;
    		}
    		else {
    			return theValues;
    		}
    	}
    	// check other types of condition:
    	boolean smaller = operator.startsWith("<");
    	boolean bigger = operator.startsWith(">");
    	boolean orEqual = operator.endsWith("=");
    	if (smaller || bigger) {
    		String compareToValue = this.dequote(right).trim();
    		double compareToD = EstimatedStatistics.VALUE_DOUBLE_UNKNOWN;
    		try {
    			compareToD = Double.parseDouble(compareToValue);
    		}
    		catch (NumberFormatException e) {
    			// can't compare nominal values by ">" or "<":
    			return null;
    		}
    		Vector remainingValues = new Vector();
    		
    		// find the values from the given listOfValues that fulfill the condition:
    		Iterator it = listOfValues.iterator();
    		while (it.hasNext()) {
				String testValue = (String) it.next();
				double testD = EstimatedStatistics.VALUE_DOUBLE_UNKNOWN;
				try {
					testD = Double.parseDouble(testValue);
				}
				catch (NumberFormatException nfe) {
					// some error, the given values should have been numeric
					return null;
				}
				if ( (bigger && testD > compareToD) ||
					 (smaller && testD < compareToD) ||
					 (bigger && orEqual && testD >= compareToD) ||
					 (smaller && orEqual && testD <= compareToD) ) {
					remainingValues.add(testValue);
				}
			}
    		
    		if (remainingValues.isEmpty()) {
    			return null;
    		}
    		else {
    			return remainingValues;
    		}
    	}
    	return null;
    }
	
	// checks if any mapping exists that maps an input attribute to the given 
	// output attribute; if yes, the corresponding input attribute is returned
	private BaseAttribute getMappedInput(BaseAttribute outputBa, Concept inputConcept, Step whichStep) throws M4Exception {
		// see which parameter the given BA belongs to:
		Collection outputPar = outputBa.getParameterReferences();
		if (outputPar != null) {
			Iterator paramIt = outputPar.iterator();
			Parameter outputParam = null;
			while (paramIt.hasNext()) {
				outputParam = (Parameter) paramIt.next();
				if ( ! outputParam.isInputParam()) {
					break; // we want the output param, there can be only one
				}
			}
			if (outputParam == null || outputParam.isInputParam()) {
				return null;
			}
			// see if there is a constraint mapping something to this output parameter:
			Iterator constrIt = whichStep.getTheOperator().getConstraints().iterator();
			while (constrIt.hasNext()) {
				Constraint myCon = (Constraint) constrIt.next();
				if (myCon.getType().equals(Constraint.TYPE_RENAME_ATTRIB) &&
					outputParam.getName().startsWith(myCon.getObj2())) {
					// now what is the corresponding input (with the same loop number)?
					Parameter inputParam = (Parameter) whichStep.getParameterTuple(myCon.getObj1(), outputParam.getLoopNr());
					ParameterObject parObj = (ParameterObject) inputParam.getTheParameterObject();
					if (parObj instanceof BaseAttribute) {
						BaseAttribute correspondingInputBa = (BaseAttribute) parObj;
						if (correspondingInputBa.getConcept().equals(inputConcept)) {
							return correspondingInputBa;
						}
					}
				}
			}
		}
		return null;
	}
	
	private Concept[] getConceptsForParamNames(String[] paramNames, Step whichStep) 
	throws M4Exception {
		Vector allConcepts = new Vector();
		for (int i = 0; i < paramNames.length; i++) {
			OpParam theOpPar = (OpParam) whichStep.getTheOperator().getOpParam(paramNames[i]);
			if (theOpPar != null && theOpPar.isConceptParameter()) {
				Collection pars = whichStep.getParameter(theOpPar, 0);
				if (pars != null) {
					Iterator it = pars.iterator();
					while (it.hasNext()) {
						allConcepts.add(it.next());
					}
				}
			}
		}
		Concept[] ret = new Concept[allConcepts.size()];
		for (int i = 0; i < ret.length; i++) {
			ret[i] = (Concept) allConcepts.get(i);
		}
		return ret;
	}
	
	// returns the minimum or maximum of the given doubles, unless
	// the UNKNOWN value is among them, then UNKNOWN is returned
	private double combineBounds(double[] someBounds, boolean useMinimum) {
		double extremeBound = (useMinimum ? Double.MAX_VALUE : Double.MIN_VALUE);
		for (int i = 0; i < someBounds.length; i++) {
			if (this.doublesAreEqual(someBounds[i], EstimatedStatistics.VALUE_DOUBLE_UNKNOWN)) {
				return EstimatedStatistics.VALUE_DOUBLE_UNKNOWN;
			}
			if ( (useMinimum && someBounds[i] < extremeBound)
				 ||
				 (( ! useMinimum) && someBounds[i] > extremeBound) )
				extremeBound = someBounds[i];
		}
		return extremeBound;
	}
	
	private boolean doublesAreEqual(double a, double b) {
		Double aA = new Double(a);
		Double bB = new Double(b);
		return (aA.compareTo(bB) == 0);
	}
	
	private String[] getInputAttribsForOutputAttribs(String[] outputAttribNames, String outputParamName, Step theStep)
	throws M4Exception {
		// we have to look at the constraints to find out about the
		// input attribs that belong to the output attribs:
		Collection constraints = theStep.getTheOperator().getConstraints();
		Iterator constrIt = constraints.iterator();
		String targetParameterName = null;
		while (constrIt.hasNext()) {
			Constraint myConstr = (Constraint) constrIt.next();
			if (myConstr.getType().equals(Constraint.TYPE_SAME_DATATYPE)) {
				if (myConstr.getObj1().equalsIgnoreCase(outputParamName)) {
					targetParameterName = myConstr.getObj2();
				}
				else if (myConstr.getObj2().equalsIgnoreCase(outputParamName)) {
					targetParameterName = myConstr.getObj1();
				}
			}
		}
		if (targetParameterName == null)
			return null;
		else {
			return this.getNamesOfAttribParameterForAllLoops(targetParameterName, theStep);
		}
	}
	
	private String[] getOutputAttribsForInputAttribs(
			String[] inputAttribNames, 
			Step theStep)
	throws M4Exception {
		// find the output attribs (with equal or similar names)
		// in the input:
		String[] ret = new String[inputAttribNames.length];
		Concept outputConcept = (Concept) theStep.getOutputConcept();
		Collection outBAs = outputConcept.getAllBaseAttributes();
		boolean allOutputAttribsFound = true;
		for (int i = 0; i < inputAttribNames.length; i++) {
			// check if an output feature exists whose name starts with
			// the input name:
			Iterator it = outBAs.iterator();
			boolean found = false;
			while (it.hasNext()) {
				BaseAttribute outBA = (BaseAttribute) it.next();
				if (outBA.getName().startsWith(inputAttribNames[i])) {
					ret[i] = outBA.getName();
					found = true;
				}
			}
			if ( ! found) {
				allOutputAttribsFound = false;
			}
		}
		if (allOutputAttribsFound) {
			return ret;
		}
		else {
			throw new M4Exception("Assertion '" + this.getAssertionType() + 
					"' for step '" + theStep.getName() +
					"': could not find matching output attribs for all input attribs!");
		}		
	}
	
	private void doValueListEstimation(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		if (this.getAssertionType().equals(ESTIMATE_NO_CHANGE)) {
			Concept outputConcept = (Concept) whichStep.getOutputConcept();
			Collection outBAs = outputConcept.getAllBaseAttributes();
			Iterator baIt = outBAs.iterator();
			while (baIt.hasNext()) {
				BaseAttribute outBA = (BaseAttribute) baIt.next();
				if (whichStep.isVisible(outBA)) {
					outputEstimation.setValueList(outBA.getName(), inputEstimation.getValueList(outBA.getName()));
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_ADD_VALUE)) {
			String[] outBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			String value = this.getValueParameter(this.getObj2(), (Step) whichStep);
			// add the value to each output list
			for (int i = 0; i < outBaNames.length; i++) {
				outputEstimation.addValueInformation(outBaNames[i], value, EstimatedStatistics.VALUE_INT_UNKNOWN);
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_FROM_LIST)) {
			String[] outAttribNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			String[] valueLists = this.getValueParametersForAllLoops(this.getObj2(), (Step) whichStep);
			// either the list is given directly:
			if (valueLists == null) {
				for (int i = 0; i < outAttribNames.length; i++) {
					StringTokenizer st = new StringTokenizer(this.getObj2(), ", ");
					while (st.hasMoreTokens()) {
						String oneValue = st.nextToken();
						outputEstimation.addValueInformation(outAttribNames[i], oneValue, EstimatedStatistics.VALUE_INT_UNKNOWN);
					}
				}
			}
			// or it is given in some value parameter:
			else {
				Vector[] valuesPerOutAttrib = new Vector[outAttribNames.length];
				for (int i = 0; i < valueLists.length; i++) {
					StringTokenizer st = new StringTokenizer(valueLists[i], ", ");
					if (st.countTokens() != outAttribNames.length) {
						throw new M4Exception("Assertion '" + this.getAssertionType() + 
								"' in Step '" + whichStep.getName() + "': found unmatching number of values in Obj2 parameter!");
					}
					int k = 0;
					while (st.hasMoreTokens()) {
						String oneValue = st.nextToken();
						if (valuesPerOutAttrib[k] == null) {
							valuesPerOutAttrib[k] = new Vector();
						}
						if ( ! valuesPerOutAttrib[k].contains(oneValue))
							valuesPerOutAttrib[k].add(oneValue);
						k++;
					}
				}
				for (int i = 0; i < outAttribNames.length; i++) {
					outputEstimation.setValueList(outAttribNames[i], valuesPerOutAttrib[i]);
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_UNCHANGED)) {
			Concept outputConcept = (Concept) whichStep.getOutputConcept();
			Concept inputConcept = (Concept) inputEstimation.getConcept();
			Iterator it = outputConcept.getAllBaseAttributes().iterator();
			while (it.hasNext()) {
				BaseAttribute outputBa = (BaseAttribute) it.next();
				String outBaName = outputBa.getName();
				BaseAttribute inputBa = (BaseAttribute) inputConcept.getBaseAttribute(outBaName);
				if (inputBa == null) {
					inputBa = this.getMappedInput(outputBa, inputConcept, (Step) whichStep);
				}
				if (inputBa != null) {
					String inBaName = inputBa.getName();
					outputEstimation.setValueList(outBaName, inputEstimation.getValueList(inBaName));
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_FROM_PARAM)) {
			String attribName = this.getNameOfAttribParameter(this.getObj1(), (Step) whichStep);
			String[] values = this.getValueParametersForAllLoops(this.getObj2(), (Step) whichStep);
			
			if (attribName != null && values != null) {
				for (int i = 0; i < values.length; i++) {
					outputEstimation.addValueInformation(attribName, values[i], EstimatedStatistics.VALUE_INT_UNKNOWN);
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_BY_SYMBOL)) {
			// this unfortunately depends on having other assertions handled first!
			int noOfSymbols = this.getIntegerFromValueParameter(this.getObj2(), (Step) whichStep);
			String attribName = this.getNameOfAttribParameter(this.getObj1(), (Step) whichStep);
			if (attribName != null) {
				// see what we have already:
				Vector vl = outputEstimation.getValueList(attribName);
				if (vl.size() >= noOfSymbols) {
					return;
				}
				for (int i = vl.size(); i < noOfSymbols; i++) {
					outputEstimation.addValueInformation(attribName, OUTVAL_SYMBOL + i, EstimatedStatistics.VALUE_INT_UNKNOWN);
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_TAKE_FROM)) {
			// this assertion means that the value list for an output
			// attribute, given in obj2, can be taken from an input
			// attribute, given in obj1

			// if obj2 is null, the parameter given in obj1 is available
			// in input and output, OR it is only in input but a CREATE_BY
			// constraint exists that maps it to the output
			
			// the whole thing only works if the step does not use aggregation!
			if ( ! whichStep.usesAggregation()) {
				String[] outBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj2(), (Step) whichStep);
				String[] inBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
				if (outBaNames == null) {
					outBaNames = this.getOutputAttribsForInputAttribs(
							inBaNames, 
							(Step) whichStep);
				}
				// a special case:
				boolean done = this.handleReversingSteps(inBaNames, outBaNames, (EstimatedStatistics) outputEstimation, (Step) whichStep);
				if (done) {
					return;
				}
				
				// transfer estimations; there may be more output than input attribs:
				if (outBaNames.length > inBaNames.length) {
					if (inBaNames.length != 1) {
						throw new M4Exception("Assertion '" + this.getAssertionType() +
								"' in Step '" + whichStep.getName() + "': don't know how to map a few input attribs to more output attribs!");
					}
				}
				for (int outputIndex = 0; outputIndex < outBaNames.length; outputIndex++) {
					// check that our input BA actually corresponds to the current 
					// input estimations:
					int inputIndex = (inBaNames.length > 1 ? outputIndex : 0);
					if (this.attribBelongsToConcept((Step) whichStep, inBaNames[inputIndex], (Concept) inputEstimation.getConcept())) {
						Vector vl = inputEstimation.getValueList(inBaNames[inputIndex]);
						if (vl != null) {
							Iterator it = vl.iterator();
							while (it.hasNext()) {
								String value = (String) it.next();
								int inputFrequency = inputEstimation.getNumberOfOccurrences(inBaNames[inputIndex], value);
								outputEstimation.addValueInformation(outBaNames[outputIndex], value, inputFrequency);
							}
						}
					}
				}
			}			
		}
		if (this.getAssertionType().equals(ESTIMATE_VALLIST_COMBINE)) {
			// EITHER obj2 is an attrib parameter, then it is an output
			// attrib and obj1 is a list of input attribs whose estimations
			// are to be combined
			
			// OR obj1 and obj2 are concept parameters, this means to combine
			// all corresponding attributes in the input concepts (given by obj1)
			
			String outBaName = this.getNameOfAttribParameter(this.getObj2(), (Step) whichStep);
			StringTokenizer st = new StringTokenizer(this.getObj1(), ", ");
			String[] inNames = new String[st.countTokens()];
			int i = 0;
			while (st.hasMoreTokens()) {
				inNames[i] = st.nextToken();
				i++;
			}
			
			// now decide between the two cases above:
			if (outBaName != null) {
				// "EITHER-case":
				Vector allValues = new Vector();
				for (int j = 0; j < inNames.length; j++) {
					String[] moreInputBas = this.getNamesOfAttribParameterForAllLoops(inNames[j], (Step) whichStep);
					if (moreInputBas != null) {
						for (int k = 0; k < moreInputBas.length; k++) {
							this.addAllIfNotContained(inputEstimation.getValueList(moreInputBas[k]), allValues);
						}
					}
				}
				outputEstimation.setValueList(outBaName, allValues);
			}
			else {
				// "OR-case":
				Concept[] inputConcepts = this.getConceptsForParamNames(inNames, (Step) whichStep);
				
				// compute the estimated statistics for each input concept only once:
				EstimatedStatistics[] inputESs = new EstimatedStatistics[inputConcepts.length];
				for (int j = 0; j < inputConcepts.length; j++) {
					inputESs[j] = (EstimatedStatistics) inputConcepts[j].getEstimatedStatistics(whichStep);
				}
				// use the first concept as reference:
				Concept refConc = inputConcepts[0];
				Collection refBAs = refConc.getAllBaseAttributes();
				Iterator baIt = refBAs.iterator();
				while (baIt.hasNext()) {
					BaseAttribute refBA = (BaseAttribute) baIt.next();
					if (whichStep.isVisible(refBA)) {
						Vector allValues = new Vector();
						allValues.addAll(inputESs[0].getValueList(refBA.getName()));
						for (int j = 1; j < inputConcepts.length; j++) {
							BaseAttribute corrBa = (BaseAttribute) inputConcepts[j].getBaseAttribute(refBA.getName());
							if (corrBa != null)
								this.addAllIfNotContained(inputESs[j].getValueList(corrBa.getName()), allValues);
						}
						outputEstimation.setValueList(refBA.getName(), allValues);
					}
				}
			}
		}
	}

	private boolean handleReversingSteps(
			String[] inputNames, 
			String[] outBaNames,
			EstimatedStatistics outputEstimation,
			Step whichStep) 
	throws M4Exception {
		if (inputNames == null || inputNames[0] == null) {
			// obj1 should be a value parameter, but it is optional:
			String[] namesOfAttribs = this.getValueParametersForAllLoops(this.getObj1(), whichStep);
			
			// ob1 is an attribute from the reversed step:
			Step originalStep = (Step) whichStep.getReversedStep();
			if (originalStep == null) {
				// throw new M4Exception("Assertion '" + this.getAssertionType() + 
				//   		"' of Step '" + whichStep.getName() + "': could not handle Obj1, expected an attribute of the reversed step!");
				
				// not all MiningMart installations may have reversing links...
				return true;
			}
			if (namesOfAttribs == null) {
				namesOfAttribs = this.getNamesOfAttribParameterForAllLoops("TheTargetAttribute", originalStep);
			}
			if (namesOfAttribs == null || namesOfAttribs.length != outBaNames.length) {
				throw new M4Exception("Assertion '" + this.getAssertionType() + 
				  		"' of Step '" + whichStep.getName() + "' (handling reversed step): unequal numbers of attributes!");				
			}
			Collection inputConcepts = originalStep.getAllInputConcepts();
			Iterator it = inputConcepts.iterator();
			while (it.hasNext()) {
				Concept oneInConc = (Concept) it.next();
				EstimatedStatistics inES = (EstimatedStatistics) oneInConc.getEstimatedStatistics(originalStep);
				for (int i = 0; i < namesOfAttribs.length; i++) {
					BaseAttribute inBa = (BaseAttribute) oneInConc.getBaseAttribute(namesOfAttribs[i]);
					if (inBa != null) {
						double min = inES.getLowestValue(namesOfAttribs[i]);
						double max = inES.getBiggestValue(namesOfAttribs[i]);
						int noMv = inES.getNumberOfMissingValues(namesOfAttribs[i]);
						if (this.getAssertionType().equals(ESTIMATE_MINMAX_TAKE_FROM)) {
							outputEstimation.setBiggestValue(outBaNames[i], max);
							outputEstimation.setLowestValue(outBaNames[i], min);
						}
						if (this.getAssertionType().equals(ESTIMATE_MV_TAKE_FROM)) {
							outputEstimation.setNumberOfMissingValues(outBaNames[i], noMv);
						}
						Vector vl = inES.getValueList(namesOfAttribs[i]);
						if (vl != null) {
							if (this.getAssertionType().equals(ESTIMATE_VALLIST_TAKE_FROM)) {
								outputEstimation.setValueList(outBaNames[i], vl);
							}
							if (this.getAssertionType().equals(ESTIMATE_VALFREQ_TAKE_FROM)) {							
								Iterator valIt = vl.iterator();
								while (valIt.hasNext()) {
									String value = (String) valIt.next();
									int valueFrequency = inES.getNumberOfOccurrences(namesOfAttribs[i], value);									
									outputEstimation.setNumberOfOccurrences(outBaNames[i], value, valueFrequency);
								}
							}
						}
					}
				}
			}
			return true; 
		}
		return false;
	}
	
	// adds all elements of 'addWhat' to 'addTo' that are not yet in it
	private void addAllIfNotContained(Collection addWhat, Collection addTo) {
		if (addTo == null || addWhat == null) return;
		Iterator it = addWhat.iterator();
		while (it.hasNext()) {
			Object o = it.next();
			if ( ! addTo.contains(o)) {
				addTo.add(o);
			}
		}
	}
	
	// checks if the attribute that is an input parameter of the given step and has
	// the given name belongs to the given concept 
	private boolean attribBelongsToConcept(Step theStep, String theAttribName, Concept theConcept)
	throws M4Exception {
		Iterator inputOpParamsIt = theStep.getTheOperator().getAllInputOperatorParameters().iterator();
		while (inputOpParamsIt.hasNext()) {
			OpParam inputOpPar = (OpParam) inputOpParamsIt.next();
			if (inputOpPar.isBaseAttribParameter() || inputOpPar.isFeatureParameter()) {
				int loopstart = 0;
				if (inputOpPar.isLoopable() && theStep.getLoopCount() > 0) loopstart = 1;
				do {
					Collection c = theStep.getParameter(inputOpPar, loopstart);
					if (c != null) {
						Iterator parObjIt = c.iterator();
						while (parObjIt.hasNext()) {
							ParameterObject parObj = (ParameterObject) parObjIt.next();
							BaseAttribute ba = (BaseAttribute) parObj;
							if (ba.getName().equalsIgnoreCase(theAttribName)
									&& theConcept.hasFeature(ba)) {
								return true;
							}
						}
					}
					loopstart++;
				}
				while (loopstart < theStep.getLoopCount());
			}
		}
		return false;
	}
	
	private void doMissingValuesEstimation(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		if (this.getAssertionType().equals(ESTIMATE_NO_CHANGE)) {
			Concept outputConcept = (Concept) whichStep.getOutputConcept();
			Collection outBAs = outputConcept.getAllBaseAttributes();
			Iterator baIt = outBAs.iterator();
			while (baIt.hasNext()) {
				BaseAttribute outBA = (BaseAttribute) baIt.next();
				if (whichStep.isVisible(outBA)) {
					outputEstimation.setNumberOfMissingValues(outBA.getName(), inputEstimation.getNumberOfMissingValues(outBA.getName()));
				}
			}
		}
		if (this.getAssertionType().equals(ASSERTION_NOT_NULL)) {
			String[] outBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			if (outBaNames != null) {
				for (int i = 0; i < outBaNames.length; i++) {
					outputEstimation.setNumberOfMissingValues(outBaNames[i], 0);
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_MV_BY_SELECTIVITY)) {
			// estimate the number of missing values according to the selectivity of the operator:
			if (inputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN &&
				outputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN) {
				double selectivity = (double) outputEstimation.getNumberOfRows() / (double) inputEstimation.getNumberOfRows();

				// apply the selectivity to all attributes:
				Concept outputConcept = (Concept) whichStep.getOutputConcept();
				Concept inputConcept = (Concept) inputEstimation.getConcept();
				Iterator it = outputConcept.getAllBaseAttributes().iterator();
				while (it.hasNext()) {
					BaseAttribute outputBa = (BaseAttribute) it.next();
					String outBaName = outputBa.getName();
					BaseAttribute inputBa = (BaseAttribute) inputConcept.getBaseAttribute(outBaName);
					if (inputBa == null) {
						inputBa = this.getMappedInput(outputBa, inputConcept, (Step) whichStep);
					}
					if ( ! this.existsMoreImportantMvAssertion(whichStep, outputBa, inputBa)) {
						if (inputBa != null) {
							String inBaName = inputBa.getName();
							int estimatedNoOfMv = (int) (selectivity * (double) inputEstimation.getNumberOfMissingValues(inBaName));
							outputEstimation.setNumberOfMissingValues(outBaName, estimatedNoOfMv);
						}
					}
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_MV_TAKE_FROM)) {
			String[] inBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			String[] outBaNames = this.getNamesOfAttribParameterForAllLoops(this.getObj2(), (Step) whichStep);
			boolean done = this.handleReversingSteps(inBaNames, outBaNames, (EstimatedStatistics) outputEstimation, (Step) whichStep);
			if (done) {
				return;
			}
			if (inBaNames.length != outBaNames.length) {
				throw new M4Exception("Assertion '" + this.getAssertionType() +
						"' in Step '" + whichStep.getName() + "': found unequal number of output and target attributes!");
			}
			for (int i = 0; i < outBaNames.length; i++) {
				outputEstimation.setNumberOfMissingValues(outBaNames[i], inputEstimation.getNumberOfMissingValues(inBaNames[i]));
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_MV_ADD_INPUTS)) {
			// EITHER obj2 is an attrib parameter, then it is an output
			// attrib and obj1 is a list of input attribs whose estimations
			// are to be combined
			
			// OR obj1 and obj2 are concept parameters, this means to combine
			// all corresponding attributes in the input concepts (given by obj1)
			
			String outBaName = this.getNameOfAttribParameter(this.getObj2(), (Step) whichStep);
			StringTokenizer st = new StringTokenizer(this.getObj1(), ", ");
			String[] inNames = new String[st.countTokens()];
			int i = 0;
			while (st.hasMoreTokens()) {
				inNames[i] = st.nextToken();
				i++;
			}
			
			// now decide between the two cases above:
			if (outBaName != null) {
				// "EITHER-case":
				int sumOfMvNumbers = 0;
				for (int j = 0; j < inNames.length; j++) {
					String[] moreInputs = this.getNamesOfAttribParameterForAllLoops(inNames[j], (Step) whichStep);
					if (moreInputs != null) {
						for (int k = 0; k < moreInputs.length; k++) {
							int inputMv = inputEstimation.getNumberOfMissingValues(moreInputs[k]);
							if (inputMv == EstimatedStatistics.VALUE_INT_UNKNOWN) {
								outputEstimation.setNumberOfMissingValues(outBaName, EstimatedStatistics.VALUE_INT_UNKNOWN);
								return;
							}
							sumOfMvNumbers += inputMv;
						}
					}
				}
				outputEstimation.setNumberOfMissingValues(outBaName, sumOfMvNumbers);
			}
			else {
				// "OR-case":
				Concept[] inputConcepts = this.getConceptsForParamNames(inNames, (Step) whichStep);
				
				// compute the estimated statistics for each input concept only once:
				EstimatedStatistics[] inputESs = new EstimatedStatistics[inputConcepts.length];
				for (int j = 0; j < inputConcepts.length; j++) {
					inputESs[j] = (EstimatedStatistics) inputConcepts[j].getEstimatedStatistics(whichStep);
				}
				// use the first concept as reference:
				Concept refConc = inputConcepts[0];
				Collection refBAs = refConc.getAllBaseAttributes();
				Iterator baIt = refBAs.iterator();
				while (baIt.hasNext()) {
					BaseAttribute refBA = (BaseAttribute) baIt.next();
					if (whichStep.isVisible(refBA)) {
						int inputMv = inputESs[0].getNumberOfMissingValues(refBA.getName());
						if (inputMv == EstimatedStatistics.VALUE_INT_UNKNOWN) {
							outputEstimation.setNumberOfMissingValues(refBA.getName(), EstimatedStatistics.VALUE_INT_UNKNOWN);
							return;
						}
						int sumOfMvNumbers = inputMv;
						for (int j = 1; j < inputConcepts.length; j++) {
							BaseAttribute corrBa = (BaseAttribute) inputConcepts[j].getBaseAttribute(refBA.getName());
							if (corrBa != null) {
								inputMv = inputESs[j].getNumberOfMissingValues(corrBa.getName());
								if (inputMv == EstimatedStatistics.VALUE_INT_UNKNOWN) {
									outputEstimation.setNumberOfMissingValues(refBA.getName(), EstimatedStatistics.VALUE_INT_UNKNOWN);
									return;
								}
								sumOfMvNumbers += inputMv;
							}
						}
						outputEstimation.setNumberOfMissingValues(refBA.getName(), sumOfMvNumbers);
					}
				}
			}
		}
	}
	
	// checks if an assertion concerning the estimation of missing values
	// other than ESTIMATE_MV_BY_SELECTIVITY holds for one of the given BAs
	private boolean existsMoreImportantMvAssertion(
			edu.udo.cs.miningmart.m4.Step theStep, 
			BaseAttribute outBa,
			BaseAttribute inBa)
	throws M4Exception {
		if (outBa == null && inBa == null)
			return false;
		Collection params;
		if (outBa != null)
			params = outBa.getParameterReferences();
		else
			params = new Vector();
		if (inBa != null)
			params.addAll(inBa.getParameterReferences());
		Iterator it = params.iterator();
		while (it.hasNext()) {
			Parameter par = (Parameter) it.next();
			if (par.getTheStep().equals(theStep)) {
				Collection assertions = par.getTheOperator().getAssertions();
				Iterator it2 = assertions.iterator();
				while (it2.hasNext()) {
					Assertion a = (Assertion) it2.next();
					if (a.getAssertionType().equals(Assertion.ASSERTION_NOT_NULL) ||
						a.getAssertionType().equals(Assertion.ESTIMATE_MV_ADD_INPUTS) ||
						a.getAssertionType().equals(Assertion.ESTIMATE_MV_TAKE_FROM))
						return true;
 				}
			}
		}
		return false;
	}
	
	private void doSizeEstimationSingleInput(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics inputEstimation,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		// if the input is unknown, in most cases so is the output:
		if (inputEstimation.getNumberOfRows() == EstimatedStatistics.VALUE_INT_UNKNOWN) {
			outputEstimation.setNumberOfRows(EstimatedStatistics.VALUE_INT_UNKNOWN);
		}
		// but let's check the other assertions:
		if (this.getAssertionType().equals(ESTIMATE_NO_CHANGE)) {
			outputEstimation.setNumberOfRows(inputEstimation.getNumberOfRows());
		}
		if (this.getAssertionType().equals(ESTIMATE_SIZE_BY_VALUE)) {
			// the output size is given directly by a parameter:
			long newOutputSize = this.getIntegerFromValueParameter(this.getObj1(), (Step) whichStep);
			// but it cannot increase:
			if (inputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN) {
				newOutputSize = Math.min(newOutputSize, inputEstimation.getNumberOfRows());
			}
			outputEstimation.setNumberOfRows(newOutputSize);
		}
		if (this.getAssertionType().equals(ESTIMATE_SIZE_BY_DIVISION) &&
			inputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN) {
			Parameter attribParam = (Parameter) whichStep.getParameterTuple(this.getObj1(), 0);
			if (attribParam != null && attribParam.getTheParameterObject() != null) {
				ParameterObject parObj = (ParameterObject) attribParam.getTheParameterObject();
				// first see if it is an attribute parameter:
				if (parObj instanceof BaseAttribute) {
					String attribName = attribParam.getTheParameterObject().getName();
					// divide the input size by the number of distinct values of the attrib:
					Vector vl = inputEstimation.getValueList(attribName);
					if (vl == null || vl.isEmpty()) {
						outputEstimation.setNumberOfRows(EstimatedStatistics.VALUE_INT_UNKNOWN);
					}
					else {
						int divisor = vl.size();
						outputEstimation.setNumberOfRows(inputEstimation.getNumberOfRows() / divisor);
					}
				}
				else {
					// not an attribute paramter, so it should be a value parameter:
					int divisor = this.getIntegerFromValueParameter(this.getObj1(), (Step) whichStep);
					outputEstimation.setNumberOfRows(inputEstimation.getNumberOfRows() / divisor);
				}
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_SIZE_MULTIPLY_BY) &&
				inputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN) {
			String[] namesOfAttribs = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			int factor = 1;
			if (namesOfAttribs == null) {
				String theFactor = this.getValueParameter(this.getObj1(), (Step) whichStep);
				try {
					factor = Integer.parseInt(theFactor);
				}
				catch (NumberFormatException nfe) {
					throw new M4Exception("Assertion '" + this.getAssertionType() +
							"' in Step '" + whichStep.getName() + "': expected an attrib or numeric value parameter or a numeric constant in Obj1!");
				}
			}
			else {
				factor = namesOfAttribs.length;
			}
			outputEstimation.setNumberOfRows(factor * inputEstimation.getNumberOfRows());
		}
		if (this.getAssertionType().equals(ESTIMATE_SIZE_REDUCE_MV) &&
				inputEstimation.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN) {
			String nameOfTargetAttrib = this.getNameOfAttribParameter(this.getObj1(), (Step) whichStep);
			int noOfMvInInput = inputEstimation.getNumberOfMissingValues(nameOfTargetAttrib);
			if (noOfMvInInput != EstimatedStatistics.VALUE_INT_UNKNOWN) {
				outputEstimation.setNumberOfRows(inputEstimation.getNumberOfRows() - noOfMvInInput);
			}
		}
		if (this.getAssertionType().equals(ESTIMATE_SIZE_BY_VALLIST)) {
			String[] nameOfAttribs = this.getNamesOfAttribParameterForAllLoops(this.getObj1(), (Step) whichStep);
			// the parameter is optional:
			if (nameOfAttribs == null) {
				// no aggregation:
				long outSize = inputEstimation.getNumberOfRows();
				if (whichStep.getTheOperator().getName().equals("SpecifiedStatistics")) { // special case...
					outSize = 1;
				}
				outputEstimation.setNumberOfRows(outSize);
				return;
			}
			// the output size is given by the number of combinations of distinct values of the attribs:
			int outputSize = 1;
			for (int i = 0; i < nameOfAttribs.length; i++) {
				Vector vl = inputEstimation.getValueList(nameOfAttribs[i]);
				if (vl == null) {
					outputEstimation.setNumberOfRows(EstimatedStatistics.VALUE_INT_UNKNOWN);
					return;
				}
				else {
					outputSize *= vl.size();
				}
			}
			outputEstimation.setNumberOfRows(outputSize);
		}
	}

	private void doSizeEstimationSeveralInputs(
			edu.udo.cs.miningmart.m4.Step whichStep,
			edu.udo.cs.miningmart.m4.EstimatedStatistics[] inputEstimations,
			edu.udo.cs.miningmart.m4.EstimatedStatistics outputEstimation) 
	throws M4Exception {
		if (this.getAssertionType().equals(ESTIMATE_SIZE_ADD_INPUTS)) {
			int sum = 0;
			for (int i = 0; i < inputEstimations.length; i++) {
				long oneInputSize = inputEstimations[i].getNumberOfRows();
				if (oneInputSize == EstimatedStatistics.VALUE_INT_UNKNOWN) {
					outputEstimation.setNumberOfRows(EstimatedStatistics.VALUE_INT_UNKNOWN);
					return;
				}
				sum += oneInputSize;
			}
			outputEstimation.setNumberOfRows(sum);
		}
		if (this.getAssertionType().equals(ESTIMATE_SIZE_BY_RELATIONS)) {
			// get the relationships over which the join takes place, if they exist:
			Relation[] theRelations = this.getRelationsForJoin(this.getObj1(), (Step) whichStep);
			
			// if it is a join by keys, check that the keys for joining are equal
			// to the relationship keys:
			boolean ok = this.checkKeysAreUsedInRelations(theRelations, (Step) whichStep);
			if (ok && theRelations != null && theRelations.length > 0) {
				// for the moment we can only handle two joined concepts:
				if (theRelations.length == 1) {
					if (theRelations[0].isOneToManyRelation()) {
						Concept manyConcept = (Concept) theRelations[0].getTheFromConcept();
						if (manyConcept != null) {
							long noOfRowsOnManySide = manyConcept.getEstimatedStatistics(whichStep).getNumberOfRows();
							outputEstimation.setNumberOfRows(noOfRowsOnManySide);
						}
					}
					if (theRelations[0].isManyToManyRelation()) {
						// we can only estimate by the cross table or by the product
						// of input sizes:
						long outputSize = EstimatedStatistics.VALUE_INT_UNKNOWN;
						Columnset cs = (Columnset) theRelations[0].getCrossLinkColumnSet();
						if (cs == null) {
							Concept from = (Concept) theRelations[0].getTheFromConcept();
							Concept to = (Concept) theRelations[0].getTheToConcept();
							if (from != null && to != null) {
								long in1 = from.getEstimatedStatistics(whichStep).getNumberOfRows();
								long in2 = to.getEstimatedStatistics(whichStep).getNumberOfRows();
								if (in1 != EstimatedStatistics.VALUE_INT_UNKNOWN &&
									in2 != EstimatedStatistics.VALUE_INT_UNKNOWN) {
									outputSize = in1 * in2;
								}									
							}
						}
						else {
							outputSize = cs.getStatisticsAll();
						}
						outputEstimation.setNumberOfRows(outputSize);
					}
				}
			}
		}
	}
	
	private boolean checkKeysAreUsedInRelations(Relation[] theRels, Step theStep)
	throws M4Exception {
		// see if the step uses relation parameters:
		OpParam theOpPar = (OpParam) theStep.getTheOperator().getOpParam(this.getObj1());
		if (theOpPar.isRelationParameter()) {
			return true;
		}
		else {
			if (theRels == null || theRels.length == 0) {
				return false;
			}
			if (theOpPar.isConceptParameter()) {
				OpParam theKeyOpPar = (OpParam) theStep.getTheOperator().getOpParam(JoinByKey.PARAMETER_KEYS);
				if (theKeyOpPar == null) {
					return false;
				}
				Collection allKeys = theStep.getParameter(theKeyOpPar, 0);
				boolean allKeysAreUsedAsRelationKeys = true;
				Iterator keyIt = allKeys.iterator();
				while (keyIt.hasNext()) {
					ParameterObject theKeyAttrib = (ParameterObject) keyIt.next();
					// check if this keyAttrib occurs somewhere in the relations as key:
					boolean found = false;
					for (int i = 0; i < theRels.length; i++) {
						// actually the key info may not be present in the relation!!
						// so we simply check if the key attrib is somewhere in the concept...
						Concept theConc = (Concept) theRels[i].getTheFromConcept();
						if (theConc.hasFeature((BaseAttribute) theKeyAttrib)) {
							found = true;
						}
						theConc = (Concept) theRels[i].getTheToConcept();
						if (theConc.hasFeature((BaseAttribute) theKeyAttrib)) {
							found = true;
						}						
					}
					if ( ! found) {
						allKeysAreUsedAsRelationKeys = false;
					}
				}
				return allKeysAreUsedAsRelationKeys;
			}
		}
		return false;
	}
	
	private Relation[] getRelationsForJoin(String nameOfParameter, Step theStep)
	throws M4Exception {
		OpParam theOpPar = (OpParam) theStep.getTheOperator().getOpParam(nameOfParameter);
		if (theOpPar.isRelationParameter()) {
			Collection theRelParams = theStep.getParameter(theOpPar, 0);
			Relation[] ret = new Relation[theRelParams.size()];
			Iterator it = theRelParams.iterator();
			int i = 0;
			while (it.hasNext()) {
				ParameterObject parObj = (ParameterObject) it.next();
				ret[i] = (Relation) parObj;				
				i++;
			}
			return ret;
		}
		if (theOpPar.isConceptParameter()) {
			Collection theConceptParams = theStep.getParameter(theOpPar, 0);
			Iterator concIt = theConceptParams.iterator();
			Vector allFoundRelationsBetweenTheConcepts = new Vector();
			while (concIt.hasNext()) {
				Concept oneConcept = (Concept) concIt.next();
				Iterator relIt = oneConcept.getTheFromRelationships().iterator();
				while (relIt.hasNext()) {
					Relation oneRel = (Relation) relIt.next();
					if (theConceptParams.contains(oneRel.getTheToConcept())) {
						allFoundRelationsBetweenTheConcepts.add(oneRel);
					}
				}
				// leave out this concept as ToConcept, such relations will be
				// found from the FromConcept
			}
			if (allFoundRelationsBetweenTheConcepts.isEmpty()) {
				return null;
			}
			Relation[] ret = new Relation[allFoundRelationsBetweenTheConcepts.size()];
			Iterator relIt = allFoundRelationsBetweenTheConcepts.iterator();
			int i = 0;
			while (relIt.hasNext()) {
				ret[i] = (Relation) relIt.next();
				i++;
			}
			return ret;
		}
		return null;
	}
	
	// returns the name of the attribute given by the parameter name;
	// assumes loop number 0.
	private String getNameOfAttribParameter(String paramName, Step whichStep)
	throws M4Exception {
		if (paramName == null || whichStep == null) {
			return null;
		}
		Parameter theParam = (Parameter) whichStep.getParameterTuple(paramName, 0);
		if (theParam != null) {
			ParameterObject parObj = (ParameterObject) theParam.getTheParameterObject();
			if (parObj != null && (parObj instanceof Feature)) {
				return parObj.getName();
			}
			else {
				return null;
			}
		}
		return null;
	}

	// returns the names of the attributes given by the parameter name;
	// assumes loop number 0.
	private String[] getNamesOfAttribParameter(String paramName, Step whichStep)
	throws M4Exception {
		if (paramName == null || whichStep == null) {
			return null;
		}
		OpParam theOpPar = (OpParam) whichStep.getTheOperator().getOpParam(paramName);
		Collection allParams = whichStep.getParameter(theOpPar, 0);
		if (allParams != null) {
			String[] allNames = new String[allParams.size()];
			Iterator it = allParams.iterator();
			int i = 0;
			while (it.hasNext()) {
				ParameterObject parObj = (ParameterObject) it.next();
				if (parObj != null && (parObj instanceof Feature)) {
					allNames[i] = parObj.getName();
				}
				i++;
			}
			return allNames;
		}
		return null;
	}

	// returns the names of the attributes given by the parameter name for all loops
	private String[] getNamesOfAttribParameterForAllLoops(String paramName, Step whichStep)
	throws M4Exception {
		if (paramName == null || whichStep == null) {
			return null;
		}
		OpParam theOpPar = (OpParam) whichStep.getTheOperator().getOpParam(paramName);
		if (theOpPar == null) {
			return null;
		}
		// collect all parameter objects for all loops:
		Collection allParams = new Vector();
		int startLoop = 0; //, endLoop = 0;
		if (theOpPar.isLoopable() && whichStep.getLoopCount() > 0) {
			startLoop = 1;
			// endLoop = whichStep.getLoopCount();
		}
		boolean end = false;
		int loop = startLoop;
		while ( ! end) {
			Collection c = whichStep.getParameter(theOpPar, loop);
			if (c == null || c.isEmpty()) {
				end = true;
			}
			else {
				allParams.addAll(c);
				loop++;
			}
		}
		// return the names in an array:
		if ( ! allParams.isEmpty()) {
			String[] allNames = new String[allParams.size()];
			Iterator it = allParams.iterator();
			int i = 0;
			while (it.hasNext()) {
				ParameterObject parObj = (ParameterObject) it.next();
				if (parObj != null && (parObj instanceof Feature)) {
					allNames[i] = parObj.getName();
				}
				i++;
			}
			return allNames;
		}
		return null;
	}
	
	// gets the value of the value parameter in the given step;
	// loop number 0 or 1 is assumed
	private String getValueParameter(String paramName, Step whichStep) 
	throws M4Exception {
		OpParam theOpParam = (OpParam) whichStep.getTheOperator().getOpParam(paramName);
		int loopNr = 0;
		if (theOpParam != null && theOpParam.isLoopable() && whichStep.getLoopCount() > 0)
			loopNr = 1;
		Parameter theParam = (Parameter) whichStep.getParameterTuple(paramName, loopNr);
		if (theParam != null) {
			ParameterObject parObj = (ParameterObject) theParam.getTheParameterObject();
			if (parObj != null && (parObj instanceof Value)) {
				return ((Value) parObj).getValue();
			}
			else {
				throw new M4Exception("Assertion '" + this.getAssertionType() +
						"' in Step '" + whichStep.getName() + "': expected value parameter!");
			}
		}
		return null;
	}
	
	// gets the values of the value parameter in the given step for all loops
	private String[] getValueParametersForAllLoops(String paramName, Step whichStep) 
	throws M4Exception {
		if (paramName == null || whichStep == null) {
			return null;
		}
		OpParam theOpPar = (OpParam) whichStep.getTheOperator().getOpParam(paramName);
		if (theOpPar == null) {
			return null;
		}
		// collect all parameter objects for all loops:
		Collection allParams = new Vector();
		int startLoop = 0; 
		if (theOpPar.isLoopable() && whichStep.getLoopCount() > 0) {
			startLoop = 1;
		}
		boolean end = false;
		int loop = startLoop;
		while ( ! end) {
			Collection c = whichStep.getParameter(theOpPar, loop);
			if (c == null || c.isEmpty()) {
				end = true;
			}
			else {
				allParams.addAll(c);
				loop++;
			}
		}
		// return the names in an array:
		if ( ! allParams.isEmpty()) {
			String[] allNames = new String[allParams.size()];
			Iterator it = allParams.iterator();
			int i = 0;
			while (it.hasNext()) {
				ParameterObject parObj = (ParameterObject) it.next();
				if (parObj != null && (parObj instanceof Value)) {
					allNames[i] = ((Value) parObj).getValue();
				}
				i++;
			}
			return allNames;
		}
		return null;
	}
	
	private int getIntegerFromValueParameter(String paramName, Step whichStep) 
	throws M4Exception {
		String value = this.getValueParameter(paramName, whichStep);
		if (value == null) {
			value = paramName;
		}
		int theNumber = EstimatedStatistics.VALUE_INT_UNKNOWN;
		if (value != null) {
			try {
				theNumber = Integer.parseInt(value);
			}
			catch (NumberFormatException nfe) {
				throw new M4Exception("Assertion '" + this.getAssertionType() +
						"' in Step '" + whichStep.getName() + "': could not handle '" +
						paramName + "', expected integer or value parameter!");
			}
		}
		return theNumber;
	}

	private double getDoubleFromValueParameter(String paramName, Step whichStep) 
	throws M4Exception {
		String value = this.getValueParameter(paramName, whichStep);
		if (value == null) {
			value = paramName;
		}
		double theNumber = EstimatedStatistics.VALUE_DOUBLE_UNKNOWN;
		if (value != null) {
			try {
				theNumber = Double.parseDouble(value);
			}
			catch (NumberFormatException nfe) {
				throw new M4Exception("Assertion '" + this.getAssertionType() +
						"' in Step '" + whichStep.getName() + "': could not handle '" +
						paramName + "', expected integer or value parameter!");
			}
		}
		return theNumber;
	}
	
	private void assertNotNull(ExecutableOperator exOp, int loop) throws M4CompilerError {
		try {
			String obj1 = this.getObj1();
			ParameterObject[] parArray = (ParameterObject[]) exOp.getParameter(obj1, loop);
			Collection columns = Condition.getColumnsForParameter(parArray);
			
			Iterator it = columns.iterator();
			while (it.hasNext()) {
				edu.udo.cs.miningmart.m4.core.Column column = (edu.udo.cs.miningmart.m4.core.Column) it.next();
				column.setNumberOfMissingValues(0);
			}
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("M4 exception caught when checking assertion '" + this.getAssertionType() + "': " + m4e.getMessage());
		}
	}
	
	// Unique attributes are keys with a "NOT_NULL" constraint. 
	private void assertUniqueness(ExecutableOperator exOp, int loop) throws M4CompilerError {
		try {
			String obj1 = this.getObj1();
			ParameterObject[] parArray = (ParameterObject[]) exOp.getParameter(obj1, loop);
			Collection columns = Condition.getColumnsForParameter(parArray);
			
			Iterator it = columns.iterator();
			while (it.hasNext()) {
				edu.udo.cs.miningmart.m4.core.Column column = (edu.udo.cs.miningmart.m4.core.Column) it.next();
				String countS = column.getColumnset().readOrComputeCount();
				try {
					int count = Integer.parseInt(countS);
					column.setNumberOfUniqueValues(count);
					column.setNumberOfMissingValues(0);
				}
				catch (NumberFormatException e) {
					this.doPrint(Print.WARNING, "Could not compute count for Columnset with ID "
							+ column.getColumnset().getId());
				}
			}	
		}
		catch (M4Exception m4e) {
			throw new M4CompilerError("M4 exception caught when checking assertion '" + this.getAssertionType() + "': " + m4e.getMessage());
		}
	}
}

/*
 * Historie
 * --------
 * 
 * $Log: Assertion.java,v $
 * Revision 1.15  2006/09/27 14:59:59  euler
 * New version 1.1
 *
 * Revision 1.14  2006/09/20 15:23:44  euler
 * Bugfixes and extensions
 *
 * Revision 1.13  2006/09/04 17:21:40  euler
 * Bugfixes around statistics estimation
 *
 * Revision 1.12  2006/09/02 12:59:33  euler
 * *** empty log message ***
 *
 * Revision 1.11  2006/08/31 14:47:49  euler
 * *** empty log message ***
 *
 * Revision 1.10  2006/08/30 11:51:50  euler
 * *** empty log message ***
 *
 * Revision 1.9  2006/08/25 13:06:23  euler
 * *** empty log message ***
 *
 * Revision 1.8  2006/08/24 17:59:28  euler
 * *** empty log message ***
 *
 * Revision 1.7  2006/08/24 13:01:24  euler
 * Started implementation of statistics estimation
 *
 * Revision 1.6  2006/08/21 15:18:52  euler
 * Bugfix
 *
 * Revision 1.5  2006/04/11 14:10:14  euler
 * Updated license text.
 *
 * Revision 1.4  2006/04/06 16:31:13  euler
 * Prepended license remark.
 *
 * Revision 1.3  2006/03/23 11:13:45  euler
 * Improved exception handling.
 *
 * Revision 1.2  2006/03/20 14:02:24  scholz
 * added assertion handling
 *
 * Revision 1.1  2006/01/03 09:54:17  hakenjos
 * Initial version!
 *
 */
