/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import java.util.Collection;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.exception.ParameterError;

/**
 * @author Timm Euler, Daniel Hakenjos
 * @version $Id: OpParam.java,v 1.4 2006/09/27 14:59:58 euler Exp $
 */
public interface OpParam extends M4Data {

	//methods from core

	/**
	 * Gets the input/output property.
	 * 
	 * @return Returns <i>true</i> iff the parameter is an input.
	 */
	public boolean isInput();

	/**
	 * Similar to <code>isInput()</code>: Gets the input/output property,
	 * but returns a <code>String</code>.
	 * 
	 * @return Returns a <code>String</code> indicating if the parameter is
	 *         an input or output.
	 */
	public String getInput();

	/**
	 * Gets the maxArg.
	 * @return Returns an int. A return value of &quot;-1&quot; represents a
	 * <i>NULL</i> in the database, a return value of &quot;0&quot; is not
	 * possible.
	 */
	public int getMaxArg();

	/**
	 * Gets the minArg.
	 * @return Returns a int
	 */
	public int getMinArg();

	/**
	 * Gets the type.
	 * @return Returns a short
	 */
	public short getType();

	/**
	 * @see getType()
	 * 
	 * @return the type in its database <code>String</code> representation.
	 */
	public String getTypeS() throws ParameterError;


	/**
	 * Sets the input/output property.
	 * @param ioType Either "IN" for input parameters, or "OUT".
	 */
	public void setInput(String ioType) throws M4Exception;

	/**
	 * Sets the maxArg.
	 * @param maxArg The maxArg to set
	 */
	public void setMaxArg(int maxArg);

	/**
	 * Sets the minArg.
	 * @param minArg The minArg to set
	 */
	public void setMinArg(int minArg);

	/**
	 * Sets the type.
	 * @param type The type <i>String</i> read from the database.
	 */
	public void setType(String typeString) throws M4Exception;

	/**
	 * Gets the Operator.
	 * @return Returns an Operator
	 */
	public Operator getTheOperator();

	/**
	 * Sets the Operator.
	 * @param operator The Operator to set
	 */
	public void setTheOperator(Operator operator) throws M4Exception;
	
	/*  Methods dealing with the constraints for this operator parameter */
	
	/**
	 * Returns all Constraint objects that constrain this operator parameter.
	 * 
	 * @return a Collection of Constraint objects
	 */
	public Collection getApplicableConstraints() throws M4Exception;
	
	/**
	 * @return TRUE iff this parameter can take different values for each loop.	
	 * This can only occur if the operator is loopable.         
	 * @throws M4Exception
	 */
	public boolean isLoopable() throws M4Exception;
	
	/**
	 * @return TRUE iff this parameter is an array whose values must be 
	 * coordinated positionwise with the values of another parameter array.
	 * Compare the method <code>getCoordinatedParameterArrays()</code>
	 * in the class <code>Operator</code>.
	 * @throws M4Exception
	 */
	public boolean isCoordinated() throws M4Exception;
	
	/** 
	 * @return TRUE iff this parameter can only take the values specified in
	 * a list. The list is given by calling <code>getOneOfList()</code>.
	 * @throws M4Exception
	 */
	public boolean isOneOf() throws M4Exception;
	
	/**
	 * @return TRUE iff this parameter is optional.
	 * @throws M4Exception
	 */
	public boolean isOptional() throws M4Exception;
	
	/**
	 * @return TRUE iff this parameter takes the form of an array. FALSE iff
	 * it is a single parameter.
	 * 
	 * @throws M4Exception
	 */
	public boolean isArray() throws M4Exception;
	
	/**
	 * Convenience method to <code>getType()</code> and <code>getTypeS()</code>.
	 * @return TRUE iff this parameter is of type Concept.
	 */
	public boolean isConceptParameter() throws M4Exception;
	
	/**
	 * Convenience method to <code>getType()</code> and <code>getTypeS()</code>.
	 * @return TRUE iff this parameter is of type Value.
	 */
	public boolean isValueParameter() throws M4Exception;

	/** 
	 * @return TRUE iff this parameter should only take the values that occur
	 * in a certain BaseAttribute
	 * 
	 * @throws M4Exception
	 */
	public boolean isDependentValueParameter() throws M4Exception;
	
	/**
	 * Returns a conceptual data type that can be used for a Value Parameter Object
	 * of this OpParam, or null if this OpParam does not represent a Value Parameter.
	 * 
	 * @return a conceptual data type String
	 * @throws M4Exception
	 */
	public String getConceptualDatatypeForValue() throws M4Exception;
	
	/**
	 * Convenience method to create a Value object of the right
	 * datatype for this parameter. This method throws an exception if 
	 * <code>isValueParameter()</code> returns FALSE.
	 * 
	 * @return a Value object with matching conceptual datatype
	 * @throws M4Exception is thrown if this is no Value parameter
	 */
	public Value createValueObject() throws M4Exception;
	
	/**
	 * Convenience method to <code>getType()</code> and <code>getTypeS()</code>.
	 * @return TRUE iff this parameter is of type Relation.
	 */
	public boolean isRelationParameter() throws M4Exception;
	
	/**
	 * Convenience method to <code>getType()</code> and <code>getTypeS()</code>.
	 * @return TRUE iff this parameter is of type Feature.
	 */
	public boolean isFeatureParameter() throws M4Exception;
	
	/**
	 * Convenience method to <code>getType()</code> and <code>getTypeS()</code>.
	 * @return TRUE iff this parameter is of type BaseAttribute.
	 */
	public boolean isBaseAttribParameter() throws M4Exception;
	
	/**
	 * Convenience method to <code>getType()</code> and <code>getTypeS()</code>.
	 * @return TRUE iff this parameter is of type MultiColumnFeature.
	 */
	public boolean isMcfParameter() throws M4Exception;
	
	/**
	 * This method makes sense only if <code>isOneOf()</code> returns TRUE. In that
	 * case this method returns the list of possible values that this parameter can
	 * take. A collection of Strings is returned. Otherwise an empty collection is 
	 * returned.
	 * @return a collection of Strings
	 * @throws M4Exception
	 */
	public Collection getOneOfList() throws M4Exception;
	
	/**
	 * This method makes sense only if <code>isArray()</code> returns TRUE.
	 * In that case this method returns the maximum number of positions of the
	 * array, if such a maximum exists; otherwise this method returns NULL.
	 * The minimum number of array positions can be taken to be 1 if this 
	 * parameter is not optional (see method <code>isArray()</code>).
	 * 
	 * @return The maximum number of array positions or NULL if the number is unconstrained.
	 * @throws M4Exception
	 */
	public Integer getMaximumNumberOfArrayFields() throws M4Exception;	
}
/*
 * Historie
 * --------
 * 
 * $Log: OpParam.java,v $
 * Revision 1.4  2006/09/27 14:59:58  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:12  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:23  hakenjos
 * Initial version!
 *
 */
