/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import java.sql.ResultSet;
import java.util.logging.Level;

import edu.udo.cs.miningmart.db.DB;
import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * @author Timm Euler, Daniel Hakenjos
 * @version $Id: M4Object.java,v 1.4 2006/09/27 14:59:58 euler Exp $
 */
public interface M4Object{

	//Methods in core-classes

	/**
	 * Get the DB Object
	 */
	public DB getM4Db();

    /**
     * Load the M4 object with the given ID.
     */
    public M4Object load(long id) throws M4Exception; 
    
	public Print getCasePrintObject();

	/**
	 * Method to print messages to
	 * the screen or log file.
	 * 
	 * @param verbosity A verbosity level. Use one of the public
	 * 		  static variables of the class <code>Print</code>.
	 * @param printString The string with the message.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.utils.Print
	 */
    public void doPrint(Level verbosity, String printString);


	/**
	 * Method to print exception messages
	 * to the screen or log file.
	 * 
	 * @param ex An exception object. Its message string will be printed.
	 */
    public void doPrint(Exception ex);

    /**
     * Method to print data about this M4Object.
     */
    public void print();
    
    /**
	 * Returns id for this object.
	 */
	public long getId();
	
	/**
	 * Sets id for this object.
	 */
	public void setId(long id) throws M4Exception;
	
 	/**
 	 * Get the name for this object.
 	 */
 	public String getName();

 	/**
 	 * Set the name for this object.
 	 */
	public void setName(String name);

	/**
	 * Service method for M4Object names that are also used as DB names.
	 * @param name the original name for an M4 object (or <code>null</code>)
	 * @return the specified name with all spaces replaced by underscores 
	 * (or <code>null</code>)
	 */
	public String replaceSpacesInName(String name);

	/**
	 * Check for M4Object if it was newly created by the
	 * user, or if it is already in the database.
	 */
	public boolean isNew();

	/**
	 * Allows to set the "isNew" flag to FALSE.
	 * Use with care.
	 */
	public void setNotNew();
	
	/**
	 * This method returns the object with the given Id if it is in the Cache.
	 * 
	 * @param Id The unique M4 Id of the object to be loaded.
	 */
    public M4Object getM4ObjectFromCache(long Id);	

	/**
	 * This method stores an M4 object in the Cache, using its ID as
	 * the key for the underlying data structure.
	 * 
	 * @param An M4Object to be stored in the Cache.
	 */
	public void putM4ObjectToCache(M4Object m4o) throws M4Exception;

	/** 
	 * Method to comfortably write to the M4 database.
	 * @param query an SQL query to be executed. This has to be a write operation to the M4 database,
	 * or an SQL string to execute a procedure in the M4 schema.
	 * */
	public void executeM4SqlWrite(String query)	throws M4Exception;
	
	/** 
	 * Method to comfortably write to the business database.
	 * @param query an SQL query to be executed. This has to be a write operation to the business database,
	 * or an SQL string to execute a procedure in the business schema.
	 * */
	public void executeBusinessSqlWrite(String query) throws M4Exception;

	/** 
	 * Method to comfortably read from the M4 database. The caller <b>has</b> to close the
	 * returned <code>ResultSet</code> after usage!
	 * @param query an SQL query to be executed. This has to be a read operation on the M4 database.
	 * */
	public ResultSet executeM4SqlRead(String query)
		throws M4Exception;
		
	/** 
	 * Method to comfortably read from the business database. The caller <b>has</b> to close the
	 * returned <code>ResultSet</code> after usage!
	 * @param query an SQL query to be executed. This has to be a read operation on the business database.
	 * */
	public ResultSet executeBusinessSqlRead(String query)
		throws M4Exception;		

	public Long executeM4SingleValueSqlReadL(String query)
		throws M4Exception;
		
	public Long executeBusinessSingleValueSqlReadL(String query)
		throws M4Exception;		

	public String executeM4SingleValueSqlRead(String query)
		throws M4Exception;

	public String executeBusinessSingleValueSqlRead(String query)
		throws M4Exception;
		
	public long getNextM4SequenceValue() throws M4Exception;
	
	/** 
	 * Two <code>M4Object</code>s are defined to be equal, if they refer to the
	 * same object. This is true, if their ID is the same.
	 * */
	public boolean equals(Object obj);	


	//Methoden aus dem alten Interface

	// get/setDocumentation -> see M4Data

	/*
	 * Getter for the "changed" property, indicating if changes need saving.
	 * @return true if there are changes to be saved.
	 */
	// public boolean isChanged(); // TE: use M4Data.isDirty instead
}
/*
 * Historie
 * --------
 * 
 * $Log: M4Object.java,v $
 * Revision 1.4  2006/09/27 14:59:58  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:12  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:24  hakenjos
 * Initial version!
 *
 */
