/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import java.util.Vector;

import edu.udo.cs.miningmart.exception.M4Exception;

/**
 * @author Timm Euler
 *
 * Interface to a class whose objects represent estimated values of data characteristics
 * (statistics) for M4 concepts that do not (yet) have a columnset. In the current
 * version no information about the validity of the information or the quality
 * of the estimation is given. Some of the information that objects of this class
 * hold will be known and true while other information will be estimated. A future
 * enhancement could be to qualify each item of information in this sense.
 * 
 * Returned estimates should be compared with the public constants defined in this
 * interface to see if they are unknown.
 */
public interface EstimatedStatistics {
	public static final String VALUE_DISCRETE_UNKNOWN = "<unknown>";
	public static final double VALUE_DOUBLE_UNKNOWN = Double.NaN;
	public static final int VALUE_INT_UNKNOWN = Integer.MIN_VALUE; 

	/**
	 * Returns the concept this EstimatedStatistics object applies to.
	 */
	public Concept getConcept();
	
	/**
	 * Returns a Vector with the values this attribute takes.
	 * 
	 * @param nameOfAttribute
	 * @return
	 */
	public Vector getValueList(String nameOfAttribute);
	
	/**
	 * If the concept that these ES belong to is of type MINING,
	 * it may happen that some statistics have been computed
	 * for the columnset that may be attached to it. If such statistics
	 * are stored in the M4 schema they can be used as the estimations
	 * with this method. 
	 * 
	 * @return TRUE iff any estimations were updated with information 
	 * from M4.
	 * 
	 * @throws M4Exception
	 */
	public boolean readAvailableStatisticsFromDb() throws M4Exception;
	
	/**
	 * Removes the given value from the list of values of the given
	 * attribute. 
	 * 
	 * @param value
	 * @param nameOfAttribute
	 */
	public void removeValue(String value, String nameOfAttribute);
	
	public int getNumberOfMissingValues(String nameOfAttribute);
	public void setNumberOfMissingValues(String nameOfAttribute, int number);
	
	public double getBiggestValue(String nameOfAttribute);
	public void setBiggestValue(String nameOfAttribute, double value);
	
	public double getLowestValue(String nameOfAttribute);
	public void setLowestValue(String nameOfAttribute, double value);
	
	public long getNumberOfRows();	
	public void setNumberOfRows(long numberOfRows);
	
	public int getNumberOfOccurrences(String nameOfAttribute, String value);
	public void setNumberOfOccurrences(String nameOfAttribute, String value, int number);
	
	/**
	 * Sets the list of occurring values of the given attribute. The given
	 * vector must contain Strings with the values.
	 * 
	 * @param nameOfAttribute
	 * @param theValues a Vector with Strings
	 */
	public void setValueList(String nameOfAttribute, Vector theValues);

	/**
	 * Copy the list of values (with their number of occurrences) of the 
	 * attribute with name <code>nameOfSourceAttribute</code>from
	 * the given EstimatedStatistics object to this object, to the 
	 * attribute with name <code>nameOfDestinationAttribute</code>.
	 *  
	 * @param nameOfDestinationAttribute name of attribute to which to set
	 *        the copied list
	 * @param from the given EstimatedStatistics object
	 * @param nameOfSourceAttribute name of attribute from which to copy
	 */
	public void copyValueList( String nameOfDestinationAttribute,
			                   EstimatedStatistics from,
							   String nameOfSourceAttribute);
	
	public EstimatedStatistics copy(Concept toConcept) throws M4Exception;
	
	public void addAttribute(String nameOfAttribute);
	
	public void addValueInformation(String nameOfAttribute, String value, int noOfOccurrences);
}
