/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.exception.DbConnectionClosed;
import java.util.Collection;

/**
 * @author Timm Euler, Daniel Hakenjos
 * @version $Id: Column.java,v 1.9 2006/09/27 14:59:58 euler Exp $
 */
public interface Column extends M4Data {

	//metgods from core
	/**
	 * Makes a copy of this <code>Column</code> that belongs to the given <code>Columnset</code>.
	 * 
	 * @param cs A Columnset
	 * @return A copy of this Column that points to the given Columnset
	 */
	public Column copyColToCS(Columnset cs) throws M4Exception;

	/**
	 * Set this column's BaseAttribute.
	 * 
	 * @param ba The new BaseAttribute.
	 */
	public void setBaseAttribute(BaseAttribute ba) throws M4Exception;

	/**
	 * Get this column's BaseAtttribute.
	 * 
	 * @return this column's BaseAttribute.
	 */
	public BaseAttribute getTheBaseAttribute() throws M4Exception;

	/**
	 * Set the sql definition of this column.
	 * 
	 * @param sqlDefinition the new definition
	 */
	public void setSQLDefinition(String sqlDefinition);

	/**
	 * @return this column's sql definition as a String
	 */
	public String getSQLDefinition();

	/**
	 * @return this column's sql definition together with
	 * 		   schema name and table name (<code>schema.table.sql</code>)
	 */
	public String getSQLPlusLocation();

	/**
	 * Set the column data type for this column.
	 * 
	 * @param type the new column data type
	 */
	public void setColumnDataType(long type);

	/**
	 * @return  this column's column data type
	 */
	public long getColumnDataType();

	/**
	 * Set the name of this column's column data type.
	 * 
	 * @param dtname the new name
	 */
	public void setColumnDataTypeName(String dtname)
		throws DbConnectionClosed, M4Exception;

	/**
	 * @return the name of this column's column data type.
	 */
	public String getColumnDataTypeName() throws M4Exception;

	/**
	 * Active getter for the <code>KeyMember</code>s referencing this object
	 * as their primary key.
	 * 
	 * @return a <code>Collection</code> of <code>KeyMember</code> objects
	 */
	public Collection getPrimaryKeyMembers() throws M4Exception;

	/**
	 * Active getter for the <code>KeyMember</code>s referencing this object
	 * as their foreign key.
	 * 
	 * @return a <code>Collection</code> of <code>KeyMember</code> objects
	 */
	public Collection getForeignKeyMembers() throws M4Exception;
	/**
	 * Active getter for all basic statistics tuples pointing to this Columm.
	 * @return <code>Collection</code> of <code>ColumnStatistics1</code> objects
	 */
	public Collection getBasicColStats() throws M4Exception;

	/**
	 * Active getter for all distribution statistics tuples pointing to this Columm.
	 * @return <code>Collection</code> of <code>ColumnStatistics2</code> objects
	 */
	public Collection getDistributionStatistics() throws M4Exception;

	/** 
	 * Sets the distribution statistics collection at once to the specified collection.
	 * @param dStats a <code>Collection</code> of <code>ColumnStatistics2</code> objects.
	 */
	public void setDistributionStatistics(Collection dStats)
		throws M4Exception;

	public void setMaxValue(double newMax) throws M4Exception;
	
	public void setMinValue(double newMin) throws M4Exception;
	
	/**
	 * @return If known, the number of unique values of this column.
	 */
	public int getNumberOfUniqueValues() throws M4Exception;

	/**
	 * @return If known, the number of missing values of this column.
	 */
	public int getNumberOfMissingValues() throws M4Exception;
	
	public void setNumberOfMissingValues(int newNumber) throws M4Exception;
	
	/**
	 * @return If known, the smallest value in this column.
	 */
	public String getMinValue() throws M4Exception;

	/**
	 * @return If known, the biggest value in this column.
	 */
	public String getMaxValue() throws M4Exception;

	/**
	 * @return If known, the median value in this column.
	 */
	public String getMedianValue() throws M4Exception;

	/**
	 * @return If known, the modal value in this column.
	 */
	public String getModalValue() throws M4Exception;

	/**
	 * @return If known, the average of the values in this column.
	 */
	public String getAverageValue() throws M4Exception;

	/**
	 * @return If known, the standard deviation of the values in this column.
	 */
	public String getStandardDeviation() throws M4Exception;

	/**
	 * @return If known, the variance of the values in this column.
	 */
	public String getVariance() throws M4Exception;

	/**
	 * @return this column's ColumnSet.
	 */
	public Columnset getColumnset();

	/**
	 * Sets the ColumnSet for this column.
	 * @param cs The new ColumnSet
	 */
	public void setColumnset(Columnset cs) throws M4Exception;

	/**
	 * Reads, or computes the minimum value of this column and returns it as
	 * a string.
	 * 
	 * @return the minimum value in the column as a String
	 */
	public String readOrComputeMinimum() throws M4Exception;

	/**
	 * Reads, or computes the average value of this column and returns it as
	 * a string.
	 * 
	 * @return the average value in the column as a String
	 */
	public String readOrComputeAverage() throws M4Exception;

	/**
	 * Reads, or computes the standard deviation of this column and returns
	 * the result as a string.
	 * 
	 * @return the average value in the column as a String
	 */
	public String readOrComputeStdDev() throws M4Exception;
	
	/**
	 * Reads, or computes the maximum value of this column and returns it as
	 * a string.
	 * 
	 * @return the maximum value in the column as a String
	 */
	public String readOrComputeMaximum() throws M4Exception;

	/**
	 * Reads, or computes the number of missing values for this column and
	 * returns it as an int.
	 * 
	 * @return the maximum value in the column as a String
	 */
	public int readOrComputeNumMissingValues() throws M4Exception;

	/**
	 * Reads, or computes the number of unique values for this column and
	 * returns it as an int.
	 * 
	 * @return the maximum value in the column as a String
	 */
	public int readOrComputeUniqueValues() throws M4Exception;
	
	/**
	 * Calculates statistics for the specified column and stores it in the M4 tables
	 * COLSTATIST1/2.
	 * */
	public void updateStatistics() throws M4Exception;

    /**
     * If data changes in the database, e.g. a DB Concept is edited,
     * then this method allows to delete the deprecated statistics objects
     * without running a garbage collection.
     * 
     * @throws M4Exception
     */
    public void clearStatistics() throws M4Exception;

	/**
	 * This method stores the pseudo foreign key reference to this objects's
	 * <code>BaseAttribute</code>, which is still realized by a cross-table
	 *(<i>BA_COLUMN_T</i>).
	 */
	public void storeLocal() throws M4Exception;

	//methods from old interface	

	/**
	 * Disassociates the BaseAttrbute from this colum. Does not delete the BA involved.
	 */
	void removeBaseAttribute() throws M4Exception;
}
/*
 * Historie
 * --------
 * 
 * $Log: Column.java,v $
 * Revision 1.9  2006/09/27 14:59:58  euler
 * New version 1.1
 *
 * Revision 1.8  2006/09/05 15:27:12  euler
 * More functions around statistics
 *
 * Revision 1.7  2006/04/11 14:10:13  euler
 * Updated license text.
 *
 * Revision 1.6  2006/04/06 16:31:12  euler
 * Prepended license remark.
 *
 * Revision 1.5  2006/03/20 13:21:39  scholz
 * added readOrComputeStdDev()
 *
 * Revision 1.4  2006/03/19 21:16:51  scholz
 * conditions are now checked before executing an operator
 *
 * Revision 1.3  2006/01/13 01:19:46  scholz
 * improved statistics computation
 *
 * Revision 1.2  2006/01/12 20:35:18  scholz
 * bugfix statistics
 *
 * Revision 1.1  2006/01/03 09:54:24  hakenjos
 * Initial version!
 *
 */
