/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.m4;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.exception.M4CompilerError;

import java.util.Collection;

/**
 * @author Timm Euler, Daniel Hakenjos
 * @version $Id: BaseAttribute.java,v 1.6 2006/09/27 14:59:57 euler Exp $
 */
public interface BaseAttribute extends Feature {

	// Unfortunately there is a constraint on Alias lengths in Oracle!
	public static final int MAX_LENGTH_OF_ATTRIB_NAME = 30;
	
	
	//methods from core

	/**
	 * Makes a copy of this BaseAttribute.
	 * 
	 * @return A clone of this BaseAttribute, but with Id 0 since it 
	 * 	 	   is not written into the database yet.
	 */
	public BaseAttribute copy() throws M4Exception;

	/**
	 * Add a column to this BaseAttribute's columns.
	 * 
	 * @param c The new column.
	 */
	public void addColumn(Column column) throws M4Exception;

	/**
	 * Create a Column with the given name and add it to this BaseAttribute's
	 * columns.
	 *  
	 * @param name name for the Column
	 * @return the Columm
	 * @throws M4Exception
	 */
	public Column createColumn(String name) throws M4Exception; 
	
	/**
	 * @param col a <code>Column</code>
	 * @return <code>true</code> if a column with the same ID is already linked to this
	 * <code>BaseAttribute</code>
	 * */
	public boolean hasColumn(Column col) throws M4Exception;

	/**
	 * @return <code>true</code> if at least one column is linked to this
	 * <code>BaseAttribute</code>
	 * */
	public boolean hasColumns() throws M4Exception;

	/**
	 * Remove a column from this BaseAttribute's columns.
	 * 
	 * @param c The column to remove
	 * @return <code>true</code> if removing succeeded
	 */
	public boolean removeColumn(Column column) throws M4Exception;

	/**
	 * Set the conceptual data type.
	 * 
	 * @param type The new type.
	 */
	public void setConceptualDataType(long type);

	/**
	 * @return the conceptual data type.
	 */
	public long getConceptualDataType();

	/**
	 * Set the name of the conceptual data type.
	 * 
	 * @param dtname The new name of the conceptual data type.
	 */
	public void setConceptualDataTypeName(String dtname)
		throws M4Exception;

	/**
	 * @return The name of this BaseAttribute's conceptual data type.
	 */
	public String getConceptualDataTypeName()
		throws M4Exception;

	/**
	 * Set the role.
	 * 
	 * @param role The new role.
	 */
	public void setRole(long role);

	/**
	 * @return the role.
	 */
	public long getRole() throws M4Exception;

	/**
	 * Set the name of the role.
	 * 
	 * @param rolename The new name of the role.
	 */
	public void setRoleName(String rolename)
		throws M4Exception;

	/**
	 * @return The name of this BaseAttribute's role.
	 */
	public String getRoleName()
		throws M4Exception;
	
	/**
	 * Set all columns of this BaseAttribute.
	 * 
	 * @param theColumns A <code>Collection</code> of <code>Column</code> objects
	 */
	public void setColumns(Collection theColumns) throws M4Exception;

	/**
	 * @return The columns that belong to this BaseAttribute.
	 */
	public Collection getColumns() throws M4Exception;

	/**
	 * Get a specific column that belongs to this BaseAttribute.
	 * 
	 * @param The number of the column to be returned.
	 * @return The column with the given number.
	 */
	public Column getColumn(int index) throws M4Exception;

	/**
	 * This method replaces "getColumnForColumnSet". The current column is
	 * the one that belongs to the current columnset of the concept that this
	 * BaseAttribute belongs to. 
	 * 
	 * @return The current column.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.Concept
	 */
	public Column getCurrentColumn() throws M4Exception;


    /**
     * Returns the Step in the Case where this BaseAttribute is created as
     * an output attribute.
     * 
     * @return a Step
     */
    public Step getStepWhereThisIsOutputBa() throws M4Exception ;
	
	/**
	 * If this BaseAttribute belongs to a MultiColumnFeature, set it here.
	 * 
	 * @param mcf The MultiColumnFeature this BaseAttribute belongs to.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.MultiColumnFeature
	 */
	public void setMCFeature(MultiColumnFeature mcf) throws M4Exception;

	/**
	 * If this BaseAttribute belongs to a MultiColumnFeature, get it here.
	 * 
	 * @return The MultiColumnFeature if this BaseAttribute belongs to one,
	 * 	       <code>null</code> otherwise.
	 * 
	 * @see edu.udo.cs.miningmart.m4.core.MultiColumnFeature
	 */
	public MultiColumnFeature getMCFeature();

	/**
	 * @return TRUE if this BaseAttribute is relevant.
	 */
	public boolean isRelevant();

	/**
	 * @return the relevance of this BaseAttribute using the
	 *         database <code>String</code> representation
	 */
	public String getRelevantS();

	/**
	 * Set the relevance of this BaseAttribute.
	 * 
	 * @param r The new boolean value of the relevance.
	 */
	public void setRelevant(boolean r);

	/**
	 * Set the relevance of this BaseAttribute.
	 * 
	 * @param r The value of the relevance in thedatabase <code>String</code>
	 * 		  representation
	 */
	public void setRelevantS(String r);

	/**
	 * @return TRUE if this BaseAttribute is a DB Attribute.
	 */
	public boolean isDBAttrib();

	/**
	 * Getter for the DB Attribute property
	 * @return the boolean flag in its database <code>String</code> representation
	 */
	public String getDBAttribS();

	/**
	 * Setter for DBAttrib.
	 * 
	 * @param a The new boolean value.
	 */
	public void setDBAttrib(boolean a);

	/**
	 * Setter for DBAttrib.
	 * 
	 * @param attrib The boolean value in its database <code>String</code> representation.
	 */
	public void setDBAttribS(String attrib);

	/**
	 * @return <code>true</code> iff this <code>BaseAttribute</code> has
	 * been &quot;deselected&quot; by a <code>FeatureSelection</code> operator
	 * or has not been connected by the user.
	 * */
	public boolean isDeselected() throws M4CompilerError;

	//methods fom old interface

	/* Constants */

	/**
	 * Concepts and BaseAttributes of type BASE can be used to define a Conceptual Model.
	 */
	public static final String TYPE_BASE = "BASE";

	/**
	 * BaseAttributes of type DB are based directly on a column or a table. They are not the result of some operator.
	 */
	public static final String TYPE_DB = "DB";

	/**
	 * BaseAttributes that are the result of an operator are of type MINING.
	 */
	public static final String TYPE_MINING = "MINING";


	/**
	 * The type of the BaseAttribute
	 * @see TYPE_BASE
	 * @see TYPE_DB
	 * @see TYPE_MINING
	 */
	public void setType(String type) throws M4Exception;

	public String getType(); 
}
/*
 * Historie
 * --------
 * 
 * $Log: BaseAttribute.java,v $
 * Revision 1.6  2006/09/27 14:59:57  euler
 * New version 1.1
 *
 * Revision 1.5  2006/08/24 13:01:24  euler
 * Started implementation of statistics estimation
 *
 * Revision 1.4  2006/06/19 15:37:13  euler
 * Bugfixes
 *
 * Revision 1.3  2006/04/11 14:10:12  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:12  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:24  hakenjos
 * Initial version!
 *
 */
