/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.stepsettings;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.jhotdraw.util.Iconkit;
import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.gui.application.MiningMartApplication;
import edu.udo.cs.miningmart.gui.application.ValueSelectionDialog;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Step;

/**
 * A Dialog for selecting multiple data-items of a list.
 * You can specify a range of selected items.
 * @author Daniel Hakenjos
 * @version $Id: ListSelectDialog.java,v 1.4 2006/09/27 14:59:59 euler Exp $
 */
public class ListSelectDialog extends JDialog implements ActionListener, ListSelectionListener{

	public static final int CANCEL = 0;
	public static final int OK = 1;

	private JList datalist;
	private JTextField valuefield;
	private JComboBox valuebox;
	private JList selectedlist, attributeList;
	private JButton cancel, ok;
	private JButton useEstimatesButton;
	private JButton add,up,down,remove,order;
	private int exitAction;
	
	private String[] data;
	private boolean sortdata;
	private String[] selectdata;
	private int minselection;
	private int maxselection;
	
	private BaseAttribute[] attribsWhoseValuesCanBeTaken;
	private String estButtonText, currentEstimationAttrib;
	private Step currentStep;
	
	private String selecttitle;
	
	private Collection list;

	private int list_type;
	
	public final int LIST_TO_LIST			= 0;
	public final int VALUE_TO_LIST			= 1;
	public final int ONE_OF_VALUE_TO_LIST 	= 2;

	/**
	 * Constructor for ListSelectDialog.
	 */
	public ListSelectDialog(MiningMartApplication parent, String title,String selecttitle, String[] data,String[] selected_data, boolean sortdata, int minselection, int maxselection) {
		super(parent, title, true);

		this.exitAction = CANCEL;
		this.data=data;
		this.selectdata=selected_data;
		this.sortdata=sortdata;
		this.minselection=minselection;
		this.maxselection=maxselection;
		if (maxselection==-1){
			this.maxselection=Integer.MAX_VALUE;
		}
		
		this.selecttitle=selecttitle;
		list_type=LIST_TO_LIST;
		initComponents(false);
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);

		this.pack();

		setSize(600, 300);

		Dimension size = this.getSize();

		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}

	/**
	 * A Dialog for creating a list of values.
	 * Constructor for ListSelectDialog.
	 */
	public ListSelectDialog(MiningMartApplication parent, String title,String selecttitle, String[] selected_data, int minselection, int maxselection) {
		this(parent, title, selecttitle, selected_data, minselection, maxselection, false, null, null);
	}

	/**
	 * A Dialog for creating a list of values.
	 * Constructor for ListSelectDialog.
	 */
	private ListSelectDialog(
			MiningMartApplication parent, 
			String title,
			String selecttitle, 
			String[] selected_data, 
			int minselection, 
			int maxselection,
			boolean useValueEstimations,
			BaseAttribute[] theAttribs,
			Step theStep) {
		super(parent, title, true);

		this.attribsWhoseValuesCanBeTaken = theAttribs;
		this.currentStep = theStep;
		
		this.exitAction = CANCEL;
		this.selectdata=selected_data;
		this.sortdata=false;
		this.minselection=minselection;
		this.maxselection=maxselection;
		if (maxselection==-1){
			this.maxselection=Integer.MAX_VALUE;
		}
		
		this.selecttitle=selecttitle;
		list_type=VALUE_TO_LIST;
		initComponents(useValueEstimations);
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);

		this.pack();

		setSize(600, 300);

		Dimension size = this.getSize();

		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}
	
	/**
	 * A Dialog for creating a list of values from a fixed list.
	 * Constructor for ListSelectDialog.
	 */
	public ListSelectDialog(MiningMartApplication parent, String title,String selecttitle,Collection list, String[] selected_data, int minselection, int maxselection) {
		super(parent, title, true);

		this.exitAction = CANCEL;
		this.list=list;
		this.selectdata=selected_data;
		this.sortdata=false;
		this.minselection=minselection;
		this.maxselection=maxselection;
		if (maxselection==-1){
			this.maxselection=Integer.MAX_VALUE;
		}
		
		this.selecttitle=selecttitle;
		list_type=ONE_OF_VALUE_TO_LIST;
		initComponents(false);
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);

		this.pack();

		setSize(600, 300);

		Dimension size = this.getSize();

		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}


	/**
	 * A Dialog for creating a list of values, with the option to
	 * select from the estimated values of the given BaseAttributes.
	 * Constructor for ListSelectDialog.
	 */
	public ListSelectDialog(
			MiningMartApplication parent, 
			String title,
			String selecttitle, 
			String[] selected_data, 
			int minselection, 
			int maxselection,
			BaseAttribute[] attribsToSelectValuesFrom,
			Step currentStep) {

		this(parent, title, selecttitle, selected_data, minselection, maxselection, true, attribsToSelectValuesFrom, currentStep);
	}

	/**
	 * Implementation of selection-Sort.
	 * 
	 */
	private String[] selectionSort(String[] data) {
		for (int i = 0; i < data.length - 1; i++) {
			// Erstes minimales Element in (a[i],...,a[n-1]) finden ...
			int min = i; // Position des aktuellen minimalen Elements
			for (int j = i + 1; j < data.length; j++)
				if (data[j].compareTo(data[min]) < 0)
					min = j;
			// ... und mit a[i] vertauschen:
			swap(data, min, i);
		}
		return data;
	}

	/** Hilfsmethode zur Vertauschung der Array-Elemente a[i] und a[j]. 
	  * Wir setzen 0 &le;= i, j &le; a.length voraus. 
	  */
	private void swap(Comparable[] a, int i, int j) {
		Comparable tmp = a[i];
		a[i] = a[j];
		a[j] = tmp;
	}

	/**
	 * Init the components of the dialog.
	 */
	private void initComponents(boolean useEstimations) {
		Iconkit kit = Iconkit.instance();

		JPanel top_panel = new JPanel();
		top_panel.setLayout(new BorderLayout(5, 5));

		JPanel center_panel=new JPanel();
		center_panel.setLayout(new GridLayout(1,2,5,5));
		top_panel.add(center_panel,BorderLayout.CENTER);


		JPanel left_panel = new JPanel();
		left_panel.setLayout(new BorderLayout());

		if (list_type==LIST_TO_LIST){
			initDataList();
			JScrollPane scrollpane = new JScrollPane(datalist);

			left_panel.add(scrollpane, BorderLayout.CENTER);
		}
		if (list_type==VALUE_TO_LIST){
			left_panel.add(initValueTextField(),BorderLayout.CENTER);
			if (useEstimations && this.attribsWhoseValuesCanBeTaken != null && this.attribsWhoseValuesCanBeTaken.length > 0) {
				
				JPanel estimPanel = new JPanel();
				estimPanel.setLayout(new BorderLayout());
				
				attributeList=new JList();
				attributeList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
				attributeList.setLayoutOrientation(JList.VERTICAL);
				attributeList.setVisibleRowCount(3);
				attributeList.addListSelectionListener(this);
				attributeList.setListData(this.getAttribNames());
				this.estButtonText = Resource.getString("STEPSETTINGS_LIST_USE_ESTIMATES");
				useEstimatesButton = new JButton(this.estButtonText);
				useEstimatesButton.setEnabled(false);
				useEstimatesButton.addActionListener(this);
				JLabel heading = new JLabel(Resource.getString("STEPSETTINGS_USE_ESTIMATES_FROM"));
				JScrollPane scrollpane=new JScrollPane(attributeList);
				estimPanel.add(heading, BorderLayout.NORTH);
				estimPanel.add(scrollpane, BorderLayout.CENTER);
				estimPanel.add(useEstimatesButton, BorderLayout.SOUTH);
				left_panel.add(estimPanel, BorderLayout.SOUTH);
			}
		}
		if (list_type==ONE_OF_VALUE_TO_LIST){
			left_panel.add(initValueComboBox(), BorderLayout.CENTER);
		}

		center_panel.add(left_panel);


		add=new JButton(new ImageIcon(kit.loadImageResource(Resource.getString("STEPSETTINGS_LIST_ADD_ICON"))));
		if (list_type==LIST_TO_LIST){
			add.setEnabled(false);
		}
		add.setToolTipText(Resource.getString("STEPSETTINGS_LIST_ADD_TOOLTIP"));
		add.setName("add");
		add.addActionListener(this);
		
		up=new JButton(new ImageIcon(kit.loadImageResource(Resource.getString("STEPSETTINGS_LIST_UP_ICON"))));
		up.setEnabled(false);
		up.setToolTipText(Resource.getString("STEPSETTINGS_LIST_UP_TOOLTIP"));
		up.setName("up");
		up.addActionListener(this);

		down=new JButton(new ImageIcon(kit.loadImageResource(Resource.getString("STEPSETTINGS_LIST_DOWN_ICON"))));
		down.setEnabled(false);
		down.setToolTipText(Resource.getString("STEPSETTINGS_LIST_DOWN_TOOLTIP"));
		down.setName("down");
		down.addActionListener(this);

		remove=new JButton(new ImageIcon(kit.loadImageResource(Resource.getString("STEPSETTINGS_LIST_REMOVE_ICON"))));
		remove.setEnabled(false);
		remove.setToolTipText(Resource.getString("STEPSETTINGS_LIST_REMOVE_TOOLTIP"));
		remove.setName("remove");
		remove.addActionListener(this);

		order=new JButton(new ImageIcon(kit.loadImageResource(Resource.getString("STEPSETTINGS_LIST_SORT_ICON"))));
		order.setEnabled(false);
		order.setToolTipText(Resource.getString("STEPSETTINGS_LIST_SORT_TOOLTIP"));
		order.setName("order");
		order.addActionListener(this);

		JPanel midbuttons2=new JPanel();
		midbuttons2.setLayout(new BorderLayout(5,5));
		midbuttons2.add(add,BorderLayout.NORTH);
		left_panel.add(midbuttons2,BorderLayout.EAST);

		JPanel midbuttons=new JPanel();
		midbuttons.setLayout(new BorderLayout(5,5));
		
		JPanel downbuttons=new JPanel();
		if (sortdata){
			downbuttons.setLayout(new GridLayout(4,1,5,5));
		}else{
			downbuttons.setLayout(new GridLayout(3,1,5,5));
		}
		downbuttons.add(up);
		downbuttons.add(down);
		downbuttons.add(remove);
		if (sortdata){
			downbuttons.add(order);
		}
		midbuttons.add(downbuttons,BorderLayout.SOUTH);


		selectedlist=new JList();
		selectedlist.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		selectedlist.setLayoutOrientation(JList.VERTICAL);
		selectedlist.setVisibleRowCount(-1);
		selectedlist.addListSelectionListener(this);
		selectedlist.setListData(selectdata);

		JScrollPane scrollpane=new JScrollPane(selectedlist);
		
		JPanel right_panel=new JPanel();
		right_panel.setLayout(new BorderLayout());		
		right_panel.add(scrollpane,BorderLayout.CENTER);
		right_panel.add(midbuttons,BorderLayout.WEST);
		center_panel.add(right_panel);

		//Buttons
		cancel = new JButton(Resource.getString("CANCEL"));
		ok = new JButton(Resource.getString("OK"));
		ok.setEnabled(false);
		checkMinMaxSelection();

		cancel.addActionListener(this);
		ok.addActionListener(this);

		JPanel bottom = new JPanel();
		bottom.setLayout(new BoxLayout(bottom, BoxLayout.X_AXIS));
		bottom.add(Box.createHorizontalGlue());
		bottom.add(ok);
		bottom.add(Box.createRigidArea(new Dimension(10, 0)));
		bottom.add(cancel);

		top_panel.add(bottom, BorderLayout.SOUTH);

		JLabel label = new JLabel(selecttitle);

		left_panel.add(label, BorderLayout.NORTH);
	
		String labeltext="";
		labeltext+="min: "+minselection;
		if (maxselection!=Integer.MAX_VALUE){
			labeltext+="    max: "+maxselection;
		}
		label = new JLabel(labeltext);

		right_panel.add(label, BorderLayout.NORTH);

		top_panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		this.getContentPane().add(top_panel);
	}
	
	private void initDataList(){
		datalist = new JList();
		datalist.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		datalist.setLayoutOrientation(JList.VERTICAL);
		datalist.setVisibleRowCount(-1);
		datalist.addListSelectionListener(this);

		if (sortdata)
			data = selectionSort(data);

		datalist.setListData(data);
	}
	
	private String[] getAttribNames() {
		String[] s = new String[this.attribsWhoseValuesCanBeTaken.length];
		for (int i = 0; i < s.length; i++) {
			s[i] = this.attribsWhoseValuesCanBeTaken[i].getName();			
		}
		return s;
	}
	
	private JPanel initValueTextField(){
		this.valuefield=new JTextField("",100);
		valuefield.setForeground(Color.BLACK);
		valuefield.setBackground(Color.WHITE);
		JPanel valuefieldpanel=new JPanel(new BorderLayout());
		valuefieldpanel.add(valuefield,BorderLayout.NORTH);
		return valuefieldpanel;
	}
	
	private JPanel initValueComboBox(){
		Vector vector=new Vector();
		vector.addAll(this.list);
		this.valuebox=new JComboBox(vector);
		valuebox.setForeground(Color.BLACK);
		valuebox.setBackground(Color.WHITE);
		JPanel valueboxpanel=new JPanel(new BorderLayout());
		valueboxpanel.add(valuebox,BorderLayout.NORTH);
		return valueboxpanel;
	}

	/**
	 * Called if one of the buttons was pressed.
	 */
	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();
		
		if (cmd.equals(Resource.getString("CANCEL"))) {
			exitAction = CANCEL;
			this.dispose();
		}else if (cmd.equals(Resource.getString("OK"))) {
			exitAction = OK;
			this.dispose();
		}else if (e.getSource() instanceof JButton){
			JButton source=(JButton) e.getSource();
			if (source.getText() != null && source.getText().equals(this.estButtonText)) {
				BaseAttribute selectedAttrib = this.getSelectedAttribute();
				if (selectedAttrib == null) return;
				ValueSelectionDialog vsd = new ValueSelectionDialog(selectedAttrib, this.currentStep);
				if (vsd.wasCancelled()) return;
				String values = vsd.getSelectionsAsOneString();
				if (values != null)
					addSelection(values);
			}
			
			if (source.getName() != null && source.getName().equals("add")){
				addSelection(null);
			}
			if (source.getName() != null && source.getName().equals("up")){
				int index=selectedlist.getSelectedIndex();
				if (index>0){
					String tmp=selectdata[index];
					selectdata[index]=selectdata[index-1];
					selectdata[index-1]=tmp;
					selectedlist.setListData(selectdata);
					selectedlist.setSelectedIndex(index-1);
					checkMinMaxSelection();
				}
			}
			if (source.getName() != null && source.getName().equals("down")){
				int index=selectedlist.getSelectedIndex();
				if (index<selectdata.length-1){
					String tmp=selectdata[index];
					selectdata[index]=selectdata[index+1];
					selectdata[index+1]=tmp;
					selectedlist.setListData(selectdata);
					selectedlist.setSelectedIndex(index+1);
					checkMinMaxSelection();
				}
			}
			if (source.getName() != null && source.getName().equals("remove")){
				int[] indices=selectedlist.getSelectedIndices();
				String[] newselectdata=new String[selectdata.length-indices.length];
				int adding=0;
				for(int s=0;s<selectdata.length;s++){
					if (!containsIndex(s,indices)){
						newselectdata[adding]=selectdata[s];
						adding++;
					}
				}
				selectdata=newselectdata;
				selectedlist.setListData(selectdata);
				checkMinMaxSelection();
			}
			if (source.getName() != null && source.getName().equals("order")){
				selectdata = selectionSort(selectdata);
				selectedlist.setListData(selectdata);
			}
		}
	}
	
	private void addSelection(String what){
		if (list_type==LIST_TO_LIST){
			Object[] selection=datalist.getSelectedValues();
			int adding=0;
			for(int s=0;s<selection.length;s++){
				if (containsItem((String) selection[s])){
					selection[s]=null;
				}else{
					adding++;
				}
			}
			
			String[] newselectdata=new String[selectdata.length+adding];
			System.arraycopy(selectdata,0,newselectdata,0,selectdata.length);
/*			for(int s=0;s<selectdata.length;s++){
				newselectdata[s]=selectdata[s];
			}*/
			adding=selectdata.length;
			for(int s=0;s<selection.length;s++){
				if (selection[s]!=null){
					newselectdata[adding]=(String) selection[s];
					adding++;
				}
			}
			selectdata=newselectdata;
			selectedlist.setListData(selectdata);
			checkMinMaxSelection();
		}
		if (list_type==VALUE_TO_LIST){
			String selection = (what == null ? valuefield.getText() : what);
			if (selection==null){
				return;
			}
			if (selection.length()==0){
				return;
			}
			String[] newselectdata=new String[selectdata.length+1];
			System.arraycopy(selectdata,0,newselectdata,0,selectdata.length);
			newselectdata[newselectdata.length-1]=selection;

			selectdata=newselectdata;
			selectedlist.setListData(selectdata);
			checkMinMaxSelection();
			valuefield.setText("");
		}
		if (list_type==ONE_OF_VALUE_TO_LIST){
			String selection=(String) valuebox.getSelectedItem();
			if (selection==null){
				return;
			}
			String[] newselectdata=new String[selectdata.length+1];
			System.arraycopy(selectdata,0,newselectdata,0,selectdata.length);
			newselectdata[newselectdata.length-1]=selection;

			selectdata=newselectdata;
			selectedlist.setListData(selectdata);
			checkMinMaxSelection();
		}
	}
	
	private boolean containsIndex(int index, int[] indices){
		for(int i=0;i<indices.length;i++){
			if (indices[i]==index){
				return true;
			}
		}
		return false;
	}
	
	private boolean containsItem(String item){
		for(int s=0;s<selectdata.length;s++){
			if (selectdata[s].equals(item)){
				return true;
			}
		}
		return false;
	}
	
	private BaseAttribute getSelectedAttribute() {
		if (this.attribsWhoseValuesCanBeTaken == null)
			return null;
		for (int i = 0; i < this.attribsWhoseValuesCanBeTaken.length; i++) {
			if (this.attribsWhoseValuesCanBeTaken[i].getName().equalsIgnoreCase(this.currentEstimationAttrib)) {
				return this.attribsWhoseValuesCanBeTaken[i];
			}
		}
		return null;
	}
	
	private void checkMinMaxSelection(){
		if (selectdata.length>1){
			order.setEnabled(true);
		}else{
			order.setEnabled(false);
		}
		if ((minselection<=selectdata.length)&&(selectdata.length<=maxselection)){
			ok.setEnabled(true);
		}else{
			ok.setEnabled(false);
		}
	}

	/**
	 * If the user selects something in the list then the ok-button is enabled.
	 */
	public void valueChanged(ListSelectionEvent e) {
		
		if (e.getSource().equals(this.attributeList)) {
			this.currentEstimationAttrib = (String) this.attributeList.getSelectedValue();
			this.useEstimatesButton.setEnabled(true);
		}
		
		if (list_type==LIST_TO_LIST){
			if (datalist.isSelectionEmpty()) {
				add.setEnabled(false);
			}else{
				add.setEnabled(true);
			}
		}

		if (selectedlist.isSelectionEmpty()){
			up.setEnabled(false);
			down.setEnabled(false);
			remove.setEnabled(false);
		}else{
			if (selectedlist.getSelectedValues().length==1){
				if (selectedlist.getSelectedIndex()!=0){
					up.setEnabled(true);
				}
				if (selectedlist.getSelectedIndex()!=selectdata.length-1){
					down.setEnabled(true);
				}
			}
			remove.setEnabled(true);
		}
	}

        /**
         *Gets the action when the dialog was exited.
         */
	public int getExitAction() {
		return exitAction;
	}

        /**
         *Gets the selected data-items.
         */
	public String[] getSelectedDataItems() {
		if (exitAction == OK)
			return selectdata;
		return null;
	}

}
/*
$Log: ListSelectDialog.java,v $
Revision 1.4  2006/09/27 14:59:59  euler
New version 1.1

Revision 1.3  2006/04/11 14:10:13  euler
Updated license text.

Revision 1.2  2006/04/06 16:31:13  euler
Prepended license remark.

Revision 1.1  2006/01/03 09:54:14  hakenjos
Initial version!

*/
