/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.stepsettings;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingConstants;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.gui.application.MiningMartApplication;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.PrimaryKey;
import edu.udo.cs.miningmart.m4.Step;

/**
 * Shows the attributes and columns of a concept
 * 
 * @author Timm Euler
 * @version $Id: $
 */
public class ConceptViewDialog extends JDialog implements ActionListener {

	private MiningMartApplication app;
	private JButton closebutton;
	private Concept myConcept;
	private Step myStep;
	private int noOfRowsInTable;
	
	/**
	 * Displays a non-editable dialog with information about the given
	 * concept.
	 *  
	 * @param app the parent component for the dialog 
	 * @param theConcept the concept whose details are to be shown
	 * @param theStep a Step where the concept plays a role
	 */
	public ConceptViewDialog(MiningMartApplication app, Concept theConcept, Step theStep) {
		
		super(app, Resource.getString("STEPSETTINGS_SHOW_CONCEPT_TITLE"), true);
		
		this.app = app;
		this.myConcept = theConcept;
		this.myStep = theStep;
		
		int heightOfTable = initComponents();
		if (heightOfTable == -1) {
			return;
		}
		
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		this.pack();
		setSize(600, 100 + heightOfTable * 25);
		Dimension size = this.getSize();
		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}	

	/**
	 * @see java.awt.event.ActionListener#actionPerformed(ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		this.dispose();
	}

	// returns the height of the table created
	private int initComponents(){
		JPanel mainpanel=new JPanel(new BorderLayout(5,5));
		this.getContentPane().add(mainpanel);
		
		JPanel northpanel=new JPanel(new BorderLayout(5,5));
		northpanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		mainpanel.add(northpanel,BorderLayout.NORTH);
		
		JLabel namelabel=new JLabel();
		String text = "Concept " + this.myConcept.getName();
		try {
			if (this.myConcept.getCurrentColumnSet() != null) {
				text += " (Columnset: " + this.myConcept.getCurrentColumnSet().getName() + ")";
			}
		}
		catch (M4Exception m4e) {
			return -1;
		}
		namelabel.setText(text);
		namelabel.getFont().deriveFont(Font.BOLD,15);
		namelabel.setHorizontalAlignment(SwingConstants.CENTER);	
		northpanel.add(namelabel,BorderLayout.CENTER);
				
		JPanel buttonpanel=new JPanel(new BorderLayout(5,5));
		mainpanel.add(buttonpanel,BorderLayout.SOUTH);
		
		closebutton = new JButton(Resource.getString("STEPSETTINGS_CLOSE"));
		closebutton.addActionListener(this);
		buttonpanel.add(closebutton, BorderLayout.EAST);
		
		// add table to mainpanel CENTER
		JScrollPane theInfoTable = this.createInfoTable();
		if (theInfoTable == null) {
			JOptionPane.showMessageDialog(this.app, "No attribute information found!", "Error displaying concept information", JOptionPane.ERROR_MESSAGE);
			this.dispose();
			return -1;
		}
		mainpanel.add(theInfoTable, BorderLayout.CENTER);
		return this.noOfRowsInTable;
	}	
	
	private JScrollPane createInfoTable() {

		Collection allAttribs = null;
		try {
			allAttribs = this.myConcept.getBaseAttributesVisible(myStep);			
		}
		catch (M4Exception m4e) {
			return null;
		}
		if (allAttribs == null || allAttribs.size() == 0) {
			this.noOfRowsInTable = -1;
			return null;
		}
		
		this.noOfRowsInTable = allAttribs.size();
		String[][] tabledata = new String[this.noOfRowsInTable][6];
		String[] columnNames = new String[] {
				"BA Name", "BA Role", "Data Type", "Column Name", "Column Type", "Primary Key"
			};
		
		try {
			Iterator attribIt = allAttribs.iterator();
			int rowIndex = 0;
			Collection keyCols = null;
			PrimaryKey pk = this.myConcept.getPrimaryKey();
			if (pk != null) 
				keyCols = pk.getAllColumns();
			
			while (attribIt.hasNext()) {
				BaseAttribute myBA = (BaseAttribute) attribIt.next();
				if (myBA != null) {
					tabledata[rowIndex][0] = myBA.getName();
					tabledata[rowIndex][1] = myBA.getRoleName();
					tabledata[rowIndex][2] = myBA.getConceptualDataTypeName();
					Column col = myBA.getCurrentColumn();
					if (col != null) {
						tabledata[rowIndex][3] = col.getName();
						tabledata[rowIndex][4] = col.getColumnDataTypeName();
						if (keyCols != null && keyCols.contains(col)) 
							tabledata[rowIndex][5] = "YES";
						else tabledata[rowIndex][5] = "";
					}
					else {
						tabledata[rowIndex][3] = Resource.getString("MAPPING_NONE");
						tabledata[rowIndex][4] = Resource.getString("MAPPING_NONE");
						tabledata[rowIndex][5] = "";
					}
				}
				rowIndex++;
			}
		}
		catch (M4Exception m4e) {
			return null;
		}
		
		JTable table=new JTable(tabledata,columnNames);
		JScrollPane scrollpane=new JScrollPane(table);
		return scrollpane;
	}
}