/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.model;

import java.awt.Color;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.Collection;
import java.util.Iterator;
import java.util.Vector;

import org.musoft.limo.application.Application;
import org.musoft.limo.application.Resource;
import org.musoft.limo.model.ModelAssociationEnd;
import org.musoft.limo.model.ModelAttribute;
import org.musoft.limo.model.ModelFigureElement;
import org.musoft.limo.model.ModelPrimitiveAttribute;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.gui.application.MiningMartApplication;
import edu.udo.cs.miningmart.m4.M4Data;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 *A wrapper for <code>Step</code>s.
 *A <code>MiningMartStep</code> could be a child of a <code>MiningMartChain</code>.
 * @author Daniel Hakenjos
 * @version $Id: MiningMartStep.java,v 1.6 2006/09/27 15:00:00 euler Exp $
 */
public class MiningMartStep extends MiningMartModelFigureElement{

	private MiningMartChain parentchain;

	private Step step;

	private MiningMartApplication app;

	private boolean deleteInDatabase=true;

    /**
     * Creates a new <code>MiningMartStep</code> with the given parameters.
     */
    public MiningMartStep(MiningMartApplication app,Step step,ModelFigureElement parent,Point position, Rectangle bounds) {
        super(step.getName(), bounds, parent);
		this.parentchain=(MiningMartChain) parent;
		this.step=step;
		this.app=app;
		String opName=step.getTheOperator().getName();
		this.setPosition(position);

		this.addAttribute(new ModelPrimitiveAttribute(this,"Operator", ModelPrimitiveAttribute.ACCESS_READ,ModelPrimitiveAttribute.TYPE_STRING) );

        addAttribute(new ModelAssociationEnd(this, "incoming", ModelAttribute.ACCESS_READWRITE, true, StepTransition.class, "end"));
        addAttribute(new ModelAssociationEnd(this, "outgoing", ModelAttribute.ACCESS_READWRITE, true, StepTransition.class, "start"));

        try{
	        ((ModelPrimitiveAttribute)getAttribute("notes")).setString(step.getDocumentation());
        }catch(M4Exception error){
        }        
        
        ((ModelPrimitiveAttribute)getAttribute("Operator")).setString(opName);
        ((ModelPrimitiveAttribute)getAttribute(Resource.getString("ATTR_COLOR"))).setColor(Color.WHITE);
    }

	/**
	 * Sets the flag that this object could be deleted in the database.
	 */    
	public void setDeleteInDatabase(boolean bool){
		this.deleteInDatabase=bool;
	}
    
	/**
	 * Gets the flag indicating that this obkect can be deleted in the database.
	 */
	public boolean canDeleteInDatabase(){
		return this.deleteInDatabase;
	}


    /**
     *Gets the <code>MiningMartApplication</code>.
     */
	public MiningMartApplication getMMartApplication(){
		return app;
	}


	public boolean canSetParent(ModelFigureElement parent) {
        
        return true;
	}
	
	public String getType(){
		return Resource.getString("STEP");
	}

        
    /**
     *Gets the name of the <code>MiningMartStep</code>.
     *@return returns the name of the <code>Step</Case>
     *@see edu.udo.cs.miningmart.m4.Step#getName()
     */
	public String getName(){
		if (step!=null)
			return this.step.getName();
		return "";
	}

        /**
         *Sets the name of the <code>MiningMartStep</code>.
         *It means it sets the name of the <code>Step</code>.
         *@see edu.udo.cs.miningmart.m4.Step#setName(String)
         */
    public void setName(String name){
    	if (step!=null)
			this.step.setName(name);
		super.setName(name);
		if ((app!=null)&&(app.isShowingSettings(this))){
			app.getStepSettings().setStepName(name);
		}
    }
    
    public Step getStep(){
		return step;
    }

	M4Data getAttachedM4Object() {
		return this.getStep();
	}
	
	/**
         * If the description of this <code>MiningMartStep</code> changed then set the documentation of the underlying <code>Step</code>.
	 * @see org.musoft.limo.model.ModelElement#primitiveAttributeChanged(ModelPrimitiveAttribute)
	 */
	public void primitiveAttributeChanged(ModelPrimitiveAttribute mpa) {
		super.primitiveAttributeChanged(mpa);
		if ((mpa.getName().equals(Resource.getString("ATTR_NAME")))&&(step!=null)){
			setName(mpa.getString());
		}
	}

	/**
         *Sets the bounds of the <code>MiningMartStep</code>.
         * The location of the underlying <code>Step</code> is set to the location of the bounds.
	 * @see org.musoft.limo.model.ModelFigureElement#setBounds(Rectangle)
	 */
	public void setBounds(Rectangle bounds) {
		super.setBounds(bounds);

		if (bounds==null){
			return;
		}
		
		if (step==null){
			return;
		}

		try{
			step.setPoint(bounds.getLocation());
		}catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
	}

        /**
         *Sets the position of this <code>MiningMartStep</code>.
         *The position of the underlying <code>Step</code> also changes.
         */
	public void setPosition(Point p){
		super.setPosition(p);
		try{
			step.setPoint(p);
		}catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
			
	}

	public boolean canDestroy() {
		if (app.getViewMode()==Application.VIEWER){
			return false;
		}
		return true;
	}
	
	/**
         *Destroys the <code>MiningMartStep</code> and deletes the underlying <code>Step</code>.
	 * @see org.musoft.limo.model.ModelElement#destroy()
	 */
	public void destroy() {
		if (app.getViewMode()==Application.VIEWER){
			//Im viewer-mode darf nicht zerstoert werden
			return;
		}

		// keep the successors of this step before deleting the transitions:
		Collection allSuccessorSteps = null; 
		try {
			allSuccessorSteps = new Vector(step.getSuccessors());
		}
		catch(M4Exception error){
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			return;
		}
		
        for (int i=parentchain.getConnectionCount()-1;i>=0;i--){
        	StepTransition trans=(StepTransition) parentchain.getConnection(i);
        	if ((trans.getStart().getName().equals(getName()))||(trans.getEnd().getName().equals(getName()))){
	        	trans.destroy();
        	}
        }

		super.destroy();

        for (int i = 0; i < getListenerCount(); i++) {
            getListener(i).onDestroy(this);
        }

		if (this.canDeleteInDatabase()){		
			try{
				String stepname=step.getName();
				this.step.deleteSoon();
				// because the step has lost its transitions, 
				// start updating all its successors:
				Iterator it = allSuccessorSteps.iterator();
				while (it.hasNext()) {
					Step succStep = (Step) it.next();
					if (succStep.adaptOutputToChangedInput()) {
						succStep.propagateOutputChanges();
					}
				}
				M4Interface.print.doPrint(Print.COMPILER_CASE_CONTROL,Resource.getString("DELETED_STEP")+" "+stepname);
			}catch(M4Exception error){
				M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
			}
		}
		
		if (app.isShowingSettings(this))
			app.removeStepSettingsPanel();
		
	}

}
/*
$Log: MiningMartStep.java,v $
Revision 1.6  2006/09/27 15:00:00  euler
New version 1.1

Revision 1.5  2006/09/04 17:21:41  euler
Bugfixes around statistics estimation

Revision 1.4  2006/04/11 14:10:15  euler
Updated license text.

Revision 1.3  2006/04/06 16:31:14  euler
Prepended license remark.

Revision 1.2  2006/03/02 16:49:59  euler
Many bugfixes

Revision 1.1  2006/01/03 09:54:36  hakenjos
Initial version!

*/
