/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.concepteditor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.gui.application.MiningMartApplication;
import edu.udo.cs.miningmart.gui.util.M4ObjectComparator;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Column;
import edu.udo.cs.miningmart.m4.ColumnStatistics2;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.ConceptualDatatypes;
import edu.udo.cs.miningmart.m4.EstimatedStatistics;
import edu.udo.cs.miningmart.m4.Step;

/**
 * A <code>JDialog</code> to show the statistics of a <code>Columnset</code>.
 * To include the estimated statistics functionality, uncomment the lines of 
 * code that add the point to the concept editor context menu, in the class
 * edu.udo.cs.miningmart.gui.application.MiningMartDrawingView, method
 * createConceptJPopupMenu(int x, int y).
 * 
 * @author Timm Euler
 * @version $Id: EstimatedStatisticsDialog.java,v 1.12 2006/09/27 15:00:03 euler Exp $
 */
public class EstimatedStatisticsDialog extends JDialog implements ActionListener{
			
	private static final String EMPTY_ATTRIBUTE = "<AttributeName>";
	private static final String EMPTY_VALUE = "<NewValue>";
	
	private Vector baseAttributes;	
	private EstimatedStatistics myEstStat;
	private String conceptName;
	private JButton closeButton, storeButton, addValueButton, removeValueButton;	
	private JTable statalltable, stat1table, stat2table;
	private Statistics2TableModel stat2tableModel;
	private MiningMartApplication myApplication;
	
	public EstimatedStatisticsDialog(
			MiningMartApplication app, 
			Concept theConcept,
			Step estimationsForThisStep){
		super(app,true);
		this.myApplication = app;
		try {
			this.conceptName = theConcept.getName();
			this.myEstStat = theConcept.getEstimatedStatistics(estimationsForThisStep);
			if (this.myEstStat == null) {
				throw new M4Exception("Got no estimations from this concept (NULL was returned).");
			}
			this.baseAttributes = new Vector();
			this.baseAttributes.addAll(theConcept.getAllBaseAttributes());
			Collections.sort(this.baseAttributes, new M4ObjectComparator());
		} catch (M4Exception error) {
			JOptionPane.showMessageDialog(app, "Error estimating statistics: " + error.getMessage(), "Estimation problem", JOptionPane.ERROR_MESSAGE);
			return;
		}

		this.updateDisplay();
	}
	
	private void updateDisplay() {
		this.setVisible(false);
		
		this.initComponents();
		
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);	
		this.pack();	
		setSize(750, 600);	
		Dimension size = this.getSize();	
		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}
	
	private void initComponents(){
		
		JPanel toppanel=new JPanel();
		toppanel.setLayout(new BorderLayout(5,5));
		toppanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		this.getContentPane().add(toppanel);

		//Info
		JPanel infopanel=new JPanel();
		infopanel.setLayout(new GridLayout(2,1,5,5));
		infopanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		toppanel.add(infopanel,BorderLayout.NORTH);	

		JLabel label=new JLabel();
		label.setText(Resource.getString("ESTIMATES"));
		Font font=new Font("SansSerif",Font.BOLD,15);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);	
		infopanel.add(label);

		label=new JLabel();
		label.setText(this.conceptName);
		font=new Font("SansSerif",Font.PLAIN,13);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);	
		label.setForeground(Color.BLUE);
		infopanel.add(label);
		
		/* This is the label for the statalltable-scrollpane (see below).
		 * Where to put it?
		label=new JLabel();
		label.setText(Resource.getString("ESTIMATED_GLOBAL"));
		font=new Font("SansSerif",Font.BOLD,12);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);
		*/
		
		//Buttons
		boolean columnsetExists = false;
		try {
			if (this.myEstStat.getConcept().getCurrentColumnSet() != null) {
				columnsetExists = true;
			}
		}
		catch (M4Exception m4e) {
			JOptionPane.showMessageDialog(myApplication, "M4 error accessing columnset: " + m4e.getMessage(), "Internal error", JOptionPane.ERROR_MESSAGE);
		}
		boolean isEditable = ( ! this.myApplication.isReadOnly());
		closeButton=new JButton(Resource.getString("CLOSE"));
		closeButton.addActionListener(this);
		storeButton=new JButton(Resource.getString("ESTIMATE_STORE"));
		storeButton.addActionListener(this);
		storeButton.setEnabled(columnsetExists && isEditable);
		addValueButton=new JButton(Resource.getString("ESTIMATE_ADD_VALUE"));
		addValueButton.addActionListener(this);
		//addValueButton.setEnabled(columnsetExists && isEditable);
		addValueButton.setEnabled(isEditable);
		removeValueButton=new JButton(Resource.getString("ESTIMATE_REMOVE_VALUE"));
		removeValueButton.addActionListener(this);
		//removeValueButton.setEnabled(columnsetExists && isEditable);
		removeValueButton.setEnabled(isEditable);

		JPanel buttonPanel=new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel,BoxLayout.X_AXIS));
		buttonPanel.add(Box.createHorizontalGlue());
		
		buttonPanel.add(Box.createRigidArea(new Dimension(5, 0)));			
		buttonPanel.add(storeButton);	
		buttonPanel.add(addValueButton);
		buttonPanel.add(removeValueButton);
		buttonPanel.add(closeButton);
		toppanel.add(buttonPanel,BorderLayout.SOUTH);

		JPanel midpanel=new JPanel();
		midpanel.setLayout(new BorderLayout(5,5));		
		toppanel.add(midpanel,BorderLayout.CENTER);
	

		//Statistics all	
		
		statalltable=new JTable();
		statalltable.setSelectionMode(javax.swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		statalltable.getTableHeader().setReorderingAllowed(false);
		statalltable.setRowHeight(statalltable.getRowHeight() + 2);
		statalltable.setModel(new StatisticsAllTableModel());
		JScrollPane scrollpane=new JScrollPane(statalltable);
		scrollpane.setPreferredSize(new Dimension(scrollpane.getWidth(),36));
		midpanel.add(scrollpane,BorderLayout.NORTH);
										
		JPanel centerPanel=new JPanel();
		centerPanel.setLayout(new GridLayout(2,1,5,5));
		midpanel.add(centerPanel,BorderLayout.CENTER);

		//Statistics1
		JPanel stat1panel=new JPanel();
		stat1panel.setLayout(new BorderLayout(5,5));
		centerPanel.add(stat1panel);
		
		label=new JLabel();
		label.setText(Resource.getString("ESTIMATED_ATTRIBDATA"));
		font=new Font("SansSerif",Font.BOLD,12);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);	
		stat1panel.add(label,BorderLayout.NORTH);

		stat1table=new JTable();
		stat1table.setSelectionMode(javax.swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		stat1table.getTableHeader().setReorderingAllowed(false);
		stat1table.setRowHeight(stat1table.getRowHeight() + 2);
		stat1table.setModel(new Statistics1TableModel());
		stat1panel.add(new JScrollPane(stat1table),BorderLayout.CENTER);
		
		//Statistics2
		JPanel stat2panel=new JPanel();
		stat2panel.setLayout(new BorderLayout(5,5));
		centerPanel.add(stat2panel);
		
		label=new JLabel();
		label.setText(Resource.getString("ESTIMATED_DISTRIBUTION"));
		font=new Font("SansSerif",Font.BOLD,12);
		label.setFont(font);
		label.setHorizontalAlignment(SwingConstants.CENTER);	
		stat2panel.add(label,BorderLayout.NORTH);

		stat2tableModel = new Statistics2TableModel();
		stat2table = new JTable(stat2tableModel);
		stat2table.setSelectionMode(javax.swing.ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		stat2table.getTableHeader().setReorderingAllowed(false);
		stat2table.setRowHeight(stat2table.getRowHeight() + 2);
		stat2table.setModel(stat2tableModel);
		stat2panel.add(new JScrollPane(stat2table),BorderLayout.CENTER);

	}	

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		if (!(event.getSource() instanceof JButton))
			return;
		JButton source=(JButton) event.getSource();
		
		if (source.getText().equals(closeButton.getText())){
			dispose();
		}
		if (source.getText().equals(storeButton.getText())){
			this.takeEstimatedStatisticsAsActualStatistics();
		}
		if (source.getText().equals(addValueButton.getText())){
			stat2tableModel.addValue();
		}
		if (source.getText().equals(removeValueButton.getText())){
			int[] selected = stat2table.getSelectedRows();
			stat2tableModel.removeValues(selected);
		}		
	}

	private void takeEstimatedStatisticsAsActualStatistics() {
		// this can only be called if there is a columnset!
		try {
			Columnset myCs = this.myEstStat.getConcept().getCurrentColumnSet();
			if (myCs == null) {
				return;
			}
			if (this.myEstStat.getNumberOfRows() != EstimatedStatistics.VALUE_INT_UNKNOWN)
				myCs.setStatisticsAll(this.myEstStat.getNumberOfRows());
			
			Iterator baIt = this.baseAttributes.iterator();
			while (baIt.hasNext()) {
				BaseAttribute myBa = (BaseAttribute) baIt.next();
				String attribName = myBa.getName();
				Column myCol = myBa.getCurrentColumn();
				if (myCs.hasColumn(myCol)) {
					if (Double.compare(this.myEstStat.getBiggestValue(attribName), EstimatedStatistics.VALUE_DOUBLE_UNKNOWN) != 0) {
						myCol.setMaxValue(this.myEstStat.getBiggestValue(attribName));
					}
					if (Double.compare(this.myEstStat.getLowestValue(attribName), EstimatedStatistics.VALUE_DOUBLE_UNKNOWN) != 0) {
						myCol.setMinValue(this.myEstStat.getLowestValue(attribName));
					}
					if (this.myEstStat.getNumberOfMissingValues(attribName) != EstimatedStatistics.VALUE_INT_UNKNOWN) {
						myCol.setNumberOfMissingValues(this.myEstStat.getNumberOfMissingValues(attribName));
					}
					Vector vl = this.myEstStat.getValueList(attribName);
					if (vl != null) {
						Vector allDistribInfos = new Vector();
						Iterator it = vl.iterator();
						while (it.hasNext()) {
							String value = (String) it.next();
							if (this.myEstStat.getNumberOfOccurrences(attribName, value) != EstimatedStatistics.VALUE_INT_UNKNOWN) {
								ColumnStatistics2 cs2 = (ColumnStatistics2) myCs.getM4Db().createNewInstance(ColumnStatistics2.class);
								cs2.setDistributionValue(value);
								cs2.setDistributionCount(this.myEstStat.getNumberOfOccurrences(attribName, value));
								allDistribInfos.add(cs2);
							}
						}
						myCol.setDistributionStatistics(allDistribInfos);
					}
				}
			}
		}
		catch (M4Exception m4e) {
			JOptionPane.showMessageDialog(this.myApplication, "M4 error when storing these estimations as actual statistics: " + m4e.getMessage(), "Internal error", JOptionPane.ERROR_MESSAGE);
		}
		
	}
	
	// returns the String that represents the given Value, or 
	// the String representing "unknown"
	private String getEstimatedValue(String dataTypeOfAttrib, int value) {
		if (this.isNotNumericType(dataTypeOfAttrib)) {
			return Resource.getString("ESTIMATE_NOT_APPLICABLE");
		}
		if (value == EstimatedStatistics.VALUE_INT_UNKNOWN) {
			return Resource.getString("ESTIMATE_UNKNOWN");
		}
		else {
			return (new Integer(value)).toString();
		}
	}

	// returns the String that represents the given Value, or 
	// the String representing "unknown"
	private String getEstimatedValue(int value) {
		if (value == EstimatedStatistics.VALUE_INT_UNKNOWN) {
			return Resource.getString("ESTIMATE_UNKNOWN");
		}
		else {
			return (new Integer(value)).toString();
		}
	}
	
	// returns the String that represents the given Value, or 
	// the String representing "unknown"
	private String getEstimatedValue(String dataTypeOfAttrib, double value) {
		if (this.isNotNumericType(dataTypeOfAttrib)) {
			return Resource.getString("ESTIMATE_NOT_APPLICABLE");
		}
		if ((new Double(value)).equals(new Double(EstimatedStatistics.VALUE_DOUBLE_UNKNOWN))) {
			return Resource.getString("ESTIMATE_UNKNOWN");
		}
		else {
			return (new Double(value)).toString();
		}
	}
	
	private boolean isNotNumericType(String conceptualDataType) {
		if (conceptualDataType == null) {
			return false;
		}
		if (conceptualDataType.equals(ConceptualDatatypes.CDT_CATEGORIAL) ||
			conceptualDataType.equals(ConceptualDatatypes.CDT_NOMINAL) ||
			conceptualDataType.equals(ConceptualDatatypes.CDT_KEYATTRIB)) {
			return true;
		}
		return false;
	}
	
	// returns the String that represents the given Value, or 
	// the String representing "unknown"
	private String getEstimatedValue(String value) {
		if (value == null || value.equals(EstimatedStatistics.VALUE_DISCRETE_UNKNOWN)) {
			return Resource.getString("ESTIMATE_UNKNOWN");
		}
		else {
			return value;
		}
	}

	private int getIntValue(Object theValue) {
		String value = null;
		try {
			value = (String) theValue;
		}
		catch (ClassCastException cce) {
			// should never happen
			JOptionPane.showMessageDialog(myApplication, "Please enter a String!", "Wrong format", JOptionPane.ERROR_MESSAGE);
			return -1;
		}
		if (value.equals(Resource.getString("ESTIMATE_UNKNOWN")) ||
			value.equals(Resource.getString("ESTIMATE_NOT_APPLICABLE"))) {
			return EstimatedStatistics.VALUE_INT_UNKNOWN;
		}
		int val = -1;
		try {
			val = Integer.parseInt(value);
		}
		catch (NumberFormatException nfe) {
			JOptionPane.showMessageDialog(myApplication, "Please enter a nonnegative integer!", "Wrong format", JOptionPane.ERROR_MESSAGE);
		}
		return val;
	}

	private double getDoubleValue(Object theValue) {
		String value = null;
		try {
			value = (String) theValue;
		}
		catch (ClassCastException cce) {
			// should never happen
			JOptionPane.showMessageDialog(myApplication, "Please enter a String!", "Wrong format", JOptionPane.ERROR_MESSAGE);
			return Double.MAX_VALUE;
		}
		if (value.equals(Resource.getString("ESTIMATE_UNKNOWN")) ||
			value.equals(Resource.getString("ESTIMATE_NOT_APPLICABLE"))) {
			return EstimatedStatistics.VALUE_DOUBLE_UNKNOWN;
		}
		double val = Double.MAX_VALUE;
		try {
			val = Double.parseDouble(value);
		}
		catch (NumberFormatException nfe) {
			JOptionPane.showMessageDialog(myApplication, "Please enter a double!", "Wrong format", JOptionPane.ERROR_MESSAGE);
		}
		return val;
	}
	
	private class StatisticsAllTableModel extends AbstractTableModel{
		
			/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnClass(int)
		 */
		public Class getColumnClass(int columnIndex) {
			return String.class;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnName(int)
		 */
		public String getColumnName(int column) {
			switch(column){
				case 0: return Resource.getString("ESTIMATED_ROWCOUNT");
				default: return ":-)";
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#isCellEditable(int, int)
		 */
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return ( ! myApplication.isReadOnly());
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return 1;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			return 1;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int rowIndex, int columnIndex) {
			switch(columnIndex){
				case 0: {
					return getEstimatedValue(null, myEstStat.getNumberOfRows());
				}
				default: return Resource.getString("ESTIMATE_UNKNOWN");
			}
		}
		
		public void setValueAt(Object theValue, int rowIndex, int columnIndex) {
			if (columnIndex != 0) {
				return;
			}
			int val = getIntValue(theValue);
			if (val != -1) {
				myEstStat.setNumberOfRows(val);
			}
			else {
				String oldValue = (String) getValueAt(rowIndex, columnIndex);
				setValueAt(oldValue, rowIndex, columnIndex);
			}
		}

}

	private class Statistics1TableModel extends AbstractTableModel{
		
		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnClass(int)
		 */
		public Class getColumnClass(int columnIndex) {
			return String.class;
		}


    	/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnName(int)
		 */
		public String getColumnName(int column) {
			switch (column){
			    case 0: return Resource.getString("ESTIMATED_ATTRIBNAME");
				case 1: return Resource.getString("ESTIMATED_MIN");
				case 2: return Resource.getString("ESTIMATED_MAX");
				case 3: return Resource.getString("ESTIMATED_MISSING");
				default: return ":-)";
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#isCellEditable(int, int)
		 */
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			if (( ! myApplication.isReadOnly()) && (columnIndex > 0)) {
				return true;
			}
			return false;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return 4;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			return baseAttributes.size();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int rowIndex, int columnIndex) {
			try {
				BaseAttribute ba = (BaseAttribute) baseAttributes.get(rowIndex);
				String type = ba.getConceptualDataTypeName();
				String attrName = ba.getName();
				switch (columnIndex){
				case 0: return attrName;
				case 1: return getEstimatedValue(type, myEstStat.getLowestValue(attrName));
				case 2: return getEstimatedValue(type, myEstStat.getBiggestValue(attrName));
				case 3: return getEstimatedValue(myEstStat.getNumberOfMissingValues(attrName));
				default: return Resource.getString("ESTIMATE_UNKNOWN");
				}
			}
			catch (M4Exception m4e) {
				// who cares
				return Resource.getString("ESTIMATE_UNKNOWN");
			}
		}
		
		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#setValueAt(Object, int, int)
		 */
		public void setValueAt(Object theValue, int rowIndex, int columnIndex) {
			if (columnIndex == 0) {
				return;
			}
			
			BaseAttribute ba = (BaseAttribute) baseAttributes.get(rowIndex);
			String attrName = ba.getName();
				
			if (columnIndex == 1 || columnIndex == 2) {
				double val = getDoubleValue(theValue);
				if (Double.compare(val, Double.MAX_VALUE) == 0) {
					String oldValue = (String) getValueAt(rowIndex, columnIndex);
					setValueAt(oldValue, rowIndex, columnIndex);
				}
				else {
					if (columnIndex == 1) 
						myEstStat.setLowestValue(attrName, val);
					else
						myEstStat.setBiggestValue(attrName, val);
				}
			}
			if (columnIndex == 3) {
				int val = getIntValue(theValue);
				if (val != -1)
					myEstStat.setNumberOfMissingValues(attrName, val);
				else {
					String oldValue = (String) getValueAt(rowIndex, columnIndex);
					setValueAt(oldValue, rowIndex, columnIndex);
				}
			}			
		}
	}

	private class Statistics2TableModel extends AbstractTableModel{
		
		private String[] attribNames;
		private String[] values;
		
		public Statistics2TableModel(){
			initNameArrays();
		}
		
		public void initNameArrays(){
			// create lists with all values and their attribute names:
			Vector attrNames = new Vector();
			Vector attrValues = new Vector();
			Iterator attrIt = baseAttributes.iterator();
			while (attrIt.hasNext()) {
				BaseAttribute myBa = (BaseAttribute) attrIt.next();
				String name = myBa.getName();
				Vector vals = myEstStat.getValueList(name);
				Iterator valIt = vals.iterator();
				while (valIt.hasNext()) {
					String value = (String) valIt.next();
					attrNames.add(name);
					attrValues.add(value);
				}
			}
			// make the arrays:
			this.attribNames = new String[attrNames.size()];
			this.values = new String[attrValues.size()];
			if (this.attribNames.length != this.values.length) {
				System.err.println("Error reading estimated statistics!");
				return;
			}
			for (int i = 0; i < this.attribNames.length; i++) {
				this.attribNames[i] = (String) attrNames.get(i);
				this.values[i] = (String) attrValues.get(i);
			}
		}
		
		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnClass(int)
		 */
		public Class getColumnClass(int columnIndex) {
			return String.class;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnName(int)
		 */
		public String getColumnName(int column) {
			switch (column){
				case 0: return Resource.getString("ESTIMATED_ATTRIBNAME");
				case 1: return Resource.getString("ESTIMATED_VALUE");
				case 2: return Resource.getString("ESTIMATED_OCCUR");
				default: return ":-)";
			}
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#isCellEditable(int, int)
		 */
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			return ( ! myApplication.isReadOnly());
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return 3;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			return this.attribNames.length;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int rowIndex, int columnIndex) {
			switch (columnIndex){
				case 0: return this.attribNames[rowIndex];
				case 1: return this.values[rowIndex];
				case 2: {
					if (this.attribNames[rowIndex].trim().equals("") ||
						this.values[rowIndex].trim().equals("")) 
						return "";
					return getEstimatedValue(null, myEstStat.getNumberOfOccurrences(this.attribNames[rowIndex], this.values[rowIndex]));
				}
				default: return ":-)";
			}
		}
		
		public void setValueAt(Object theValue, int rowIndex, int columnIndex) {
			String val = (String) theValue;
			if (columnIndex == 0) { // cannot add or change an attribute name
				// the only extra "attribute name" we have to accept is "<empty>",
				// which is ignored when using the estimations as actual statistics:
				if (val.equals(EMPTY_ATTRIBUTE)) {
					return;
				}
				if ( ! checkOccurs(val, baseAttributes)) {
					String oldValue = (String) getValueAt(rowIndex, columnIndex);
					setValueAt(oldValue, rowIndex, columnIndex);
				}
			}
			if (columnIndex == 1) {
				if (val.equals(EMPTY_VALUE)) {
					return;
				}
				// it may be a new value for this attribute!
				boolean valueIsNew = true;
				Vector vl = myEstStat.getValueList(this.attribNames[rowIndex]);
				if (vl != null && ( ! vl.isEmpty())) {
					Iterator it = vl.iterator();
					while (it.hasNext()) {
						String existingValue = (String) it.next();
						if (existingValue.equalsIgnoreCase(val)) {
							valueIsNew = false;
						}
					}
				}
				if (valueIsNew) {
					String noOfOcc = (String) getValueAt(rowIndex, 2);
					int newEntry = EstimatedStatistics.VALUE_INT_UNKNOWN;
					if (noOfOcc != null && ( ! noOfOcc.trim().equals(""))) {
						newEntry = getIntValue(noOfOcc);
					}
					try {
						// find the BA:
						Iterator attrIt = baseAttributes.iterator();
						BaseAttribute theBA = null;
						while (attrIt.hasNext()) {
							BaseAttribute oneBa = (BaseAttribute) attrIt.next();
							if (oneBa.getName().equals(this.attribNames[rowIndex]))
								theBA = oneBa;
						}
						if (theBA == null) {
							throw new M4Exception("Could not find BA with name '" + this.attribNames[rowIndex] + "'!");
						}
						String type = theBA.getConceptualDataTypeName();
						if ( ! isNotNumericType(type)) {
							// new value must be a number:
							double num = getDoubleValue(val);
							if (Double.compare(num, Double.MAX_VALUE) == 0) {
								return;
							}
							val = "" + num;
						}
						myEstStat.addValueInformation(this.attribNames[rowIndex], val, newEntry);
					}
					catch (M4Exception m4e) {
						JOptionPane.showMessageDialog(myApplication, "M4 error: " + m4e.getMessage(), "Internal error", JOptionPane.ERROR_MESSAGE);
					}
				} 
				else {
					int oldValue = getIntValue(getValueAt(rowIndex, 2));
					myEstStat.setNumberOfOccurrences(this.attribNames[rowIndex], val, oldValue);
				}
				this.values[rowIndex] = val;
			}
			if (columnIndex == 2) {
				int intval = getIntValue(theValue);
				if (intval != -1)
					myEstStat.setNumberOfOccurrences(this.attribNames[rowIndex], this.values[rowIndex], intval);
				else {
					String oldValue = (String) getValueAt(rowIndex, columnIndex);
					setValueAt(oldValue, rowIndex, columnIndex);
				}
			}
		}
		
		public void addValue() {
			String[] temp1 = new String[this.attribNames.length + 1];
			String[] temp2 = new String[this.values.length + 1];
			for (int i = 0; i < this.attribNames.length; i++) {
				temp1[i] = this.attribNames[i];
				temp2[i] = this.values[i];
			}
			temp1[this.attribNames.length] = EMPTY_ATTRIBUTE;
			temp2[this.attribNames.length] = EMPTY_VALUE;
			this.attribNames = new String[temp1.length];
			this.values = new String[temp1.length];
			for (int i = 0; i < temp1.length; i++) {
				this.attribNames[i] = temp1[i];
				this.values[i] = temp2[i];
			}
			this.fireTableDataChanged();
		}

		public void removeValues(int[] rowIndicesToBeRemoved) {
			int newLength = this.attribNames.length - rowIndicesToBeRemoved.length;
			String[] temp1 = new String[newLength];
			String[] temp2 = new String[newLength];
			int newIndex = 0;
			for (int i = 0; i < this.attribNames.length; i++) {
				if (this.contains(rowIndicesToBeRemoved, i)) {
					myEstStat.removeValue(this.values[i], this.attribNames[i]);
				}
				else {
					temp1[newIndex] = this.attribNames[i];
					temp2[newIndex] = this.values[i];
					newIndex++;
				}
			}
			this.attribNames = new String[temp1.length];
			this.values = new String[temp1.length];
			for (int i = 0; i < temp1.length; i++) {
				this.attribNames[i] = temp1[i];
				this.values[i] = temp2[i];
			}
			this.fireTableDataChanged();
		}
		
		private boolean contains(int[] array, int number) {
			for (int i = 0; i < array.length; i++) {
				if (array[i] == number) {
					return true;
				}
			}
			return false;
		}
		
		private boolean checkOccurs(String name, Collection baseattribs) {
			Iterator it = baseattribs.iterator();
			while (it.hasNext()) {
				BaseAttribute ba = (BaseAttribute) it.next();
				if (ba.getName().equals(name)) {
					return true;
				}
			}
			JOptionPane.showMessageDialog(myApplication, "Unknown attribute!", "Wrong input", JOptionPane.ERROR_MESSAGE);
			return false;
		}
	}
}
/*
 * Historie
 * --------
 *
 * $Log: EstimatedStatisticsDialog.java,v $
 * Revision 1.12  2006/09/27 15:00:03  euler
 * New version 1.1
 *
 * Revision 1.11  2006/09/05 15:27:12  euler
 * More functions around statistics
 *
 * Revision 1.10  2006/09/04 17:21:40  euler
 * Bugfixes around statistics estimation
 *
 * Revision 1.9  2006/09/02 12:59:33  euler
 * *** empty log message ***
 *
 * Revision 1.8  2006/08/31 14:47:50  euler
 * *** empty log message ***
 *
 * Revision 1.7  2006/08/30 11:51:50  euler
 * *** empty log message ***
 *
 * Revision 1.6  2006/08/24 13:01:25  euler
 * Started implementation of statistics estimation
 *
 * Revision 1.5  2006/04/11 14:10:17  euler
 * Updated license text.
 *
 * Revision 1.4  2006/04/06 16:31:16  euler
 * Prepended license remark.
 *
 * Revision 1.3  2006/03/30 16:07:16  scholz
 * fixed author tags for release
 *
 * Revision 1.2  2006/01/24 12:31:45  euler
 * Added recognition of key type for new columns.
 * Removed EstimatedStatistics from context menu
 * because they are in too basic status for the release.
 *
 * Revision 1.1  2006/01/18 16:58:58  euler
 * Added some basic estimations of statistics.
 * Will need improvements.
 *
 */
