/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.BaseAttribute;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.EstimatedStatistics;
import edu.udo.cs.miningmart.m4.Step;

/**
 * @author Timm Euler
 * @version $Id: $
 */
public class ValueSelectionDialog extends JDialog implements ActionListener,ListSelectionListener {

	public final int CANCEL = 0;
	public final int OK = 1;

	private int exitAction;
	
	private BaseAttribute myAttrib;
	private Step myStep;
	
	private JButton ok,cancel;
	
	private JList tables;
	
	private String[] selectedValues;
	
	private Component app;

	public ValueSelectionDialog(
			BaseAttribute theAttrib,
			Step theStep) {
		super(getApplication(), Resource.getString("DIALOG_VALUELIST_TITLE"), true);

		this.myAttrib = theAttrib;
		this.myStep = theStep;
		if (this.myAttrib == null) 
			return;
		this.exitAction = CANCEL;
		initComponents();
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		
		this.pack();

		setSize(250, 300);

		Dimension size = this.getSize();

		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}

	/**
	 * Inits the components.
	 */
	public void initComponents() {

		JPanel top_panel = new JPanel();
		top_panel.setLayout(new BorderLayout(5, 5));

		cancel = new JButton(Resource.getString("CANCEL"));
		ok = new JButton(Resource.getString("OK"));
		ok.setEnabled(false);

		cancel.addActionListener(this);
		ok.addActionListener(this);

		String text = Resource.getString("DIALOG_VALUELIST_MULT");
		JLabel label = new JLabel(text);
		top_panel.add(label,BorderLayout.NORTH);
		
		String[] tablenames = this.getValuesFromAttrib();
		
		this.tables=new JList(tablenames);
		tables.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		
		tables.addListSelectionListener(this);
		
		JPanel center = new JPanel();
		center.setLayout(new BorderLayout(5,5));
		JScrollPane scrollpane=new JScrollPane(tables);
		center.add(scrollpane, BorderLayout.CENTER);
		
		top_panel.add(center,BorderLayout.CENTER);

		JPanel bottom = new JPanel();
		bottom.setLayout(new BoxLayout(bottom, BoxLayout.X_AXIS));
		bottom.add(Box.createHorizontalGlue());
		bottom.add(ok);
		bottom.add(Box.createRigidArea(new Dimension(10, 0)));
		bottom.add(cancel);
		top_panel.add(bottom, BorderLayout.SOUTH);

		top_panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		this.getContentPane().add(top_panel);
	}

	private String[] getValuesFromAttrib() {
		try {
			Concept inCon = this.myAttrib.getConcept();
			if (inCon == null) 
				return null;
			EstimatedStatistics es = inCon.getEstimatedStatistics(this.myStep);
			if (es == null)
				return null;
			Collection theValues = es.getValueList(this.myAttrib.getName());
			if (theValues == null)
				return null;
			String[] allValues = new String[theValues.size()];
			Iterator it = theValues.iterator();
			int i = 0;
			while (it.hasNext()) {
				allValues[i] = (String) it.next();
				i++;
			}
			return allValues;
		}
		catch (M4Exception m4e) {
			JOptionPane.showConfirmDialog(this.app, "M4 error when reading estimated values for attrib '" + this.myAttrib.getName() + "':\n" + m4e.getMessage(), "Error", JOptionPane.ERROR_MESSAGE);
			return null;
		}
	}

	private static MiningMartApplication getApplication() {
		return null;
	}
	
	public boolean wasCancelled() {
		return (this.getExitAction() == CANCEL); 
	}
	
	public String getSelectionsAsOneString() {
		if (this.selectedValues == null)
			return null;
		String ret = "";
		for (int i = 0; i < this.selectedValues.length; i++) {
			ret += this.selectedValues[i] + ", ";
		}
		ret = ret.substring(0, ret.length() - 2);
		return ret;
	}
	
	public String[] getSelectedValues() {
		return this.selectedValues;
	}
	
	/**
	 * @see java.awt.event.ActionListener#actionPerformed(ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		String cmd = event.getActionCommand();
		
		if (cmd.equals(Resource.getString("CANCEL"))) {
			exitAction = CANCEL;
			this.dispose();
		}
		if (cmd.equals(Resource.getString("OK"))) {
			exitAction = OK;
			Object[] selection = tables.getSelectedValues();
			if (selection == null) 
				this.selectedValues = null;
			else {
				this.selectedValues = new String[selection.length];
				for (int i = 0; i < selection.length; i++) {
					this.selectedValues[i] = (String) selection[i];
				}
			}
			this.dispose();
		}
	}

	/* (non-Javadoc)
	 * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
	 */
	public void valueChanged(ListSelectionEvent e) {
		if (!tables.isSelectionEmpty()){
			ok.setEnabled(true);
		}else{
			ok.setEnabled(false);
		}
	}

	/**
	 * Gets the exit-action.
	 */
	public int getExitAction() {
		return exitAction;
	}
}
/*
 * Historie
 * --------
 *
 * $Log: ConceptFromTableDialog.java,v $
 * Revision 1.4  2006/09/27 15:00:01  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:16  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:15  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:27  hakenjos
 * Initial version!
 *
 */
