/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Step;

/**
 * @author Daniel Hakenjos
 * @version $Id: ConceptFromTableDialog.java,v 1.4 2006/09/27 15:00:01 euler Exp $
 */
public class StepSelectionDialog extends JDialog implements ActionListener,ListSelectionListener {

	public final int CANCEL = 0;
	public final int OK = 1;

	private int exitAction;
	
	private JButton ok,cancel;
	
	private JList tables;
	
	private String singleSelectedConcept;
	private String[] selectedObjects;
	
	private boolean singleSelectionRequired;
	
	private MiningMartApplication app;

	public StepSelectionDialog(MiningMartApplication parent, boolean selectOnlyOne) {
		super(parent, Resource.getString("DIALOG_STEPSELECT_TITLE"), true);

		this.app = parent;
		this.exitAction = CANCEL;
		initComponents();
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);

		this.singleSelectionRequired = selectOnlyOne;
		
		this.pack();

		setSize(250, 300);

		Dimension size = this.getSize();

		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}

	/**
	 * Inits the components.
	 */
	public void initComponents() {

		JPanel top_panel = new JPanel();
		top_panel.setLayout(new BorderLayout(5, 5));

		cancel = new JButton(Resource.getString("CANCEL"));
		ok = new JButton(Resource.getString("OK"));
		ok.setEnabled(false);

		cancel.addActionListener(this);
		ok.addActionListener(this);

		String text = (this.singleSelectionRequired ? Resource.getString("DIALOG_STEPSELECT_SINGLE") : Resource.getString("DIALOG_STEPSELECT_MULT"));
		JLabel label = new JLabel(text);
		top_panel.add(label,BorderLayout.NORTH);
		
		String[] tablenames = this.getListOfSteps();
		
		this.tables=new JList(tablenames);
		if ( ! this.singleSelectionRequired)
			tables.setSelectionMode(ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		
		tables.addListSelectionListener(this);
		
		JPanel center = new JPanel();
		center.setLayout(new BorderLayout(5,5));
		JScrollPane scrollpane=new JScrollPane(tables);
		center.add(scrollpane, BorderLayout.CENTER);
		
		top_panel.add(center,BorderLayout.CENTER);

		JPanel bottom = new JPanel();
		bottom.setLayout(new BoxLayout(bottom, BoxLayout.X_AXIS));
		bottom.add(Box.createHorizontalGlue());
		bottom.add(ok);
		bottom.add(Box.createRigidArea(new Dimension(10, 0)));
		bottom.add(cancel);
		top_panel.add(bottom, BorderLayout.SOUTH);

		top_panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		this.getContentPane().add(top_panel);
	}

	private String[] getListOfSteps() {
		String[] tableEntries = null;
		try{
			int tableSize = MiningMartApplication.m4Interface.getCurrentCase().getNumberOfSteps();
			tableEntries = new String[tableSize];
			int index = 0;
			Iterator stepIt = MiningMartApplication.m4Interface.getCurrentCase().getStepIterator();
			while (stepIt.hasNext()) {
				Step oneStep = (Step) stepIt.next();
				tableEntries[index] = oneStep.getName() + "\t (Chain:" + oneStep.getTheChain().getName() + ")";
				index++;
			}
		}catch(M4Exception error) {
			JOptionPane.showMessageDialog(this.app, "Error when reading tables and views from business schema:\n" + error.getMessage(), "DB error", JOptionPane.ERROR_MESSAGE);
			return new String[0];
		}
		return tableEntries;
	}

	/**
	 * @see java.awt.event.ActionListener#actionPerformed(ActionEvent)
	 */
	public void actionPerformed(ActionEvent event) {
		String cmd = event.getActionCommand();
		
		if (cmd.equals(Resource.getString("CANCEL"))) {
			exitAction = CANCEL;
			this.dispose();
		}
		if (cmd.equals(Resource.getString("OK"))) {
			exitAction = OK;
			Object[] selection = tables.getSelectedValues();
			if (selection == null) 
				this.selectedObjects = null;
			else {
				this.selectedObjects = new String[selection.length];
				for (int i = 0; i < selection.length; i++) {
					this.selectedObjects[i] = (String) selection[i];
					int index = -1;
					if ((index = this.selectedObjects[i].indexOf("(Chain:")) > -1) {
						this.selectedObjects[i] = this.selectedObjects[i].substring(0, index).trim();
					}
				}
				if (this.selectedObjects != null && this.selectedObjects.length > 0)
					this.singleSelectedConcept = this.selectedObjects[0];
			}
			this.dispose();
		}
	}

	/* (non-Javadoc)
	 * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
	 */
	public void valueChanged(ListSelectionEvent e) {
		if (!tables.isSelectionEmpty()){
			ok.setEnabled(true);
		}else{
			ok.setEnabled(false);
		}
	}

	/**
	 * Gets the exit-action.
	 */
	public int getExitAction() {
		return exitAction;
	}

	/**
	 * Gets the name of the step.
	 */
	public String getStepName() {
		if (exitAction == OK)
			return this.singleSelectedConcept;
		return null;
	}
	
	public String[] getSelectionOfSteps() {
		if (exitAction == OK)
			return this.selectedObjects;
		return null;
	}
}
/*
 * Historie
 * --------
 *
 * $Log: ConceptFromTableDialog.java,v $
 * Revision 1.4  2006/09/27 15:00:01  euler
 * New version 1.1
 *
 * Revision 1.3  2006/04/11 14:10:16  euler
 * Updated license text.
 *
 * Revision 1.2  2006/04/06 16:31:15  euler
 * Prepended license remark.
 *
 * Revision 1.1  2006/01/03 09:54:27  hakenjos
 * Initial version!
 *
 */
