/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Iterator;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.M4Interface;
import edu.udo.cs.miningmart.m4.utils.Print;

/**
 * A dialog to choose a case to open.
 * @author Daniel Hakenjos
 * @version $Id: OpenCaseDialog.java,v 1.4 2006/09/27 15:00:02 euler Exp $
 */
public class OpenCaseDialog
	extends JDialog
	implements ActionListener, ListSelectionListener {

	public static final int CANCEL = 0;
	public static final int SELECT = 1;

	private MiningMartApplication app;

	private JList list;
	private JButton cancel, select;
	private JRadioButton view, update;
	private JRadioButton caseeditor,concepteditor;
	private int exitAction;

	/**
	 * Constructor for OpenCaseDialog.
	 */
	public OpenCaseDialog(MiningMartApplication parent) {
		super(parent, Resource.getString("OPEN_CASE"), true);

		this.app = parent;
		this.exitAction = CANCEL;
		initComponents();
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);

		this.pack();

		setSize(300, 300);

		Dimension size = this.getSize();

		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}

	/**
	 * Implementierung von Selection-Sort.
	 * 
	 */
	private String[] selectionSort(String[] data) {
		for (int i = 0; i < data.length - 1; i++) {
			// Erstes minimales Element in (a[i],...,a[n-1]) finden ...
			int min = i; // Position des aktuellen minimalen Elements
			for (int j = i + 1; j < data.length; j++)
				if (data[j].compareTo(data[min]) < 0)
					min = j;
			// ... und mit a[i] vertauschen:
			swap(data, min, i);
		}
		return data;
	}

	/** Hilfsmethode zur Vertauschung der Array-Elemente a[i] und a[j]. 
	  * Wir setzen 0 &le;= i, j &le; a.length voraus. 
	  */
	private void swap(Comparable[] a, int i, int j) {
		Comparable tmp = a[i];
		a[i] = a[j];
		a[j] = tmp;
	}

	/**
	 * Inits the components.
	 */
	public void initComponents() {

		JPanel top_panel = new JPanel();
		top_panel.setLayout(new BorderLayout(5, 5));

		list = new JList();
		list.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		list.setLayoutOrientation(JList.VERTICAL);
		list.setVisibleRowCount(-1);

		Collection coll = null;
		try {
			coll = MiningMartApplication.m4Interface.getAllCaseNames();
		}
		catch (M4Exception error) {
			M4Interface.print.doPrint(Print.ERROR,error.getMessage(),error);
		}
		if (coll == null) return;
		Iterator iter = coll.iterator();
		String[] data = new String[coll.size()];
		int index = 0;
		while (iter.hasNext()) {
			//one_case=(String) iter.next();
			data[index] = (String) iter.next();
			index++;
		}

		data = selectionSort(data);

		list.setListData(data);
		list.addListSelectionListener(this);

		JScrollPane scrollpane = new JScrollPane(list);

		JPanel mid_panel = new JPanel();
		mid_panel.setLayout(new BorderLayout());

		mid_panel.add(scrollpane, BorderLayout.CENTER);

		JPanel viewPanel = new JPanel();
		viewPanel.setLayout(new BoxLayout(viewPanel, BoxLayout.X_AXIS));

		view = new JRadioButton(Resource.getString("OPEN_CASE_FOR_VIEW"));
		view.setSelected(!app.openedForUpdate);
		view.setEnabled(false);
		update = new JRadioButton(Resource.getString("OPEN_CASE_FOR_UPDATE"));
		update.setSelected(app.openedForUpdate);
		update.setEnabled(false);

		ButtonGroup bGroup = new ButtonGroup();
		bGroup.add(update);
		bGroup.add(view);

		viewPanel.add(update);
		viewPanel.add(view);
		
		JPanel editorpanel=new JPanel();
		editorpanel.setLayout(new BoxLayout(editorpanel, BoxLayout.X_AXIS));
		
		caseeditor=new JRadioButton(Resource.getString("COMP_CASEEDITOR"));
		caseeditor.setSelected(app.openedCaseEditor);
		caseeditor.setEnabled(false);
		concepteditor=new JRadioButton(Resource.getString("COMP_CONCEPTEDITOR"));
		concepteditor.setSelected(!app.openedCaseEditor);
		concepteditor.setEnabled(false);
		
		ButtonGroup BGroup=new ButtonGroup();
		BGroup.add(caseeditor);
		BGroup.add(concepteditor);

		editorpanel.add(caseeditor);
		editorpanel.add(concepteditor);

		JPanel grouppanel=new JPanel();
		grouppanel.setLayout(new GridLayout(2,2));
		grouppanel.add(caseeditor);
		grouppanel.add(concepteditor);
		grouppanel.add(update);
		grouppanel.add(view);
		
		JPanel grouppanel2=new JPanel();
		grouppanel2.setLayout(new BorderLayout());
		grouppanel2.add(grouppanel,BorderLayout.WEST);
		
		mid_panel.add(grouppanel2, BorderLayout.SOUTH);
		top_panel.add(mid_panel, BorderLayout.CENTER);

		cancel = new JButton(Resource.getString("CANCEL"));
		select = new JButton(Resource.getString("SELECT"));
		select.setEnabled(false);

		cancel.addActionListener(this);
		select.addActionListener(this);

		JPanel bottom = new JPanel();
		bottom.setLayout(new BoxLayout(bottom, BoxLayout.X_AXIS));
		bottom.add(Box.createHorizontalGlue());
		bottom.add(select);
		bottom.add(Box.createRigidArea(new Dimension(10, 0)));
		bottom.add(cancel);

		top_panel.add(bottom, BorderLayout.SOUTH);

		JLabel label = new JLabel(Resource.getString("CASES") + ":");

		top_panel.add(label, BorderLayout.NORTH);

		top_panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		this.getContentPane().add(top_panel);
	}

	/**
	 * @see java.awt.event.ActionListener#actionPerformed(ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		String cmd = e.getActionCommand();

		if (cmd.equals(Resource.getString("CANCEL"))) {
			exitAction = CANCEL;
			this.dispose();
		}
		if (cmd.equals(Resource.getString("SELECT"))) {
			app.openedCaseEditor=caseeditor.isSelected();
			app.openedForUpdate=update.isSelected();
			exitAction = SELECT;
			this.dispose();
		}
	}

	/**
	 * @see javax.swing.event.ListSelectionListener#valueChanged(ListSelectionEvent)
	 */
	public void valueChanged(ListSelectionEvent e) {
		if (!list.isSelectionEmpty()) {
			select.setEnabled(true);
			view.setEnabled(true);
			update.setEnabled(true);
			caseeditor.setEnabled(true);
			concepteditor.setEnabled(true);
		}
	}

	/**
	 * Gets the exit-action.
	 */
	public int getExitAction() {
		return exitAction;
	}

	/**
	 * Could you update the case?
	 */
	public boolean openCaseForUpdating() {
		return update.isSelected();
	}
	
	/**
	 * Open the case with the CaseEditor?
	 * If return false then the case will be opened with the ConceptEditor.
	 */
	public boolean openCaseEditor(){
		return caseeditor.isSelected();
	}

	/**
	 * Gets the selected case.
	 */
	public String getSelectedCase() {
		if (exitAction == SELECT)
			return (String) list.getSelectedValue();
		return null;
	}

}
/*
$Log: OpenCaseDialog.java,v $
Revision 1.4  2006/09/27 15:00:02  euler
New version 1.1

Revision 1.3  2006/04/11 14:10:16  euler
Updated license text.

Revision 1.2  2006/04/06 16:31:15  euler
Prepended license remark.

Revision 1.1  2006/01/03 09:54:27  hakenjos
Initial version!

*/
