/*
 * MiningMart Version 1.1
 * 
 * Copyright (C) 2006 Martin Scholz, Timm Euler, 
 *                    Daniel Hakenjos, Katharina Morik
 *
 * Contact: miningmart@ls8.cs.uni-dortmund.de
 *
 * A list of contributing developers (other than the copyright 
 * holders) can be found at
 * http://mmart.cs.uni-dortmund.de/downloads/download.html
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program, see the file MM_HOME/LICENSE; if not, write
 * to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301, USA.
 */
package edu.udo.cs.miningmart.gui.application;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Collection;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;

import org.musoft.limo.application.Application;
import org.musoft.limo.application.Resource;

import edu.udo.cs.miningmart.exception.M4Exception;
import edu.udo.cs.miningmart.m4.Columnset;
import edu.udo.cs.miningmart.m4.Concept;
import edu.udo.cs.miningmart.m4.Step;
import edu.udo.cs.miningmart.schemamatching.DataModelConnection;
import edu.udo.cs.miningmart.schemamatching.MatchingResult;
import edu.udo.cs.miningmart.schemamatching.SchemaMatchException;

/**
 * Shows the result of matching data models, and allows to edit them.
 * 
 * @author Timm Euler
 * @version $Id: $
 */
public class MatchingConceptsDialog extends JDialog implements ActionListener {

	private MiningMartApplication app;
	private JButton closebutton, matchButton;
	private Collection<MatchingResult<Concept>> myMappings;
	private DataModelConnection myDmc;
	private Collection<Concept> conceptualModel, targetModel, myJoinResults;
	private String[] allTargetNames;
	private String[][] tableData;
	private int noOfRowsInTable;
	
	/**
	 * Displays a non-editable dialog with the matching results
	 *  
	 * @param app the parent component for the dialog 
	 * @param theConcept the concept whose details are to be shown
	 */
	public MatchingConceptsDialog(
				MiningMartApplication app,
				Collection<Concept> conceptualModel,
				Collection<Concept> targetModel,
				DataModelConnection dmc,
				Collection<Concept> joinResults) {
		
		super(app, Resource.getString("SHOW_MATCHINGS_TITLE"), true);
		
		this.app = app;
		this.conceptualModel = conceptualModel;
		this.targetModel = targetModel;
		this.myDmc = dmc;
		this.myMappings = this.myDmc.getTheConceptMappings();
		this.myJoinResults = joinResults;
		if (this.myMappings == null || this.targetModel == null || this.conceptualModel == null)
			return;
		
		this.allTargetNames = new String[this.targetModel.size() + 1];
		this.allTargetNames[0] = Resource.getString("MAPPING_NONE");
		int i = 1;
		for (Concept target : this.targetModel) {
			this.allTargetNames[i] = target.getName();
			i++;
		}
		
		int heightOfTable = initComponents();
		if (heightOfTable == -1) {
			return;
		}
		
		this.setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
		this.pack();
		setSize(600, 150 + heightOfTable * 25);
		Dimension size = this.getSize();
		this.setLocation(
			(Toolkit.getDefaultToolkit().getScreenSize().width - size.width)
				/ 2,
			(Toolkit.getDefaultToolkit().getScreenSize().height - size.height)
				/ 2);
		this.setVisible(true);
	}	

	/**
	 * @see java.awt.event.ActionListener#actionPerformed(ActionEvent)
	 */
	public void actionPerformed(ActionEvent e) {
		if ( ! (e.getSource() instanceof JButton))
			return;
		JButton source=(JButton) e.getSource();
		
		if (source.getText().equals(this.matchButton.getText())){
			// go through the data table and adjust the mapping
			this.myMappings = new Vector<MatchingResult<Concept>>();
			for (int i = 0; i < this.tableData.length; i++) {
				String conceptualConcept = this.tableData[i][0];
				String targetConcept = this.tableData[i][1];
				if (targetConcept != null && ( ! targetConcept.equalsIgnoreCase(Resource.getString("MAPPING_NONE")))) {
					if (targetConcept.indexOf("(") > -1) 
						targetConcept = targetConcept.substring(0, targetConcept.indexOf("(")).trim();
					MatchingResult<Concept> mapping = this.collectConcepts(conceptualConcept, targetConcept);
					if (mapping != null)
						this.myMappings.add(mapping);
				}
			}
		}
		if (source.getText().equals(this.closebutton.getText())) 
			this.myMappings = null;
		
		this.dispose();
	}
	
	public Collection<MatchingResult<Concept>> getUserSelectedMapping() {
		return this.myMappings;
	}

	private MatchingResult<Concept> collectConcepts(String concName, String targetName) {
		MatchingResult<Concept> mr = new MatchingResult<Concept>();
		try {
			for (Concept cc : this.conceptualModel) {
				if (cc.getName().equalsIgnoreCase(concName))
					mr.setObjectOfFirstSchema(cc);
			}
			for (Concept tc : this.targetModel) {
				if (tc.getName().equalsIgnoreCase(targetName))
					mr.setObjectOfSecondSchema(tc);
			}
		}
		catch (SchemaMatchException sm) {}
		if (mr.getObjectOfFirstSchema() != null && mr.getObjectOfSecondSchema() != null)
			return mr;
		
		return null;
	}
	
	// returns the height of the table created
	private int initComponents(){
		JPanel mainpanel=new JPanel(new BorderLayout(5,5));
		this.getContentPane().add(mainpanel);
		
		JPanel northpanel=new JPanel(new BorderLayout(5,5));
		northpanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		mainpanel.add(northpanel,BorderLayout.NORTH);
		
		JLabel namelabel=new JLabel(Resource.getString("SHOW_MATCHINGS_LABEL"));
		JLabel simLabel = new JLabel("\nOverall similarity: " + this.myDmc.getSimilarity());
		
		String text = "";
		if (this.myDmc.isInitialDataModel())
			text += "\nThe initial data model was matched.";
		Step s = this.myDmc.getDataModelProducingStep();
		if (s != null)
			text += "\nThe Step that produces this data model is '" + s.getName() + "' from Chain '" + s.getTheChain().getName() + "'."; 
		JLabel infoLabel = new JLabel(text);
		namelabel.getFont().deriveFont(Font.BOLD,15);
		infoLabel.getFont().deriveFont(Font.BOLD,15);
		simLabel.getFont().deriveFont(Font.BOLD,15);
		namelabel.setHorizontalAlignment(SwingConstants.CENTER);
		infoLabel.setHorizontalAlignment(SwingConstants.CENTER);
		simLabel.setHorizontalAlignment(SwingConstants.CENTER);
		northpanel.add(namelabel,BorderLayout.NORTH);
		northpanel.add(simLabel, BorderLayout.CENTER);
		northpanel.add(infoLabel, BorderLayout.SOUTH);
				
		JPanel buttonpanel=new JPanel(new GridLayout(1,2));
		mainpanel.add(buttonpanel,BorderLayout.SOUTH);
		
		closebutton = new JButton(Resource.getString("CANCEL"));
		closebutton.addActionListener(this);
		this.matchButton = new JButton(Resource.getString("CONNECT"));
		this.matchButton.addActionListener(this);
		
		buttonpanel.add(this.matchButton);
		buttonpanel.add(closebutton);
		
		// add table to mainpanel CENTER
		JScrollPane theInfoTable = this.createInfoTable();
		if (theInfoTable == null) {
			JOptionPane.showMessageDialog(this.app, "No mapping information found!", "Error displaying matching information", JOptionPane.ERROR_MESSAGE);
			this.dispose();
			return -1;
		}
		mainpanel.add(theInfoTable, BorderLayout.CENTER);
		return this.noOfRowsInTable;
	}	
	
	private JScrollPane createInfoTable() {
		
		this.noOfRowsInTable = Math.max(this.myMappings.size(), Math.max(this.conceptualModel.size(), this.targetModel.size()));
		tableData = new String[this.noOfRowsInTable][3];
		String[] columnNames = new String[] {
				"Concepts of current Case", "DB objects", "Similarity"
			};
		
		try {
			int rowIndex = 0;
			for (MatchingResult<Concept> oneMapping : this.myMappings) {
				Concept left = oneMapping.getObjectOfFirstSchema();
				Concept right = oneMapping.getObjectOfSecondSchema();
				String ext = " (" + (right.getCurrentColumnSet().getType().equals(Columnset.TYPE_TABLE) ? "Table" : "View" );
				if (this.myJoinResults.contains(right)) {
					ext += ", Join result";
				}
				ext += ")";
				tableData[rowIndex][0] = left.getName();
				tableData[rowIndex][1] = right.getName() + ext;
				tableData[rowIndex][2] = "" + oneMapping.getSimilarity();
				rowIndex++;
			}
			for (Concept leftover : this.conceptualModel) {
				if (this.notMatched(leftover)) {
					tableData[rowIndex][0] = leftover.getName();
					rowIndex++;
				}
			}
		}
		catch (M4Exception m4e) {
			return null;
		}
		
		MappingTable table=new MappingTable(columnNames);
		table.createEditor();
		JScrollPane scrollpane=new JScrollPane(table);
		return scrollpane;
	}
	
	private boolean notMatched(Concept c) {
		for (MatchingResult<Concept> oneMapping : this.myMappings) {
			Concept left = oneMapping.getObjectOfFirstSchema();
			if (left.equals(c)) {
				return false;
			}
		}
		return true;
	}
	

	private class MappingTable extends JTable{
		
		public MappingTable(String[] columnNames){
			setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
			getTableHeader().setReorderingAllowed(false);
			setRowHeight(getRowHeight() + 4);
			setModel(new MappingTableModel(columnNames));
			this.getColumnModel().getColumn(2).setMaxWidth(60);
			// this.getColumnModel().getColumn(3).setMaxWidth(20);
		}
		
		public void createEditor(){
			TableColumn tablecolumn = getColumnModel().getColumn(1);
			JComboBox comboBox = new JComboBox(allTargetNames);
			tablecolumn.setCellEditor(new DefaultCellEditor(comboBox));			
		}
		
	}

	private class MappingTableModel extends AbstractTableModel {
		
		private String[] colNames;
		
		public MappingTableModel(String[] columnNames) {
			super();
			this.colNames = columnNames;
		}
		
		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnCount()
		 */
		public int getColumnCount() {
			return 3;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getRowCount()
		 */
		public int getRowCount() {
			return conceptualModel.size();
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getValueAt(int, int)
		 */
		public Object getValueAt(int rowIndex, int columnIndex) {
			if (columnIndex >= 0 && columnIndex <= 2)
				return tableData[rowIndex][columnIndex];
			return null;
		}	

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnClass(int)
		 */
		public Class getColumnClass(int columnIndex) {
			return String.class;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#getColumnName(int)
		 */
		public String getColumnName(int column) {
			return this.colNames[column];
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#isCellEditable(int, int)
		 */
		public boolean isCellEditable(int rowIndex, int columnIndex) {
			if (app.getViewMode()!=Application.EDITOR){
				return false;
			}
			if (columnIndex == 1)
				return true;
			return false;
		}

		/* (non-Javadoc)
		 * @see javax.swing.table.TableModel#setValueAt(java.lang.Object, int, int)
		 */
		public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
						
			if (columnIndex==1){
				String value=(String) aValue;
				if (value.equals(Resource.getString("MAPPING_NONE"))){
					tableData[rowIndex][1] = Resource.getString("MAPPING_NONE");
					tableData[rowIndex][2] = null;
				}
				else {
					tableData[rowIndex][1] = value;
					tableData[rowIndex][2] = null;					
				}
				fireTableDataChanged();
			}			
		}

	}
}